/* btree.c - binary trees insert only (AVL) */

#include "glo.h"
#include "btree.h"
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

PUBLIC Tree 
newtree(void)
     /* constructor */
{
  return NULL;
}

PRIVATE Tree 
makecell(void * what)
     /* create a new cell if possible; return new root or NULL */
{
  Tree t;

  t = (Tree) malloc(sizeof(struct treecell));
  if (t == NULL) 
    return NULL;         /* this is an error */
  t->entry = what;
  t->left = t->right = NULL;
  t->depth = 1;
  return t;
}

PUBLIC Tree 
killtree(Tree e, Kill killinfo)
     /* release the cell after its sons; return NULL */
{
  if (e == NULL) return NULL;
  /* e->entry = */ killinfo(e->entry);
  /* e->left =  */ killtree(e->left,killinfo);
  /* e->right = */ killtree(e->right, killinfo);
  free(e);
  return NULL;
}

PUBLIC void * 
find(Tree r, void * s, Compare infocmp)
     /* look for s */
{
    int c;
    
    if (r == NULL) return NULL;
    c = infocmp(s, r->entry);
    if (c == 0) return r->entry;
    if (c < 0) return find(r->left, s, infocmp); 
    else return find(r->right, s, infocmp);
}

PUBLIC int
depth(Tree r)
     /* homogeneous depth */
{
  if (r == NULL) return 0;
  else return r->depth;
}

PRIVATE Tree 
equilibrate(Tree r)
     /* if tree is asymetrical balance it */
{
  long dl, dr;
  Tree aux, rez;

  dl = depth(r->left);
  dr = depth(r->right);
  if (dl - dr >= 2l) {
    /* rotate towards right */
    rez = r->left;
    aux = rez->right;
    rez->right = r;
    r->left = aux;
    r->depth -= 2;
    return rez;
  }
  else if (dr - dl >= 2l) {
    /* roatate towards left */
    rez = r->right;
    aux = rez->left;
    rez->left = r;
    r->right = aux;
    r->depth -= 2;
    return rez;
  }
  else return r;
}

PUBLIC Tree 
addcell(Tree r, void * what, Compare infocmp, Itis isthere)
     /* add a new cell in the tree; return new root or NULL for error;
	Call isthere when the cell already exists; it'll know what to do */
{
  int c; 
  Tree cell;

  if ( r == NULL ) return makecell(what);
  
  c = infocmp(what, r->entry);
  if (c == 0) {
    /* the cell is there ! */
    r->entry = isthere(r->entry, what);  /* already there! */
    return r;
  }
  if (c < 0) {
    cell = addcell(r->left, what, infocmp, isthere);
    if (cell != NULL) {
      r->left = cell;
      if (r->depth < cell->depth + 1) r->depth = cell->depth + 1;
      return equilibrate(r);
    }
    else return NULL;  /* error */
  }
  else {
    cell = addcell(r->right, what, infocmp, isthere);
    if (cell != NULL) {
      r->right = cell;
      if (r->depth < cell->depth + 1) r->depth = cell->depth + 1;
      return equilibrate(r);
    }
    else return NULL;
  }
}

PUBLIC void 
writetree(Tree t, FILE * f, Write w)
     /* write the data to the file - inorder */
{
  if (t == NULL) return;
  writetree(t->left, f, w);
  w(f, t->entry);
  writetree(t->right, f, w);
}

PRIVATE void * 
rte(Tree t, int ddepth)
     /* return at random a node at this ddepth */
{
  bool lr;  /* go to left or right ? */

  if (ddepth == 0) return t->entry;
  else {
    if (depth(t->left) < ddepth) return rte(t->right, ddepth -1);
    else if (depth(t->right) < ddepth) return rte(t->left, ddepth - 1);
    else {
      lr = rand() % 2;
      if (lr == 0) return rte(t->left, ddepth - 1);
      else return rte(t->right, ddepth - 1);
    }
  }
}

PUBLIC void * 
randomnodeentry(Tree t)
     /* return a random node entry */
{
  static bool been = NO;
  int dep;

  if (! been) {
    been = YES;
    srand(time(0));
  }
  dep = rand() % depth(t);
  return rte(t, dep);
}
