/*
 *  linux/fs/ext4/super.c
 * April 1997
 * by Mihai Budiu (budiu@cs.cornell.edu)
 * 
 *  from 
 *
 * Copyright (C) 1992, 1993, 1994, 1995
 * Remy Card (card@masi.ibp.fr)
 * Laboratoire MASI - Institut Blaise Pascal
 * Universite Pierre et Marie Curie (Paris VI)
 *
 *  from
 *
 *  linux/fs/minix/inode.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 */

#include <linux/module.h>

#include <stdarg.h>

#include <asm/bitops.h>
#include <asm/segment.h>
#include <asm/system.h>

#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/ext4_fs.h>
#include <linux/malloc.h>
#include <linux/sched.h>
#include <linux/stat.h>
#include <linux/string.h>
#include <linux/locks.h>

static char error_buf[1024];

void ext4_error (struct super_block * sb, const char * function,
		 const char * fmt, ...)
{
	va_list args;
	int i;

	if (!(sb->s_flags & MS_RDONLY)) {
		for (i=0; i < 2; i++) {
			sb->u.ext4_sb[i].s_mount_state |= EXT4_ERROR_FS;
			sb->u.ext4_sb[i].s_es->s_state |= EXT4_ERROR_FS;
			mark_buffer_dirty(sb->u.ext4_sb[i].s_sbh, 1);
		}
		sb->s_dirt = 1;
	}
	va_start (args, fmt);
	vsprintf (error_buf, fmt, args);
	va_end (args);
	if (ext4_test_opt (sb, ERRORS_PANIC) ||
	    (sb->u.ext4_sb[0].s_es->s_errors == EXT4_ERRORS_PANIC &&
	     !ext4_test_opt (sb, ERRORS_CONT) && !ext4_test_opt (sb, ERRORS_RO)))
		panic ("EXT4-fs panic (device %s): %s: %s\n",
		       kdevname(sb->s_dev), function, error_buf);
	printk (KERN_CRIT "EXT4-fs error (device %s): %s: %s\n",
		kdevname(sb->s_dev), function, error_buf);
	if (ext4_test_opt (sb, ERRORS_RO) ||
	    (sb->u.ext4_sb[0].s_es->s_errors == EXT4_ERRORS_RO &&
	     !ext4_test_opt (sb, ERRORS_CONT) && !ext4_test_opt (sb, ERRORS_PANIC))) {
		printk ("Remounting filesystem read-only\n");
		sb->s_flags |= MS_RDONLY;
	}
}

NORET_TYPE void ext4_panic (struct super_block * sb, const char * function,
			    const char * fmt, ...)
{
	va_list args;
	int i;

	if (!(sb->s_flags & MS_RDONLY)) {
		for (i=0; i < 2; i++) {
			sb->u.ext4_sb[i].s_mount_state |= EXT4_ERROR_FS;
			sb->u.ext4_sb[i].s_es->s_state |= EXT4_ERROR_FS;
			mark_buffer_dirty(sb->u.ext4_sb[i].s_sbh, 1);
		}
		sb->s_dirt = 1;
	}
	va_start (args, fmt);
	vsprintf (error_buf, fmt, args);
	va_end (args);
	/* this is to prevent panic from syncing this filesystem */
	if (sb->s_lock)
		sb->s_lock=0;
	sb->s_flags |= MS_RDONLY;
	panic ("EXT4-fs panic (device %s): %s: %s\n",
	       kdevname(sb->s_dev), function, error_buf);
}

void ext4_warning (struct super_block * sb, const char * function,
		   const char * fmt, ...)
{
	va_list args;

	va_start (args, fmt);
	vsprintf (error_buf, fmt, args);
	va_end (args);
	printk (KERN_WARNING "EXT4-fs warning (device %s): %s: %s\n",
		kdevname(sb->s_dev), function, error_buf);
}

void ext4_put_super (struct super_block * sb)
{
	int db_count;
	int i, j;

	lock_super (sb);
	sb->s_dev = 0;
	for (j=0; j < 2; j++) {
		if (!(sb->s_flags & MS_RDONLY)) {
			sb->u.ext4_sb[j].s_es->s_state = 
				sb->u.ext4_sb[j].s_mount_state;
			mark_buffer_dirty(sb->u.ext4_sb[j].s_sbh, 1);
		}
		db_count = sb->u.ext4_sb[j].s_db_per_group;
		for (i = 0; i < db_count; i++)
			if (sb->u.ext4_sb[j].s_group_desc[i])
				brelse (sb->u.ext4_sb[j].s_group_desc[i]);
		kfree_s (sb->u.ext4_sb[j].s_group_desc,
			 db_count * sizeof (struct buffer_head *));
		for (i = 0; i < EXT4_MAX_GROUP_LOADED; i++)
			if (sb->u.ext4_sb[j].s_inode_bitmap[i])
				brelse (sb->u.ext4_sb[j].s_inode_bitmap[i]);
		for (i = 0; i < EXT4_MAX_GROUP_LOADED; i++)
			if (sb->u.ext4_sb[j].s_block_bitmap[i])
				brelse (sb->u.ext4_sb[j].s_block_bitmap[i]);
		brelse (sb->u.ext4_sb[j].s_sbh);
	}
	unlock_super (sb);
	MOD_DEC_USE_COUNT;
	return;
}

static struct super_operations ext4_sops = { 
	ext4_read_inode,
	NULL,
	ext4_write_inode,
	ext4_put_inode,
	ext4_put_super,
	ext4_write_super,
	ext4_statfs,
	ext4_remount
};

/*
 * This function has been shamelessly adapted from the msdos fs
 */
static int parse_options (char * options, unsigned long * sb_block,
			  unsigned short *resuid, unsigned short * resgid,
			  unsigned long * mount_options)
{
	char * this_char;
	char * value;

	if (!options)
		return 1;
	for (this_char = strtok (options, ",");
	     this_char != NULL;
	     this_char = strtok (NULL, ",")) {
		if ((value = strchr (this_char, '=')) != NULL)
			*value++ = 0;
		if (!strcmp (this_char, "bsddf"))
			ext4_clear_opt (*mount_options, MINIX_DF);
		else if (!strcmp (this_char, "check")) {
			if (!value || !*value)
				ext4_set_opt (*mount_options, CHECK_NORMAL);
			else if (!strcmp (value, "none")) {
				ext4_clear_opt (*mount_options, CHECK_NORMAL);
				ext4_clear_opt (*mount_options, CHECK_STRICT);
			}
			else if (!strcmp (value, "normal"))
				ext4_set_opt (*mount_options, CHECK_NORMAL);
			else if (!strcmp (value, "strict")) {
				ext4_set_opt (*mount_options, CHECK_NORMAL);
				ext4_set_opt (*mount_options, CHECK_STRICT);
			}
			else {
				printk ("EXT4-fs: Invalid check option: %s\n",
					value);
				return 0;
			}
		}
		else if (!strcmp (this_char, "debug"))
			ext4_set_opt (*mount_options, DEBUG);
		else if (!strcmp (this_char, "errors")) {
			if (!value || !*value) {
				printk ("EXT4-fs: the errors option requires "
					"an argument");
				return 0;
			}
			if (!strcmp (value, "continue")) {
				ext4_clear_opt (*mount_options, ERRORS_RO);
				ext4_clear_opt (*mount_options, ERRORS_PANIC);
				ext4_set_opt (*mount_options, ERRORS_CONT);
			}
			else if (!strcmp (value, "remount-ro")) {
				ext4_clear_opt (*mount_options, ERRORS_CONT);
				ext4_clear_opt (*mount_options, ERRORS_PANIC);
				ext4_set_opt (*mount_options, ERRORS_RO);
			}
			else if (!strcmp (value, "panic")) {
				ext4_clear_opt (*mount_options, ERRORS_CONT);
				ext4_clear_opt (*mount_options, ERRORS_RO);
				ext4_set_opt (*mount_options, ERRORS_PANIC);
			}
			else {
				printk ("EXT4-fs: Invalid errors option: %s\n",
					value);
				return 0;
			}
		}
		else if (!strcmp (this_char, "grpid") ||
			 !strcmp (this_char, "bsdgroups"))
			ext4_set_opt (*mount_options, GRPID);
		else if (!strcmp (this_char, "minixdf"))
			ext4_set_opt (*mount_options, MINIX_DF);
		else if (!strcmp (this_char, "nocheck")) {
			ext4_clear_opt (*mount_options, CHECK_NORMAL);
			ext4_clear_opt (*mount_options, CHECK_STRICT);
		}
		else if (!strcmp (this_char, "nogrpid") ||
			 !strcmp (this_char, "sysvgroups"))
			ext4_clear_opt (*mount_options, GRPID);
		else if (!strcmp (this_char, "resgid")) {
			if (!value || !*value) {
				printk ("EXT4-fs: the resgid option requires "
					"an argument");
				return 0;
			}
			*resgid = simple_strtoul (value, &value, 0);
			if (*value) {
				printk ("EXT4-fs: Invalid resgid option: %s\n",
					value);
				return 0;
			}
		}
		else if (!strcmp (this_char, "resuid")) {
			if (!value || !*value) {
				printk ("EXT4-fs: the resuid option requires "
					"an argument");
				return 0;
			}
			*resuid = simple_strtoul (value, &value, 0);
			if (*value) {
				printk ("EXT4-fs: Invalid resuid option: %s\n",
					value);
				return 0;
			}
		}
		else if (!strcmp (this_char, "sb")) {
			if (!value || !*value) {
				printk ("EXT4-fs: the sb option requires "
					"an argument");
				return 0;
			}
			*sb_block = simple_strtoul (value, &value, 0);
			if (*value) {
				printk ("EXT4-fs: Invalid sb option: %s\n",
					value);
				return 0;
			}
		}
		/* Silently ignore the quota options */
		else if (!strcmp (this_char, "grpquota")
		         || !strcmp (this_char, "noquota")
		         || !strcmp (this_char, "quota")
		         || !strcmp (this_char, "usrquota"))
			/* Don't do anything ;-) */ ;
		else {
			printk ("EXT4-fs: Unrecognized mount option %s\n", this_char);
			return 0;
		}
	}
	return 1;
}

static void ext4_setup_super (struct super_block * sb,
			      struct ext4_super_block * es)
{
	if (es->s_rev_level > EXT4_MAX_SUPP_REV) {
		printk ("EXT4-fs warning: revision level too high, "
			"forcing read/only mode\n");
		sb->s_flags |= MS_RDONLY;
	}
	if (!(sb->s_flags & MS_RDONLY)) {
		if (!(sb->u.ext4_sb[0].s_mount_state & EXT4_VALID_FS))
			printk ("EXT4-fs warning: mounting unchecked fs, "
				"running fsck is recommended\n");
		else if ((sb->u.ext4_sb[0].s_mount_state & EXT4_ERROR_FS))
			printk ("EXT4-fs warning: mounting fs with errors, "
				"running fsck is recommended\n");
		else if (es->s_max_mnt_count >= 0 &&
		         es->s_mnt_count >= 
			 (unsigned short) es->s_max_mnt_count)
			printk ("EXT4-fs warning: maximal mount count "
				"reached, running fsck is recommended\n");
		else if (es->s_checkinterval &&
			 (es->s_lastcheck + es->s_checkinterval 
			  <= CURRENT_TIME))
			printk ("EXT4-fs warning: checktime reached, "
				"running fsck is recommended\n");
		es->s_state &= ~EXT4_VALID_FS;
		if (!es->s_max_mnt_count)
			es->s_max_mnt_count = EXT4_DFL_MAX_MNT_COUNT;
		es->s_mnt_count++;
		es->s_mtime = CURRENT_TIME;
		mark_buffer_dirty(sb->u.ext4_sb[0].s_sbh, 1);
		sb->s_dirt = 1;
		if (ext4_test_opt (sb, DEBUG))
			printk ("[EXT IV FS %s, %s, bs=%lu, fs=%lu, gc=%lu, "
				"bpg=%lu, ipg=%lu, mo=%04lx]\n",
				EXT4FS_VERSION, EXT4FS_DATE, sb->s_blocksize,
				sb->u.ext4_sb[0].s_frag_size,
				sb->u.ext4_sb[0].s_groups_count,
				EXT4_BLOCKS_PER_GROUP(sb, 0),
				EXT4_INODES_PER_GROUP(sb, 0),
				sb->u.ext4_sb[0].s_mount_opt);
		if (ext4_test_opt (sb, CHECK)) {
			ext4_check_blocks_bitmap (sb);
			ext4_check_inodes_bitmap (sb);
		}
	}
}

static int ext4_check_descriptors (struct super_block * sb, int j)
{
	int i;
	int desc_block = 0;
	unsigned long block = sb->u.ext4_sb[j].s_es->s_first_data_block;
	struct ext4_group_desc * gdp = NULL;

	ext4_debug ("Checking group descriptors, partition %d\n", j);

	for (i = 0; i < sb->u.ext4_sb[j].s_groups_count; i++)
	{
		if ((i % EXT4_DESC_PER_BLOCK(sb, j)) == 0)
		        gdp = (struct ext4_group_desc *) 
		        sb->u.ext4_sb[j].s_group_desc[desc_block++]->b_data;
		if (gdp->bg_block_bitmap < block ||
		    gdp->bg_block_bitmap >= 
		    block + EXT4_BLOCKS_PER_GROUP(sb, j))
		{
			ext4_error (sb, "ext4_check_descriptors",
				    "Block bitmap for group %d"
				    " not in group (block %lu)!",
				    i, (unsigned long) gdp->bg_block_bitmap);
			return 0;
		}
		if (gdp->bg_inode_bitmap < block ||
		    gdp->bg_inode_bitmap >= 
		    block + EXT4_BLOCKS_PER_GROUP(sb, j))
		{
			ext4_error (sb, "ext4_check_descriptors",
				    "Inode bitmap for group %d"
				    " not in group (block %lu)!",
				    i, (unsigned long) gdp->bg_inode_bitmap);
			return 0;
		}
		if (gdp->bg_inode_table < block ||
		    gdp->bg_inode_table + sb->u.ext4_sb[j].s_itb_per_group >=
		    block + EXT4_BLOCKS_PER_GROUP(sb, j))
		{
			ext4_error (sb, "ext4_check_descriptors",
				    "Inode table for group %d"
				    " not in group (block %lu)!",
				    i, (unsigned long) gdp->bg_inode_table);
			return 0;
		}
		block += EXT4_BLOCKS_PER_GROUP(sb, j);
		gdp++;
	}
	return 1;
}

#define log2(n) ffz(~(n))

static void release_descriptors(struct super_block * sb, 
				int db_count, int partition, int loaded)
{
	int k;

	for (k = 0; k < loaded; k++)
		brelse (sb->u.ext4_sb[partition].s_group_desc[k]);
	kfree_s (sb->u.ext4_sb[partition].s_group_desc,
		 db_count * sizeof (struct buffer_head *));
	ext4_debug("Done releasing descriptors\n");
}

struct super_block * ext4_read_super (struct super_block * sb, void * data,
				      int silent)
{
	struct buffer_head * bh[2];
	struct ext4_super_block * es[2];
	unsigned long sb_block = 1;
	unsigned short resuid = EXT4_DEF_RESUID;
	unsigned short resgid = EXT4_DEF_RESGID;
	unsigned long logic_sb_block = 1;
	unsigned long offset = 0l;
	kdev_t dev[2];
	int db_count[2];
	int i, j;

	bh[0] = bh[1] = NULL;
	dev[0] = sb->s_dev;
	sb->u.ext4_sb[0].s_mount_opt = 0;
	ext4_set_opt (sb->u.ext4_sb[0].s_mount_opt, CHECK_NORMAL);
	if (!parse_options ((char *) data, &sb_block, &resuid, &resgid,
			    &sb->u.ext4_sb[0].s_mount_opt)) {
		sb->s_dev = 0;
		return NULL;
	}

	MOD_INC_USE_COUNT;
	lock_super (sb);
	set_blocksize (dev[0], BLOCK_SIZE);
	if (!(bh[0] = bread (dev[0], sb_block, BLOCK_SIZE))) {
		sb->s_dev = 0;
		unlock_super (sb);
		printk ("EXT4-fs: unable to read superblock\n");
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	/*
	 * Note: s_es[] must be initialized as soon as possible because
	 * some ext4 macro-instructions depend on its value
	 */
	es[0] = (struct ext4_super_block *) bh[0]->b_data;
	sb->u.ext4_sb[0].s_es = es[0];
	sb->s_magic = es[0]->s_magic;
	if (sb->s_magic != EXT4_SUPER_MAGIC) {
		if (!silent)
			printk ("VFS: Can't find an ext4 filesystem on dev "
				"%s.\n", kdevname(dev[0]));
	failed_mount:
		sb->s_dev = 0;
		unlock_super (sb);
		if (bh[0]) 
			brelse(bh[0]);
		if (bh[1])
			brelse(bh[1]);
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	
	sb->s_blocksize_bits = sb->u.ext4_sb[0].s_es->s_log_block_size + 10;
	sb->s_blocksize = 1 << sb->s_blocksize_bits;
	if (sb->s_blocksize != BLOCK_SIZE && 
	    (sb->s_blocksize == 1024 || sb->s_blocksize == 2048 ||  
	     sb->s_blocksize == 4096)) {
		brelse (bh[0]);
		set_blocksize (dev[0], sb->s_blocksize);
		logic_sb_block = (sb_block*BLOCK_SIZE) / sb->s_blocksize;
		offset = (sb_block*BLOCK_SIZE) % sb->s_blocksize;
		bh[0] = bread (dev[0], logic_sb_block, sb->s_blocksize);
		if(!bh[0]) {
			printk("EXT4-fs: Couldn't read superblock on "
			       "2nd try.\n");
			goto failed_mount;
		}
		es[0] = (struct ext4_super_block *) 
			(((char *)bh[0]->b_data) + offset);
		sb->u.ext4_sb[0].s_es = es[0];
		if (es[0]->s_magic != EXT4_SUPER_MAGIC) {
			printk ("EXT4-fs: Magic mismatch, very weird !\n");
			goto failed_mount;
		}
	}

	/* Quickly load second superblock to check in parallel */
	dev[1] = es[0]->s_other_device;
	if (dev[0] == dev[1]) {
		printk("s_other_device is SAME device: %s\n",
		       kdevname(dev[0]));
		goto failed_mount;
	}
	if (!es[0]->s_directory_flag) {
		printk("Trying to load secondary device of ext4 fs "
		       "instead of %s\n", kdevname(dev[1]));
		goto failed_mount;
	}
	set_blocksize (dev[1], sb->s_blocksize);
	if (!(bh[1] = bread (dev[1], sb_block, sb->s_blocksize))) {
		printk ("EXT4-fs: unable to read second superblock\n");
		goto failed_mount;
	}
	es[1] = (struct ext4_super_block *) bh[1]->b_data + offset;
	sb->u.ext4_sb[1].s_es = es[1];
	if (es[1]->s_magic != EXT4_SUPER_MAGIC) {
		if (!silent)
			printk ("VFS: Can't find ext4 fs on second dev "
				"%s.\n", kdevname(dev[1]));
		goto failed_mount;
	}
	if ((es[1]->s_other_device != dev[0]) ||
	    (es[0]->s_other_device != dev[1])) {
		printk("Cross device check mismatch: %d != %d or %d != %d\n",
		       dev[0], es[1]->s_other_device,
		       dev[1], es[0]->s_other_device);
		goto failed_mount;
	}
	if (es[1]->s_directory_flag) {
		printk("Directory flag set on secondary partition %s\n",
		       kdevname(dev[1]));
		goto failed_mount;
	}
	if ((es[0]->s_log_block_size != es[1]->s_log_block_size) ||
	    (es[0]->s_first_ino != es[1]->s_first_ino) ||
	    (es[0]->s_inode_size != es[1]->s_inode_size)) {
		printk("Different sizes: block bits %d, %d, "
		       "first inode %ld, %ld, inode size %d, %d\n",
		       es[0]->s_log_block_size, es[1]->s_log_block_size,
		       es[0]->s_first_ino, es[1]->s_first_ino,
		       es[0]->s_inode_size, es[1]->s_inode_size);
		goto failed_mount;
	}

        /* Proceed to check in parallel the two superblocks */
	for (j = 0; j < 2; j++) {
		if (es[j]->s_rev_level > EXT4_GOOD_OLD_REV) {
			if (es[j]->s_feature_incompat & 
			    ~EXT4_FEATURE_INCOMPAT_SUPP) {
				printk("EXT4-fs: %s: couldn't mount because "
				       "unsupported optional features.\n", 
				       kdevname(dev[j]));
				goto failed_mount;
			}
			if (!(sb->s_flags & MS_RDONLY) &&
			    (es[j]->s_feature_ro_compat & 
			     ~EXT4_FEATURE_RO_COMPAT_SUPP)) {
				printk("EXT4-fs: %s: couldn't mount RDWR "
				       "bec. of unsupported opt features.\n", 
				       kdevname(dev[j]));
				goto failed_mount;
			}
		}

		if (es[j]->s_rev_level == EXT4_GOOD_OLD_REV) {
			sb->u.ext4_sb[j].s_inode_size = 
				EXT4_GOOD_OLD_INODE_SIZE;
			sb->u.ext4_sb[j].s_first_ino = 
				EXT4_GOOD_OLD_FIRST_INO;
		} else {
			sb->u.ext4_sb[j].s_inode_size = 
				es[j]->s_inode_size;
			sb->u.ext4_sb[j].s_first_ino = 
				es[j]->s_first_ino;
			if (sb->u.ext4_sb[j].s_inode_size 
			    != EXT4_GOOD_OLD_INODE_SIZE) {
				printk ("EXT4-fs: bad inode size: %d\n",
					sb->u.ext4_sb[j].s_inode_size);
				goto failed_mount;
			}
		}
		sb->u.ext4_sb[j].s_frag_size = EXT4_MIN_FRAG_SIZE <<
			es[j]->s_log_frag_size;
		if (sb->u.ext4_sb[j].s_frag_size)
			sb->u.ext4_sb[j].s_frags_per_block = 
				sb->s_blocksize /
				sb->u.ext4_sb[j].s_frag_size;
		else
			sb->s_magic = 0;               /* ? */
		sb->u.ext4_sb[j].s_blocks_per_group = 
			es[j]->s_blocks_per_group;
		sb->u.ext4_sb[j].s_frags_per_group = 
			es[j]->s_frags_per_group;
		sb->u.ext4_sb[j].s_inodes_per_group = 
			es[j]->s_inodes_per_group;
		sb->u.ext4_sb[j].s_inodes_per_block = 
			sb->s_blocksize / EXT4_INODE_SIZE(sb);
		sb->u.ext4_sb[j].s_itb_per_group = 
			sb->u.ext4_sb[j].s_inodes_per_group /
			sb->u.ext4_sb[j].s_inodes_per_block;
		sb->u.ext4_sb[j].s_desc_per_block = sb->s_blocksize /
			sizeof (struct ext4_group_desc);
		sb->u.ext4_sb[j].s_sbh = bh[j];
		if (resuid != EXT4_DEF_RESUID)
			sb->u.ext4_sb[j].s_resuid = resuid;
		else
			sb->u.ext4_sb[j].s_resuid = es[j]->s_def_resuid;
		if (resgid != EXT4_DEF_RESGID)
			sb->u.ext4_sb[j].s_resgid = resgid;
		else
			sb->u.ext4_sb[j].s_resgid = es[j]->s_def_resgid;
		sb->u.ext4_sb[j].s_mount_state = es[j]->s_state;
		sb->u.ext4_sb[j].s_rename_lock = 0;
		sb->u.ext4_sb[j].s_rename_wait = NULL;
		sb->u.ext4_sb[j].s_addr_per_block_bits =
			log2 (EXT4_ADDR_PER_BLOCK(sb));
		sb->u.ext4_sb[j].s_desc_per_block_bits =
			log2 (EXT4_DESC_PER_BLOCK(sb, j));
		if (sb->s_magic != EXT4_SUPER_MAGIC) {
			if (!silent)
				printk ("VFS: Can't find an ext4 filesystem "
					"on dev %s.\n",
					kdevname(dev[j]));
			goto failed_mount;
		}
		if (sb->s_blocksize != bh[j]->b_size) {
			if (!silent)
				printk ("VFS: Unsupported blocksize on dev "
					"%s.\n", kdevname(dev[j]));
			goto failed_mount;
		}

		if (sb->s_blocksize != sb->u.ext4_sb[j].s_frag_size) {
			printk ("EXT4-fs: fragsize %lu != blocksize "
				"%lu (not supported yet)\n",
				sb->u.ext4_sb[j].s_frag_size, sb->s_blocksize);
			goto failed_mount;
		}

		if (sb->u.ext4_sb[j].s_blocks_per_group > 
		    sb->s_blocksize * 8) {
			printk ("EXT4-fs: #blocks per group too big: %lu\n",
				sb->u.ext4_sb[j].s_blocks_per_group);
			goto failed_mount;
		}
		if (sb->u.ext4_sb[j].s_frags_per_group > 
		    sb->s_blocksize * 8) {
			printk ("EXT4-fs: #fragments per group too big: %lu\n",
				sb->u.ext4_sb[j].s_frags_per_group);
			goto failed_mount;
		}
		if (sb->u.ext4_sb[j].s_inodes_per_group > 
		    sb->s_blocksize * 8) {
			printk ("EXT4-fs: #inodes per group too big: %lu\n",
				sb->u.ext4_sb[j].s_inodes_per_group);
			goto failed_mount;
		}

		sb->u.ext4_sb[j].s_groups_count = 
			(es[j]->s_blocks_count -
			 es[j]->s_first_data_block +
			 EXT4_BLOCKS_PER_GROUP(sb, j) - 1) /
			EXT4_BLOCKS_PER_GROUP(sb, j);
		db_count[j] = (sb->u.ext4_sb[j].s_groups_count + 
			    EXT4_DESC_PER_BLOCK(sb, j) - 1) /
			EXT4_DESC_PER_BLOCK(sb, j);
		sb->u.ext4_sb[j].s_group_desc =
			kmalloc (db_count[j] * sizeof (struct buffer_head *), 
				 GFP_KERNEL);
		if (sb->u.ext4_sb[j].s_group_desc == NULL) {
			printk ("EXT4-fs: not enough memory\n");
			if (j) release_descriptors(sb, db_count[0], 
						   0, db_count[0]);
			goto failed_mount;
		}
		for (i = 0; i < db_count[j]; i++) {
			sb->u.ext4_sb[j].s_group_desc[i] = 
				bread(dev[j], logic_sb_block + i + 1,
				      sb->s_blocksize);
			if (!sb->u.ext4_sb[j].s_group_desc[i]) {
				if (j) {
					/*partition 0 descriptors loaded all*/
					release_descriptors(sb, db_count[0], 
							    0, db_count[0]);
				}
				/* this partition descriptors partially ld.*/
				release_descriptors(sb, db_count[j], j, i);
				printk ("EXT4-fs: unable to read group descriptors\n");
				goto failed_mount;
			}
		}
		if (!ext4_check_descriptors (sb, j)) {
			if (j) release_descriptors(sb, db_count[0], 
						   0, db_count[0]);
			release_descriptors(sb, db_count[j], j, db_count[j]);
			printk ("EXT4-fs: group descriptors corrupted !\n");
			goto failed_mount;
		}
		for (i = 0; i < EXT4_MAX_GROUP_LOADED; i++) {
			sb->u.ext4_sb[j].s_inode_bitmap_number[i] = 0;
			sb->u.ext4_sb[j].s_inode_bitmap[i] = NULL;
			sb->u.ext4_sb[j].s_block_bitmap_number[i] = 0;
			sb->u.ext4_sb[j].s_block_bitmap[i] = NULL;
		}
		sb->u.ext4_sb[j].s_loaded_inode_bitmaps = 0;
		sb->u.ext4_sb[j].s_loaded_block_bitmaps = 0;
		sb->u.ext4_sb[j].s_db_per_group = db_count[j];
	}
	unlock_super (sb);
	/*
	 * set up enough so that it can read an inode
	 */
	sb->s_dev = dev[0];
	sb->s_op = &ext4_sops;
	sb->u.ext4_sb[0].realdevice = 
		sb->u.ext4_sb[1].otherdevice = dev[0];
	sb->u.ext4_sb[0].otherdevice = 
		sb->u.ext4_sb[1].realdevice = dev[1];

	if (!(sb->s_mounted = iget (sb, EXT4_ROOT_INO))) {
		sb->s_dev = 0;
		release_descriptors(sb, db_count[0], 0, db_count[0]);
		release_descriptors(sb, db_count[1], 1, db_count[1]);
		brelse (bh[0]);
		brelse (bh[1]);
		printk ("EXT4-fs: get root inode failed\n");
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	ext4_setup_super(sb, es[0]);
	ext4_debug("Succesfully mounted ext4 filesystem partition 0: %s\n",
		   kdevname(dev[0]));
	ext4_debug("partition 1: %s\n", kdevname(dev[1]));
	/* you can't print them at once, as kdevname returns pointer to
	   static data */
	return sb;
}

static void ext4_commit_super (struct super_block * sb,
			       struct ext4_super_block * es[2])
{
	es[0]->s_wtime = CURRENT_TIME;
	mark_buffer_dirty(sb->u.ext4_sb[0].s_sbh, 1);
	es[1]->s_wtime = CURRENT_TIME;
	mark_buffer_dirty(sb->u.ext4_sb[1].s_sbh, 1);
	sb->s_dirt = 0;
}

/*
 * In the second extended file system, it is not necessary to
 * write the super block since we use a mapping of the
 * disk super block in a buffer.
 *
 * However, this function is still used to set the fs valid
 * flags to 0.  We need to set this flag to 0 since the fs
 * may have been checked while mounted and e2fsck may have
 * set s_state to EXT4_VALID_FS after some corrections.
 */

void ext4_write_super (struct super_block * sb)
{
	struct ext4_super_block * es[2];

	if (!(sb->s_flags & MS_RDONLY)) {
		es[0] = sb->u.ext4_sb[0].s_es;
		es[1] = sb->u.ext4_sb[1].s_es;

		ext4_debug ("setting valid to 0\n");

		if (es[0]->s_state & EXT4_VALID_FS) {
			es[0]->s_state &= ~EXT4_VALID_FS;
			es[0]->s_mtime = CURRENT_TIME;
			es[1]->s_state &= ~EXT4_VALID_FS;
			es[1]->s_mtime = CURRENT_TIME;
		}
		ext4_commit_super (sb, es);
	}
	sb->s_dirt = 0;
}

int ext4_remount (struct super_block * sb, int * flags, char * data)
{
	struct ext4_super_block * es[2];
	unsigned short resuid = sb->u.ext4_sb[0].s_resuid;
	unsigned short resgid = sb->u.ext4_sb[0].s_resgid;
	unsigned long new_mount_opt;
	unsigned long tmp;

	/*
	 * Allow the "check" option to be passed as a remount option.
	 */
	new_mount_opt = EXT4_MOUNT_CHECK_NORMAL;
	if (!parse_options (data, &tmp, &resuid, &resgid,
			    &new_mount_opt))
		return -EINVAL;

	sb->u.ext4_sb[0].s_mount_opt = new_mount_opt;
	sb->u.ext4_sb[0].s_resuid = resuid;
	sb->u.ext4_sb[0].s_resgid = resgid;
	es[0] = sb->u.ext4_sb[0].s_es;
	es[1] = sb->u.ext4_sb[1].s_es;
	if ((*flags & MS_RDONLY) == (sb->s_flags & MS_RDONLY))
		return 0;
	if (*flags & MS_RDONLY) {
		if (es[0]->s_state & EXT4_VALID_FS ||
		    !(sb->u.ext4_sb[0].s_mount_state & EXT4_VALID_FS))
			return 0;
		/*
		 * OK, we are remounting a valid rw partition rdonly, so set
		 * the rdonly flag and then mark the partition as valid again.
		 */
		es[0]->s_state = sb->u.ext4_sb[0].s_mount_state;
		es[0]->s_mtime = CURRENT_TIME;
		mark_buffer_dirty(sb->u.ext4_sb[0].s_sbh, 1);
		es[1]->s_state = sb->u.ext4_sb[1].s_mount_state;
		es[1]->s_mtime = CURRENT_TIME;
		mark_buffer_dirty(sb->u.ext4_sb[1].s_sbh, 1);
		sb->s_dirt = 1;
		ext4_commit_super (sb, es);
	}
	else {
		/*
		 * Mounting a RDONLY partition read-write, so reread and
		 * store the current valid flag.  (It may have been changed 
		 * by e2fsck since we originally mounted the partition.)
		 */
		sb->u.ext4_sb[0].s_mount_state = es[0]->s_state;
		sb->s_flags &= ~MS_RDONLY;
		ext4_setup_super (sb, es[0]);
	}
	return 0;
}

static struct file_system_type ext4_fs_type = {
        ext4_read_super, "ext4", 1, NULL
};

int init_ext4_fs(void)
{
        return register_filesystem(&ext4_fs_type);
}

#ifdef MODULE
int init_module(void)
{
	int status;

	if ((status = init_ext4_fs()) == 0)
		register_symtab(0);
	return status;
}

void cleanup_module(void)
{
        unregister_filesystem(&ext4_fs_type);
}

#endif

void ext4_statfs (struct super_block * sb, struct statfs * buf, int bufsiz)
{
	unsigned long overhead;
	unsigned long overhead_per_group;
	struct statfs tmp;

	if (ext4_test_opt (sb, MINIX_DF))
		overhead = 0;
	else {
		/*
		 * Compute the overhead (FS structures)
		 */
		overhead_per_group = 1 /* super block */ +
			sb->u.ext4_sb[0].s_db_per_group /* descriptors */ +
			1 /* block bitmap */ +
			1 /* inode bitmap */ +
			sb->u.ext4_sb[0].s_itb_per_group /* inode table */;
		overhead = sb->u.ext4_sb[0].s_es->s_first_data_block +
			(sb->u.ext4_sb[0].s_groups_count +
			 sb->u.ext4_sb[1].s_groups_count) 
			* overhead_per_group;
	}

	tmp.f_type = EXT4_SUPER_MAGIC;
	tmp.f_bsize = sb->s_blocksize;
	tmp.f_blocks = sb->u.ext4_sb[0].s_es->s_blocks_count +
		sb->u.ext4_sb[1].s_es->s_blocks_count - overhead;
	tmp.f_bfree = ext4_count_free_blocks (sb);
	tmp.f_bavail = tmp.f_bfree - 
		sb->u.ext4_sb[0].s_es->s_r_blocks_count -
		sb->u.ext4_sb[1].s_es->s_r_blocks_count;
/* refine this ! */
	if (tmp.f_bfree < sb->u.ext4_sb[0].s_es->s_r_blocks_count +
	    sb->u.ext4_sb[0].s_es->s_r_blocks_count)
		tmp.f_bavail = 0;
	tmp.f_files = sb->u.ext4_sb[0].s_es->s_inodes_count +
		sb->u.ext4_sb[1].s_es->s_inodes_count;
	tmp.f_ffree = ext4_count_free_inodes (sb);
	tmp.f_namelen = EXT4_NAME_LEN;
	memcpy_tofs(buf, &tmp, bufsiz);
}
