/*
 *  linux/fs/ext4/balloc.c
 * April 1997
 * by Mihai Budiu (budiu@cs.cornell.edu)
 * 
 *  from 
 *
 * Copyright (C) 1992, 1993, 1994, 1995
 * Remy Card (card@masi.ibp.fr)
 * Laboratoire MASI - Institut Blaise Pascal
 * Universite Pierre et Marie Curie (Paris VI)
 *
 *  Enhanced block allocation by Stephen Tweedie (sct@dcs.ed.ac.uk), 1993
 */

/*
 * balloc.c contains the blocks allocation and deallocation routines
 */

/*
 * The free blocks are managed by bitmaps.  A file system contains several
 * blocks groups.  Each group contains 1 bitmap block for blocks, 1 bitmap
 * block for inodes, N blocks for the inode table and data blocks.
 *
 * The file system contains group descriptors which are located after the
 * super block.  Each descriptor contains the number of the bitmap block and
 * the free blocks count in the block.  The descriptors are loaded in memory
 * when a file system is mounted (see ext4_read_super).
 */

#include <linux/fs.h>
#include <linux/ext4_fs.h>
#include <linux/stat.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/locks.h>

#include <asm/bitops.h>

#define in_range(b, first, len)		((b) >= (first) && (b) <= (first) + (len) - 1)

static struct ext4_group_desc * get_group_desc (struct super_block * sb,
						unsigned int block_group,
						struct buffer_head ** bh,
						int partition)
{
	unsigned long group_desc;
	unsigned long desc;
	struct ext4_group_desc * gdp;

	if (block_group >= sb->u.ext4_sb[partition].s_groups_count)
		ext4_panic (sb, "get_group_desc",
			    "block_group >= groups_count - "
			    "block_group = %d, groups_count = %lu",
			    block_group, sb->u.ext4_sb[partition].s_groups_count);

	group_desc = block_group / EXT4_DESC_PER_BLOCK(sb, partition);
	desc = block_group % EXT4_DESC_PER_BLOCK(sb, partition);
	if (!sb->u.ext4_sb[partition].s_group_desc[group_desc])
		ext4_panic (sb, "get_group_desc",
			    "Group descriptor not loaded - "
			    "block_group = %d, group_desc = %lu, desc = %lu",
			     block_group, group_desc, desc);
	gdp = (struct ext4_group_desc *) 
		sb->u.ext4_sb[partition].s_group_desc[group_desc]->b_data;
	if (bh)
		*bh = sb->u.ext4_sb[partition].s_group_desc[group_desc];
	return gdp + desc;
}

static void read_block_bitmap (struct super_block * sb,
			       unsigned int block_group,
			       unsigned long bitmap_nr,
			       int partition)
{
	struct ext4_group_desc * gdp;
	struct buffer_head * bh;
	
	gdp = get_group_desc (sb, block_group, NULL, partition);
	bh = bread (sb->u.ext4_sb[partition].realdevice, 
		    gdp->bg_block_bitmap, sb->s_blocksize);
	if (!bh)
		ext4_panic (sb, "read_block_bitmap",
			    "Cannot read block bitmap - "
			    "block_group = %d, block_bitmap = %lu",
			    block_group, (unsigned long) gdp->bg_block_bitmap);
	sb->u.ext4_sb[partition].s_block_bitmap_number[bitmap_nr] = 
		block_group;
	sb->u.ext4_sb[partition].s_block_bitmap[bitmap_nr] = bh;
}

/*
 * load_block_bitmap loads the block bitmap for a blocks group
 *
 * It maintains a cache for the last bitmaps loaded.  This cache is managed
 * with a LRU algorithm.
 *
 * Notes:
 * 1/ There is one cache per mounted file system.
 * 2/ If the file system contains less than EXT4_MAX_GROUP_LOADED groups,
 *    this function reads the bitmap without maintaining a LRU cache.
 */
static int load__block_bitmap (struct super_block * sb,
			       unsigned int block_group,
			       int partition)
{
	int i, j;
	unsigned long block_bitmap_number;
	struct buffer_head * block_bitmap;

	if (block_group >= sb->u.ext4_sb[partition].s_groups_count)
		ext4_panic (sb, "load_block_bitmap",
			    "block_group >= groups_count - "
			    "block_group = %d, groups_count = %lu",
			    block_group, 
			    sb->u.ext4_sb[partition].s_groups_count);

	if (sb->u.ext4_sb[partition].s_groups_count 
	    <= EXT4_MAX_GROUP_LOADED) {
		if (sb->u.ext4_sb[partition].s_block_bitmap[block_group]) {
			if (sb->u.ext4_sb[partition].s_block_bitmap_number[block_group] !=
			    block_group)
				ext4_panic (sb, "load_block_bitmap",
					    "block_group != block_bitmap_number");
			else
				return block_group;
		} else {
			read_block_bitmap (sb, block_group, 
					   block_group, partition);
			return block_group;
		}
	}

	for (i = 0; i < sb->u.ext4_sb[partition].s_loaded_block_bitmaps &&
		     sb->u.ext4_sb[partition].s_block_bitmap_number[i] 
		     != block_group; 
	     i++)
		;
	if (i < sb->u.ext4_sb[partition].s_loaded_block_bitmaps &&
  	    sb->u.ext4_sb[partition].s_block_bitmap_number[i] == block_group) {
		block_bitmap_number = 
			sb->u.ext4_sb[partition].s_block_bitmap_number[i];
		block_bitmap = 
			sb->u.ext4_sb[partition].s_block_bitmap[i];
		for (j = i; j > 0; j--) {
			sb->u.ext4_sb[partition].s_block_bitmap_number[j] =
				sb->u.ext4_sb[partition].s_block_bitmap_number[j - 1];
			sb->u.ext4_sb[partition].s_block_bitmap[j] =
				sb->u.ext4_sb[partition].s_block_bitmap[j - 1];
		}
		sb->u.ext4_sb[partition].s_block_bitmap_number[0] = 
			block_bitmap_number;
		sb->u.ext4_sb[partition].s_block_bitmap[0] = block_bitmap;
	} else {
		if (sb->u.ext4_sb[partition].s_loaded_block_bitmaps < 
		    EXT4_MAX_GROUP_LOADED)
			sb->u.ext4_sb[partition].s_loaded_block_bitmaps++;
		else
			brelse (sb->u.ext4_sb[partition].s_block_bitmap[EXT4_MAX_GROUP_LOADED - 1]);
		for (j = sb->u.ext4_sb[partition].s_loaded_block_bitmaps - 1; 
		     j > 0;  
		     j--) {
			sb->u.ext4_sb[partition].s_block_bitmap_number[j] =
				sb->u.ext4_sb[partition].s_block_bitmap_number[j - 1];
			sb->u.ext4_sb[partition].s_block_bitmap[j] =
				sb->u.ext4_sb[partition].s_block_bitmap[j - 1];
		}
		read_block_bitmap (sb, block_group, 0, partition);
	}
	return 0;
}

static inline int load_block_bitmap (struct super_block * sb,
				     unsigned int block_group,
				     int partition)
{
	if (sb->u.ext4_sb[partition].s_loaded_block_bitmaps > 0 &&
	    sb->u.ext4_sb[partition].s_block_bitmap_number[0] == block_group)
		return 0;
	
	if (sb->u.ext4_sb[partition].s_groups_count 
	    <= EXT4_MAX_GROUP_LOADED && 
	    sb->u.ext4_sb[partition].s_block_bitmap_number[block_group] 
	    == block_group &&
	    sb->u.ext4_sb[partition].s_block_bitmap[block_group]) 
		return block_group;

	return load__block_bitmap (sb, block_group, partition);
}

void ext4_free_blocks (const struct inode * inode, unsigned long block,
		       unsigned long count)
{
	struct buffer_head * bh;
	struct buffer_head * bh2;
	unsigned long block_group;
	unsigned long bit;
	unsigned long i;
	int bitmap_nr;
	struct super_block * sb;
	struct ext4_group_desc * gdp;
	struct ext4_super_block * es;
	int partition;

	sb = inode->i_sb;
	if (!sb) {
		printk ("ext4_free_blocks: nonexistent device");
		return;
	}
	lock_super (sb);
	partition = EXT4_INO_PARTITION(inode->i_ino, sb);
	es = sb->u.ext4_sb[partition].s_es;
	if (block < es->s_first_data_block || 
	    (block + count) > es->s_blocks_count) {
		ext4_error (sb, "ext4_free_blocks",
			    "Freeing blocks not in datazone - "
			    "block = %lu, count = %lu", block, count);
		unlock_super (sb);
		return;
	}

	ext4_debug ("freeing block %lu\n", block);

	block_group = (block - es->s_first_data_block) /
		      EXT4_BLOCKS_PER_GROUP(sb, partition);
	bit = (block - es->s_first_data_block) % 
		EXT4_BLOCKS_PER_GROUP(sb, partition);
	if (bit + count > EXT4_BLOCKS_PER_GROUP(sb, partition))
		ext4_panic (sb, "ext4_free_blocks",
			    "Freeing blocks across group boundary - "
			    "Block = %lu, count = %lu",
			    block, count);
	bitmap_nr = load_block_bitmap (sb, block_group, partition);
	bh = sb->u.ext4_sb[partition].s_block_bitmap[bitmap_nr];
	gdp = get_group_desc (sb, block_group, &bh2, partition);

	if (ext4_test_opt (sb, CHECK_STRICT) &&
	    (in_range (gdp->bg_block_bitmap, block, count) ||
	     in_range (gdp->bg_inode_bitmap, block, count) ||
	     in_range (block, gdp->bg_inode_table,
		       sb->u.ext4_sb[partition].s_itb_per_group) ||
	     in_range (block + count - 1, gdp->bg_inode_table,
		       sb->u.ext4_sb[partition].s_itb_per_group)))
		ext4_panic (sb, "ext4_free_blocks",
			    "Freeing blocks in system zones - "
			    "Block = %lu, count = %lu",
			    block, count);

	for (i = 0; i < count; i++) {
		if (!clear_bit (bit + i, bh->b_data))
			ext4_warning (sb, "ext4_free_blocks",
				      "bit already cleared for block %lu", 
				      block);
		else {
			if (sb->dq_op)
				sb->dq_op->free_block(inode, 
						      fs_to_dq_blocks(1, sb->s_blocksize));
			gdp->bg_free_blocks_count++;
			es->s_free_blocks_count++;
		}
	}
	
	mark_buffer_dirty(bh2, 1);
	mark_buffer_dirty(sb->u.ext4_sb[partition].s_sbh, 1);

	mark_buffer_dirty(bh, 1);
	if (sb->s_flags & MS_SYNCHRONOUS) {
		ll_rw_block (WRITE, 1, &bh);
		wait_on_buffer (bh);
	}
	sb->s_dirt = 1;
	unlock_super (sb);
	return;
}

/*
 * ext4_new_block uses a goal block to assist allocation.  If the goal is
 * free, or there is a free block within 32 blocks of the goal, that block
 * is allocated.  Otherwise a forward search is made for a free block; within 
 * each block group the search first looks for an entire free byte in the block
 * bitmap, and then for any free bit if that fails.
 */
int ext4_new_block (const struct inode * inode, unsigned long goal,
		    u32 * prealloc_count, u32 * prealloc_block, int * err)
{
	struct buffer_head * bh;
	struct buffer_head * bh2;
	char * p, * r;
	int i, j, k, tmp;
	int bitmap_nr;
	struct super_block * sb;
	struct ext4_group_desc * gdp;
	struct ext4_super_block * es;
	int partition;
#ifdef EXT4FS_DEBUG
	static int goal_hits = 0, goal_attempts = 0;
#endif

	*err = -ENOSPC;
	sb = inode->i_sb;
	if (!sb) {
		printk ("ext4_new_block: nonexistent device");
		return 0;
	}
	lock_super (sb);
	partition = EXT4_INO_PARTITION(inode->i_ino, sb);
	es = sb->u.ext4_sb[partition].s_es;
	if (es->s_free_blocks_count <= es->s_r_blocks_count &&
	    (!fsuser() && (sb->u.ext4_sb[partition].s_resuid 
			   != current->fsuid) &&
	     (sb->u.ext4_sb[partition].s_resgid == 0 ||
	      !in_group_p (sb->u.ext4_sb[partition].s_resgid)))) {
		unlock_super (sb);
		ext4_debug("First check fails\n");
		return 0;
	}

	ext4_debug ("goal=%lu.\n", goal);

repeat:
	/*
	 * First, test whether the goal block is free.
	 */
	if (goal < es->s_first_data_block || goal >= es->s_blocks_count)
		goal = es->s_first_data_block;
	i = (goal - es->s_first_data_block) / 
		EXT4_BLOCKS_PER_GROUP(sb, partition);
	gdp = get_group_desc (sb, i, &bh2, partition);
	if (gdp->bg_free_blocks_count > 0) {
		j = ((goal - es->s_first_data_block) 
		     % EXT4_BLOCKS_PER_GROUP(sb, partition));
#ifdef EXT4FS_DEBUG
		if (j)
			goal_attempts++;
#endif
		bitmap_nr = load_block_bitmap (sb, i, partition);
		bh = sb->u.ext4_sb[partition].s_block_bitmap[bitmap_nr];

		ext4_debug ("goal is at %d:%d.\n", i, j);

		if (!test_bit(j, bh->b_data)) {
#ifdef EXT4FS_DEBUG
			goal_hits++;
			ext4_debug ("goal bit allocated.\n");
#endif
			goto got_block;
		}
		if (j) {
			/*
			 * The goal was occupied; search forward for a free 
			 * block within the next XX blocks.
			 *
			 * end_goal is more or less random, but it has to be
			 * less than EXT4_BLOCKS_PER_GROUP. Aligning up to the
			 * next 64-bit boundary is simple..
			 */
			int end_goal = (j + 63) & ~63;
			j = find_next_zero_bit(bh->b_data, end_goal, j);
			if (j < end_goal)
				goto got_block;
		}
	
		ext4_debug ("Bit not found near goal\n");

		/*
		 * There has been no free block found in the near vicinity
		 * of the goal: do a search forward through the block groups,
		 * searching in each group first for an entire free byte in
		 * the bitmap and then for any free bit.
		 * 
		 * Search first in the remainder of the current group; then,
		 * cyclicly search through the rest of the groups.
		 */
		p = ((char *) bh->b_data) + (j >> 3);
		r = memscan(p, 0, (EXT4_BLOCKS_PER_GROUP(sb, partition) 
				   - j + 7) >> 3);
		k = (r - ((char *) bh->b_data)) << 3;
		if (k < EXT4_BLOCKS_PER_GROUP(sb, partition)) {
			j = k;
			goto search_back;
		}
		k = find_next_zero_bit ((unsigned long *) bh->b_data, 
					EXT4_BLOCKS_PER_GROUP(sb, partition),
					j);
		if (k < EXT4_BLOCKS_PER_GROUP(sb, partition)) {
			j = k;
			goto got_block;
		}
	}

	ext4_debug ("Bit not found in block group %d.\n", i);

	/*
	 * Now search the rest of the groups.  We assume that 
	 * i and gdp correctly point to the last group visited.
	 */
	for (k = 0; k < sb->u.ext4_sb[partition].s_groups_count; k++) {
		i++;
		if (i >= sb->u.ext4_sb[partition].s_groups_count)
			i = 0;
		gdp = get_group_desc (sb, i, &bh2, partition);
		if (gdp->bg_free_blocks_count > 0)
			break;
	}
	if (k >= sb->u.ext4_sb[partition].s_groups_count) {
		unlock_super (sb);
		ext4_debug("All groups checked\n");
		return 0;
	}
	bitmap_nr = load_block_bitmap (sb, i, partition);
	bh = sb->u.ext4_sb[partition].s_block_bitmap[bitmap_nr];
	r = memscan(bh->b_data, 0, EXT4_BLOCKS_PER_GROUP(sb, partition) >> 3);
	j = (r - bh->b_data) << 3;
	if (j < EXT4_BLOCKS_PER_GROUP(sb, partition))
		goto search_back;
	else
		j = find_first_zero_bit ((unsigned long *) bh->b_data,
					 EXT4_BLOCKS_PER_GROUP(sb, partition));
	if (j >= EXT4_BLOCKS_PER_GROUP(sb, partition)) {
		ext4_error (sb, "ext4_new_block",
			    "Free blocks count corrupted for block group %d", 
			    i);
		unlock_super (sb);
		return 0;
	}

search_back:
	/* 
	 * We have succeeded in finding a free byte in the block
	 * bitmap.  Now search backwards up to 7 bits to find the
	 * start of this group of free blocks.
	 */
	for (k = 0; 
	     k < 7 && j > 0 && !test_bit (j - 1, bh->b_data); 
	     k++, j--);
	
got_block:

	ext4_debug ("using block group %d(%d)\n", i, 
		    gdp->bg_free_blocks_count);

	/*
	 * Check quota for allocation of this block.
	 */
	if (sb->dq_op)
		if (sb->dq_op->alloc_block (inode, 
					    fs_to_dq_blocks(1, 
							    sb->s_blocksize))){
			unlock_super (sb);
			*err = -EDQUOT;
			return 0;
		}

	tmp = j + i * EXT4_BLOCKS_PER_GROUP(sb, partition) 
		+ es->s_first_data_block;

	if (ext4_test_opt (sb, CHECK_STRICT) &&
	    (tmp == gdp->bg_block_bitmap ||
	     tmp == gdp->bg_inode_bitmap ||
	     in_range (tmp, gdp->bg_inode_table, 
		       sb->u.ext4_sb[partition].s_itb_per_group)))
		ext4_panic (sb, "ext4_new_block",
			    "Allocating block in system zone - "
			    "block = %u", tmp);

	if (set_bit (j, bh->b_data)) {
		ext4_warning (sb, "ext4_new_block",
			      "bit already set for block %d", j);
		if (sb->dq_op)
			sb->dq_op->free_block(inode, 
					      fs_to_dq_blocks(1, 
							      sb->s_blocksize));
		goto repeat;
	}

	ext4_debug ("found bit %d\n", j);

	/*
	 * Do block preallocation now if required.
	 */
#ifdef EXT4_PREALLOCATE
	if (prealloc_block) {
		*prealloc_count = 0;
		*prealloc_block = tmp + 1;
		for (k = 1;
		     k < 8 && (j + k) < EXT4_BLOCKS_PER_GROUP(sb, partition);
		     k++) {
			if (sb->dq_op)
				if (sb->dq_op->alloc_block(inode, 
							   fs_to_dq_blocks(1, sb->s_blocksize)))
					break;
			if (set_bit (j + k, bh->b_data)) {
				if (sb->dq_op)
					sb->dq_op->free_block(inode, 
							      fs_to_dq_blocks(1, sb->s_blocksize));
				break;
			}
			(*prealloc_count)++;
		}	
		gdp->bg_free_blocks_count -= *prealloc_count;
		es->s_free_blocks_count -= *prealloc_count;
		ext4_debug ("Preallocated a further %u bits.\n",
			    *prealloc_count);
	}
#endif

	j = tmp;

	mark_buffer_dirty(bh, 1);
	if (sb->s_flags & MS_SYNCHRONOUS) {
		ll_rw_block (WRITE, 1, &bh);
		wait_on_buffer (bh);
	}

	if (j >= es->s_blocks_count) {
		ext4_error (sb, "ext4_new_block",
			    "block >= blocks count - "
			    "block_group = %d, block=%d", i, j);
		unlock_super (sb);
		return 0;
	}
	if (!(bh = getblk (sb->u.ext4_sb[partition].realdevice, 
			   j, sb->s_blocksize))) {
		ext4_error (sb, "ext4_new_block", "cannot get block %d", j);
		unlock_super (sb);
		return 0;
	}
	memset(bh->b_data, 0, sb->s_blocksize);
	mark_buffer_uptodate(bh, 1);
	mark_buffer_dirty(bh, 1);
	brelse (bh);

	ext4_debug ("allocating block %d. "
		    "Goal hits %d of %d.\n", j, goal_hits, goal_attempts);

	gdp->bg_free_blocks_count--;
	mark_buffer_dirty(bh2, 1);
	es->s_free_blocks_count--;
	mark_buffer_dirty(sb->u.ext4_sb[partition].s_sbh, 1);
	sb->s_dirt = 1;
	unlock_super (sb);
	*err = 0;
	return j;
}

unsigned long ext4_count_free_blocks (struct super_block * sb)
{
#ifdef EXT4FS_DEBUG
	struct ext4_super_block * es;
	unsigned long desc_count, bitmap_count, x;
	int bitmap_nr;
	struct ext4_group_desc * gdp;
	int i, j;
	
	lock_super (sb);
	gdp = NULL;
	for (j=0; j < 2; j++) {
          	desc_count = 0;
	        bitmap_count = 0;

		es = sb->u.ext4_sb[j].s_es;
		for (i = 0; i < sb->u.ext4_sb[j].s_groups_count; i++) {
			gdp = get_group_desc (sb, i, NULL, j);
			desc_count += gdp->bg_free_blocks_count;
			bitmap_nr = load_block_bitmap (sb, i, j);
			x = ext4_count_free (sb->u.ext4_sb[j].s_block_bitmap[bitmap_nr],
					     sb->s_blocksize);
			printk ("group %d: stored = %d, counted = %lu\n",
				i, gdp->bg_free_blocks_count, x);
			bitmap_count += x;
		}
		printk("ext4_count_free_blocks: stored = %lu, "
		       "computed = %lu, %lu\n",
		       es->s_free_blocks_count, desc_count, bitmap_count);
	}
	unlock_super (sb);
	return bitmap_count;
#else
	return sb->u.ext4_sb[0].s_es->s_free_blocks_count +
		sb->u.ext4_sb[1].s_es->s_free_blocks_count;
#endif
}

static inline int block_in_use (unsigned long block,
				struct super_block * sb,
				unsigned char * map,
				int partition)
{
	return test_bit ((block - 
			  sb->u.ext4_sb[partition].s_es->s_first_data_block) %
			 EXT4_BLOCKS_PER_GROUP(sb, partition), map);
}

void ext4_check_blocks_bitmap (struct super_block * sb)
{
	struct buffer_head * bh;
	struct ext4_super_block * es;
	unsigned long desc_count, bitmap_count, x;
	unsigned long desc_blocks;
	int bitmap_nr;
	struct ext4_group_desc * gdp;
	int i, j, k;

	lock_super (sb);
	gdp = NULL;

	for (k=0; k < 2; k++) {
	        desc_count = 0;
	        bitmap_count = 0;

		es = sb->u.ext4_sb[k].s_es;
		desc_blocks = (sb->u.ext4_sb[k].s_groups_count 
			       + EXT4_DESC_PER_BLOCK(sb, k) - 1) /
			EXT4_DESC_PER_BLOCK(sb, k);
		for (i = 0; i < sb->u.ext4_sb[k].s_groups_count; i++) {
			gdp = get_group_desc (sb, i, NULL, k);
			desc_count += gdp->bg_free_blocks_count;
			bitmap_nr = load_block_bitmap (sb, i, k);
			bh = sb->u.ext4_sb[k].s_block_bitmap[bitmap_nr];

			if (!test_bit (0, bh->b_data))
				ext4_error (sb, "ext4_check_blocks_bitmap",
					    "Superblock in group %d is marked free", i);
			
			for (j = 0; j < desc_blocks; j++)
				if (!test_bit (j + 1, bh->b_data))
					ext4_error (sb, "ext4_check_blocks_bitmap",
						    "Descriptor block #%d in group "
						    "%d is marked free", j, i);

			if (!block_in_use (gdp->bg_block_bitmap, sb, 
					   bh->b_data, k))
				ext4_error (sb, "ext4_check_blocks_bitmap",
					    "Block bitmap for group %d is marked free",
					    i);

			if (!block_in_use (gdp->bg_inode_bitmap, sb, 
					   bh->b_data, k))
				ext4_error (sb, "ext4_check_blocks_bitmap",
					    "Inode bitmap for group %d is marked free",
					    i);

			for (j = 0; j < sb->u.ext4_sb[k].s_itb_per_group; j++)
				if (!block_in_use (gdp->bg_inode_table + j, sb,
						   bh->b_data, k))
					ext4_error (sb, "ext4_check_blocks_bitmap",
						    "Block #%d of the inode table in "
						    "group %d is marked free", 
						    j, i);

			x = ext4_count_free (bh, sb->s_blocksize);
			if (gdp->bg_free_blocks_count != x)
				ext4_error (sb, "ext4_check_blocks_bitmap",
					    "Wrong free blocks count for group %d, "
					    "stored = %d, counted = %lu", i,
					    gdp->bg_free_blocks_count, x);
			bitmap_count += x;
		}
		if (es->s_free_blocks_count != bitmap_count)
			ext4_error (sb, "ext4_check_blocks_bitmap",
				    "Wrong free blocks count in super block, "
				    "stored = %lu, counted = %lu",
				    (unsigned long) es->s_free_blocks_count, 
				    bitmap_count);
	}
	unlock_super (sb);
}
