/*
 *  tools/build.c
 *  Modified by Mihai Budiu for own purposes from Linux code
 */

/*
 * This file builds a disk-image from several different files:
 *
 * - bootsect: max 510 bytes of 8086 machine code, loads the rest
 * - setup: max 4 sectors of 8086 machine code, sets up system parm
 * - system: 80386 code for actual system
 * - a # of INIT_SERVERS files: initial system servers
 * The files to use are described in the build.info file.
 *
 * It does some checking that all files are of the correct type, and
 * just writes the result to stdout, removing headers and padding to
 * the right amount. It also writes some system data to stderr.
 */

#include <stdio.h>	/* fprintf */
#include <string.h>
#include <stdlib.h>	/* contains exit */
#include <sys/types.h> /* unistd.h needs this */
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <unistd.h>	/* contains read/write */
#include <fcntl.h>
#include "../include/config.h"
#include <linux/a.out.h>
#include "../kernel/params.h"
   /* describes where to write initial system setup information */

#define MINIX_HEADER 32
static int GCC_HEADER = sizeof(struct exec);
#define N_MAGIC_OFFSET 1024
#define Kb 1024
#define AG (Kb)  /* alignament value; now 1Kb */

#define ALIGN(addr) ((((addr) + AG - 1) / AG) * AG)  /* align */

#define SYS_SIZE DEF_SYSSIZE

/* max nr of sectors of setup: don't change unless you also change
 * bootsect etc */
#define SETUP_SECTS 4

#define STRINGIFY(x) #x

typedef union {
  long l;
  short s[2];
  char b[4];
} conv;

long intel_long(long l)
{
  conv t;
  
  t.b[0] = l & 0xff; l >>= 8;
  t.b[1] = l & 0xff; l >>= 8;
  t.b[2] = l & 0xff; l >>= 8;
  t.b[3] = l & 0xff; l >>= 8;
  return t.l;
}

short intel_short(short l)
{
  conv t;
  
  t.b[0] = l & 0xff; l >>= 8;
  t.b[1] = l & 0xff; l >>= 8;
  return t.s[0];
}

char buf[1024];
struct exec *ex = (struct exec *)buf;

void die(char * str)
{
  fprintf(stderr,"%s\n",str);
  exit(1);
}

void usage(void)
{
  fprintf(stderr, 
	  "Usage: build build_information_file description_offset >image\n");
  fprintf(stderr, 
	  "The description offset is the address in the main system file\n");
  die("where the system information should be saved\n");
}

FILE * info;

void open_info(char * f)    
{
  info = fopen(f, "r");
  if (info == NULL) die("Cannot open `build.info' file");
}
 
char * next_arg(void)
{
  static char name[50];
  int r;

  r = fscanf(info, "%s ", name);
  if (r <= 0) die("Cannot read next name from `build.info'");
  return name;
}

int next_int(void)
{
  int l, r;
 
  r = fscanf(info, "%d ", &l);
  if (r <= 0) die("cannot read number from `build.info'");
  return l;
}

int sys_size = 0; /* accumulate here system size in clicks */
struct configuration init[INIT_SERVERS+1]; /* 1 for system code */
int crt = 0;  /* current part of the system */

char * mybasename(char * st)
     /* skips up to last '/' in string st inclusive */
{
  char * p;
  for (p=st; *p; p++);  /* finds end */
  for (; (p[-1] != '/') && (p != st); p--);  /* finds '/' or beginning */
  return p;
}

void gnu_build(char * file, int stack)
     /* build an image from the indicated gnu-compiled code */
{
  int id, sz, i;
  int complete;  

  fprintf(stderr, "Processing %s:\n", file);
  if ((id=open(file, O_RDONLY,0))<0) 
    die("Unable to open");
  strncpy(init[crt].proc_name, mybasename(file), 
	  sizeof(init[crt].proc_name)); 
  if (read(id,buf,GCC_HEADER) != GCC_HEADER) 
    die("Unable to read header");
  if (N_MAGIC(*ex) == ZMAGIC) {
    GCC_HEADER = N_MAGIC_OFFSET;
    lseek(id, GCC_HEADER, SEEK_SET);
  } 
  else if (N_MAGIC(*ex) != QMAGIC)
    die("Non-GCC header of 'system'");
#if 0
  if (N_MAGIC(*ex) != ZMAGIC) die("Non-GCC header");
#endif
  fprintf(stderr, "offset %d (0x%x), ", sys_size, sys_size);
  ex->a_bss = ALIGN(ex->a_bss);   /* do not leave it unaligned */
  stack = ALIGN(stack);
  fprintf(stderr,
	  "%d kB (%d kB code, %d kB data, %d kB bss, %d kB stack)\n",
	  (ex->a_text + ex->a_data + ex->a_bss + stack) / Kb,
	  ex->a_text/Kb, ex->a_data/Kb, ex->a_bss/Kb, stack/Kb);
  /* recall system configuration information */
  init[crt].base = sys_size + 0x1000;  
  /* this constant is hardcoded in ../Makefile, ../boot/setup.S; it is
     the place where the system is loaded initially */
  init[crt].code_size = ex->a_text;
  init[crt].data_size = ex->a_data + ex->a_bss;
  init[crt].stack_size = stack;
  init[crt].entry = ex->a_entry;
  complete = stack + ex->a_bss; /* that many bytes 0 at the end */
  sys_size += ex->a_text + ex->a_data + complete;
  if (sys_size > SYS_SIZE * 16) die("System is too big");

  sz = N_SYMOFF(*ex) - GCC_HEADER /* + 4 ?? */; /* code bytes */
  while (sz > 0) {
    int l, n;
    
    l = sz;
    if (l > sizeof(buf)) l = sizeof(buf);
    if ((n=read(id, buf, l)) != l) {
      if (n == -1) perror(file);
      else fprintf(stderr, "Unexpected EOF\n");
      die("Can't read");
    }
    if (write(1, buf, l) != l) die("Write failed");
    sz -= l;
  }
  close(id);

  /* now fill bss and stack */
  for (i=0; i < sizeof(buf); i++) buf[i] = '\0';
  while (complete > 0) {
    int l;
    
    l = complete;
    if (l > sizeof(buf)) l = sizeof(buf);
    if (write(1, buf, l) != l) die("Write failed");
    complete -= l;
  }
  crt++;  /* next process */
}

int main(int argc, char ** argv)
{
  int i,c,id;
  int write_configuration;
  
  if (argc != 3) usage();
  open_info(argv[1]);
  sscanf(argv[2], "%x", &write_configuration);
  /* at this offset in the System (3rd file of the image file) is
     an array of structures `struct configuration' where the array
     build in THIS file is copyed when the image is known */
  
  for (i=0;i<sizeof buf; i++) buf[i]=0;
  if ((id=open(next_arg(),O_RDONLY,0))<0) die("Unable to open 'boot'");
  if (read(id,buf,MINIX_HEADER) != MINIX_HEADER) 
    die("Unable to read header of 'boot'");
  if (((long *) buf)[0]!=intel_long(0x04100301)) 
    die("Non-Minix header of 'boot'");
  if (((long *) buf)[1]!=intel_long(MINIX_HEADER))
    die("Non-Minix header of 'boot'");
  if (((long *) buf)[3] != 0) die("Illegal data segment in 'boot'");
  if (((long *) buf)[4] != 0) die("Illegal bss in 'boot'");
  if (((long *) buf)[5] != 0) die("Non-Minix header of 'boot'");
  if (((long *) buf)[7] != 0) die("Illegal symbol table in 'boot'");
  i=read(id,buf,sizeof buf);
  fprintf(stderr,"Boot sector %d bytes.\n",i);
  if (i != 512) die("Boot block must be exactly 512 bytes");
  if ((*(unsigned short *)(buf+510)) 
      != (unsigned short)intel_short(0xAA55))
    die("Boot block hasn't got boot flag (0xAA55)");
#if 0
  buf[508] = (char) minor_root;
  buf[509] = (char) major_root;	
#endif
  i=write(1,buf,512);
  if (i != 512) die("Write call failed");
  close (id);
  
  if ((id=open(next_arg(),O_RDONLY,0))<0) die("Unable to open 'setup'");
  if (read(id,buf,MINIX_HEADER) != MINIX_HEADER)
    die("Unable to read header of 'setup'");
  if (((long *) buf)[0]!=intel_long(0x04100301))
    die("Non-Minix header of 'setup'");
  if (((long *) buf)[1]!=intel_long(MINIX_HEADER))
    die("Non-Minix header of 'setup'");
  if (((long *) buf)[3] != 0) die("Illegal data segment in 'setup'");
  if (((long *) buf)[4] != 0) die("Illegal bss in 'setup'");
  if (((long *) buf)[5] != 0) die("Non-Minix header of 'setup'");
  if (((long *) buf)[7] != 0) die("Illegal symbol table in 'setup'");
  for (i=0 ; (c=read(id,buf,sizeof buf))>0 ; i+=c )
    if (write(1,buf,c)!=c) die("Write call failed");
  if (c != 0) die("read-error on 'setup'");
  close (id);
  if (i > SETUP_SECTS*512)
    die("Setup exceeds " STRINGIFY(SETUP_SECTS)
	" sectors - rewrite build/boot/setup");
  fprintf(stderr,"Setup is %d bytes.\n",i);
  for (c=0 ; c<sizeof(buf) ; c++) buf[c] = '\0';
  while (i<SETUP_SECTS*512) {
    c = SETUP_SECTS*512-i;
    if (c > sizeof(buf)) c = sizeof(buf);
    if (write(1,buf,c) != c) die("Write call failed");
    i += c;
  }
    
  gnu_build(next_arg(), 0);  /* process the System image */

  for (i = 0; i < INIT_SERVERS; i++) {/* process each server code separately */
    char * n = next_arg();
    int st = next_int();
    gnu_build(n, st);
  }

  fprintf(stderr, "Initial system size : %d\n", sys_size);
  if (lseek(1,500,0) == 500) {
    sys_size = (sys_size + 15) / 16;
    buf[0] = (sys_size & 0xff);
    buf[1] = ((sys_size >> 8) & 0xff);
    if (write(1, buf, 2) != 2) die("Write failed");
  }
  write_configuration += 512 * (SETUP_SECTS + 1);  /* over boot & setup */
  write_configuration -= 4096;   /* text loaded at this address */
  if (lseek(1, (long)write_configuration, 0) != write_configuration)
    die("Cannot lseek into system image to write configuration");
  fprintf(stderr, "Writing data to offset %d (0x%x) in image file\n", 
	  write_configuration, write_configuration);
  for (i = 0; i <= INIT_SERVERS; i++) { /* start with second file */
     c = write(1, (char *)&init[i], sizeof(struct configuration));
     if (c != sizeof(struct configuration)) 
       die("cannot write configuration to image");
  }
  return(0);
}






