/*
 * keyboard/scan.c
 * @(#) keyboard server; reads and queues raw scan codes
 * (c) 1995 by Mihai Budiu
 */

/* This is the keyboard server.  It is a driver.
   Messages it accepts :
   -OPERATION|-arg--|-description-----------------------------
   INT       | int  | a keyboard interrupt has happen
   SCAN_GET  | -    | client wants a scan code.  Blocking.
   SCAN_NOW  | -    | client wants a scan code.  Return error if none.
   SCAN_MODE | int  | new functioning mode. Def. in operation.h
   ------------------------------------------------------------
   The GET_SCAN call blocks the first caller if no key available.  A second
   caller of GET_SCAN or SCAN_NOW is rejected with error (it is in fact a 
   race for the terminal ...)
 */

#define __SERVER__

#include "../include/globals.h"
#include "../include/message.h"
#include "../include/operation.h"
#include "../include/pid.h"
#include "../include/error.h"
#include "../include/config.h"     /* for KEYBOARD_INT */
#include "../include/string.h"
#include "../include/asm.h"
#include "./queue.h"
#include "../include/server.h"

#define DEBUG_LOCAL 0

PRIVATE struct message msg, reply;
PRIVATE struct {
  proc_id pid;             /* process waiting for key */
  msg_id_t msg_id; /* msg_id waiting for answer */
} blocked;
PRIVATE char isblocked;     /* some process already blocked for message */
PRIVATE unsigned int mode_flags;    
  /* current functioning mode; defined in include/operation.h */

PRIVATE void init_keyboard(void);
PRIVATE void do_keyboard_int(void);
PRIVATE do_thing_t do_scan_get, do_scan_now, do_scan_mode, do_scan_get_mode;

QUEUE(key_q, unsigned char, 24)  /*scan codes queue (from queue.h) */

PUBLIC void Main(void)  /* entry point */
{
  int err;
  char answer;

  init_keyboard();     /* initializing */

  for (;;) {
  again:
    while ((err = receive(&msg)) != E_OK) /* retry */;
#if DEBUG_LOCAL
    tell_ker("scan got msg", msg.OPERATION, SRV_MESS);
#endif
    answer = ANSWER_EXPECTED(msg.flags);
    switch(msg.OPERATION) {
    case INT:           /* a keyboard interrupt came */
      do_keyboard_int();
      goto again;       /* faster, as this happens often */
    case SCAN_GET:  
      if (! answer) break;  /* can't reply - no action */
      err = do_scan_get();
      if (err == E_BLOCK && CAN_BLOCK(msg.flags)) 
	answer = NO;  /* no reply now */
      break;
    case SCAN_NOW:
      if (! answer) break;
      err = do_scan_now();
      break;
    case SCAN_MODE:
      err = do_scan_mode();
      break;
    case SCAN_GET_MODE:
      if (! answer) break;
      err = do_scan_get_mode();
      break;
    default:
      err = E_BADOP;
    }
    if (answer) {
      (void)send_reply(msg.SOURCE.local_pid, msg.msg_id, 
		       msg.OPERATION, err, &reply);
#if DEBUG_LOCAL
      tell_ker("'scan' replyes", msg.ERROR, SRV_MESS);
#endif
    }
  }
}

/********************** keyboard server functions ******************/

PRIVATE void init_keyboard(void)
     /* initialize keyboard server */
{
  int err;

    /* change priority to become driver */
  msg.INFOi[0] = DRIVER_PRIO;
  err = send_receive(SYS_PID, CH_PRIO, &msg);
  if (err != E_OK) tell_ker("can't send for prio", err, SRV_PANIC);
  err = (int)msg.ERROR;
  if (err != E_OK) tell_ker("can't change prio", err, SRV_PANIC);

    /* register as handler for keyboard interrupts */
  msg.INFOi[0] = KEYBOARD_INT;
  err = send_receive(SYS_PID, HAND_INT, &msg);
  if (err != E_OK) tell_ker("can't send for int", err, SRV_PANIC);
  err = (int)msg.ERROR;
  if (err != E_OK) tell_ker("can't get int", err, SRV_PANIC);

  init_queue(&key_q);
  isblocked = NO;   /* no process blocked */
  mode_flags = SCAN_FILTER_MOST | SCAN_FILTER_PREFIX;   /* linear buffering */
}


PRIVATE inline unsigned char filter(unsigned char s)
     /* filter most of the released keys */
{
  unsigned char c;            /* direction bit extracted */

  if (mode_flags & SCAN_FILTER_MOST) {
    c = s & 0x7f;
    if (s == c) return c;                  /* key pressed : no filter */
    if (s == 0xe0 || s == 0xe1) 
      return (mode_flags & SCAN_FILTER_PREFIX ? 0 : s);  /* prefix key */
    if (c == 0x2a  /* LShift */ ||     /* for these the direction matters */
	c == 0x36  /* RShift */ ||
	c == 0x1d  /* L/R Ctrl*/||
	c == 0x38  /* L/R Alt */) return s;
    return 0;
  }
  return s;
}

PRIVATE void do_keyboard_int(void)
     /* a keyboard interrupt happened; process it; 
	The message contains msg.ERROR characters */
{
  unsigned char scancode;
  int i;

  for (i=0; msg.ERROR != 0; i++, msg.ERROR--) {
    scancode = msg.INFOc[i];
    if ((scancode = filter(scancode)) == 0) continue;

    /* we've got a key to memorize */
#if DEBUG && DEBUG_LOCAL
    __asm__("gs; movw %0, (0)" : : "r"(scancode | 7 << 8) : "memory");
#endif
    if (isblocked) {
      reply.INFOc[0] = scancode;
      /* somebody was blocked for a key; send it to him */
      (void)send_reply(blocked.pid.local_pid, blocked.msg_id, SCAN_GET, 
		       E_OK, &reply);
      isblocked = NO;
    }
    else put_queue(scancode, &key_q, mode_flags & SCAN_CIRCULAR);
  }
}

PRIVATE int do_scan_get(void)
     /* read first key in queue; block if none */
{
  char scancode;

  if (isblocked) return E_BUSY; /* somebody already blocked */
  do {
    if (empty(&key_q)) {
      isblocked = YES;
      blocked.pid = msg.SOURCE;  /* recall whom to wake up later */
      blocked.msg_id = msg.MSGID;
      return E_BLOCK;
    }
    /* queue not empty */
    scancode = get_queue(&key_q); /* filter queued chars */
    scancode = filter(scancode);
    if (scancode) {  /* valid code */
      reply.INFOc[0] = scancode;
      return E_OK;
    }
  } while (scancode == 0);
  return E_BLOCK;   /* should be never reached */
}

PRIVATE int do_scan_get_mode(void)
     /* read the mode flags */
{
  reply.INFOi[0] = mode_flags;
  return E_OK;
}

PRIVATE int do_scan_now(void)
     /* read first key in queue; return error if none */
{
  if (isblocked) return E_BUSY;
  if (empty(&key_q)) return E_AGAIN;
  reply.INFOc[0] = get_queue(&key_q);
  return E_OK;
}

PRIVATE int do_scan_mode(void)
{
  if (isblocked) return E_BUSY;
  mode_flags = (unsigned int)msg.INFOi[0];
  return E_OK;
}
