/*
 * keyboard/key.c
 * @(#) upper keyboard server; reads and queues key codes
 * (c) 1995 by Mihai Budiu
 */

/* This is a second keyboard server, running on top of `scan'
   It decodes the scan codes read by scan.
   Messages it accepts :
   -OPERATION|-arg--|-description-----------------------------
   KEY_GET   | -    | give me a depressed key.  Blocking.
   ------------------------------------------------------------
 */

#define __SERVER__

#include "../include/globals.h"
#include "../include/message.h"
#include "../include/operation.h"
#include "../include/pid.h"
#include "../include/error.h"
#include "../include/string.h"
#include "../include/asm.h"
#include "./keycodes.h"
#include "../include/server.h"

#define DEBUG_LOCAL 0

PRIVATE struct message msg, reply;
PRIVATE void init_keydecode(void);
PRIVATE do_thing_t do_key_get, do_reply_key;
PRIVATE struct {          /* describe who's pending */
  proc_id pid;            /* who waits for key */
  msg_id_t msg_id;
} pending;
PRIVATE char ispending;   /* is somebody pending ? */

PUBLIC void Main(void)  /* entry point */
{
  int err;
  char answer;
  
  init_keydecode();     /* initializing */

  for (;;) {
    while ((err = receive(&msg)) != E_OK) /* retry */;
#if DEBUG_LOCAL
    tell_ker("`key' got msg", msg.OPERATION, SRV_MESS);
#endif
    answer = ANSWER_EXPECTED(msg.flags);
    switch(msg.OPERATION) {
    case KEY_GET:          
      if (! answer) break;     /* can't reply if no answer expected ! */
      err = do_key_get();
      if (err == E_BLOCK && CAN_BLOCK(msg.flags)) 
	answer = NO;  /* no reply now */
    /* if the caller is a server, the reply will be sent now, with error
       E_BLOCK, and a new reply will be sent when some answer will be
       present.  The first answer is necessary to wake-up the server,
       which cannot be pending */
#if DEBUG_LOCAL
      else tell_ker("reply now", err, SRV_MESS);
#endif
      break;
    case REPLY_TO(SCAN_GET):
      if (answer) tell_ker("reply requests answer", 0, SRV_ERROR);
      err = do_reply_key();
      break;
    default:
      err = E_BADOP;
    }
    if (answer) {
      (void)send_reply(msg.SOURCE.local_pid, msg.MSGID, 
		       msg.OPERATION, err, &reply);
#if DEBUG_LOCAL 
      tell_ker("reply", msg.ERROR, SRV_MESS);
#endif
    }
  }
}

/********************** keyboard server functions ******************/

PRIVATE void init_keydecode(void)
{
  int err;

  ispending = NO;  /* no client request pending */
  msg.INFOi[0] = SCAN_FILTER_MOST | SCAN_FILTER_PREFIX;  /* operation mode */
  err = send_receive(SCAN_PID, SCAN_MODE, &msg);
  if (err != E_OK) tell_ker("can't tell mode", err, SRV_PANIC);
  err = msg.ERROR;
  if (err != E_OK) tell_ker("can't set mode", err, SRV_PANIC);  
}

PRIVATE int reg_pending(struct message * m)
     /* mark the current client's request */
{
#if DEBUG_LOCAL
  tell_ker("register", m->SOURCE.local_pid, SRV_MESS);
#endif
  ispending = YES;
  pending.pid = m->SOURCE; 
  pending.msg_id = m->MSGID;
  return E_BLOCK;
}

PRIVATE int unregister(int err, struct message * m)
     /* unregister current pending request and send reply */
{
  if (! ispending) tell_ker("unreg. but no reg", 0, SRV_ERROR);
  ispending = NO;
  return send_reply(pending.pid.local_pid, pending.msg_id, KEY_GET,
		    err, m);
  /* message type always KEY_GET */
}

PRIVATE void show_leds(char caps, char scroll, char num)
     /* light the keyboard leds */
{
  static char oldleds = 0, leds;
  /* don't know how to do that ! */

  leds = scroll << LED_SCR | num << LED_NUM | caps << LED_CAPS;
  if (leds != oldleds) {
    oldleds = leds;
  }
}

PRIVATE int translate(unsigned char c)
     /* c is a scan code; If it forms a key code return that code.
	else return -1; uses the mapsh/mapnosh arrays in keycodes.h */
{
  static char lshift=0, rshift=0;  /* no shift */
  static char ctrl = 0;
  static char alt = 0;
  /* leds */
  static char caps = 0;
  static char num = 0;
  static char scroll = 0;
  int s, down;
  
  s = (int)(c & 0x7f);
  if (s > sizeof(mapsh)) return 0;     /* impossible, but ... */
  if (s < TOP_ROW) s = (int)((lshift | rshift) ? mapsh[s] : mapnosh[s]);
  else if (s > PAD) s = (int)(num ? mapsh[s] : mapnosh[s]);
  else s = (int)(((lshift | rshift) ^ caps) ? mapsh[s] : mapnosh[s]);
  down = (c & 0x80) ? 0 : 1;           /* key down or up ? */
  switch (s) {    /* additional processing on resulted code */
  case ALT:
    alt = down;   /* depressed / released ALT */
    return -1;
  case SH:
    lshift = down;
    return -1;
  case RSH:
    rshift = down;
    return -1;
  case CTRL:
    ctrl = down;
    return -1;
  case CAPS:
    caps ^= 1;    /* toggle */
    show_leds(caps, scroll, num);
    return -1;
  case SCR:
    scroll ^= 1;
    show_leds(caps, scroll, num);
    return -1;
  case NUM:
    num ^= 1;
    show_leds(caps, scroll, num);
    return -1;
  default:
    if (alt) s |= 0200;
    if (ctrl) {
      s &= 0x1f;
      if (s == 0) s = CTRL_AT;  /* C-@ == 0 on the above rule; correct */
    }
  }
#if DEBUG_LOCAL
  tell_ker("`key' decodes", c, SRV_MESS);
  tell_ker("`key' decoded", s, SRV_MESS);
#endif
  return s;
}

PRIVATE int have_data(void)
     /* asks scan about data; converts it; returns data or error */
{
  struct message m;
  int i, err;

  while(1) {
    err = send_receive(SCAN_PID, SCAN_GET, &m);
#if DEBUG_LOCAL
    tell_ker("key tryed again", m.ERROR, SRV_MESS);
#endif
    if (err != E_OK) return err;
    if (ANSWER_EXPECTED(m.flags)) {
      send(SCAN_PID, NOOP, &m);
      tell_ker("reply expects reply", 0, SRV_ERROR);
    }
    err = m.ERROR;
    if (err == E_BLOCK) return err;
    else if (err != E_OK) return err;
    i = translate(m.INFOc[0]);   /* this is the reply */
    if (i >= 0) return i;
    /* not a valid code, try to get another */
  }
}

PRIVATE int do_key_get(void)
     /* read first key; block if none */
{
  static struct message m;
  int i;

  if (ispending) return E_BUSY;
#if DEBUG && DEBUG_LOCAL
  __asm__("gs; movw %0, (2)" : : "r"(m.INFOc[0] | 7 << 8) : "memory");
#endif
  i = have_data();
  if (i == E_BLOCK) return reg_pending(&msg);
  if (i < 0) return i;   /* error */
  /* a valid reply ! */
  reply.INFOc[0] = i;
  return E_OK;
}

PRIVATE int do_reply_key(void)
     /* a delayed answer from `scan' came with the expected key.
	Deliver it to the pending client */
{
  int i;

  if (! ispending) return E_ILL;
  i = translate(msg.INFOc[0]);   /* use the char just arrived */
  if (i < 0) i = have_data();    /* not good enough this one; try again */
  if (i >= 0) {                  /* we have a key code ! */
    reply.INFOc[0] = i;
    return unregister(msg.ERROR, &reply);
  }
  /* no data now; some other time */
  return i;
}
