/* 
 * task.c
 * @(#) task management - creation, destruction, etc.
 * (c) 1995 by Mihai Budiu
 */

#include "../include/limits.h"
#include "../include/globals.h"
#include "../include/asm.h"
#include "./proc.h"
#include "../include/segment.h"
#include "../include/error.h"
#include "../include/pid.h"
#include "../include/message.h"   /* for ANY */
#include "./desc.h"
#include "../include/config.h"
#include "./params.h"

#define DEBUG_LOCAL 0
#define USER_QUANTUM 25  /* ticks before rescheduling */

/****************************
 * the main process table ! *
 ****************************/
PUBLIC struct proc_struct proc[NR_TASKS];

PRIVATE struct proc_struct * free_slot;

PRIVATE proc_id 
newpid(void)
     /* rewrite this ! */
     /* generate a new pid */
{
  static struct proc_id cur_pid = {-1};

  cur_pid.local_pid++; return cur_pid;
}

PUBLIC struct proc_struct * 
find_proc(proc_id * s)
{  
  struct proc_struct * p;
  unsigned long fl;

  if (s->local_pid == ANY_PID) 
    return ANY; 
  save_flags(fl);
  cli();
  for (p = proc; p < proc + NR_TASKS; p++) 
    if (p->status != SLOT_FREE && p->pid.local_pid == s->local_pid) {
      restore_flags(fl);
      return p;
    }
  restore_flags(fl);
  return NULL;
}

PUBLIC void print_name(int task_no)
{
  extern struct configuration sys_description[];

  printk("Proc %d ", task_no);
  if (task_no == IDLE_PROC) printk("`idle'");
  else if (task_no == CLOCK_TASK) printk("`clock'");
  else if (task_no == SYS_TASK) printk("`sys'");
  else {
    task_no -= 2;
    if (task_no <= 0 || task_no > INIT_SERVERS) return;
    printk("`%s'", sys_description[task_no].proc_name);
  }
}
  
PUBLIC void show_task(int task_no)
{
  struct proc_struct * p;
  struct tss_struct * ts;
  struct descriptor * ldt;
  extern void show_senders(struct proc_struct *);
  unsigned long fl;

  if (task_no < 0 || task_no > NR_TASKS) return;
  p = proc_addr(task_no);
  save_flags(fl);
  cli();
  if (p->status == SLOT_FREE) {
    printk("Slot %d free\n", task_no);
    restore_flags(fl);
    return;
  }
  ts = &p->tss;
  print_name(task_no);
  printk(" pid %ld status %d priority %d flags 0x%x int %d\n",
	 p->pid.local_pid, p->status, p->priority, p->flags, p->interrupt);
  if (p->status == SENDING) {
    printk("Sends to ");
    print_name(proc_slot(p->send_to));
    printk(" in queue with :");
    show_senders(p->send_to);
  }
  if (p->status == RECEIVING) { 
    printk("Receives from ");
    print_name(proc_slot(p->recv_from));
    printk("\n");
  }
  printk("Times : user %ld sys %ld quant %ld remain %ld\n",
	 p->u_time, p->s_time, p->quantum, p->remained);
#if 0
  printk("Status at last task switch :\n");
#endif
  printk("cs %x, ss %x, ds %x, es %x, fs %x, gs %x, ldt %x, ss0 %x\n",
	 ts->cs, ts->ss, ts->ds, ts->es, ts->fs, ts->gs, ts->ldt, ts->ss0);
  printk("eip %lx, esp %lx, esp0 %lx, eflags %lx\n",
	 ts->eip, ts->esp, ts->esp0, ts->eflags);
  printk("eax %lx, ebx %lx, ecx %lx, edx %lx, esi %lx, edi %lx\n",
	 ts->eax, ts->ebx, ts->ecx, ts->edx, ts->esi, ts->edi);
  if (ts->ldt != NULL_SELECTOR) {
    ldt = (struct descriptor *)get_base(&gdt[LDT_DESCRIPTOR(task_no)]); 
    show_descriptor(&gdt[LDT_DESCRIPTOR(task_no)]);
    show_descriptor(&ldt[0]);
    show_descriptor(&ldt[1]);
    show_descriptor(&ldt[2]);
  }    
  restore_flags(fl);
}

PUBLIC void
init_task(void)
     /* initialize task handling data structures */
{
  struct proc_struct * p;
  
  p = free_slot = proc;         /* free list begins here */
  for (; p < proc + NR_TASKS; p++) {
    p->next_ready = p+1;        /* chain on the free list */
    p->status = SLOT_FREE;      /* this slot is unused */
    p->send_head = p->send_tail = NULL;  /* no senders to me */
    p->next_send = p->prev_send = NULL;  /* i am on no queue */
    p->recv_from = NULL;                 /* I wait no reply */
    p->msg_ptr = NULL;                   /* no message */
  }
  proc[NR_TASKS - 1].next_ready = NULL;  /* end of free list */
}

   /* two functions for manipulating the free slot list */
PRIVATE struct proc_struct * 
get_slot(void)
     /* get a free slot to create a new process */
{
  unsigned long fl;
  struct proc_struct * p;

  save_flags(fl);
  cli();
  p = free_slot;
  if (free_slot != NULL) free_slot = p->next_ready;   /* extract from queue */
    /* chain the slot as free */
#if DEBUG
  if (p->status != SLOT_FREE) 
    panic("Inconsistent free slot list : slot %d", proc_slot(p));
#endif
  p->status = INCOMPLETE;
  restore_flags(fl);
  return p;
}

PRIVATE void 
release_slot(struct proc_struct * p)
     /* this process has exited; release its slot; */
{
  unsigned long fl;

  save_flags(fl);
  cli();
  p->status = SLOT_FREE;
    /* chain the slot as free */
  p->next_ready = free_slot;
  free_slot = p;
  restore_flags(fl);
  p->prev_ready = p->send_head = p->send_tail = 
    p->next_send = p->prev_send = NULL;
  p->recv_from = NULL;
  p->msg_ptr = NULL;
  p->flags = 0;
  p->task_address = 0;
}

PRIVATE inline void 
fill_pad_zero(struct tss_struct * t)
     /* fills with zero the padding fields in a tss struct 
      * They should never change, but who knows ... */
{
  t->__ss0h = t->__ss1h = t->__ss2h = (unsigned short) 0;
  t->__csh=t->__dsh=t->__esh=t->__fsh=t->__gsh=t->__ssh=(unsigned short)0;
  t->__ldth = t->__blh = (unsigned short) 0;
}

PUBLIC int 
make_kernel_task(long * stack_base, 
		 unsigned long stack_size, unsigned long entry)
     /* This function builds a kernel task.
      * It has same segments as the idle task
      * except a separate stack, given as argument. Size in LONGS.
      * Should be big enough, as it has no separate segment.
      * Returns task number.
      */
{
  struct proc_struct * p;
  struct tss_struct * ts;
  int r; 

  p = get_slot();  /* p->status == INCOMPLETE now */
  if (p == NULL) return E_NOSLOT;
  p->pid = newpid();
  p->flags = 0;
  p->priority = KERNEL_PRIO;
  p->interrupt= NO_INT;  /* no interrupt */
  p->send_to = NULL;
  p->next_send = p->prev_send = NULL;
  p->send_head = p->send_tail = NULL;
  p->msg_ptr = NULL;

  p->u_time = p->s_time = 0L;
  p->quantum = p->remained = 0L; /* unused, anyhow by kernel tasks */
  /* set up the tss now */

  ts = &p->tss;
  ts->cs = KERNEL_CS;
  ts->ds = ts->es = ts->fs = KERNEL_DS;
#if DEBUG
  ts->gs = SCREEN_SELECTOR;
#else
  ts->gs = KERNEL_DS;
#endif
  ts->ss = KERNEL_DS;  
  if (stack_base != NULL) 
    ts->esp = (unsigned long) &stack_base[stack_size]; /* stack end ! */
  ts->eflags = 1L << 9;
  ts->eip = (unsigned long) entry;
  ts->ldt = NULL_SELECTOR;

  ts->trace = 0;
  ts->bitmap = 0;

  /* the rest of the tss fields must be 0 */
  ts->ss0 = ts->ss1 = ts->ss2 = NULL_SELECTOR;
  ts->esp0= ts->esp1= ts->esp2= 0L;
  
  fill_pad_zero(ts);

  ts->cr3 = 0L;
  ts->eax = ts->ebx = ts->ecx = ts->edx = 0L;
  ts->ebp = ts->edi = ts->esi = 0L;
  /* that is all ... */

  if ((r = set_tss_descriptor(proc_slot(p))) < 0) {
    release_slot(p);
    return r;
  }
  
  /* finally set up the address used for a far jump to this task
   * to start this task
   */
  p->tr = TSS(proc_slot(p));
  p->k_stk_frame = 0L;
  p->pend_sig = 0;
  for (r = 0; r < SGN_MAP_SIZE; r++) p->sgn_map[r] = 0l;
  
  return proc_slot(p);  /* task no; the caller should make it ready */
}

extern void ret_from_sys_call(void);

PUBLIC int
make_task(struct descriptor ldt_image[], unsigned long ldt_size, 
	  long * k_stk_base, unsigned long k_stk_size, /* in LONGS */
	  unsigned long entry)
     /* create a task having the ldt segment the one given;
	ldt size given in BYTES. Kernel stack in LONGS.
        entry should be relative (in the given code segment) */
{
  struct proc_struct * p;
  struct tss_struct * ts;
  unsigned long stk_size;
  int task_no;
  int r;
  unsigned long * esp;

  /* things proceed awhile like for a kernel task */

  if (ldt_size < 3 * sizeof(struct descriptor)) return E_BADARG;
  if (k_stk_size < MIN_K_STK_SIZE) return E_STKSML;
  stk_size = get_limit(&ldt_image[USER_SS_DESCRIPTOR]);  /* bytes */
  task_no = make_kernel_task(k_stk_base, k_stk_size,
			     (unsigned long)ret_from_sys_call /* entry point */);
  if (task_no < 0) return task_no;  /* error */

        /* start filling user-specific part */
  p = proc_addr(task_no);
  p->quantum = p->remained = USER_QUANTUM; /* change this */
  ts = &p->tss;
  ts->ss0 = KERNEL_DS;         /* this task HAS kernel stack */
  ts->esp0= (unsigned long)&k_stk_base[k_stk_size];
  ts->ldt = LDT(proc_slot(p));
  if ((r = set_ldt_descriptor(task_no, ldt_image, ldt_size, USER_PRIV) < 0)) {
    release_slot(p);
    return r;
  }
  /* 
     Build the initial stack image for the new task.
     This has to match the one expected by sys_call.c.
     ts->esp points absolutely (in kernel DS) to the future kernel stack
   */
  esp = (unsigned long *)ts->esp;
#define PUSHL(v) (*--(esp) = (unsigned long)(v))

  /* push the stack frame for iret : */
  PUSHL(USER_SS);      /* SS of user; there will be a stack change ! */
  PUSHL(stk_size);     /* user stack initial esp */
  PUSHL(1L << 9);      /* eflags = IF */
  PUSHL(USER_CS);      /* code segment */
  PUSHL(entry);        /* eip */
  /* push what ret_from_sys_call pops */
#if DEBUG
  PUSHL(SCREEN_SELECTOR);  /* gs */
#else
  PUSHL(USER_DS);      /* gs */
#endif
  PUSHL(USER_DS);      /* fs */
  PUSHL(USER_DS);      /* es */
  PUSHL(USER_DS);      /* ds */
  PUSHL(0L);           /* ebx */
  PUSHL(0L);           /* ecx */
  PUSHL(0L);           /* edx */
  PUSHL(0L);           /* esi */
  PUSHL(0L);           /* edi */
  PUSHL(0L);           /* ebp */
  PUSHL(0L);           /* eax */
#undef PUSHL
  p->k_stk_frame = ts->esp = (unsigned long)esp;
#if DEBUG_LOCAL
  show_task(task_no);
#endif
  return task_no; /* the caller should make it ready */
}

PUBLIC int
build_ldt(struct descriptor * ldt_base, unsigned long ldt_size, 
	  unsigned long code_base, unsigned long code_size,
	  unsigned long data_base, unsigned long data_size,
	  unsigned long sk_base,   unsigned long sk_size)
  /* 
     Build a ldt containing at least these four segment descriptors.
     The ldt_size is in bytes.  Stack size in bytes.
     Use 3 segments like this :
     code_base      data_base      sk_base        sk_base+sk_size
     |------code-------|
     |<--code_size---->|
     |---------------------------data----------------------------|
     |<--------code_size+data_size+sk_size--(whole_size)-------->|
     |---------------------------stack---------------------------|
     This is necessary due to the facts that :
     - data is compiled at offset (code_size), thus its segment must
     begin at 0, where the code begins;
     - pointers from stack are accessed as data (e.g. in DS) and viceversa,
     thus the stack & data segments have to overlap on their whole length;
     - there's no gap between segments
     All this is the fault of the C compiler.
   */
{
  unsigned long whole_size;

  if (ldt_size < 3 * sizeof(struct descriptor)) return E_OUTSP;
  if (data_base != code_base + code_size) return E_ILL;   
      /* no gap code-data allowed */
  /* the sk_base is always superflous; no checks about it */
  set_descriptor(&ldt_base[USER_CS_DESCRIPTOR], (char *) code_base,
		 code_size, TYPE_CODE_RO, USER_PRIV);
  whole_size = code_size + data_size + sk_size;
  set_descriptor(&ldt_base[USER_DS_DESCRIPTOR], (char *) code_base,
		 whole_size, TYPE_DATA_RW, USER_PRIV);
  set_descriptor(&ldt_base[USER_SS_DESCRIPTOR], (char *) code_base, 
		 whole_size, TYPE_DATA_RW, USER_PRIV);
  return E_OK;
}

PUBLIC int kill_task(struct proc_struct * p)
     /* this task dies */
{
  int err, status;
  extern int unregister_handler(int);
  extern int mess_clean(struct proc_struct *, int oldstatus); /* messages.c */
  struct proc_struct * server;  /* server servicing this process now */
  unsigned long fl;

  if (p < proc || p >= &proc[NR_TASKS]) return E_BADARG;
  save_flags(fl);
  cli();
  status = p->status;                  /* recall it */
  err = unready(proc_slot(p), ZOMBIE); /* scheduler clean-up */
  restore_flags(fl);
  if (err != E_OK) return err;  
       /* this means that this process cannot be killed */
  if (p->interrupt != NO_INT) {
    err = unregister_handler(p->interrupt);
    if (err != E_OK) 
       printk("cannot unregister dying process %d as handler", proc_slot(p));
  }
  server = p->recv_from;
  (void) mess_clean(p, status);  /* message clean-up */
#if 0  /* thinking yet if to implement signaling of client's death */
  if (status == RECEIVING && p->recv_from != ANY) {
    /* process remains zombie.  Chain it to the list of its server's
       zombies
       */
    p->next_ready = server->zombie;
    server->zombie = p;
    server->flags |= DEAD_CLIENT;
    return E_ZOMBIE;             /* slot released later */
  }
#endif
  release_slot(p);               /* slot clean-up; do it now */
  return E_OK;                   /* dead; not ZOMBIE anymore */
}

PUBLIC void show_frame(void)
     /* call this during kernel context, and you show the saved
	registers of the current running process */
{
  struct stack_regs * p;
  unsigned long fl;

  save_flags(fl);
  cli();
  p = (struct stack_regs *)cur_proc_ptr->k_stk_frame;    
  print_name(cur_proc);
  if (p == NULL) {
    printk(" - no kernel stack frame !\n");
    restore_flags(fl);
    return;
  }
  printk(" registers before switch to kernel mode\n");
  printk("eax 0x%lx ebx 0x%lx ecx 0x%lx edx 0x%lx edi 0x%lx esi 0x%lx\n",
	 p->eax, p->ebx, p->ecx, p->edx, p->edi, p->esi);
  printk("cs 0x%x, ss 0x%x, eip 0x%lx, esp 0x%lx, flags 0x%lx ebp 0x%lx\n",
	 (unsigned short)p->cs, (unsigned short)p->ss, p->eip, 
	 p->esp, p->flags, p->ebp);
  restore_flags(fl);
}
