/*
 * sys.c
 * @(#) system task; this is a kernel task
 * (c) 1995 by Mihai Budiu
 */

/*
 *  The system task is the main responsable with the low level system calls
 * It is a server, linked with the kernel.  It ocassionally calls kernel
 * functions.  The functions are numerous, and the replyes are often very 
 * specific
 */

#define __SERVER__

#include "../include/error.h"
#include "../include/globals.h"
#include "../include/segment.h"
#include "./proc.h"
#include "../include/message.h"
#include "../include/operation.h"
#include "../include/pid.h"
#include "./params.h"
#include "../include/server.h"
#include "../include/asm.h"

#define DEBUG_LOCAL 0

PRIVATE struct message 
  msg,   /* here it receives the messages */
  reply; /* here build reply */
extern   void show_task(int);

PRIVATE void init_sys_task(void)
     /* initialize the system task */
{
  /* actually nothing special to do */
}

/* the main routines; parameters extracted from the message */
PRIVATE do_thing_t 
  do_new_proc, do_kill_proc, do_hand_int, do_data,
  do_unhand_int, do_change_prio, do_error, do_signal;

PUBLIC void sys_task(void)
     /* the only entry point of this task */
{
  int err;
  unsigned char answer; /* NO if no answer is needed */

  init_sys_task();   /* initialize myself */

  for (;;)   
    {
      while ((err = receive(&msg)) != E_OK)
	printk("Failed receive for System task : error %d\n", err);
      answer = ANSWER_EXPECTED(msg.flags);
      switch(msg.OPERATION) {
      case NEW_PROC: 
	err = do_new_proc();
	break;
      case KILL_PROC:
	err = do_kill_proc();
	break;
      case HAND_INT:
	err = do_hand_int();
	break;
      case UNHAND_INT:
	err = do_unhand_int();
	break;
      case CH_PRIO:
	err = do_change_prio();
	break;
      case SRV_PANIC:
      case SRV_ERROR:
      case SRV_MESS:
	err = do_error();
	break;
      case GET_DATA:
	if (answer) err = do_data();
	break;
      case DO_SIGNAL:
	err = do_signal();
	break;
      default:
	err = E_BADOP;
      }
      if (answer) {
	err = send_reply(msg.SOURCE.local_pid, msg.MSGID, 
			 msg.OPERATION, err, &reply);
	if (err != E_OK) printk("Sys task can't reply process %ld, error %d",
				reply.DESTINATION.local_pid, err);
      }
    }
}

/*********** implementation of the sys process functions *************/

extern struct proc_struct * find_proc(proc_id * pd);

PRIVATE int do_hand_int(void)
     /* some process (SENDER) wants to handle one interrupt 
      INFOi[0] = interrupt */
{
  extern int register_handler(int, struct proc_struct *);
  extern struct proc_struct * find_proc(proc_id * s);
  int interrupt,             /* interrupt requested */
      result;
  unsigned long fl;
  struct proc_struct * p;  /* process treating it */
  
  save_flags(fl);
  cli();
  p = find_proc(&msg.SOURCE);            /* who handles it */
  if (p == NULL) result = E_NOPID;
  else {
    interrupt = msg.INFOi[0];
#if DEBUG_LOCAL
    printk("registering %d handler for int %d treating %d\n",
	   proc_slot(p), interrupt, p->interrupt);
#endif
    if (p->interrupt != NO_INT) result = E_BUSY;  /* already some handled */
    else result = register_handler(interrupt, p); /* here is the bulk done */
  }
  restore_flags(fl);
  return result;
}

PRIVATE int do_unhand_int(void)
     /* some process gives up handling an interrupt; INFOi[0] = inter */
{
  extern int unregister_handler(int);
  int interrupt;

  interrupt = msg.INFOi[0];
#if DEBUG_LOCAL
  printk("unregistering handler for int %d\n", interrupt);
#endif
  return unregister_handler(interrupt); /* relay on this function */
}

PRIVATE int do_new_proc(void)
{
  return E_OK;
}

PRIVATE int do_kill_proc(void)
     /* one process should die; INFO holds the proc_id */
{
  proc_id * pd;
  struct proc_struct * p;
  extern int kill_task(struct proc_struct * p);
  unsigned long fl;
  int result;

  pd = (proc_id *)&msg.INFO;
  save_flags(fl);
  cli();
  p = find_proc(pd);
  if (p == NULL) return E_NOPID;
#if DEBUG_LOCAL
  printk("killing process %d\n", proc_slot(p));
#endif
  result = kill_task(p);
  restore_flags(fl);
  return result;
}

PRIVATE int do_change_prio(void)
     /* INFOi[0] = newprio */
{
  struct proc_struct * p;
  int new_prio;
  extern int change_priority(int task, int);  /* sched.c */
  int err;
  unsigned long fl;

  save_flags(fl);
  cli();
  p = find_proc(&msg.SOURCE);
  if (p == NULL) return E_NOPID;
  new_prio = msg.INFOi[0];
  err = change_priority(proc_slot(p), new_prio);
#if DEBUG_LOCAL
  printk("Changing priority of process %d - err %d\n", proc_slot(p), err);
#endif
  restore_flags(fl);
  return err;
}

PRIVATE int do_error(void)
     /* some server reports an error; useful mainly for loging
      INFOi[0] = error code; INFOi[1..] = text */
{
    /* for the moment just print some inane message */
  msg.INFOc[sizeof (msg.INFO) - 1] = '\0'; 
  switch(msg.OPERATION) {
  case SRV_ERROR: 
    printk("*SERVER ERROR");
    break;
  case SRV_PANIC:
    printk("*SERVER PANIC");
    break;
  case SRV_MESS:
    printk("*SERVER MESSAGE");
    break;
  }
  printk(" (pid %lu) : %s", (unsigned long)(msg.SOURCE.local_pid), 
	 (char *)&msg.INFOi[1]);
  if (msg.INFOi[0] < E_OK) printk(", error");
  printk(" %d\n", msg.INFOi[0]);
  return E_OK;
}

PRIVATE int do_data(void)
     /* INFOi[0] = what data */
{
  int wdata = msg.INFOi[0];
  extern unsigned char * get_vid_mem_start(void);
  struct screen_info * p = SCREEN_INFOP;
  int i;
  char * s, *d;

  switch (wdata) {
  case SCREEN_DATA:
    /* pack data as follows :
       reply.INFOl[0] = base address of video ram
       reply.INFOl[1..] = struct screen_info */
    reply.INFOl[0] = (long) get_vid_mem_start();
    s = (char *) p;
    d = (char *) &reply.INFOl[1];
    for (i = 0; i < sizeof (struct screen_info); i++, s++, d++)
      *d = *s;      /* copy the structure; there's enough place */
    return E_OK;
  case RAM_DATA:
    break;
  case HD_DATA:
    break;
  case CONFIG_DATA:
    break;
  }
  return E_BADARG;
}

PRIVATE int do_signal(void)
     /* signal a process */
{
  struct proc_struct * p;
  signal_t signal;
  unsigned long fl;
  unsigned long * x;
  unsigned long bit;
  extern void build_signal_message(struct proc_struct * p);

  signal = msg.SIGNAL_ME;
  p = find_proc((proc_id *) &msg.INFO);
  if (p == NULL) return E_NOPID;
  if (signal == 0) return E_ILL;
  x = &p->sgn_map[signal >> LOG_LONG_BITS]; /* byte to change */
  bit = 1ul << (signal & LOG_LONG_BITS);
  save_flags(fl);
  cli();
  if (! (*x & bit)) {
    *x |= bit;     /* new signal */
    p->pend_sig++;
  }
  if (p->status == RECEIVING && p->recv_from == ANY) {
    /* ready this process ! */
    build_signal_message(p);
    ready(proc_slot(p));
  }
  restore_flags(fl);
  return E_OK;
}
