/* 
 * sys_call.c
 * @(#) system call code - kernel mode switch
 * (c) 1995 by Mihai Budiu
 */

#include "../include/asm.h"
#include "../include/segment.h"
#include "../include/globals.h"
#include "../include/error.h"
#include "../include/message.h"
#include "../include/pid.h"
#include "../include/unistd.h"
#include "./proc.h"

/*****************************************************************
 * PUBLIC functions defined here:
 *
 * sys_entry - entered via a trap from a trap gate
 *****************************************************************/ 
#define DEBUG_LOCAL 0
#define DEBUG_COND 0
/* 
   This is the default system call handler;
   it is called through a trap (interrupt) gate, with
   the operation code in eax and with the message pointer in
   ebx. It just saves registers, calls the unique sytem call function, 
   restores registers and returns.  Changes here should be mimicked by
   changes in ../include/unistd.h
*/

#define _STR(c) #c
#define STR(c) _STR(c)

__asm__(	
".globl _sys_entry\n"
".align 4, 0x90\n"
"_sys_entry:\n\t"
	SAVE_REGS         /* last reg. on stack is eax = op. code ! */
	"cli\n\t"
	/* now prepare arguments for the future function call */
	
	"pushl %ebx\n\t"  /* this is the message ptr in user space */
	"pushl %ds\n\t"   /* this is the user space selector */
	
	/* now switch to the kernel segments */	
	
	"movl $" STR(KERNEL_DS) ", %eax\n\t"
	"movw %ax, %ds\n\t"
	"movw %ax, %es\n\t"
	"call _sys_handle\n\t" 	/* call the function */ 

	"addl $12, %esp\n\t"  	/* arguments discarded; also old eax ! */
	"pushl %eax\n\t" /* eax has now the return value from function ! */    
".globl _ret_from_sys_call\n"
"_ret_from_sys_call:\n\t"       
       	RESTORE_REGS 
	"iret"	         /* back to user space; %eax has return value ! */
);

PUBLIC void ret_from_sys_call(void);

PRIVATE inline void 
copy_pid(unsigned short selector, proc_id * user_sr, proc_id * ds)
     /* get the pid of the party to examine */
   /* 
     We transfer the message from the user space pointed by gs:esi
     to the kernel space pointed by es:edi.
     Load thus gs with the given selector, and es with KERNEL_DS
   */
{
   __asm__ __volatile__(
     "push %%es\n\t"
     "push %%gs\n\t"
     "push %%ds\n\t"
     "pop %%es\n\t"
     "movw %%ax, %%gs\n\t"
     "rep; gs; movsb\n\t"
     "pop %%gs\n\t"
     "pop %%es"
     : /* no output */
     : "ax" (selector), "r"(KERNEL_DS), "D" (ds), "S" (user_sr), 
       "c" (sizeof(proc_id))
     : "memory", "%edi", "%esi", "%ecx"
     );
} 

extern struct proc_struct * find_proc(proc_id * p);
    /* process slot with given pid; in task.c */

/* PRIVATE */  /* make it public to have its code generated ! */
int            /* returns error code through eax */
sys_handle(long selector, long message_ptr, long operation)
     /* this function is called to handle system calls by
	the above assembly code; arguments are received al longs
	to simplify things; it might cause process preemption ! */
{
  proc_id party;       /* the other end of the line */
  struct message * m;  /* the user space message */
  int send;            /* I send */
  int receive;         /* I receive */
  struct proc_struct * p;  /* pointer to real party */
  int ret_val;

  if (DEBUG_COND) printk("Sys handle : sel %ld mesg %lx op %ld cur %d\n", 
			 selector, message_ptr, operation, cur_proc);
  send = (operation == SEND || operation == SEND_REC);
  receive = (operation == RECEIVE || operation == SEND_REC);
  m = (struct message *)message_ptr;
  copy_pid((unsigned short)selector, send ? &m->to : &m->from, &party);
  p = find_proc(&party);      /* process slot */
  ret_val = E_BADOP;          /* some initial value = wrong operation */

  if (send) {
#if 0
    if (p == ANY) p = stub_proc;   /* send to another machine ! */
#endif
    if (p == NULL || p == ANY) return E_NOPID;  /* destination not found ! */
    if (DEBUG_COND)
      printk("sys_call : Sending %d to %d\n", cur_proc, proc_slot(p));
    ret_val = k_send(p, (unsigned short)selector, m, 
		     (operation == SEND_REC) ? 1 : 0/* wait_answ flag */);
    if (ret_val != E_OK) return ret_val;
  }    
  if (receive) { /* p may be ANY ! */
    if (DEBUG_COND)
      printk("sys_call : receiving %d from %d\n", cur_proc, proc_slot(p));
    ret_val = k_receive(p, (unsigned short)selector, m);
  }
  if (DEBUG_COND)
    printk("sys_call : proc %d completed with %ld code\n", cur_proc, ret_val);
  return ret_val;
}
