/*
 * kernel/sched.c
 * @(#) PicOs process scheduling
 * (c) 1995 by Mihai Budiu
 */

#include "../include/globals.h"
#include "./proc.h"
#include "../include/error.h"
#include "../include/limits.h"
#include "../include/pid.h"
#include "../include/asm.h"

/**********************************************************************
 * PUBLIC functions of the scheduler :
 *
 * init_sched - initialization; called only once
 * ready(task_no) - the task in this slot is ready
 * unready(task_no, reason) - reverse of above
 * change_priority(task_no, prio) - ditto
 * next_ready() - who should run next ?
 * show_queues() - dumps scheduler queues for debug
 * reschedule(task_no) - this task has run too long
 * switch_to(task_no) - run this task
 **********************************************************************/

#define DEBUG_LOCAL 0
#define DEBUG_COND 0
/* This file contains all code for process scheduling */

/* queues of processes on priorities heads & tails */
PRIVATE struct proc_struct 
  * head[PRIO_LEVELS], 
  * tail[PRIO_LEVELS];

PUBLIC struct proc_struct *cur_proc_ptr;   /*current process pointer */ 
PUBLIC struct proc_struct *prev_proc_ptr;  /*previous proc.interrupted*/
PUBLIC struct proc_struct *bill_ptr;       /* process which pays now */
PUBLIC int cur_proc;

PUBLIC void 
init_sched(void)     /* initialize scheduler */
{
  int i;
    /* null priority queues */
  for (i=0; i < PRIO_LEVELS; i++) {
    head[i] = NULL;
    tail[i] = NULL;
  }
#if DEBUG_LOCAL
  printk("Scheduler initialized.\n");
#endif
}

/* macros for priority queues handling */
#define READY(p) \
do { \
    int prio = p->priority; \
    p->prev_ready = tail[prio]; \
    if (tail[prio] == NULL) head[prio] = p; \
    else tail[prio]->next_ready = p; \
    tail[prio] = p; \
    p->next_ready = NULL; \
    p->status = READY; \
} while (0)

#define UNREADY(p, newstatus) \
do {\
     int prio = p->priority; \
     p->status = newstatus; \
     if (p->prev_ready != NULL) p->prev_ready->next_ready = p->next_ready; \
     else head[prio] = p->next_ready; \
     if (p->next_ready != NULL) p->next_ready->prev_ready = p->prev_ready; \
     else tail[prio] = p->prev_ready; \
     p->prev_ready = p->next_ready = NULL; \
} while (0)

PUBLIC int 
ready(int task_no)
     /* make the indicated process ready to run */
     /* the slot SHOULD NEVER be on the free list too ! */
{
  int prio;
  unsigned long fl;
  struct proc_struct * p;

  if (task_no < 0 || task_no >= NR_TASKS) return E_BADARG;
  p = proc_addr(task_no);
  save_flags(fl);
  cli();
#if DEBUG
  if (p->status == READY) panic("ready : bad initial task %d status", task_no);
#endif
  prio = p->priority;
  if (prio < 0 || prio >= PRIO_LEVELS) 
    panic("Bad priority for task %d", task_no);
  /* chain the process at the end of the doubly linked list */
  if (DEBUG_COND)
    printk(" %d readyed by %d # ", task_no, cur_proc);
  READY(p);
  restore_flags(fl);
  return E_OK;
}
  
PUBLIC int 
unready(int task_no, int newstatus)
     /* for some reason (newstatus) this process is blocked.
	returns error ONLY if this process *cannot* be unreadyed */
{
  unsigned long fl;
  struct proc_struct * p;

  if (task_no < 0 || task_no >= NR_TASKS) return E_BADARG;
  if (newstatus == READY) return E_BADARG;
  if (task_no == IDLE_PROC) return E_BADARG;   /* Idle always awake */
  p = proc_addr(task_no);
  
  save_flags(fl);
  cli();
#if DEBUG
  if (p->priority >= PRIO_LEVELS) 
    panic("unready : bad priority level for task %d", task_no);
#endif
  if (p->status != READY) p->status = newstatus;
  else UNREADY(p, newstatus);  /* also have to dequeue it */
  restore_flags(fl);
  return E_OK;
}

PUBLIC int change_priority(int task_no, int newprio)
     /* return error code */
{
  int oldprio;
  unsigned long fl;
  struct proc_struct * p;
  
  if (task_no < 0 || task_no >= NR_TASKS) return E_BADARG;
  if (newprio < 0 || newprio >= PRIO_LEVELS) return E_BADARG;
  if (task_no == IDLE_PROC && newprio < THRESHOLD_PRIO) return E_ILL;
  p = proc_addr(task_no);
  save_flags(fl);
  cli();
  if (newprio >= THRESHOLD_PRIO && p->interrupt != NO_INT) {
    restore_flags(fl);
    return E_BUSY;  /* a handler cannot have a too low priority ! */
  }
  oldprio = p->priority;
  if (p->status != READY) {
    p->priority = newprio; /* that's all I have to do */
    restore_flags(fl);     /* don't forget this ! */
    return E_OK;           /* done */
  } 
   /* the task is READY */
  UNREADY(p, INCOMPLETE); 
  p->priority = newprio;
  READY(p);             /* task WAS ready; but now on a new queue */
  restore_flags(fl);
  return E_OK;
}

PUBLIC int 
next_ready(void)
     /* get next process ready to run with highest priority */
{
  int i;
  unsigned long fl;
    /* scan all queues in decreasing priority */

  save_flags(fl);
  cli();
  for (i=0; i < PRIO_LEVELS; i++) if (head[i] != NULL) break;
  /* this cannot be null, as on the last queue IDLE lyes forever ready */
#if DEBUG
  if (i == PRIO_LEVELS) panic("%d tasks ready", 0);
#endif
  i = proc_slot(head[i]);  /* this is first ready task */
  restore_flags(fl);
  return i;
}

#if DEBUG
extern void print_name(int);

PUBLIC void 
show_queue(int prio)
     /* dump a queue's contents */
{
  struct proc_struct * p;
  unsigned long fl;

  save_flags(fl);
  cli();
  p = head[prio];
  printk("*** Queue %d, tasks : ", prio);
  while (p != NULL) {
    print_name(proc_slot(p));
    p = p->next_ready;
  }
  printk("\n");
  restore_flags(fl);
}

PUBLIC void 
show_queues(void)
     /* dump all queues */
{
  int i;
  for (i=0; i<PRIO_LEVELS; i++) show_queue(i);
}
#endif

PUBLIC int 
reschedule(int task_no)
     /* The indicated task has run enough (him or user processes
      * on behalf of him).  Move it to the end of its queue.
      * Called only by the timer interrupt. Return success status */
{
  unsigned long fl;
  struct proc_struct * p;

  if (task_no < 0 || task_no >= NR_TASKS) return E_BADARG;
  p = proc_addr(task_no);
  save_flags(fl);
  cli();
  if (p->priority > THRESHOLD_PRIO) {
    /* only such a process is preempted */
#if DEBUG_LOCAL
    printk("Rescheduling task %d\n", task_no);
#endif
    UNREADY(p, INCOMPLETE);
    READY(p);                /* these move it to the end of its queue */
    restore_flags(fl);
    return E_OK;
  }
  restore_flags(fl);             /* cannot preempt such a process ! */
  return E_ILL;                  
}

PUBLIC void 
switch_to(int task_no)
     /* this routine does the actual task switch;
	It updates cur_task_ptr and cur_task     */
{
  unsigned long fl;

  save_flags(fl);
  cli();
  if (task_no == cur_proc) {
    restore_flags(fl);
    return;  /* no switch ! */
  }
#if DEBUG  /* extra checking */
  if (proc_slot(cur_proc_ptr) != cur_proc) 
    panic("Inconsistent cur_proc %d", cur_proc);
#endif
  if (proc_addr(task_no)->status != READY) 
    panic("Switching to task %d not ready", task_no);
#if DEBUG_LOCAL
  printk("Switching from %d to %d\n", cur_proc, task_no);
  show_queues();
#endif
  cur_proc = task_no;
  prev_proc_ptr = cur_proc_ptr;
    /* 
       this is used by the clock task, to know which process it just
       preempted, in order to make propper accounting
     */
  cur_proc_ptr = proc_addr(cur_proc);
  if (cur_proc_ptr->priority > THRESHOLD_PRIO) /* this is a user process */
    bill_ptr = cur_proc_ptr;   /* so it pays for everything */
  jmp_far(&cur_proc_ptr->task_address);
  /* this is equivalent with
     EIP = cur_proc_ptr->task_address;
     CS  = cur_proc_ptr->tr;
     which in fact is a jump to a TSS segment (the tr is a selector
     for the task's TSS).  Thus this is a task switch !
  */
  restore_flags(fl);
  /* This function completes when somebody runs the current task again ! */
}
