/* 
 * proc.h
 * @(#) process table contents & other data structures
 * (c) 1995 by Mihai Budiu
 */

#ifndef _PROC_H
#define _PROC_H

/* 
 * This file holds practically all
 * the data structure & function declarations for the kernel's globals
 */

#include "../include/config.h"
#include "../include/pid.h"
#include "../include/message.h"
#include "../include/globals.h"
#include "./desc.h"

typedef int PRE_INT(struct message *);
typedef int (*PRE_INT_P)(struct message *);
/* type of function pre-processing an interrupt */

/* the next assumes a long has 4 bytes, a short 2 */
struct tss_struct {
  unsigned short back_link,__blh;  /* __xxx means 0, for padding */
  unsigned long	 esp0;
  unsigned short ss0,__ss0h;
  unsigned long	 esp1;
  unsigned short ss1,__ss1h;
  unsigned long  esp2;
  unsigned short ss2,__ss2h;
  unsigned long	 cr3;
  unsigned long  eip;
  unsigned long	 eflags;
  unsigned long	 eax,ecx,edx,ebx;
  unsigned long	 esp;
  unsigned long	 ebp;
  unsigned long	 esi;
  unsigned long	 edi;
  unsigned short es, __esh;
  unsigned short cs, __csh;
  unsigned short ss, __ssh;
  unsigned short ds, __dsh;
  unsigned short fs, __fsh;
  unsigned short gs, __gsh;
  unsigned short ldt, __ldth;
  unsigned short trace, bitmap;
/*
  unsigned long	 io_bitmap[IO_BITMAP_SIZE+1];  
*/
};

struct proc_struct {
  unsigned char status;   /* ready, blocked, etc. */
  unsigned char priority; /* current prio (0-PRIO_LEVELS-1) */
  short interrupt;        /* interrupt handled by this task (NO_INT if none) */
  unsigned char flags;    /* pending int, etc. */
  signal_t pend_sig;      /* no. of pending signals */
#define NO_INT -1

    /* time information */
  unsigned long u_time;   /* user time run */
  unsigned long s_time;   /* system time run */
  unsigned long quantum;  /* how much time between two preemptions */
  long remained;          /* how much time before next preemption */

  struct tss_struct tss;  /* registers and all the stuff */
  struct proc_id 
    pid,                  /* process identifier */
    ppid;                 /* father; unused yet */
  struct proc_struct 
    * zombies,     /* dead clients chained here */
    * send_to,     /* process waited to receive (a stub, maybe) 
		     but on this machine, anyhow (so I may use a *) */
    * next_ready,  /* list of ready processes || 
		      list of free proc_struct slots ||
		      list of zombies of a server (not used yet) */
    * prev_ready,    /* the list is doubly linked */
    * next_send,     /* me as sender in queue */
    * prev_send,     /* also two-way linked */
    * send_head,     /* head of the list of senders to me */
    * send_tail,     /* tail of the list of senders to me */
    * recv_from;     /* process I am waiting to reply */
  struct message msg_copy;  /* messages buffered in kernel 
			     * this is necessary to avoid
			     * simultaneous access to
			     * two different user spaces */
  struct message * msg_ptr; /* the receiver has here a message 
			       (in kernel space) */
  unsigned long k_stk_frame;  /*address of the stack frame in kernel stk*/
  unsigned long task_address;
    /* this precedes the tr, as a switch_to(this task) makes a far jump to 
       the address held in this pair of words (task_address, tr);
       Its value does not matter, as switch through a TSS ignores offset.
     */
  unsigned long tr;  /* the value of the task register TR for this task */
#define SGN_MAP_SIZE (sizeof(signal_t) << 3)
  /* << 3 because of long = 32 bits = 2**8 - 2**3 */
  unsigned long sgn_map[SGN_MAP_SIZE]; 
    /* signal bit map */ 
#define LOG_LONG_BITS 5
  /* change this if long is not 4 bytes ! */
};

/* process status values */
enum { 
  READY,     /* process is ready to run */
  SENDING,   /* process is blocked sending */
  RECEIVING, /* process is blocked receiving */
  INCOMPLETE,/* this slot is ocupied, but process not ready */
  SLOT_FREE, /* here lyes no process */
  SLEEPING,  /* sleeps */
  ZOMBIE     /* this task is dead but not yet gone */
};

/* process flags values */
enum {
  PENDING_INT = 01,   /* there is a pending interrupt for this process */
  MESG_ERROR = 02,    /* an error occured during message transmission */
  DEAD_CLIENT = 04,   /* a client waiting for this server died */
};

/* space in longs for the kernel stack (becomes stack of the idle task) 
   it also is the size of the kernel tasks stacks ! */
#define KERNEL_STK_SIZE 1024   /* LONGS */
#define MM_STK_SIZE     1024   /* LONGS */
#define MIN_K_STK_SIZE   256   /* minimum kernel stack size (in LONGS) */

#define IDLE_PROC  0     /* this slot belongs to the idle process */
#define SYS_TASK   1
#define CLOCK_TASK 2

/*****************(some) kernel PUBLIC prototypes *****************/

/* the following are defined in sched.c */

extern struct proc_struct proc[];           /* process table */
extern struct proc_struct * cur_proc_ptr;   /* current process */
extern struct proc_struct * bill_ptr;     /* process which pays now */
extern struct proc_struct * prev_proc_ptr; /* prev. running proc */
extern int cur_proc;                       /* slot nr of crt. proc */

#define proc_addr(n) (&proc[(n)])
#define proc_slot(p) ( (p) - proc)

/* these return the success status */
int ready(int task_no);               /* this task ready to run */
int unready(int task_no, int status); /* task not ready */
int change_priority(int task_no, int newprio);
int reschedule(int task_no);                  /* this has run enough */
void switch_to(int task_no);                /* start this new task */
int next_ready(void);                       /* who's first ready ? */

/* a function from interrupt.c */
int send_pending_int(struct proc_struct * p);
  /* send the interrupt pending for the task p; return success status */

/* from misc.c */
int printk(const char *, ...);

/* from task.c */
void show_frame(void);  /* show registers saved during an interrupt */
#endif
