/* 
 * messages.c
 * @(#) implement the message passing
 * (c) 1995 by Mihai Budiu
 */

/* 
 * The message handling procedures are called via a trap/inter. gate.
 * They have to fetch their parameters from the user space.
 * (this is in file sys_call.c).
 * In order not to access simultaneously two different user spaces
 * (somehow a difficult task),
 * the message itself is temporary buffered in kernel memory, 
 * in the proc table (field msgcopy). The buffering is like follows :
 * o if a sender has to block waiting for a receiver the message
 *   is copyed to the sender's buffer. When the receiver executes,
 *   the message is directly fetched from that buffer (in kernel space)
 *   to the receiver (user space) in whose context all this happens.
 * o if a receiver has to block for a sender, when the sender arrives
 *   the message is moved to the receiver's buffer. The sender continues
 *   to execute. 
 * In both cases the process arriving second at the rendez-vous continues
 * execution.
 *
 *  The senders are queued in the order of appearance at the expected
 * destination of their messages.  They use a doubly linked list with
 * head and tail in the destination structure, and using the links
 * prev_send & next_send of the senders.
 *  On the other hand a waiting receiver just points (recv_from) to the 
 * expected sender (or to ANY if any sender is to be accepted).
 *  Signals are patched in the messages received from ANY.  If no message is
 * available the kernel builds one.
 *
 *  This file contains two PUBLIC functions (except k_send & k_receive)
 * that are needed by the interrupt handling upon
 * interrupts, to check for receiver status and to transmit it the message
 * (can_send_to and give_message).
 */

#include "../include/limits.h"
#include "../include/globals.h"
#include "../include/message.h"
#include "./proc.h"
#include "../include/asm.h"
#include "../include/error.h"
#include "../include/operation.h"  /* for SIGNAL */

#define DEBUG_MESG 0
#define DEBUG_COND 0

/************************ send part *********************/

PRIVATE inline void 
copy_message_from_user(unsigned short selector,
		       struct message * user_src,
		       struct message * dest)
   /* the user_src + selector is a pointer 
      in the current user's context ! */
{
  /* 
     We transfer the message from the user space pointed by gs:esi
     to the kernel space pointed by es:edi.
     Load thus gs with the given selector, and es with KERNEL_DS
   */
  __asm__ __volatile__ 
    (
     "push %%es\n\t"
     "push %%gs\n\t"
     "push %%ds\n\t"
     "pop  %%es\n\t"
     "movw %%ax, %%gs\n\t"
     "rep; gs; movsb\n\t"
     "pop %%gs\n\t"
     "pop %%es"
     : /* no output */
     : "ax" (selector), "D" (dest), "S" (user_src), 
       "c" (sizeof(struct message))
     : "memory", "%edi", "%esi", "%ecx"
     );
} 

PUBLIC int 
can_send_to(struct proc_struct * me, 
	    struct proc_struct * destination)
     /* true if me is waited by the receiving destination 
	this should be better called in a critical region ! 
	This function is called by k_send (context of me process) and
	interrupt (any context) */
{
  if (destination->status != RECEIVING) return 0;
  if (destination->recv_from == ANY ||
      destination->recv_from == me) return 1;
  return 0;
}

#define QUEUE(p, dest) \
do {  \
  p->send_to = dest; \
  if (dest->send_tail != NULL) dest->send_tail->next_send = p; \
  else { \
    if (dest->send_head != NULL) \
      panic("Bad link list for senders to %d\n", proc_slot(dest)); \
    dest->send_head = p;  \
  } \
  p->prev_send = dest->send_tail; \
  p->next_send = NULL; \
  dest->send_tail = p; \
} while (0)

#define DEQUEUE(p) \
do {  \
  struct proc_struct * dest = p->send_to; \
	\
  if (dest < proc || dest >= proc + NR_TASKS) \
    panic("Bad destination for sending for %d\n", proc_slot(p)); \
  if (dest->send_head == p) dest->send_head = p->next_send; \
  else { \
    if (p->prev_send < proc || p->prev_send >= proc+NR_TASKS) \
      panic("Bad sender links for %d\n", proc_slot(p)); \
    p->prev_send->next_send = p->next_send; \
  } \
  if (dest->send_tail == p) dest->send_tail = p->prev_send; \
  else { \
    if (p->next_send < proc || p->next_send >= proc+NR_TASKS) \
      panic("Bad sender links for %d\n", proc_slot(p)); \
    p->next_send->prev_send = p->prev_send; \
  } \
  p->prev_send = p->next_send = p->send_to = NULL; \
} while (0)

PRIVATE inline void 
block_to_send(struct proc_struct * dest)
     /* the current process blocks as sender to dest */
{
#if DEBUG
  if (dest == NULL) panic("Bad destination for sending by %d", cur_proc);
#endif
  if (unready(cur_proc, SENDING) != E_OK) return;
     /* could not unready myself ! */
  QUEUE(cur_proc_ptr, dest);
  if (DEBUG_COND) printk("%d blocks to send to %d #", 
			 cur_proc, proc_slot(dest));
  switch_to(next_ready());                    /* run somebody else */
  /* now I am ready and dequeued by the receiver ! */
#if DEBUG
  if (DEBUG_COND) printk("%d unblocks to send to %d #", 
			 cur_proc, proc_slot(dest));
  if (cur_proc_ptr->status != READY) 
    panic("k_send completes with non ready process %d", cur_proc);
#endif
}

PUBLIC void 
give_message(struct proc_struct * receiver, unsigned short selector, 
	     struct message * src, struct message * dest)
     /* 
	This function moves the whole message & readies the receiver 
	process if needed (which might not sleep
	if this function is called from block_to_receive).
	It is called by k_send (context of sender process) &
	interrupt (any context !) to send an interrupt message. 
	The receiver is GUARANTEED to wait for the message 
	Called inside critical region !
      */
{
  copy_message_from_user(selector, src, dest);
  receiver->msg_ptr = dest;
  /* 
     In case of call from k_send, this picks message from user space(selector);
     In case of call from interrupt, the selector points to kernel space;
     In case of call from send_pending_int, the selector points to kernel spc;
     */
#if DEBUG
  if (receiver->status == SLOT_FREE || 
      receiver->status == INCOMPLETE ||
      receiver->status == ZOMBIE)
    panic("Moving message to task %d with bad status", proc_slot(receiver));
#endif
  if (receiver->status != READY) ready(proc_slot(receiver));
}

PUBLIC int k_send(struct proc_struct * dest, unsigned short selector, 
		 struct message * user_msgbuf, int wait_answ)
     /* internal fast version of send; cur_proc sends to dest.
	it uses process pointers instead of pids;
	the second pointer is in USER space (selector+msg*);
	if an answer will be waited must tell now */
{
  unsigned long fl;
  unsigned char mflg;  /* build here the ronly message flags */

  if (dest == NULL || dest == ANY ||   /* cannot send to ANY */
      dest->status == SLOT_FREE ||
      dest->status == ZOMBIE) return E_NOPID;
          /* prepare the message flags */
  mflg = (wait_answ ? WAIT_ANSW : 0);  /* this is SEND_RECEIVE */
#if 0
  if (dest->flags & SIGNALS)    /* kernel has some signals to send */
    mflg |= SIGNAL;
#endif
  save_flags(fl);
  cli();     /* delicate operations ! */
  if (dest == cur_proc_ptr) {
    restore_flags(fl);
    return E_ILL;    /* cannot send to myself */
  }
  if (cur_proc_ptr->priority <= SERVER_PRIO) mflg |= FROM_SERVER;
          /* the originator is a server */
  if (can_send_to(cur_proc_ptr, dest)) {  
    /* dest is blocked receiving -
       the message is moved to the receiver user-space buffer; 
       let crt_proc run; give_message makes the dest ready */
    give_message(dest, selector, user_msgbuf, &dest->msg_copy);
    dest->msg_ptr->SOURCE = cur_proc_ptr->pid;  /* remember this ! */
    dest->msg_ptr->flags = (dest->msg_ptr->flags & MSG_RWMASK) | mflg;
  }
  else {     /* block sender */
       /* copy message in the sender's buffer waiting for receiver */
    copy_message_from_user(selector, user_msgbuf, 
			   &cur_proc_ptr->msg_copy);  
    cur_proc_ptr->msg_copy.SOURCE = cur_proc_ptr->pid;  /* remember this */
    cur_proc_ptr->msg_copy.flags = 
      (cur_proc_ptr->msg_copy.flags & MSG_RWMASK) | mflg;
    cur_proc_ptr->flags &= ~MESG_ERROR;
    block_to_send(dest);
      /* this returns only when dest receives or dies */
    if (cur_proc_ptr->flags & MESG_ERROR) {  /* dest dead ! */
      restore_flags(fl);
      return E_NOPID;
    }
  }
  /* at this point the message is sent - i.e. it is in some kernel buffer */
  if (DEBUG_COND) printk("%d completes send to %d #",
			 cur_proc, proc_slot(dest));
  restore_flags(fl);
  return E_OK;
}

/******************** receive part *************************/

PRIVATE inline void 
copy_message_to_user(struct message * src, 
		     unsigned short selector,
		     struct message * user_dest)
     /* the user_dest is a pointer in the current user's context */
{
  /* We transfer a message to the user space pointed by es:edi
     from the kernel space pointed by ds:edi.
     Load thus es with the given selector.
   */
  __asm__ __volatile__ 
    (
     "push %%es\n\t"
     "movw %%ax, %%es\n\t"
     "rep; movsb\n\t"
     "pop %%es"
     : /* no output */
     : "ax" (selector), "D" (user_dest), "S" (src),
       "c" (sizeof(struct message))
     : "memory", "%edi", "%esi", "%ecx"
     );
}

PRIVATE inline int 
can_receive_now(struct proc_struct * source)
     /* true if current task is waited by the sending source;
	or if the source is ANY and there are some senders
	call this inside a critical region */
{
  if (source == ANY) 
    return ((cur_proc_ptr->send_head != NULL) ||
	    (cur_proc_ptr->flags & PENDING_INT) ||
	    (cur_proc_ptr->pend_sig != 0));

  /* source != ANY */
  if (source->status != SENDING) return 0;
  if (source->send_to == cur_proc_ptr) return 1;
  return 0;
}

PRIVATE inline 
void block_to_receive(struct proc_struct * from)
     /* the current process is blocking waiting for that sender.
	This function causes a context switch. */
{
  cur_proc_ptr->recv_from = from;
  if (DEBUG_COND) printk("%d blocks to rec from %d #", 
			 cur_proc, proc_slot(from));
  if (unready(cur_proc, RECEIVING) == E_OK) switch_to(next_ready());   
                /* find me somebody to run */
  else panic("process %d cannot unready to receive", cur_proc);
  if (cur_proc_ptr->status != READY) 
    panic("k_receive completed for nonready process %d", cur_proc);
  cur_proc_ptr->recv_from = NULL;    /* some clean-up */
  if (DEBUG_COND) printk("%d unblocks to rec from %d #", 
			 cur_proc, proc_slot(from));
}

PUBLIC void
build_signal_message(struct proc_struct * p)
     /* send the signal no to the RECEIVING from ANY process p */
{
  p->msg_ptr = &p->msg_copy;
  p->msg_copy.flags = 0;
  p->msg_copy.SOURCE.local_pid = IDLE_PID;
  p->msg_copy.OPERATION = SIGNAL;
}

PUBLIC int 
k_receive(struct proc_struct * from, 
	  unsigned short selector,
	  struct message * user_msgbuf)
     /* internal receive; selector + user_msgbuf is a (far) address */
{
  unsigned long fl;
  void dump_mess(struct message *);
  signal_t signal;
  int err;

  if (from != NULL && (from->status == SLOT_FREE ||
		       from->status == ZOMBIE ||
		       from->status == INCOMPLETE)) return E_NOPID;
  save_flags(fl);
  cli();
  if (DEBUG_COND) printk("%d starts rec from %d #",
			 cur_proc, proc_slot(from));
  if (from == cur_proc_ptr) {
    restore_flags(fl);
    return E_ILL;
  }
  if (can_receive_now(from)) { 
    if (DEBUG_COND) printk("%d can receive now :", cur_proc);
    if (cur_proc_ptr->flags & PENDING_INT && from == ANY) {
      if (DEBUG_COND) 
	printk(" a pending int (%d)\n", cur_proc_ptr->interrupt);
      if ((err = send_pending_int(cur_proc_ptr)) == E_OK) 
	/* this puts interrupt message in msg_copy of cur_proc. */
	cur_proc_ptr->msg_ptr = &cur_proc_ptr->msg_copy;
      else panic("could not send pending interrupt, err %d", err);
    }
    else if (cur_proc_ptr->send_head == NULL) {
	/* no sender, but signal; build a fake message ! */
#if DEBUG
      if (cur_proc_ptr->pend_sig == 0)
	panic("Process %d receives no message !", cur_proc);
      if (from != ANY) 
	panic("Signal sent on receive from %d", proc_slot(from));
#endif
      build_signal_message(cur_proc_ptr);
    }
    else {    /* the message is in the sender buffer */
      if (from == ANY) from = cur_proc_ptr->send_head;  /* sender pointer */
      cur_proc_ptr->msg_ptr = &from->msg_copy; 
      if (DEBUG_COND) printk(" from = %d ...", proc_slot(from));
      if (from->send_to != cur_proc_ptr)
	panic("Inconsistent send list for %d", proc_slot(from));
      if (from <= proc || from >= proc + NR_TASKS) 
	panic("Bad source at receiving (%d)", proc_slot(from));
      DEQUEUE(from);   /* dequeue from the senders' list */
      if (DEBUG_COND) printk(" now readyes srce #");
      if (ready(proc_slot(from)) != E_OK)
	panic("Could not ready %d", proc_slot(from));
    }
  }

  else {  /* block. the message will be put in *RECEIVER* (my) buffer */
    cur_proc_ptr->flags &= ~MESG_ERROR;
    cur_proc_ptr->msg_ptr = &cur_proc_ptr->msg_copy;
    block_to_receive(from);  
    /* Wake up because
       - a good message came
       - waited source died
       - a signal came
       When source dies it sets my flags to MESG_ERROR to signal this ! */

    if (DEBUG_COND) printk("%d unblocked to rec # ", cur_proc);
    if (cur_proc_ptr->flags & MESG_ERROR) {
      restore_flags(fl);
      if (DEBUG_COND) printk("message error for %d; done\n", cur_proc);
      return E_NOPID;
    }
  }
  /* when arrived here the message exists at cur_proc_ptr->msg_ptr; 
     take it from kernel to user space ! */
  if (DEBUG_COND) {
    if (cur_proc_ptr->msg_ptr == &cur_proc_ptr->msg_copy)
      printk("MY buffer ! # ");
    else printk("SOURCE buffer ! # ");
  }
  /* see if signals are to be sent */
  signal = 0;                   /* no signal */
  if (cur_proc_ptr->pend_sig && from == ANY) {
    /* signals are not sent if receiving from smbdy */
    unsigned long * s;
    unsigned long bit;
    /* search for the first signal bit present */
    
    for (s = cur_proc_ptr->sgn_map; *s == 0; s++)
      signal += 8 * sizeof(long); /* search a bit */
#if DEBUG
    if (s >= &cur_proc_ptr->sgn_map[SGN_MAP_SIZE])
      panic("Bad signal map for proc %d", cur_proc);
#endif
    for (bit = 1; *s & bit; bit <<= 1)
      signal++;                   /* search THE bit */
    *s &= ~bit;
    cur_proc_ptr->pend_sig--;
  }
  cur_proc_ptr->msg_ptr->SIGNAL_NO = signal;
  copy_message_to_user(cur_proc_ptr->msg_ptr /* here is the message */
		       ,selector, user_msgbuf /* here it goes */);
  if (DEBUG_COND) {
    printk("%d completes rec : # ", cur_proc);
    dump_mess(cur_proc_ptr->msg_ptr);
  }
  restore_flags(fl);
  return E_OK;
}

PUBLIC void 
show_senders(struct proc_struct * p)
     /* dump message related data structures */
{
  struct proc_struct * s;
  extern void print_name(int);

  if ((s = p->send_head) == NULL) printk("No senders\n");
  while (s != NULL) {
    print_name(proc_slot(s));
    if (s->send_to != p) 
      printk("\n* Inconsistent send list for %d : %d sends to %d\n", 
	     proc_slot(p), proc_slot(s), proc_slot(s->send_to));
    s = s->next_send;
    if (s != NULL && s->prev_send->next_send != s)
      printk("\n* Bad linked list for senders to %d\n", proc_slot(p));
  }
  printk("\n");
}

PUBLIC int mess_clean(struct proc_struct * p, int status)
     /* called when p dies.  Remove it from all queues.  p->status
	is already changed; rely on status and not on p->status */
{
  struct proc_struct * peer;
  int i;
  
#if DEBUG
  if (p == NULL) panic("Bad argument to dequeue (%d)", 0);
#endif
  if (status == SENDING) DEQUEUE(p); /* remove it from the destination queue */

  /* have to announce all senders to it that destination is dead */
  for (peer = p->send_head; peer != NULL; peer = peer->next_send) {
    DEQUEUE(peer);
    peer->flags |= MESG_ERROR;
    (void) ready(proc_slot(peer));  /* answer unusable anyhow */
  }

  /* announce all receivers from it that source is dead; no better way than
     than to search the whole process table : */
  for (i=0, peer = proc; i < NR_TASKS; i++, peer++) 
    if (peer->status == RECEIVING && peer->recv_from == p) {
      peer->flags |= MESG_ERROR;    /* signal source is dead */
      (void) ready(i);              /* no more reasons to wait */
    }
  return E_OK;
}

PUBLIC void dump_mess(struct message * m)
     /* show a message's contents; in kernel space.
	This depends strongly on the message structure, unlike the other
	kernel functions.  It is still useful for debugging. */
{
  printk("Message %d from pid %ld to pid %ld flags %#x error %d op %d # ",
	 m->MSGID,m->SOURCE,m->DESTINATION, m->flags, m->ERROR, m->OPERATION);
#if 0
  printk("data ");
  for (i = 0; i < sizeof (m->INFOc); i++) printk("%x ", m->INFOc[i]);
  printk("\n");
#endif
}
