/*
 * main.c
 * (c) 1995 by Mihai Budiu
 * #(@) "start the ball rolling" (A.S.T. quoted)
 */

/*
 * The kernel actually does not do many things;
 * It consists of :
 * - a very simple scheduler : sched.c
 * - two message handling routines : message.c
 * - code to convert interrupts into messages :interrupts.c trap.c
 * - descriptor handling : desc.c
 * - printing : misc.c
 * - some data structures that will be passed to various drivers (params.h)
 */

#define DEBUG_LOCAL 0

#include "../include/config.h"
#include "../include/globals.h"
#include "../include/limits.h"
#include "../include/segment.h"
#include "./proc.h"
#include "../include/asm.h"
#include "../include/error.h"
#include "./params.h"
#include "../include/unistd.h"	/* for SYS_CALL_INT */

PUBLIC union prm save_boot_params;
/*
 * Here boot/head.S saved the boot parameters
 * detected by boot/setup.S
 * (and some other information); they are fetched by various kernel parts;
 * declaration in ./params.h, but it is hardcoded in boot/setup.S
 */

PRIVATE unsigned long kstack[KERNEL_STK_SIZE];
/*
 * sometimes we'll have to change this to a separate segment
 * to catch stack faults; KERNEL_STK_SIZE defined in proc.h
 */

PUBLIC unsigned long x86;	/* processor type - set by head.S */
PUBLIC unsigned long hard_math;	/* FPU emulation. Actually not used */

PUBLIC struct
  {
    unsigned long *a;
    unsigned short b;
  }

k_stack_start =
{
  &kstack[KERNEL_STK_SIZE],	/* end element of the big array */
  KERNEL_DS			/* kernel data segment */
};

void init_puts (void);
void init_sched (void);
void init_task (void);
void init_interrupts (void);
void init_traps (void);
void sys_entry (void);
int build_ldt (struct descriptor *ldt, unsigned long,
	       unsigned long code, unsigned long,
	       unsigned long data, unsigned long,
	       unsigned long stack, unsigned long);
int set_sys_call_gate (int int_no, void (*handler) (void));
int make_kernel_task (long *stack, unsigned long size, unsigned long entry);
int make_task (struct descriptor *ldt, unsigned long ldt_size, long *k_stk,
	       long k_stk_size, unsigned long entry);
void show_task (int);

/* the following are entry points in the kernel tasks ! */
void sys_task (void);
void clock_task (void);

/* the following are stacks for the kernel tasks */
long sys_stack[KERNEL_STK_SIZE];
long clock_stack[KERNEL_STK_SIZE];

#if INIT_SERVERS
/* stacks for the initial servers : */
PRIVATE long k_server_stacks[INIT_SERVERS][KERNEL_STK_SIZE];
PRIVATE struct descriptor server_ldt[INIT_SERVERS][3];	/* ldts for servers */

/* PRIVATE */
struct configuration sys_description[INIT_SERVERS + 1] =
{
  {0, 1, 2, 3, 4, "xx"}};
/* The struct configuration HAS to be initialized, as to be kept in the
     data segment, because the bss is zeroed on start.  In this array the
     program tools/build places the informations about the initial system
     configuration (each server described sepparately).
   */
#endif

PUBLIC void 
start_kernel (void)
{
  int task_no, i, err;

  init_puts ();
  printk ("PicOs welcomes you !\n");
  init_sched ();
  init_interrupts ();
  init_traps ();
  init_task ();

  /* set up structures for the system calls */

#if DEBUG
  /* screen access descriptor */
  set_descriptor (&gdt[1], (char *) 0xb8000, 80L * 25 * 2, TYPE_DATA_RW, USER_PRIV);
#endif
  err = set_sys_call_gate (SYS_CALL_INT, sys_entry);
  if (err != E_OK)
    panic ("Cannot set up system call gate; error %d", err);

  /* set up structures for task 0 = IDLE_PROC */

  task_no = make_kernel_task (kstack, KERNEL_STK_SIZE, 0L);
  /* these values do not matter, in fact. They will be overwritten by
	the task switch */
  if (task_no != IDLE_PROC)
    panic ("IDLE slot not first in list but %d", task_no);
  if ((err = change_priority (IDLE_PROC, IDLE_PRIO)) != E_OK)
    panic ("cannot set priority of the idle proc : error %d", err);
  if ((err = ready (IDLE_PROC)) != E_OK)
    panic ("Cannot ready idle proc : error %d", err);
  ltr (TSS (IDLE_PROC));	/* this is the current task ! */
  cur_proc = IDLE_PROC;		/* this is the only time this is done by hand */
  prev_proc_ptr = bill_ptr = cur_proc_ptr = proc_addr (cur_proc);

  /* interrupts can come now ... */
  sti ();
#if DEBUG_LOCAL
  printk ("Interrupts enabled.\n");
#endif

  /* set up structures for the other tasks : */

  /* First the KERNEL tasks : */
  /* the system task : */
  task_no = make_kernel_task (sys_stack, KERNEL_STK_SIZE,
			      (unsigned long) sys_task);
  if (task_no != SYS_TASK)
    panic ("cannot initialize the system task; error %d", task_no);
  if ((err = ready (task_no)) != E_OK)
    panic ("Cannot ready system task : error %d", err);

  /* the clock task : */
  task_no = make_kernel_task (clock_stack, KERNEL_STK_SIZE,
			      (unsigned long) clock_task);
  if (task_no != CLOCK_TASK)
    panic ("Cannot initialize the clock task; error %d", task_no);
  if ((err = ready (task_no)) != E_OK)
    panic ("Cannot ready clock task; error %d", err);

  /* Now the start-up servers
   Their images have been loaded together with the kernel, and
   their description put in the array sys_description by the
   program building the system image.
 */
  printk ("Initial system composition : [kernel + 3 kernel processes]");
  printk (" + %d servers :\n", INIT_SERVERS);
  for (i = 0; i <= INIT_SERVERS; i++)
    {				/* 0 is `System' itself */
      struct configuration *s = &sys_description[i];
      /* here is described the binary image loaded */

      printk ("%d.`%s':base 0x%lx code 0x%lx data 0x%lx stk 0x%lx entry 0x%lx\n",
	      i, s->proc_name, s->base, s->code_size, s->data_size,
	      s->stack_size, s->entry);
      if (i)
	{			/* build them now ! 0 is kernel itself and is not built */
	  int task = i - 1;	/* this one has been a bug very hard to notice !*/

	  err = build_ldt (server_ldt[task], sizeof (server_ldt[task]),
			   s->base, s->code_size,
			   s->base + s->code_size, s->data_size,
			   s->base + s->code_size + s->data_size,
			   s->stack_size);
	  if (err != E_OK)
	    panic ("Cannot build ldt, err %d", err);
	  task_no = make_task (server_ldt[task], sizeof (server_ldt[task]),
			       k_server_stacks[task], KERNEL_STK_SIZE,
			       s->entry);
	  if (task_no < 0)
	    panic ("Cannot build task, err %d", task_no);
	  err = change_priority (task_no, SERVER_PRIO);	/* all are servers */
	  if (err != E_OK)
	    panic ("Cannot change priority, err %d", err);
	  err = ready (task_no);
	  if (err != E_OK)
	    panic ("Cannot ready, err %d", err);
#if DEBUG_LOCAL
	  show_task (task_no);
#endif
	}
    }

  /* NOW THE SYSTEM IS READY and UP.  SCHEDULE WHO MIGHT BE READY ! */

  switch_to (next_ready ());	/* youpii */

  /* this is proc 0 - IDLE; it will run (only) when none else ready */
  for (;;) /* spend the time as my name shows */ ;
}
