/* 
 * kernel/interrupt.c
 * @(#) converts interrupts into messages
 * (c) 1995 by Mihai Budiu
 */
  
#include "../include/globals.h"
#include "./proc.h"
#include "../include/error.h"
#include "../include/segment.h"
#include "../include/asm.h"
#include "../include/config.h"  /* for FIRST_HARD_INT */
#include "../include/operation.h"  /* for messages codes */
/* 
 *   Each (hardware) interrupt is transformed into a message
 * and sent to the registered process handler.
 * If handler is busy, buffer the message.
 *   Each message is built by a specific function depending on
 * the interrupt type.  These `pre-process' functions are in the
 * file kernel/int_proc.c.
 *   If a certain interrupt could not be sent count it in the
 * field ERROR of the message being sent.  This allows for
 * several pending interrupts to be resolved at once.
 */
#define DEBUG_LOCAL 0

  /* pointers to tasks handling each interrupt */
PRIVATE struct proc_struct * handler[HARD_INT_VECTORS];
  /* messages that could not be sent yet */
PRIVATE struct message pending_mess[HARD_INT_VECTORS];

#define INT_ROUTINE(intr) interrupt_ ## intr

/* build a set of interrupt treating routines;
 * the ideas come from linux/kernel/irq.c
 */

#define _STR(x) #x
#define STR(x) _STR(x)

/* acknowledge interrupt to one of the 8259 */

#define ACK_MASTER(mask) \
	"inb $0x21,%al\n\t" \
	"jmp 1f\n" \
	"1:\tjmp 1f\n" \
	"1:\tjmp 1f\n" \
	"1:\torb $" #mask ",_cache_21\n\t" \
	"movb _cache_21,%al\n\t" \
	"outb %al,$0x21\n\t" \
	"jmp 1f\n" \
	"1:\tjmp 1f\n" \
	"1:\tjmp 1f\n" \
	"1:\tmovb $0x20,%al\n\t" \
	"outb %al,$0x20\n\t"

#define ACK_SLAVE(mask) \
	"inb $0xA1,%al\n\t" \
	"jmp 1f\n" \
	"1:\tjmp 1f\n" \
	"1:\tjmp 1f\n" \
	"1:\torb $" #mask ",_cache_A1\n\t" \
	"movb _cache_A1,%al\n\t" \
	"outb %al,$0xA1\n\t" \
	"jmp 1f\n" \
	"1:\tjmp 1f\n" \
	"1:\tjmp 1f\n" \
	"1:\tmovb $0x20,%al\n\t" \
	"outb %al,$0xA0\n\t" \
	"jmp 1f\n" \
	"1:\tjmp 1f\n" \
	"1:\tjmp 1f\n" \
	"1:\toutb %al,$0x20\n\t"

/* unblock one of the 8259 */

#define UNBLOCK_MASTER(mask) \
	"inb $0x21,%al\n\t" \
	"jmp 1f\n" \
	"1:\tjmp 1f\n" \
	"1:\tjmp 1f\n" \
	"1:\tandb $~(" #mask "),_cache_21\n\t" \
	"movb _cache_21,%al\n\t" \
	"outb %al,$0x21\n\t"

#define UNBLOCK_SLAVE(mask) \
	"inb $0xA1,%al\n\t" \
	"jmp 1f\n" \
	"1:\tjmp 1f\n" \
	"1:\tjmp 1f\n" \
	"1:\tandb $~(" #mask "),_cache_A1\n\t" \
	"movb _cache_A1,%al\n\t" \
	"outb %al,$0xA1\n\t"

__asm__ (
"ST:     .ascii \"crt %ld ds %lx ss %lx cs %lx eax %lx ebx %lx ecx %lx edx %lx esi %lx edi %lx\\12\\0\"\n"
"INTSTR: .ascii \"interrupt %d\\12\\0\"\n"
".globl   _cur_proc\n"	
"show_status:\n\t"
	 "pushl %edi\n\t"
	 "pushl %esi\n\t"
	 "pushl %edx\n\t"
	 "pushl %ecx\n\t"
	 "pushl %ebx\n\t"
	 "pushl %eax\n\t"
	 "pushl %cs\n\t"
	 "pushl %ss\n\t"
	 "pushl %ds\n\t"
	 "pushl _cur_proc\n\t"
	 "pushl $ST\n\t"
	 "call _printk\n\t"
	 "addl $20, %esp\n\t"
	 "popl %eax\n\t"
	 "popl %ebx\n\t"
	 "popl %ecx\n\t"
	 "popl %edx\n\t"
	 "popl %esi\n\t"
	 "popl %edi\n\t"
	 "ret"
);

/* generate code for the interrupt treatment */

extern void (*ret_from_sys_call)(void);

#define BUILD_INT_CODE(inter, bit, which) \
void INT_ROUTINE(inter) (void); \
__asm__( \
"\n.align 4,0x90\n" \
"_interrupt_" #inter ":\n\t" \
	SAVE_REGS \
	"movl $" STR(KERNEL_DS) ", %eax\n\t" \
	"movw %ax, %ds\n\t" \
	"movw %ax, %es\n\t" \
	ACK_ ## which ## (bit) \
	"pushl $" #inter "+" STR(FIRST_HARD_INT) "\n\t" \
	"#movl $" #inter ", %eax\n\t" \
	"#cmpl $32, %eax\n\t" \
	"#je 1f\n\t" \
	"#pushl $INTSTR\n\t" \
	"#call _printk\n\t" \
	"#addl $4, %esp\n\t" \
	"#call show_status\n" \
"#1:\n\t"       \
	"call _interrupt\n\t" /* in eax new process to schedule ! */ \
	"movl %eax, %ebx\n\t" \
	"addl $4, %esp\n\t" \
	UNBLOCK_ ## which ## (bit) \
	"cmpl $0, %ebx\n\t" \
	"jz 1f\n\t" \
	"pushl %ebx\n\t" \
	"call _switch_to\n\t"  /* new process to run ! */ \
	"addl $4, %esp\n" \
"1:\n\t" \
	"jmp _ret_from_sys_call\n" \
)

/* for 8259 conversation - interrupt ignore mask */
PRIVATE volatile unsigned char cache_21 = 0xff; /* the MASTER 8259 */
PRIVATE volatile unsigned char cache_A1 = 0xff; /* the SLAVE */

/* this first set uses first 8259 */
/*             irq,bit, chip */
BUILD_INT_CODE(0, 0x01, MASTER);
BUILD_INT_CODE(1, 0x02, MASTER);
BUILD_INT_CODE(2, 0x04, MASTER);
BUILD_INT_CODE(3, 0x08, MASTER);
BUILD_INT_CODE(4, 0x10, MASTER);
BUILD_INT_CODE(5, 0x20, MASTER);
BUILD_INT_CODE(6, 0x40, MASTER);
BUILD_INT_CODE(7, 0x80, MASTER);
/* this second set uses the second (cascaded) chip */
BUILD_INT_CODE(8, 0x01, SLAVE);
BUILD_INT_CODE(9, 0x02, SLAVE);
BUILD_INT_CODE(10, 0x04, SLAVE);
BUILD_INT_CODE(11, 0x08, SLAVE);
BUILD_INT_CODE(12, 0x10, SLAVE);
BUILD_INT_CODE(13, 0x20, SLAVE);
BUILD_INT_CODE(14, 0x40, SLAVE);
BUILD_INT_CODE(15, 0x80, SLAVE);

/* 
 * interrupt handling functions, initialized be the
 * previously built handlers 
 */
PRIVATE void (*inter_func[])(void) = {
  interrupt_0,   interrupt_1,   interrupt_2,   interrupt_3,
  interrupt_4,   interrupt_5,   interrupt_6,   interrupt_7,
  interrupt_8,   interrupt_9,   interrupt_10,  interrupt_11,
  interrupt_12,  interrupt_13,  interrupt_14,  interrupt_15
};

PUBLIC void show_interrupts(void)
{
  int i;
  unsigned long fl;
  
  save_flags(fl);
  cli();
  printk("Interrupt handlers :\n");
  for(i=0; i < HARD_INT_VECTORS; i++) 
    if(handler[i] != NULL) 
      printk("int %d - proc# %d\n", i+FIRST_HARD_INT, proc_slot(handler[i]));
  restore_flags(fl);
}

extern PRE_INT ad_key, ad_time;
PRIVATE PRE_INT_P int_process[] = {
  /* functions pre-processing each interrupt 
     Such functions build the corresponding interrupt message */
  
  ad_time, ad_key, NULL, NULL,
  NULL, NULL, NULL, NULL,
  NULL, NULL, NULL, NULL,
  NULL, NULL, NULL, NULL
};

PRIVATE int enable_irq(int irq_no)
     /* enable from now on this hard interrupt; return error code */
{
  unsigned long fl;  /* save flags */
  unsigned char mask;

  if (irq_no < 0 || irq_no >= HARD_INT_VECTORS) return E_BADARG;
  /* now change mask and announce 8259s */
  mask = ~(1 << (irq_no & 0x07));
  save_flags(fl);
  cli();
  if (irq_no < 8) {  /* just the first chip */
    cache_21 &= mask;
    outb(cache_21, 0x21);
  }
  else {            /* SLAVE chip */
    cache_21 &= ~0x02;  /* cascade HAS to work */
    outb(cache_21, 0x21);
    cache_A1 &= mask;
    outb(cache_A1, 0xA1);
  }
  restore_flags(fl);
  return E_OK;
}

PRIVATE int disable_irq(int irq_no)
     /* enable from now on this hard interrupt */
{
  unsigned long fl;  /* save flags */
  unsigned char mask;

  if (irq_no < 0 || irq_no >= HARD_INT_VECTORS) return E_BADARG;
  if (irq_no == 2 && cache_A1 != 0xff) 
    return E_ILL;  /* cascade cannot be undone unless no slave interrupt */
  /* now change mask and announce 8259s */
  mask = 1 << (irq_no & 0x07);
  save_flags(fl);
  if (irq_no < 8) {  /* just the first chip */
    cli();
    cache_21 |= mask;
    outb(cache_21, 0x21);
  }
  else {            /* SLAVE chip */
    cli();
    cache_A1 |= mask;
    outb(cache_A1, 0xA1);
  }
#if DEBUG_LOCAL
  printk("irq %d disabled\n", irq_no);
#endif
  restore_flags(fl);
  return E_OK;
}

PUBLIC void init_interrupts(void)
     /* build the data structures */
{
  int i;

  for (i=0; i < HARD_INT_VECTORS; i++) {
    struct message * p = &pending_mess[i];

    handler[i] = NULL;
    p->SOURCE.local_pid = IDLE_PROC;
    p->OPERATION = INT;            /* interrupt operation */
    p->MSGID = i + FIRST_HARD_INT; /* interrupt no */
    p->flags &= ~WAIT_ANSW;        /* interrupts have no answer */
    p->ERROR = 0;                  /* count of int's not sent */
  }

  /* 
   * start from int 0x20 - these are the HARDWARE generated interrupts
   * Note that the code from SETUP has reprogrammed the 8259 as to
   * remap the interrupts 8,9 etc. which overlap the internally
   * generated ones (80386 traps) from 0x20 (FIRST_HARD_INT) upwards
   */
  /* set up now the interrupt gates */
  for (i=0; i < HARD_INT_VECTORS; i++)
    set_interrupt_gate(i + FIRST_HARD_INT, inter_func[i]);
#if DEBUG_LOCAL
  printk("Hardware interrupt descriptors :\n");
  addr = &idt[FIRST_HARD_INT];
  for (i=0; i < HARD_INT_VECTORS; i++, addr++) show_descriptor(addr);
#endif
  /* announce controller about initial irq map */
  outb(cache_21, 0x21);
  slow();
  outb(cache_A1, 0xA1);  
  if (enable_irq(2) != E_OK)
    panic("Cannot enable irq %d for slave handling !", 2);
#if DEBUG_LOCAL
  printk("Interrupts initialized.\n");
#endif
}
extern void show_queues(void);
extern void show_frame(void);

PRIVATE void unexpected_int(long int_no)
     /* an interrupt without handler came */
{
  printk("Unexpected interrupt : %ld\n", int_no);
}

PUBLIC int interrupt(long int_no)
     /* this routine is called from the interrupt gates;
      * it still saves flags & makes cli() to be sure;
      * Returns process that should be scheduled if rescheduling necessary, 
      * else 0 */
{
  unsigned long fl;
  struct proc_struct * p;
  int irq = int_no - FIRST_HARD_INT;
  int err;

#if DEBUG
  if (irq < 0 || irq >= HARD_INT_VECTORS) 
    panic("interrupt : bad hardware interrupt number %d", int_no);
#endif
  save_flags(fl);
  cli();
  p = handler[irq];
  if (p == NULL) {  
    /* no handler registered ! */
    unexpected_int(int_no);
    restore_flags(fl);
    return 0;  /* no rescheduling */
  }
  
  /* preprocess the interrupt building the message; result < 0
     = error in interrupt treatment -> no message sent ! */

  if (int_process[irq] != NULL)
    if ((err = int_process[irq](&pending_mess[irq]) < 0)) {
      printk("interrupt preprocess error %d!\n", err);
      restore_flags(fl);
      return 0;
    }
  
  /* a handler exists; try to send message */
  pending_mess[irq].ERROR++; /* one more int */
  if (can_send_to(ANY, p)) { /* it is blocked waiting */
    give_message(p, KERNEL_DS, &pending_mess[irq], &p->msg_copy);
    /* this readies handler ! */
    pending_mess[irq].ERROR = 0; 
    p->flags &= ~PENDING_INT;
    /* clear error as an indication of the message being sent */
    restore_flags(fl);
    return next_ready();             /* rescheduling may be necessary */
  }

  /* process is not blocked to receive : have a pending message */  
  p->flags |= PENDING_INT;
  if (p->status == READY) {  /* if ready ... */
    restore_flags(fl);
    return next_ready();     /* give it a chance to run */
  }
  restore_flags(fl);
  return 0;                  /* no rescheduling */
}

PUBLIC int send_pending_int(struct proc_struct * p) 
     /* p tryes to go RECEIVING (from ANY), and it has a pending interrupt.
	Send this interrupt to it.  Return E_OK if successfull.
	It is called only by block_to_receive, when
	the current process tries to block with pending interrupt message */
{
  int irq;
  unsigned long fl;

  save_flags(fl);
#if DEBUG
  cli();
  if (p->interrupt == NO_INT || (p->flags & PENDING_INT) == 0) {
    restore_flags(fl);
    return E_ILL;
  }
  irq = p->interrupt - FIRST_HARD_INT;
  if (irq < 0 || irq >= HARD_INT_VECTORS)
    panic("Bad hard interrupt treated : %d", p->interrupt);
  if (handler[irq] != p) 
    panic("Inconsistent handler information for interrupt %d", p->interrupt);
#endif
  give_message(p, KERNEL_DS, &pending_mess[irq], &p->msg_copy);
  p->flags &= ~PENDING_INT;  /* done with it */
  pending_mess[irq].ERROR = 0;  /* clear no of pending ints */
  restore_flags(fl);
  return E_OK;
}

PUBLIC int register_handler(int int_no, struct proc_struct * p)
     /* p will handle the interrupt int_no */
{
  unsigned long fl;
  int result, irq;
  struct proc_struct * old_handl;  /* previous handler */

  irq = int_no - FIRST_HARD_INT;
  if (irq < 0 || irq >= HARD_INT_VECTORS) return E_BADARG;
  save_flags(fl);
  cli();
  if (p->status == SLOT_FREE || 
      p->status == INCOMPLETE ||
      p->priority > DRIVER_PRIO) {
    restore_flags(fl);
    return E_ILL;  /* handlers must have high priorities ... */
  }
  old_handl = handler[irq];
    /* can be non null, as to switch handlers `on the fly' */
  handler[irq] = p;        /* p treats it */
  p->interrupt = int_no;
  p->flags &= ~PENDING_INT;   /* no pending interrupts yet */
  result = enable_irq(irq);
  if (result != E_OK) {
    handler[irq] = old_handl;  /* do not change it */
    printk("Cannot register %d as handler for interrupt %d !",
	   proc_slot(p), int_no);
  }
  else {
    old_handl->interrupt = NO_INT;     /* no interrupt */
    old_handl->flags &= ~PENDING_INT;  /* lost if any.  Anyhow, too late */
  }
#if DEBUG_LOCAL
  if (result == E_OK) 
    printk("process %d registered as handler for int %d\n", 
	   proc_slot(p), int_no);
#endif
  restore_flags(fl);
  return result;
}

PUBLIC int unregister_handler(int int_no)
     /* a process gives up handling some interrupt */
{ 
  unsigned long fl;
  struct proc_struct * p;
  int irq_no = int_no - FIRST_HARD_INT;

  if (irq_no < 0 || irq_no >= HARD_INT_VECTORS)
    return E_BADARG;
  save_flags(fl);
  cli();
  p = handler[irq_no];
  if (p == NULL) {
    restore_flags(fl);
    return E_NOPID;   /* no handler here ... */
  }
  handler[irq_no] = NULL;
  p->interrupt = NO_INT;
  p->flags &= ~PENDING_INT;
  disable_irq(irq_no);
  restore_flags(fl);
  pending_mess[irq_no].ERROR = 0;
  return E_OK;
}
