/* 
 * kernel/desc.c
 * @(#) descriptor handling; lowest level of kernel
 * (c) 1995 by Mihai Budiu
 */

#include "../include/limits.h"  /* for NR_TASKS */
#include "../include/globals.h"
#include "../include/segment.h" /* for TSS_DESCRIPTOR(t), etc. */
#include "./desc.h"
#include "../include/error.h"   
#include "../include/config.h"  /* for FIRST_HARD_INT */
#include "./proc.h"             /* for proc */

#define DEBUG_LOCAL 0

#define U_CHAR(x)  (*((unsigned char  *)(x)))
#define U_SHORT(x) (*((unsigned short *)(x)))
#define U_LONG(x)  (*((unsigned long  *)(x)))

PRIVATE inline void 
set_base(struct descriptor * addr, char * base)
     /* write in a descriptor at address addr the base value */
{
  unsigned short s;
  unsigned char  c;
  unsigned long  b = (unsigned long) base;
  unsigned char * a = (unsigned char *) addr;
    /* to be able to express all kinds of byte offsets */

  s = b & 0xffff;  /* low word */
  U_SHORT(a + 2) = s;
  b >>= 16;
  c = b & 0xff;
  U_CHAR(a + 4) = c;
  b >>= 8;
  c = b & 0xff;
  U_CHAR(addr + 7) = c;
}

PRIVATE inline void 
set_limit(struct descriptor * addr, unsigned long limit)
     /* write in a descriptor at addr the limit value */
{
  unsigned short s;  /* low word */
  unsigned char  c;  /* high byte + G + D */
  enum {G = 01,    /* granularity */
	DEC = 02}; /* decrement */
  char flgs = DEC;   
  unsigned char * a = (unsigned char *) addr;

  if (limit & 0xfff00000) {
    if ((limit & 0x00000fff) != 0L) {
      panic("set_limit : bad segment size (too many digits): %lx", limit);
      flgs &= ~DEC; /* do not decrement limit */
    }
    limit >>= 12;   /* represent limit with granularity 1 */
    flgs |= G;      /* this is granularity flag */
  }
  if (flgs & DEC) --limit; /* store limit - 1 ! */
  s = limit & 0xffff;
  U_SHORT(a) = s;
  limit >>= 16;
  limit &= 0x0f;
  c = limit | 0x40;       /* 386 flag always */
  if (flgs & G) c |= 0x80;/* set granularity */
  else c &= ~0x80;
  U_CHAR(a + 6) = c;
}

PRIVATE inline void 
set_type(struct descriptor * addr, unsigned char type, int privilege)
     /* sets up the type & DPL of the descriptor */
{
  unsigned char * a = (unsigned char *) addr;
  unsigned char v;

  v = (privilege & 3) << 5;  /* DPL */
  type &= 0x9F;              /* type */
  U_CHAR(a + 5) = type | v;
}

PUBLIC void 
set_descriptor(struct descriptor * addr, char * base, 
	       unsigned long limit, unsigned char type,
	       int privilege)
     /* set up all the fields of a segment descriptor */
{
  set_base (addr, base);
  set_limit(addr, limit);
  set_type (addr, type, privilege);
}

PUBLIC char * 
get_base(struct descriptor * addr)
     /* get the base address of the segment described by the 
	descriptor at addr */
{
  unsigned short s;
  unsigned char  c;
  unsigned long  base;
  unsigned char * a = (unsigned char *) addr;

  c = U_CHAR(a + 7);
  base = c;
  base <<= 8;
  c = U_CHAR(a + 4);
  base |= c;
  base <<= 16;
  s = U_SHORT(a + 2);
  base |= s;
  return (char *) base;
}

PUBLIC unsigned long 
get_limit(struct descriptor * addr)
     /* get the limit of a segment */  
{
  unsigned long limit;
  unsigned char c;
  int gran;
  unsigned short s;
  unsigned char * a = (unsigned char *) addr;

  c = U_CHAR(a + 6);
  gran = c & 0x80;   /* granularity */
  c &= 0x0f;
  limit = c;
  limit <<= 16;
  s = U_SHORT(a);
  limit |= s;
  ++ limit;
  if (gran) limit <<= 12;
  return limit;
}

PRIVATE inline unsigned char 
get_flags(struct descriptor * addr)
     /* read the FLAGS byte (5) of a descriptor */
{
  unsigned char * a = (unsigned char *) addr;

  return U_CHAR(a + 5);
}

PUBLIC char * 
get_displacement(struct descriptor * addr)
     /* this is a gate; get displacement */
{
  unsigned long l;
  unsigned short s;
  unsigned char * a = (unsigned char *) addr;

  s = U_SHORT(a+6);
  l = s;
  l <<= 16;
  s = U_SHORT(a);
  l |= s;
  return (char *) l;
}

PRIVATE inline unsigned short 
get_selector(struct descriptor * addr)
     /* get selector number from a gate */
{
  unsigned char * a = (unsigned char *) addr;

  return U_SHORT(a + 2);
}

PRIVATE inline unsigned int 
get_copy(struct descriptor * addr)
     /* get no. of bytes copyed on call */
{
  unsigned char * a = (unsigned char *) addr;
  unsigned char b;

  b = U_CHAR(a + 4);
  b &= 0x1f;
  return ((unsigned int)b << 2);
}

PRIVATE inline void 
set_displacement(struct descriptor * addr, void (*d)(void))
     /* this is a gate; set displacement d */
{
  unsigned short s;
  unsigned long l;
  unsigned char * a = (unsigned char *) addr;

  l = (unsigned long)d;
  s = (unsigned short)(l & 0x0000ffff);
  U_SHORT(a) = s;
  l >>= 16;
  s = (unsigned short)(l & 0x0000ffff);
  U_SHORT(a + 6) = s;
}

PRIVATE inline void 
set_selector(struct descriptor * addr, unsigned short selector)
     /* set selector number of a gate */
{
  unsigned char * a = (unsigned char *) addr;

  U_SHORT(a + 2) = selector;
}

PRIVATE inline void
set_copy(struct descriptor * addr, unsigned char Dcopy)
     /* set no. of double words copyed on call */
{
  unsigned char * a = (unsigned char *) addr;

  U_CHAR(a + 4) = Dcopy & 0x1f;
}

PUBLIC int 
set_interrupt_gate(int int_no, void (*handler)(void))
     /* set up an interrupt gate; 
	this is not necessary a hardware interrupt */
{
  struct descriptor * addr;

  if (int_no < 0 || int_no >= INT_VECTORS) return E_BADARG;
  addr = &idt[int_no];
  set_type(addr, TYPE_INT_GATE, KERNEL_PRIV);
  set_displacement(addr, handler);
  set_selector(addr, KERNEL_CS);
  set_copy(addr, 0);
  return E_OK;
}

#if DEBUG
PUBLIC void 
show_descriptor(struct descriptor * addr)
     /* dump a descriptor's contents */
{
  unsigned char * b;
  unsigned long limit;
  unsigned short flags;
  unsigned short selector;
  unsigned int stk;

  flags = get_flags(addr);
  if (flags & 0x10) {  /* this is a segment descriptor */
    b = get_base(addr);
    limit = get_limit(addr);
    printk("DESCRIPTOR Address %p, Base %p, Limit %lx, Flags %x\n", 
	   addr, b, limit, flags);
  }
  else {   /* this is a system/gate descriptor */
    switch(flags & 0x0f) { 
      case 0x02:  
      case 0x09:
      case 0x0b:  /* these are system descriptors */
        b = get_base(addr);
        limit = get_limit(addr);
	printk(((flags & 0x0f) == 0x02) ? "LDT " : "TSS ");
        printk("DESCRIPTOR Address %p, Base %p, Limit %lx, Flags %x\n", 
	       addr, b, limit, flags);
	break;
      case 0x04:
      case 0x05:
      case 0x06:
      case 0x07:
      case 0x0c:
      case 0x0e:
      case 0x0f: /* these are all gates */
	selector = get_selector(addr);
	b = (unsigned char *)get_displacement(addr);
	stk = get_copy(addr);
	printk("GATE Address %p, Selector %u, Displ. %p, Stack copy %u\n",
	       addr, selector, b, stk);
	break;
      default :
	printk("ILLEGAL TYPE - address %p\n", addr);
    }
  }
}
#endif /* debug */

PUBLIC int
set_tss_descriptor(int task_no)
     /* set up a descriptor for a tss segment */
{
  struct descriptor * addr;

  if (task_no < 0 || task_no >= NR_TASKS) return E_BADARG;
  addr = &gdt[TSS_DESCRIPTOR(task_no)];
  set_descriptor(addr, (unsigned char *)&(proc[task_no].tss), 
		 sizeof(struct tss_struct), TYPE_TSS, KERNEL_PRIV);
  return E_OK;
}

PUBLIC int
set_trap_gate(int trap_no, void (*handler)(void))
     /* build a trap gate */
{
  struct descriptor * addr;

  if (trap_no < 0 || trap_no >= INT_VECTORS) return E_BADARG;
  addr = &idt[trap_no];
  set_type(addr, TYPE_TRAP_GATE, KERNEL_PRIV);
  set_displacement(addr, handler);
  set_selector(addr, KERNEL_CS);
  set_copy(addr, 0);   /* this could miss as it is ignored */
  return E_OK;
}

PUBLIC int
set_ldt_descriptor(int task_no, struct descriptor * base, 
		   unsigned long size, int privilege)
     /* write a descriptor for a ldt segment; size in BYTES */
{
  struct descriptor * addr;

  if (task_no < 0 || task_no >= NR_TASKS) return E_BADARG;
  addr = &gdt[LDT_DESCRIPTOR(task_no)];
  set_descriptor(addr, (unsigned char *)base, size, TYPE_LDT, privilege);
  return E_OK;
}

PUBLIC int set_sys_call_gate(int int_no, void (*handler)(void))
     /* set up a system call gate - function called upon sys call */
{  /* this is like an interrupt gate, except a different privilege */
  struct descriptor * addr;

  if (int_no < 0 || int_no >= INT_VECTORS) return E_BADARG;
  if (int_no <= 16) return E_ILL;  /* these are internally generated traps */
  if (int_no >= FIRST_HARD_INT && int_no < FIRST_HARD_INT + HARD_INT_VECTORS)
    return E_ILL;                  /* these vectors for hardware */
  addr = &idt[int_no];
  set_type(addr, TYPE_INT_GATE, USER_PRIV);  /* user privilege ! */
  set_displacement(addr, handler);
  set_selector(addr, KERNEL_CS);
  set_copy(addr, 0);   /* this could miss as it is ignored */
  return E_OK;
}

