/*
 * clock.c
 * @(#) clock driver; this is a kernel task
 * (c) 1995 by Mihai Budiu
 */

/* 
 * The clock task is a server.  It is a kernel process, beacause
 * it is the one that handles scheduling calling the functions in
 * sched.c. It also handles the clock interrupts & some system calls
 * Here are its functions :
 *
 * -OPERATION|-arg--|-description-----------------------------
 * INT       | int  | clock interrupt; may cause rescheduling
 * SLEEP     | ulong| wake me up after that much time
 * TIME      | -    | get times (ticks, u_time, s_time);
 * -------------------------------------------------------------
 */

#define __SERVER__ /* this is a server */

#include "../include/globals.h"
#include "../include/segment.h"
#include "../include/message.h"
#include "./proc.h"
#include "../include/error.h"
#include "../include/config.h"
#include "../include/operation.h"
#include "../include/pid.h"
#include "../include/asm.h"
#include "../include/limits.h"
#include "../include/server.h"

#define DEBUG_LOCAL 0

PRIVATE struct message 
  msg,     /* here I receive all messages */
  reply;   /* build reply here */
PRIVATE unsigned long ticks; /* time since up */

PRIVATE struct sleep {
  long interval;     /* how much */
  LOCAL_PID lpid;    /* who (sleep works only locally !) */
  msg_id_t msg_id;   /* id of message of request */
  struct sleep * next, * prev;
} sleep_info[NR_TASKS+1];  /* one for a dummy structure */

PRIVATE struct sleep 
  * first,  /* first sleeping; doubly linked */
  * free,   /* free list; simply linked */
  * const dummy = sleep_info;

PRIVATE do_thing_t do_sleep, do_time;

PRIVATE void init_clock_task(void)
     /* initialize the clock task */
{
  int err;
  void show_task(int);

  ticks = 0L;
  msg.INFOi[0] = TIMER_INT;         /* this is int. no */
  err = send_receive(SYS_PID, HAND_INT, &msg);
  if (err != E_OK) panic("clock task: error in message to SYS TASK %d", err);
  err = msg.ERROR;
  if (err != E_OK) 
    panic("clock task: cannot register as handler for clock int, err %d",
	  err);
  /* build list of sleeping structures */
  for (free = sleep_info; 
       free < sleep_info + sizeof(sleep_info) / sizeof(struct sleep) - 1;
       free++) free->next = free+1;
  free = sleep_info + 1;    
  /* dummy structure for non-void list end */
  dummy->next = NULL;
  dummy->prev = NULL;
  first = dummy;
}

PRIVATE inline void   /* inline to be as fast as possible */
do_ck_int(void)       /* received interrupt message */
{
  unsigned long lost, fl;
  int err;
  
  /* treat this interrupt.  Account times.  The process just stopped
     by the clock task is prev_proc_ptr.  Mark its time. */
#if DEBUG_LOCAL
  printk("ck : %d %d", proc_slot(bill_ptr), proc_slot(prev_proc_ptr));
#endif
  lost = msg.ERROR;
  ticks += lost;       /* that many interrupts lost */
  save_flags(fl);
  cli();               /* critical region */
  prev_proc_ptr->u_time += lost;  /* increment user time */
  /* 
     If the previous process is a user process, that is all.  Else
     it is a server, and the process on whose behalf it runs must
     pay for its time too !
     */
  if (prev_proc_ptr != bill_ptr) bill_ptr->s_time += lost; /* system time */
  if (bill_ptr != proc_addr(IDLE_PROC) && (bill_ptr->remained-=lost) <= 0L) { 
        /* too much run; somebody else .. */
    bill_ptr->remained = bill_ptr->quantum;  /* start anew */
    err = reschedule(proc_slot(bill_ptr));
#if DEBUG
    if (err != E_OK) panic("Cannot preempt process %d", proc_slot(bill_ptr));
#endif
  }
  restore_flags(fl);

  /* handle sleeping processes */
  if (first != dummy) first->interval -= lost;
  while (first != dummy && first->interval <= 0l) {
    struct sleep * f;

    (void) send_reply(first->lpid, first->msg_id, SLEEP, E_OK, &reply);
    f = first;
    first = f->next;    /* cannot be NULL, as dummy always there on list */
    first->prev = NULL;
    f->next = free;
    free = f;
  }
}

PUBLIC void clock_task(void)
     /* main function of the clock task */
{
  int err;
  int answer;  /* NO if no answer is delivered now */

  init_clock_task();     /* initialize myself */
    /* ready to serve you, mylord. */

  for (;;) {           /* this function never returns */
  again:
    while ((err = receive(&msg)) != E_OK) 
      printk("Failed receive for Clock task : error %d", err);
    answer = ANSWER_EXPECTED(msg.flags);
    switch(msg.OPERATION) {
    case INT:  /* a clock interrupt came */
      do_ck_int();
      goto again;           /* faster, as this happens often */
    case SLEEP: /* sleep system call */
      if (! answer) break;  /* nobody sleeps ... */
      if (! CAN_BLOCK(msg.flags)) {
	err = E_ILL; /* server does not sleep */
	break;
      }
      err = do_sleep();
      if (err == E_BLOCK) 
	answer = NO;        /* no reply now = delayed reply */
      break;
    case TIME:
      if (answer) err = do_time();
      break;
    default:
       err = E_BADOP;
    }
    if (answer) {
      err = send_reply(msg.SOURCE.local_pid, msg.MSGID, 
		       msg.OPERATION, err, &reply);
#if DEBUG
      if (err != E_OK) printk("clock task can't reply process %ld, error %d",
			      reply.DESTINATION.local_pid, err);
#endif
    }
  }
}

PRIVATE int do_sleep(void)
{
  unsigned long time;
  struct sleep * crt, *p;

  if (! (msg.flags & WAIT_ANSW)) return E_ILL;  /* does not matter */
  time = (unsigned long)msg.INFOl[0];
  if (time == 0l) return E_BADARG;   /* this causes no sleep at all */

  /* search list of sleeping processes */
  p = free;
  if (p == NULL) return E_AGAIN; /* too many sleepers */
  free = free->next;
  p->lpid = msg.SOURCE.local_pid;
  for (crt = first; crt != dummy && crt->interval < time; crt = crt->next)
    time -= crt->interval;
    /* insert this process before crt on the list */
  if (crt->prev != NULL) crt->prev->next = p; 
  else first = p;
  p->prev = crt->prev;
  p->next = crt;
  crt->prev = p;
  p->interval = time;
  p->msg_id= msg.MSGID;
  if (crt != dummy) crt->interval -= time; /* tail of queue */
  return E_BLOCK;
}

PRIVATE int do_time(void)
     /* puts in message ticks, u_time and s_time of current process */
{
  struct proc_struct * p;
  extern struct proc_struct * find_proc(proc_id *);
  unsigned long fl;

  save_flags(fl);
  cli();
  p = find_proc(&msg.SOURCE);   /* whose times */
  restore_flags(fl);
  if (p == NULL) {
    restore_flags(fl);
    return E_ILL;
  }
  reply.INFOl[0] = ticks;
  reply.INFOl[1] = p->u_time;
  reply.INFOl[2] = p->s_time;
  return E_OK;
}
