/* 
 * message.h
 * @(#) message structure and other definitions
 * (c) 1995 by Mihai Budiu
 */

#ifndef _MESSAGE_H
#define _MESSAGE_H

#include "./memobj.h"
#include "./pid.h"

/* the message structure */

#define DATA_LEN 32       /* how many bytes of data in one message ? */

typedef long operation_t;
typedef unsigned short msg_id_t;
typedef short err_t;  
typedef unsigned char signal_t;

struct mess_data {        /* so looks the data sent in one message */
  operation_t operation;       /* operation code; receiver dependent */
    /* keep a `long' operation so as to align properly the info field.
       Many functions treat it as an array of longs, so it is better if it
       lyes on a long boundary */
  union {
    char   c[DATA_LEN];  /* operation data; receiver dependent */
    int    i[1];         /* this structure makes easier conversions */
    long   l[1];         /* between the char array and some other types */
    short  s[1];
  } info;
};

struct message {           /* so looks a message */
  proc_id to;              /* who's gonna get it; used for send */
  proc_id from;            /* where does it come from; for receive */
  struct mess_data data;   /* data of the message */
  msg_id_t msg_id;         /* useful for query/reply identification */
  err_t oper_error;        /* in replyes it is the error code */
  signal_t signal;         /* used to signal */
  signal_t signal_me;      /* signal I wish to receive */
  unsigned char flags;     /* flags (see below) */
};

/* flags for message */
#define MSG_ROMASK 0x0f  /* these flags are read-only (set by the kernel) */
#define MSG_RWMASK 0xf0  /* these flags are part of the message */

enum {
/* these are read-only flags, set by the kernel upon transmission */
  WAIT_ANSW = 0x01,   /* sender did send_receive */
  FROM_SERVER=0x02,   /* the originator is a server */
/* these are read-write flags, part of the message itself */
  EMERGENCY = 0x10    /* emergency delivered message; not used yet */
};

/* some macros for handy field access to messages */
#define OPERATION   data.operation
#define INFO        data.info.c
#define INFOc       data.info.c   /* message data as char array */
#define INFOi       data.info.i   /* message data as int array */
#define INFOl       data.info.l   /* message data as long array */
#define INFOs       data.info.s   /* message data as short array */
#define DESTINATION to
#define SOURCE      from
#define ERROR       oper_error
#define MSGID       msg_id
#define SIGNAL_NO   signal
#define SIGNAL_ME   signal_me

#define REPLY_TO(operation) ((operation) + 1000)
  /* the reply code to an operation `op' is op + 1000 */

  /* servers are always replied if they did send_receive. */
#define ANSWER_EXPECTED(flags) ((flags) & WAIT_ANSW)
  /* a server should not be blocked */
#define CAN_BLOCK(flags) (!((flags) & FROM_SERVER)) 

/***** user level interface to message handling through the library *****/
int send(LOCAL_PID destination, operation_t op, struct message * msg);
  /* local send */
int receive(struct message * msg);
  /* local receive from any */
int send_receive(LOCAL_PID destination, operation_t op, struct message * msg);
  /* send-rec locally */
int send_reply(LOCAL_PID dest, msg_id_t reqid, operation_t op, err_t error, 
	       struct message * reply);
#if 0
int send_remote();
     /* not implemented */
int receive_remote();
     /* not implemented */
int send_rec_remote();
     /* not implemented */
#endif

#ifdef __KERNEL__              /* only for kernel use */
#define ANY proc               /* any sender accepted */
  /* this is a struct proc_struct *; it is safe to use proc as value
     for ANY, as the first process (IDLE) never can be destination or source
     of a message transmission.  On the other hand a NULL pointer signals
     error, so it cannot be used for ANY
   */
struct proc_struct;

int can_send_to(struct proc_struct * me, struct proc_struct * destination);
int k_send(struct proc_struct * dest, unsigned short selector, 
	   struct message * user_msg, int wait_asnwer);
     /* internal fast version of send; send a COMPLETE user message 
	it uses process pointers instead of pids;
	the second pointer is in USER space; returns error code */
int k_receive(struct proc_struct * from,   /* NULL if any */
	      unsigned short selector,
	      struct message * user_msg);
     /* internal receive; receive ONLY DATA part */
void give_message(struct proc_struct * receiver, unsigned short selector, 
		  struct message * src, struct message * dest);
     /* this is called to transfer a built message */
#endif  /* __KERNEL__ */

#endif  /* MESSAGE_H */




