/*
 * ImageKeyframing.java
 */

//TODO: disable growth zones near del. zones
//TODO: hierarchical creation of smoothing regions

import java.awt.image.BufferedImage;
import java.awt.image.Raster;
import java.io.File;
import java.io.IOException;
import javax.imageio.ImageIO;
import java.awt.*;
import java.util.*;

public class ImageKeyframing extends TriangleRegions {

	public ImageKeyframing(int x, int y) {
		super(x, y);
	}
	
	public void fillFromImage(String fname) {
		state = loadImage(fname);
	}
	
	public int[][] loadImage(String fname) {
		int[][] d = new int[maxX][maxY];
		try {
			BufferedImage bi = ImageIO.read(new File(fname));
			Raster r = bi.getData();
			int numBands = r.getNumBands();
			if ((r.getHeight() != maxY) || (r.getWidth() != maxX))
			{
				System.out.println("image file " + fname + " is wrong size.");
				return null;
			}
			for (int i = 0; i < maxX; i++)
				for (int j = 0; j < maxY; j++)
				{
					if (r.getSample(i,j,0) == 0) {
						d[i][j] = FILLED;
					}
					else
					{
						d[i][j] = VOID;
					}
				}
			repaint();
		} catch (IOException e)
		{
			System.out.println("couldn't read image file:" + fname);
			return null;
		}
		return d;
	}
	
	public void regionsFromImage(String fname) {
		int [][] frameB = loadImage(fname);
		
		//compute delta
		for (int i = 0;i < maxX; i++)
			for (int j = 0; j < maxY; j++)
			{
				if (isVoid(new Point(i,j)))
				{
					if (frameB[i][j] == 0) //both void
					{ frameB[i][j] = VOID; }
					else //original is void, B is not
					{ frameB[i][j] = TO_FILL;}
				}
				else
				{
					if (frameB[i][j] == 0) //original is filled, B is not
					{ frameB[i][j] = TO_DELETE; }
					else //both filled
					{ frameB[i][j] = FILLED;}
				}
			}
		
		//create triangles
		createTriRegions();
		//determine region fill state
		determineFillStates(frameB);
		
		//find edges
		for (int i = 0;i < maxX; i++)
			for (int j = 0; j < maxY; j++)
			{
				int ctrVal = frameB[i][j];
				if ((ctrVal == VOID) || (ctrVal == FILLED)) continue;
				for (int k = 1; k < 7; k++) //ctrVal must be grow or delete
				{
					Point p = move(new Point(i,j),k);
					int nVal = frameB[p.x][p.y];
					if ((ctrVal == TO_FILL) && (nVal == FILLED))
						frameB[i][j] = EDGE;
					if ((ctrVal == TO_DELETE) && (nVal == VOID))
						frameB[i][j] = EDGE;
				}
			}
		
		//determine region normal
		setRegionNormals(frameB);
		
		//set up distanceField
		setDistanceField(frameB);
		
		//state = frameB; //for debugging only!!!
	}
	
	public void setRegionNormals(int[][] frameB) {
		//reset collapse dir of all regions
		for (int i = 0; i < regions.size(); i++) {
			TriRegion tr = (TriRegion) regions.get(i);
			tr.collapseDir = 0;
		}
		
		HashSet edgeRegions = new HashSet();
		//find regions on edge,determine normals
		for (int k = 0; k < regions.size(); k++) {
			TriRegion tr = (TriRegion) regions.get(k);
			Point min = tr.r.minPt();
			Point max = tr.r.maxPt();
			int neighborCounts[] = new int[7];
			for (int i = min.x; i < max.x; i += radius)
				for (int j = min.y; j < max.y; j += radius) {
					if (!tr.ptInside(i,j)) continue; //skip if not in triangle
					Point p = screenToArray(i,j);
					if (frameB[p.x][p.y] == EDGE) {
						for (int m = 0; m < 7; m++) 
							if (!isEmpty(move(p,m))) neighborCounts[m]++;
					}
				}
			if (arraySum(neighborCounts) > 0) {
				edgeRegions.add(tr);
				setNormalFromCounts(tr,neighborCounts);
			}
		}
		//determine normal for all other regions
		boolean normalChanged = false;
		do {
			normalChanged = false;
			for (int i = 0; i < regions.size(); i++) {
				TriRegion tr = (TriRegion) regions.get(i);
				if (tr.collapseDir != 0) continue;
				//find neighboring triangular regions
				TriRegion neighbors[] = new TriRegion[3];
				neighbors[0] = findRegion(tr.triX-1,tr.triY);
				neighbors[1] = findRegion(tr.triX+1,tr.triY);
				if (tr.triX %2 == 0) {
					neighbors[2] = findRegion(tr.triX,tr.triY+1);
				}
				else
				{
					neighbors[2] = findRegion(tr.triX,tr.triY-1);
				}
				normalChanged = normalFromNeighbors(tr,neighbors);
			}
		}
		while (normalChanged);
	}
	
	public boolean normalFromNeighbors(TriRegion tr, TriRegion n[]) {
		int dirs[] = new int[7];
		for (int i = 0; i < n.length; i++) {
			if ((n[i] != null)) 
				{
					if (n[i].collapseDir != 0)
						dirs[n[i].collapseDir]++;
					//else
						//return false;
				}
		}
		if (arraySum(dirs) == 0)return false;
		setNormalFromCounts(tr,dirs);
		
		for (int i = 0; i < n.length; i++) {
			if (n[i] != null) System.out.print(n[i].collapseDir + " ");
		}
		System.out.println(tr.collapseDir);
		return true;
	}
	
	public int arraySum(int a[]) {
		int sum = 0;
		for (int i = 0; i < a.length; i++)
			sum += a[i];
		return sum;
	}
	
	public TriRegion findRegion(int triX,int triY) {
		for (int i = 0; i < regions.size(); i++) {
			TriRegion tr = (TriRegion) regions.get(i);
			if ((tr.triX == triX) && (tr.triY == triY)) return tr;
		}
		return null;
	}
	
	public void setNormalFromCounts(TriRegion tr,int counts[]) {
		//3-way averaging
		int avgCounts[] = new int[7];
		for (int i = 1; i < 7; i++) {
			avgCounts[i] = counts[i] + counts[shiftDir(i,1)] + counts[shiftDir(i,-1)];
		}
		int maxCount = 0;
		int dir = 1;
		//pick most popular
		for (int i = 1; i < 7; i++) {
			if (avgCounts[i] > maxCount) {
				dir = i;
				maxCount = avgCounts[i];
			}
		}
		tr.collapseDir = dir;
	}
	
	public void determineFillStates(int[][] frameB) {
		Vector regionsToKill = new Vector();
		
		for (int k = 0; k < regions.size(); k++) {
			TriRegion tr = (TriRegion) regions.get(k);
			Point min = tr.r.minPt();
			Point max = tr.r.maxPt();
			int staticCount = 0;
			int growCount = 0;
			int delCount = 0;
			for (int i = min.x; i < max.x; i += radius)
				for (int j = min.y; j < max.y; j += radius) {
					if (!tr.ptInside(i,j)) continue; //skip if not in triangle
					Point p = screenToArray(i,j);
					switch(frameB[p.x][p.y]){
						case TO_FILL:
							growCount++;
							break;
						case TO_DELETE:
							delCount++;
							break;
						default:
							staticCount++;
					}
				}
			if (growCount+delCount <= inactiveThreshold) {
				regionsToKill.add(tr);
			}
			else if (growCount > delCount) {
				tr.fill = true;
			}
			else {
				tr.fill = false;
			}
		}
		//if any regions need to be removed, kill them
		for (int i = 0; i < regionsToKill.size(); i++) {
			regions.remove(regionsToKill.get(i));
		}
	}
	
	public void createTriRegions() {
		for (int i = 0; i < maxX - triangleSize;i+= triangleSize)
			for (int j = 0; j < maxY - 2*triangleSize; j += 2*triangleSize)
			{
				int xMid = i + (int)Math.round(Math.ceil(triangleSize / 2.0)) -1;
				//System.out.println(i + " " + xMid + " " + j);
				TriRegion tr = new TriRegion();
				tr.r = new Triangle();
				tr.r.a = arrayToScreen(i,j);
				tr.r.b = arrayToScreen(i+triangleSize,j);
				tr.r.c = arrayToScreen(xMid,j+triangleSize);
				tr.triX = 2*i/triangleSize;
				tr.triY = j/triangleSize;
				regions.add(tr);
				
				tr = new TriRegion();
				tr.r = new Triangle();
				tr.r.c = arrayToScreen(i,j+2*triangleSize);
				tr.r.b = arrayToScreen(i+triangleSize,j+2*triangleSize);
				tr.r.a = arrayToScreen(xMid,j+triangleSize);
				tr.triX = 2*i/triangleSize;
				tr.triY = j/triangleSize + 1;
				regions.add(tr);
				
				tr = new TriRegion();
				tr.r = new Triangle();
				tr.r.c = arrayToScreen(i+triangleSize,j+2*triangleSize);
				tr.r.b = arrayToScreen(xMid+triangleSize,j+triangleSize);
				tr.r.a = arrayToScreen(xMid,j+triangleSize);
				tr.triX = 2*i/triangleSize + 1;
				tr.triY = j/triangleSize + 1;
				regions.add(tr);
				
				tr = new TriRegion();
				tr.r = new Triangle();
				tr.r.a = arrayToScreen(i+triangleSize,j);
				tr.r.b = arrayToScreen(xMid+triangleSize,j+triangleSize);
				tr.r.c = arrayToScreen(xMid,j+triangleSize);
				tr.triX = 2*i/triangleSize + 1;
				tr.triY = j/triangleSize;
				regions.add(tr);
			}
	}
	
	public void setDistanceField (int frameB[][]) {
		for (int i = 0; i < maxX; i++)
			for (int j = 0; j  <maxY; j++) {
				switch(frameB[i][j]) {
					case VOID:
					case FILLED:
						distanceField[i][j] = 0;
						break;
					case EDGE:
						distanceField[i][j] = 1;
						distanceMax = 1;
						break;
					default:
						distanceField[i][j] = -1; 
				}
			}
		boolean changed = false;
		do {
			changed = false;
			for (int i = 0; i < maxX; i++)
				for (int j = 0; j  <maxY; j++) {
					if (distanceField[i][j] == -1) {
						int newval = newDistance(i,j);
						if (newval != -1) {
							distanceField[i][j] = newval;
							if (newval > distanceMax) distanceMax = newval;
							changed = true;
						}
					}
				}
		} while (changed);
		
		System.out.println("dmax:" + distanceMax);
	}
	
	int newDistance(int x, int y) {
		int ret = -1;
		Point p = new Point (x,y);
		for (int i = 1; i < 7; i++) {
			Point p2 = move(p,i);
			if (outOfBounds(p)) continue;
			if (distanceField[p2.x][p2.y] > -1) {
				if (ret == -1) ret = distanceField[p2.x][p2.y]+1;
				else if (ret > distanceField[p2.x][p2.y]+1)
					ret = distanceField[p2.x][p2.y]+1;
			}
		}
		return ret;
	}
	
	public void createHoles() {
		int numHoles = (int)Math.round(holeDensity * maxX * maxY);
		for (int i = 0; i < numHoles; i++)
		{
			Hole h = makeHole(new Point(_r.nextInt(maxX), _r.nextInt(maxY)));
			if (h != null) holes.add(h);
		}
	}
	
	public void init() {
		super.init();
		if (imageName != null) {
				fillFromImage(imageName);
				createHoles();
		}
		if (targetImage != null) {
			regionsFromImage(targetImage);
			repaint();
		}
	}
	
	public static void main(String[] args) {
		ImageKeyframing ik = new ImageKeyframing(100, 100);
		ik.radius = 2;
		ik.setSize(410, 410);
		ik.saveFrames = true;
		ik.sleepTime = 100;
		ik.imageName = "foo.png";
		ik.targetImage = "foo2.png";
		ik.holeDensity = 0.01f;
		ik.mainLoop();
	}
	
	String imageName = null;
	String targetImage = null;
	float holeDensity = 0.01f;
	int triangleSize = 15;
	int inactiveThreshold = 5; //number of active pixels in a region before it's used
	
	public static final int TO_FILL = 14;
	public static final int TO_DELETE = 28;
	public static final int EDGE = 19;
}
