// Spatial Index Library
//
// Copyright (C) 2003 Navel Ltd.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Contact information:
//  Mailing address:
//    Marios Hadjieleftheriou
//    University of California, Riverside
//    Department of Computer Science
//    Surge Building, Room 310
//    Riverside, CA 92521
//
//  Email:
//    marioh@cs.ucr.edu

#include "../spatialindex/SpatialIndexImpl.h"

#include "Statistics.h"

using namespace SpatialIndex::MVRTree;

Statistics::Statistics()
{
	reset();
}

Statistics::Statistics(const Statistics& s)
{
	m_reads  = s.m_reads;
	m_writes = s.m_writes;
	m_splits = s.m_splits;
	m_hits   = s.m_hits;
	m_misses = s.m_misses;
	m_nodes  = s.m_nodes;
	m_deadIndexNodes = s.m_deadIndexNodes;
	m_deadLeafNodes = s.m_deadLeafNodes;
	m_adjustments = s.m_adjustments;
	m_queryResults = s.m_queryResults;
	m_data = s.m_data;
	m_totalData = s.m_totalData;
	m_treeHeight = s.m_treeHeight;
	m_nodesInLevel = s.m_nodesInLevel;
}

Statistics::~Statistics()
{
}

Statistics& Statistics::operator=(const Statistics& s)
{
	if (this != &s)
	{
		m_reads  = s.m_reads;
		m_writes = s.m_writes;
		m_splits = s.m_splits;
		m_hits   = s.m_hits;
		m_misses = s.m_misses;
		m_nodes  = s.m_nodes;
		m_deadIndexNodes = s.m_deadIndexNodes;
		m_deadLeafNodes = s.m_deadLeafNodes;
		m_adjustments = s.m_adjustments;
		m_queryResults = s.m_queryResults;
		m_data = s.m_data;
		m_totalData = s.m_totalData;
		m_treeHeight = s.m_treeHeight;
		m_nodesInLevel = s.m_nodesInLevel;
	}

	return *this;
}

unsigned long Statistics::getReads() const
{
	return m_reads;
}

unsigned long Statistics::getWrites() const
{
	return m_writes;
}

unsigned long Statistics::getNumberOfNodes() const
{
	return m_nodes;
}

unsigned long Statistics::getNumberOfData() const
{
	return m_data;
}

unsigned long Statistics::getSplits() const
{
	return m_splits;
}

unsigned long Statistics::getHits() const
{
	return m_hits;
}

unsigned long Statistics::getMisses() const
{
	return m_misses;
}

unsigned long Statistics::getAdjustments() const
{
	return m_adjustments;
}

unsigned long Statistics::getQueryResults() const
{
	return m_queryResults;
}

unsigned long Statistics::getTreeHeight() const
{
	unsigned long ret = 0;

	for (unsigned long cIndex = 0; cIndex < m_treeHeight.size(); cIndex++)
	{
		ret = std::max(ret, m_treeHeight[cIndex]);
	}

	return ret;
}

unsigned long Statistics::getNumberOfNodesInLevel(unsigned long l) const throw (Tools::IndexOutOfBoundsException)
{
	try
	{
		return m_nodesInLevel.at(l);
	}
	catch (...)
	{
		throw Tools::IndexOutOfBoundsException(l);
	}
}

void Statistics::reset()
{
	m_reads = 0;
	m_writes = 0;
	m_splits = 0;
	m_hits = 0;
	m_misses = 0;
	m_nodes = 0;
	m_deadIndexNodes = 0;
	m_deadLeafNodes = 0;
	m_adjustments = 0;
	m_queryResults = 0;
	m_data = 0;
	m_totalData = 0;
	m_treeHeight.clear();
	m_nodesInLevel.clear();
}

std::ostream& SpatialIndex::MVRTree::operator<<(std::ostream& os, const Statistics& s)
{
	using std::endl;

	os	<< "Reads: " << s.m_reads << endl
		<< "Writes: " << s.m_writes << endl
		<< "Hits: " << s.m_hits << endl
		<< "Misses: " << s.m_misses << endl
		<< "Number of live data: " << s.m_data << endl
		<< "Total number of data: " << s.m_totalData << endl
		<< "Number of nodes: " << s.m_nodes << endl
		<< "Numer of dead index nodes: " << s.m_deadIndexNodes << endl
		<< "Numer of dead leaf nodes: " << s.m_deadLeafNodes << endl;

	for (unsigned long cTree = 0; cTree < s.m_treeHeight.size(); cTree++)
	{
		os << "Tree " << cTree << ", Height " << s.m_treeHeight[cTree] << endl;
	}

	for (unsigned long cLevel = 0; cLevel < s.m_nodesInLevel.size(); cLevel++)
	{
		os << "Level " << cLevel << " pages: " << s.m_nodesInLevel[cLevel] << endl;
	}

	os	<< "Splits: " << s.m_splits << endl
		<< "Adjustments: " << s.m_adjustments << endl
		<< "Query results: " << s.m_queryResults << endl;

	return os;
}
