// Spatial Index Library
//
// Copyright (C) 2003 Navel Ltd.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Contact information:
//  Mailing address:
//    Marios Hadjieleftheriou
//    University of California, Riverside
//    Department of Computer Science
//    Surge Building, Room 310
//    Riverside, CA 92521
//
//  Email:
//    marioh@cs.ucr.edu

#ifndef __spatialindex_mvrtree_node_h
#define __spatialindex_mvrtree_node_h

namespace SpatialIndex
{
	namespace MVRTree
	{
		class MVRTree;
		class Leaf;
		class Index;
		class Node;

		typedef Tools::PoolPointer<Node> NodePtr;

		class Node : public SpatialIndex::INode
		{
		public:
			virtual ~Node();

			//
			// Tools::IObject interface
			//
			virtual IObject* clone() throw (Tools::NotSupportedException);

			//
			// Tools::ISerializable interface
			//
			virtual unsigned long getByteArraySize();
			virtual unsigned long loadFromByteArray(byte* const data);
			virtual void storeToByteArray(unsigned long& len, byte** data);

			//
			// SpatialIndex::IEntry interface
			//
			virtual long getIdentifier() const;
			virtual void getShape(IShape** out) const throw (std::exception);

			//
			// SpatialIndex::INode interface
			//
			virtual unsigned long getChildrenCount() const;
			virtual long getChildIdentifier(unsigned long index)  const throw (Tools::IndexOutOfBoundsException);
			virtual void getChildShape(unsigned long index, IShape** out)  const throw (Tools::IndexOutOfBoundsException, std::exception);
			virtual unsigned long getLevel() const;
			virtual bool isIndex() const;
			virtual bool isLeaf() const;

		private:
			Node();
			Node(MVRTree* pTree, long id, unsigned long level, unsigned long capacity);

			virtual Node& operator=(const Node&);

			virtual void insertEntry(unsigned long dataLength, byte* pData, TimeRegion& mbr, long id);
			virtual bool deleteEntry(unsigned long index);

			virtual bool insertData(
				unsigned long dataLength, byte* pData, TimeRegion& mbr, long id, std::stack<long>& pathBuffer,
				TimeRegion& mbr2, long id2, bool bInsertMbr2 = false, bool forceAdjust = false);
			virtual void insertData(TimeRegion& mbr1, long id1, TimeRegion& mbr2, long id2, Node* oldVersion, std::stack<long>& pathBuffer);
			virtual bool deleteData(long id, double delTime, std::stack<long>& pathBuffer, bool adjustMBR = false);

			virtual void rtreeSplit(
				unsigned long dataLength, byte* pData, TimeRegion& mbr, long id, std::vector<long>& group1, std::vector<long>& group2,
				TimeRegion& mbr2, long id2, bool bInsertMbr2 = false);
			virtual void rstarSplit(
				unsigned long dataLength, byte* pData, TimeRegion& mbr, long id, std::vector<long>& group1, std::vector<long>& group2,
				TimeRegion& mbr2, long id2, bool bInsertMbr2 = false);

			virtual void pickSeeds(unsigned long& index1, unsigned long& index2, unsigned long total);

			virtual NodePtr chooseSubtree(const TimeRegion& mbr, unsigned long level, std::stack<long>& pathBuffer) = 0;
			virtual NodePtr findLeaf(const TimeRegion& mbr, long id, std::stack<long>& pathBuffer) = 0;
			virtual NodePtr findNode(const TimeRegion& mbr, long id, std::stack<long>& pathBuffer);

			virtual void split(
				unsigned long dataLength, byte* pData, TimeRegion& mbr, long id, NodePtr& left, NodePtr& right,
				TimeRegion& mbr2, long id2, bool bInsertMbr2 = false) = 0;

			MVRTree* m_pTree;
				// Parent of all nodes.

			unsigned long m_level;
				// The level of the node in the tree.
				// Leaves are always at level 0.

			long m_identifier;
				// The unique ID of this node.

			unsigned long m_children;
				// The number of children pointed by this node.

			unsigned long m_capacity;
				// Specifies the node capacity.

			TimeRegion m_nodeMBR;
				// The minimum bounding region enclosing all data contained in the node.

			byte** m_pData;
				// The data stored in the node.

			TimeRegionPtr* m_ptrMBR;
				// The corresponding data MBRs.

			long* m_pIdentifier;
				// The corresponding data identifiers.

			unsigned long* m_pDataLength;

			unsigned long m_totalDataLength;

			class RstarSplitEntry
			{
			public:
				RstarSplitEntry(TimeRegion* pr, long id, unsigned long dimension) :
					m_pRegion(pr), m_id(id), m_sortDim(dimension) {}

				static int compareLow(const void* pv1, const void* pv2)
				{
					RstarSplitEntry* pe1 = * (RstarSplitEntry**) pv1;
					RstarSplitEntry* pe2 = * (RstarSplitEntry**) pv2;

					if (pe1->m_pRegion->m_pLow[pe1->m_sortDim] < pe2->m_pRegion->m_pLow[pe2->m_sortDim]) return -1;
					if (pe1->m_pRegion->m_pLow[pe1->m_sortDim] > pe2->m_pRegion->m_pLow[pe2->m_sortDim]) return 1;
					return 0;
				}

				static int compareHigh(const void* pv1, const void* pv2)
				{
					RstarSplitEntry* pe1 = * (RstarSplitEntry**) pv1;
					RstarSplitEntry* pe2 = * (RstarSplitEntry**) pv2;

					if (pe1->m_pRegion->m_pHigh[pe1->m_sortDim] < pe2->m_pRegion->m_pHigh[pe2->m_sortDim]) return -1;
					if (pe1->m_pRegion->m_pHigh[pe1->m_sortDim] > pe2->m_pRegion->m_pHigh[pe2->m_sortDim]) return 1;
					return 0;
				}

				TimeRegion* m_pRegion;
				long m_id;
				unsigned long m_sortDim;
			}; // RstarSplitEntry

			class DeleteDataEntry
			{
			public:
				DeleteDataEntry(unsigned long l, double d) : m_index(l), m_increase(d) {}

				static bool compare(DeleteDataEntry e1, DeleteDataEntry e2) { return e1.m_increase < e2.m_increase; }

				unsigned long m_index;
				double m_increase;
			}; // DeleteDataEntry

			// Needed to access protected members without having to cast from Node.
			// It is more efficient than using member functions to access protected members.
			friend class MVRTree;
			friend class Leaf;
			friend class Index;
			friend class Tools::PointerPool<Node>;
		}; // Node
	}
}

#endif /*__spatialindex_mvrtree_node_h*/
