// Spatial Index Library
//
// Copyright (C) 2004 Navel Ltd.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Contact information:
//  Mailing address:
//    Marios Hadjieleftheriou
//    University of California, Riverside
//    Department of Computer Science
//    Surge Building, Room 310
//    Riverside, CA 92521
//
//  Email:
//    marioh@cs.ucr.edu

#ifndef __spatialindex_movingregion_h
#define __spatialindex_movingregion_h

namespace SpatialIndex
{
	class MovingRegion : public TimeRegion, public Tools::Geometry::IEvolvingShape
	{
	public:
		MovingRegion();
		MovingRegion(
			const double* pLow, const double* pHigh,
			const double* pVLow, const double* pVHigh,
			const Tools::IInterval& ti, unsigned long dimension);
		MovingRegion(
			const double* pLow, const double* pHigh,
			const double* pVLow, const double* pVHigh,
			double tStart, double tEnd, unsigned long dimension);
		MovingRegion(
			const Tools::Geometry::Point& low, const Tools::Geometry::Point& high,
			const Tools::Geometry::Point& vlow, const Tools::Geometry::Point& vhigh,
			const Tools::IInterval& ti);
		MovingRegion(
			const Tools::Geometry::Point& low, const Tools::Geometry::Point& high,
			const Tools::Geometry::Point& vlow, const Tools::Geometry::Point& vhigh,
			double tStart, double tEnd);
		MovingRegion(const Region& mbr, const Region& vbr, const IInterval& ivI);
		MovingRegion(const Region& mbr, const Region& vbr, double tStart, double tEnd);
		MovingRegion(const MovingPoint& low, const MovingPoint& high);
		MovingRegion(const MovingRegion& in);
		virtual ~MovingRegion();

		virtual MovingRegion& operator=(const MovingRegion& r);
		virtual bool operator==(const MovingRegion&) const;

		bool isShrinking() const;

		virtual double getLow(unsigned long index, double t) const throw (Tools::IndexOutOfBoundsException);
		virtual double getHigh(unsigned long index, double t) const throw (Tools::IndexOutOfBoundsException);
		virtual double getExtrapolatedLow(unsigned long index, double t) const throw (Tools::IndexOutOfBoundsException);
		virtual double getExtrapolatedHigh(unsigned long index, double t) const throw (Tools::IndexOutOfBoundsException);
		virtual double getVLow(unsigned long index) const throw (Tools::IndexOutOfBoundsException);
		virtual double getVHigh(unsigned long index) const throw (Tools::IndexOutOfBoundsException);

		virtual bool intersectsRegionInTime(const MovingRegion& r) const;
		virtual bool intersectsRegionInTime(const MovingRegion& r, Tools::IInterval& out) const;
		virtual bool intersectsRegionInTime(const Tools::IInterval& ivI, const MovingRegion& r, Tools::IInterval& ret) const;
		virtual bool containsRegionInTime(const MovingRegion& r) const;
		virtual bool containsRegionInTime(const Tools::IInterval& ivI, const MovingRegion& r) const;
		virtual bool containsRegionAfterTime(double t, const MovingRegion& r) const;

		virtual double getProjectedSurfaceAreaInTime() const;
		virtual double getProjectedSurfaceAreaInTime(const IInterval& ivI) const;

		virtual double getCenterDistanceInTime(const MovingRegion& r) const;
		virtual double getCenterDistanceInTime(const IInterval& ivI, const MovingRegion& r) const;

		virtual bool intersectsRegionAtTime(double t, const MovingRegion& r) const;
		virtual bool containsRegionAtTime(double t, const MovingRegion& r) const;

		virtual bool intersectsPointInTime(const MovingPoint& p) const;
		virtual bool intersectsPointInTime(const MovingPoint& p, Tools::IInterval& out) const;
		virtual bool intersectsPointInTime(const Tools::IInterval& ivI, const MovingPoint& p, Tools::IInterval& out) const;
		virtual bool containsPointInTime(const MovingPoint& p) const;
		virtual bool containsPointInTime(const Tools::IInterval& ivI, const MovingPoint& p) const;

		//virtual bool intersectsPointAtTime(double t, const MovingRegion& in) const;
		//virtual bool containsPointAtTime(double t, const MovingRegion& in) const;

		virtual void combineRegionInTime(const MovingRegion& r);
		virtual void combineRegionAfterTime(double t, const MovingRegion& r);
		virtual void getCombinedRegionInTime(MovingRegion& out, const MovingRegion& in) const;
		virtual void getCombinedRegionAfterTime(double t, MovingRegion& out, const MovingRegion& in) const;

		virtual double getIntersectingAreaInTime(const MovingRegion& r) const;
		virtual double getIntersectingAreaInTime(const IInterval& ivI, const MovingRegion& r) const;

		//
		// IObject interface
		//
		virtual MovingRegion* clone() throw (Tools::NotSupportedException);

		//
		// ISerializable interface
		//
		virtual unsigned long getByteArraySize();
		virtual unsigned long loadFromByteArray(byte* const data);
		virtual void storeToByteArray(unsigned long& len, byte** data);

		//
		// IEvolvingShape interface
		//
		virtual void getVMBR(Tools::Geometry::Region& out) const;
		virtual void getMBRAtTime(double t, Tools::Geometry::Region& out) const;

		//
		// ITimeShape interface
		//
		virtual double getAreaInTime() const;
		virtual double getAreaInTime(const IInterval& ivI) const;
		virtual double getIntersectingAreaInTime(const Tools::Geometry::ITimeShape& r) const;
		virtual double getIntersectingAreaInTime(const IInterval& ivI, const Tools::Geometry::ITimeShape& r) const;

		virtual void makeInfinite(unsigned long dimension);
		virtual void makeDimension(unsigned long dimension);

	private:
		void initialize(
			const double* pLow, const double* pHigh,
			const double* pVLow, const double* pVHigh,
			double tStart, double tEnd, unsigned long dimension);

	public:
		class CrossPoint
		{
		public:
			double m_t;
			unsigned long m_dimension;
			unsigned long m_boundary;
			const MovingRegion* m_to;

			struct ascending: public std::binary_function<CrossPoint&, CrossPoint&, bool>
			{
				bool operator()(const CrossPoint& __x, const CrossPoint& __y) const { return __x.m_t > __y.m_t; }
			};
		}; // CrossPoint

	public:
		double* m_pVLow;
		double* m_pVHigh;

		friend std::ostream& operator<<(std::ostream& os, const MovingRegion& r);
	}; // MovingRegion

	std::ostream& operator<<(std::ostream& os, const MovingRegion& r);
}

#endif /* __spatialindex_movingregion_h */
