#include <stdlib.h>
#include <string>
#include <stdarg.h>
#include <iostream>
#include <iomanip>
#include <list>
#include <ext/hash_set>

#include "CatomWorld.hxx"
#include "LatticeFunctions.hxx"
#include "Modeler.hxx"

#include <ode/config.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <GL/glx.h>
#include <ode/ode.h>
#include "graphics/graphics.h"
#include "graphics/internal.h"


#ifndef WORLDBUILDER_H
#define WORLDBUILDER_H


class WorldBuilder {
public:
  WorldBuilder(); //constructor. note: worldbuilder must be attached to a CatomWorld with setWorld before it can be used
  ~WorldBuilder();

  //used primarily by Gtk interface (and rarely by the simulator) to set WorldBuilder's properties
  void setWorld(CatomWorld* wp);
  void setLattice(string l);
  void setMode(int i);
  void setScrollMode(int m);
  void setColor(int r, int g, int b);
  void setFillStyle(bool solid);
  void setStamp(string s);
  void setStampMode(bool m);
  bool getStampMode();

  //called by Gtk save button; saves world in specified file
  void saveWorldAs(string filename);

  //called during each rendering step by the simulator
  //used to render catomCatomCursor and as a chance to modify
  //the world in a thread-safe manner
  void updateBuilder();

  //called by Gtk buttons; these methods set up
  //world-altering requests which are later processed
  //in updateBuilder()
  void import(string filename, int size, float precision, bool flipYZ, bool atCatomCatomCursor);
  void requestPrimitive(string type, vector<float> params, float precision, bool flipYZ, bool asStamp);
  void requestGUIDChange(long unsigned int newg);
  void requestOrganizeGUIDs();
  void clearAll();

  //callbacks used in events.cpp
  bool mouseMoved(int x, int y);
  bool buttonPressed(unsigned int button, int x, int y);

private:

  struct color{
    float r;
    float g;
    float b;
  } catomColor; 
  //keeps track of the current catom-drawing color

  struct primitiveRequest{
    bool done;
    string type;
    vector<float> params;
    float precision;
    bool solid;
    bool flipYZ;
    bool asStamp;
  } primReq; 
  //holds a request for a primitive (to be created from catoms and inserted into the world)

  struct modelRequest{
    bool done;
    string filename;
    int size;
    float precision;
    bool solid;
    bool flipYZ;
    bool asStamp;
  } modelReq; 
  //holds a request for a STL or OBJ model (to be created from catoms and inserted into the world)

  unsigned long int newGUID; //holds the new GUID for a given GUID change request
  bool GUIDChangeReq; //holds the status of GUID change request 

  bool GUIDOrgReq; //holds the status of an "organize GUIDs 1..n" request

  bool clearReq; //true if there is a pending GUID request

  string saveName; //holds the name of a file if there is a requested save operation

  //the pointer to the CatomWorld of the current simulation
  //used primarily for saving
  CatomWorld* worldPtr;
  
  //a string specifying the lattice.
  //valid choices are determined by latticeFunctions.cxx
  //currently, choices are:
  //"cubic", "hexp" (hexagonal planar), "fcc" (face-centered cubic), "bcc" (body-centered cubic)
  string lattice;

  //these keep track of stamp-related state:
  string stampName; //the filename of the stamp
  bool newStamp; //has a new stamp been requested?
  list<Point3D> stamp; //holds the locations of catoms in the current stamp

  //holds a number one greater than the largest known GUID of any catom in *worldPtr
  //this is used to create new catoms with unique GUIDs
  unsigned long int GUID;
  unsigned long int catomCount;

  //catomCatomCursor information
  Point3D catomCursor; //catomCursor's position
  Point3D cursor;      //real cursor's position
  bool legalCursor;    //true if the catomCursor is in a legal catom position

  bool fillSolid; //determines whether a given model is created solid or as a shell
  int mode; //holds an integer corresponding to the catom manipulation tool currently in use (create, erase, etc)
  int scrollmode; //holds the mouse scroll wheel mode (0 means wheel controls camera Z, 1 means wheel controls catomCursor Z)
  bool stampMode; //tells whether the user is in stamp mode

  void moveCatomCursor();
  //used internally to update the position of the catom insertion/deletion cursor when the mouse cursor's position changes


  //the following structures and functions are used by the builder to track the locations and properties of catoms placed in the world
  struct builderCatom{
    unsigned long GUID;
    struct color color;
    Point3D pos;
  };

  hash_map<unsigned long, builderCatom> builderCatoms;
  
  void clearCatoms();
  void deleteCatom(unsigned long int GUID);
  builderCatom getCatom(unsigned long int GUID);
  bool addCatom( unsigned long GUID, Point3D pos, color c);
  bool addCatom(builderCatom c); 
  unsigned long int catomAt(Point3D pos);
  void swap(unsigned long int GUID1, unsigned long int GUID2);
  void organizeGUIDs();
 
};

#endif
