#define DEG_TO_RAD (M_PI/180.0)

#include "WorldBuilder.hxx"

using namespace std;


///////////////////////////////////////////////////////////////////////////////
//WorldBuilder::WorldBuilder
//
//This constructor sets up a worldbuilder which is ready to be
//attached to a CatomWorld using setWorld

WorldBuilder::WorldBuilder() {

  lattice = "cubic"; //Initially, these settings are reflected in the Gtk window
  fillSolid = true; 

  mode=-1;      //begin in navigation mode
  scrollmode=0; //..with the scroll wheel controlling the camera's z position

  catomColor.r = 200;
  catomColor.g = 200;
  catomColor.b = 200; //initial catom color is grey

  catomCursor.setZ(CatomSim::catom_radius); //set up initial cursor positions
  legalCursor = false;
  cursor.setZ(CatomSim::catom_radius);

  modelReq.done = true; //no CatomWorld-changing requests have been made yet
  primReq.done = true;
  GUIDChangeReq=false;
  clearReq = false;
  GUIDOrgReq = false;
  
  saveName = "";

  stampName = "";   //initially there is no stamp created
  newStamp=false;
  stampMode = false;
}

WorldBuilder::~WorldBuilder(){}

///////////////////////////////////////////////////////////////////////////////
//WorldBuilder::setColor
//
//Used by the Gtk color selection widget to set the color 
//to be used for any new catoms

void WorldBuilder::setColor(int r, int g, int b)
{
  catomColor.r = r/255;
  catomColor.g = g/255;
  catomColor.b = b/255;  
}


///////////////////////////////////////////////////////////////////////////////
//WorldBuilder::setWorld
//
//Associates a particular CatomWorld with this builder
//any changes requested of the worldbuilder will be made to this world
//additionally, an initial scan sets the GUID of the next catom to be created
//to one greater than the greatest GUID in the CatomWorld

void WorldBuilder::setWorld(CatomWorld* wp) {
  worldPtr = wp;
  GUID = 1;
  hash_map<const unsigned long, CatomSim *, hash<const unsigned long>, equl>::iterator c_i;
  for (c_i = worldPtr->catomHash.begin(); c_i != worldPtr->catomHash.end(); c_i++ ) {  
    if(c_i->first>=GUID) GUID = c_i->first + 1;

    const dReal *dpos; //passed to ODE to get positions of catoms
    dpos = dGeomGetPosition(c_i->second->shape); 
    Point3D point(dpos[0], dpos[1], dpos[2]); //hold the position of the current catom
    
    catomColor.r = c_i->second->red;
    catomColor.g = c_i->second->green;
    catomColor.b = c_i->second->blue;

    addCatom(c_i->first, point, catomColor); 
  }
  catomCount = worldPtr->num_catoms;
  worldPtr->clearCatoms();
}


///////////////////////////////////////////////////////////////////////////////
//WorldBuilder state mutators
//
//These are used by the Gtk window (through callbacks.cpp) to set
//the scrolling mode, lattice, interaction mode, and fill style of the builder

void WorldBuilder::setScrollMode(int m){
  scrollmode = m;
}

void WorldBuilder::setLattice(string l) {
  lattice = l;
}

void WorldBuilder::setMode(int i) {
  mode = i;
}

void WorldBuilder::setFillStyle(bool solid){
  fillSolid = solid;
}

void WorldBuilder::setStamp(string s) {
  stampName = s;
  newStamp = true;
}

void WorldBuilder::setStampMode(bool m) {
  stampMode = m;
}


bool WorldBuilder::getStampMode() {
  return stampMode;
}

//end state mutators


///////////////////////////////////////////////////////////////////////////////
//WorldBuilder::buttonPressed
//
//Called by callbacks.cpp when a mouse button is pressed within the GL window
//WorldBuilder interprets this click depending on the mouse button and
//on the state of the builder
//If buttonPressed returns true, the event is not handled by the rest of events.cpp
//Otherwise, the event is passed on to other handlers in events.cpp
//The builder is sensitive to the left mouse button and the mouse wheel (when scrollmode = 1); the right button
//is passed for normal handling by events.cpp (as is the mouse wheel when scrollmode != 1)

bool WorldBuilder::buttonPressed(unsigned int button, int x, int y) {
  char* msg; //used to sprintf text to be passed to Gtk

  if(button == Button1) { //if the user left-clicked
    //find the GUID of the catom the user has clicked
    //(if no catom is clicked, target = 0)
    unsigned long int target = catomAt(catomCursor); 

    switch (mode) {
    case 0: //in mode 0 (creation mode) a left click creates a catom

      if(!stampMode){
	if(legalCursor && !target){ //make sure this is a legal catom position
	  addCatom(GUID, catomCursor, catomColor);
	  GUID++;
	  
	  //update the catomCursor position
	  moveCatomCursor();
	  
	  //update the catomCountLabel GtkLabel with the new number of catoms
	  msg = new char[100];
	  sprintf(msg, "%lu", catomCount);
	  update_widget("catomCountLabel", "GtkLabel", msg);
	  delete msg;
	}
      }
      else{
	if(legalCursor){
	  
	  //progress bar
	  int bar = 0;
	  int counter =0;
	  char msg4[50];
	  int s = stamp.size();
	  sprintf(msg4, "Checking %d catoms for overlaps", s);
	  update_widget("progress_bar", "GtkProgressBar", msg4);
	  
	  //delete overlaps from stamp
	  list<Point3D> tempStamp = stamp;
	  list<Point3D>::iterator s_i;
	  for(s_i=tempStamp.begin(); s_i!=tempStamp.end();){
	    	    
	    *s_i += catomCursor;
	    if(catomAt(*s_i)){
	      list<Point3D>::iterator temp = s_i;
	      s_i++;
	      tempStamp.erase(temp);
	    }
	    else{
	      s_i++;
	    }
	    
	    counter++;
	    //progress bar
	    while(int(counter/(s/100+1))>bar+1) {
	      bar++;
	      update_widget("progress_bar", "GtkProgressBar", ((float)counter)/((float)s));
	    } 
	  }
	  
	  //progress bar
	  bar = 0;
	  counter =0;
	  s = tempStamp.size();
	  sprintf(msg4, "Creating %d catoms", s);
	  update_widget("progress_bar", "GtkProgressBar", msg4);
	  
	  list<Point3D>::iterator l_i;
	  for(l_i=tempStamp.begin(); l_i!=tempStamp.end(); l_i++){
	    addCatom(GUID, *l_i, catomColor);
	    GUID++;
	    
	    counter++;
	    //progress bar
	    while(int(counter/(s/100+1))>bar+1) {
	      bar++;
	      update_widget("progress_bar", "GtkProgressBar", ((float)counter)/((float)s));
	    } 
	  }
	  
	  update_widget("progress_bar", "GtkProgressBar", 0);
	  update_widget("progress_bar", "GtkProgressBar", "");
	  
	  //update the catomCursor position
	  moveCatomCursor();
	  //update the catomCountLabel GtkLabel with the new number of catoms
	  msg = new char[100];
	  sprintf(msg, "%lu", catomCount);
	  update_widget("catomCountLabel", "GtkLabel", msg);
	  delete msg;
	}
      }
      break;
      
    case 1: //in mode 1 (deletion mode) a left click deletes a catom
      
      if(!stampMode){
	if(target){ //if there is really a catom there...
	  
	  //delete this catom from the world
	  //(this is thread-safe because mouse events in the GL window come from the main thread)
	  deleteCatom(target);

	  //update the catomCursor position
	  moveCatomCursor();
	  
	  //update the catomCountLabel GtkLabel with the new number of catoms
	  msg = new char[100];
	  sprintf(msg, "%lu", catomCount);
	  update_widget("catomCountLabel", "GtkLabel", msg);
	  delete msg;
	}
      }
      else{
	if(legalCursor){
	  
	  //progress bar
	  int bar = 0;
	  int counter =0;
	  char msg4[50];
	  int s = stamp.size();
	  sprintf(msg4, "Deleting %d catoms", s);
	  update_widget("progress_bar", "GtkProgressBar", msg4);
	  
	  list<Point3D>::iterator s_i;
	  for(s_i=stamp.begin(); s_i!=stamp.end(); s_i++){
	    
	    target = catomAt(*s_i + catomCursor);
	    	    
	    if(target) deleteCatom(target);

	    counter++;
	    //progress bar
	    while(int(counter/(s/100+1))>bar+1) {
	      bar++;
	      update_widget("progress_bar", "GtkProgressBar", ((float)counter)/((float)s));
	    } 
	  }
	  
	  //progress bar
	  update_widget("progress_bar", "GtkProgressBar", 0);
	  update_widget("progress_bar", "GtkProgressBar", "");
	  
	  //update the catomCursor position
	  moveCatomCursor();
	  //update the catomCountLabel GtkLabel with the new number of catoms
	  msg = new char[100];
	  sprintf(msg, "%lu", catomCount);
	  update_widget("catomCountLabel", "GtkLabel", msg);
	  delete msg;
	}
      }
      break;

    case 2: //in mode 2 (selection mode) a left click causes the catom's GUID to be displayed
      
      if(target){ //if a catom is actually selected

	//update the GUIDEntry GtkEntry with the catom's GUID
	msg = new char[100];
	sprintf(msg, "%lu", target);
	update_widget("GUIDEntry", "GtkEntry", msg);
	delete msg;
	
      }
      else{ //if no catom is selected clear the GUIDEntry
	update_widget("GUIDEntry", "GtkEntry",  "");
      }
       
      break;

    case 3: //origin placement mode

      //move all the catoms to move the origin
      hash_map<unsigned long, builderCatom>::iterator b_i;
      for(b_i=builderCatoms.begin(); b_i!=builderCatoms.end(); b_i++){
	b_i->second.pos -= catomCursor + Point3D(0,0,-CatomSim::catom_radius);      
      }

      //change mode back to navigate
      update_widget("moveOriginButton", "GtkToggleButton", false);
      update_widget("navigateButton", "GtkToggleToolButton", true);
      mode = -1;
      legalCursor=false;
      break;
    }
    
    //returning true keeps the rest of the simulator's 
    //event code from handling this event also
    return true;
  }
  else if(button == Button5 && mode != -1 && scrollmode == 1) { 
    //if the wheel scrolls down and the scrollmode is set to move the cursor
    cursor.setZ(cursor.getZ()+CatomSim::catom_radius); //the cursor is moved
    
    moveCatomCursor();
    return true;
  }
  else if(button == Button4 && mode != -1 && scrollmode == 1) {
      //if the wheel scrolls down and the scrollmode is set to move the cursor
    cursor.setZ(cursor.getZ()-CatomSim::catom_radius);//the cursor is moved
    
    if(cursor.getZ()<CatomSim::catom_radius) cursor.setZ(CatomSim::catom_radius); //the cursor may not go through the floor
    
    moveCatomCursor();
    return true;
  }
  
  return false;
}

///////////////////////////////////////////////////////////////////////////////
//WorldBuilder::mouseMoved
//
//Called by events.cpp; when the mouse moves, the representation of the cursor is
//updated and moveCatomCursor is called to update the catomCursor's position

bool WorldBuilder::mouseMoved(int x, int y){
  cursor.setX(x);
  cursor.setY(y);
  return false;
}


///////////////////////////////////////////////////////////////////////////////
//WorldBuilder::moveCatomCursor
//
//called by mouseMoved and buttonPressed; calculates the position of the catom
//insertion/deletion cursor based on the position of the mouse cursor. When in
//catom creation mode, tries to place the cursor on a neighbor of an exisiting 
//catom or on the working plane (usually the xy plane). When in selection or
//deletion mode, tries to place the cursor on an existing catom. When the catom
//cursor can't be placed effectively, it is turned off by setting legalCursor to
//false. 

void WorldBuilder::moveCatomCursor(){
  legalCursor = true;
  if(mode==-1) return; //if we are in navigation mode, no cursor is drawn

  float cpos[4]; //these two arrays are passed by reference
  float cvec[4]; //and used to get the camera position and selection vector
  convert2Dto3D(cursor.getX(), cursor.getY(), cpos, cvec);
  
  float x = cpos[0]; //the camera position
  float y = cpos[1];
  float z = cpos[2];

  float dx = cvec[0]; //the selection vector
  float dy = cvec[1]; //(a ray from the camera position along
  float dz = cvec[2]; // the direction indicated by the mouse)

  float dist_sq = -1.0f; //the square of the distance from the guess catom to the camera
  
  builderCatom guess;
  guess.GUID = 0;

  Point3D minproj; //point on the selection ray nearest to the guess catom 
  float mindist; //the distance from the nearest point on the selection ray to the guess catom
 
  //loop through all catoms; at the end, ctm should hold the one which is within one radius of
  //the selection ray and closest to the camera
  hash_map<unsigned long, builderCatom>::iterator c_i;
  for (c_i = builderCatoms.begin(); c_i != builderCatoms.end(); c_i++ ) {
    Point3D point = c_i->second.pos; //hold the position of the current catom
   
    //we will project the point onto the line segment
    //then we will measure the distance from the point to the projection
    //find the projected length by dividing the dotproduct by the length
    //find the param by dividing by length again
    //this will allow us to find distance from the selection ray to the catom

    Point3D selvector(dx, dy, dz);
    Point3D camera(x, y, z);
    double param = selvector.dot(point-camera)/pow(selvector.getR(),2);
  
    Point3D projection(dx*param + x, dy*param + y, dz*param + z);
    
    float md = point.distanceFrom(projection);

    //if the current catom is within one radius of the selection vector, its distance from the camera is computed
    if (md < CatomSim::catom_radius) {
      float t = (x-point.getX())*(x-point.getX()) + (y-point.getY())*(y-point.getY()) + (z-point.getZ())*(z-point.getZ());
      if (t < dist_sq || dist_sq < 0.0f) { //if this distance is better than the current guess catom...
	dist_sq = t;
	guess = c_i->second; //the current catom becomes the guess catom
	minproj = projection;
	mindist = md;
      }
    }
  }
  
  if (guess.GUID && (pow(minproj.getZ()-z, 2)<pow(cursor.getZ()-z, 2))) { //if we found a good guess       
    Point3D base(guess.pos); //base the rest of our calculations on it
    Point3D exact(base);
    
    if(mode == 0){ //in mode 0 (insert) we will try to calculate which neighbor the user wants to create
      if(mindist>.5*CatomSim::catom_radius){
	x=minproj.getX();
	y=minproj.getY();
	z=minproj.getZ();
      }
      dist_sq = -1.0f;
      list<Point3D> neighbors = latticeNeighbors(base, lattice);
      list<Point3D>::iterator n_i;
      for(n_i = neighbors.begin(); n_i != neighbors.end(); n_i++){
	float t = (n_i->getX()-x)*(n_i->getX()-x) + (n_i->getY()-y)*(n_i->getY()-y) + (n_i->getZ()-z)*(n_i->getZ()-z);
	if ((t < dist_sq || dist_sq < 0.0f) && (catomAt(*n_i) == 0) && (n_i->getZ() > 0)) {
	  dist_sq = t;
	  exact = *n_i;
	}
      }
    }
    catomCursor.setData(exact);  
  }
  else if (cvec[2]!=0) { //if we didn't find a catom, project the catom cursor down to the current drawing plane
    double param = (cursor.getZ()-cpos[2])/cvec[2];
    if(param >= 0){
      Point3D exact(cpos[0]+param*cvec[0], cpos[1]+param*cvec[1], cpos[2]+param*cvec[2]); 
      catomCursor.setData(snapToLattice(exact, lattice));
    }
    else { //if we can't get down to a drawing plane (maybe camera is facing up) then don't place the cursor
      legalCursor=false;
    }
  }

}


void WorldBuilder::updateBuilder() {

  moveCatomCursor();

  if(saveName!=""){ //save the file
    
    //progress bar
    int bar = 0;
    int counter =0;
    char msg[50];
    int s = builderCatoms.size();
    sprintf(msg, "Saving %d catoms", s);
    update_widget("progress_bar", "GtkProgressBar", msg);  
    
    
    hash_map<unsigned long, builderCatom>::iterator b_i;
    for(b_i=builderCatoms.begin(); b_i!=builderCatoms.end(); b_i++){
      builderCatom bc = b_i->second;
      uint8 r = (uint8)(bc.color.r);
      uint8 g = (uint8)(bc.color.g);
      uint8 b = (uint8)(bc.color.b);
      
      //convert builderCatoms to real catoms

      CatomSim *c = new CatomSim(bc.GUID,bc.pos,0,0,0,r,g,b,0);
      worldPtr->addCatom(c, bc.GUID);
      delete c;
      
      counter++;
      //progress bar
      while(int(counter/(s/100+1))>bar+1) {
	bar++;
	update_widget("progress_bar", "GtkProgressBar", ((float)counter)/((float)s));
      } 
    }
    
    update_widget("progress_bar", "GtkProgressBar", "");  
    update_widget("preogress_bar", "GtkProgressBar", 0);
    
    // Open the file for writing
    worldPtr->myFile.fileOpen((char*)saveName.c_str(), true );
    
    // iterate through the catom objects and save them to the file
    hash_map<const unsigned long, CatomSim *, 
      hash<const unsigned long>, equl>::iterator catom_i;
    
    vector<CodeModule*>::iterator mod_i;
    
    worldPtr->myFile.fileWrite_ModuleHeader();
    worldPtr->myFile.fileWrite_BodyStart();
    

    //progress bar
    bar = 0;
    counter =0;
    update_widget("progress_bar", "GtkProgressBar", "Writing to file");  

    for(catom_i = worldPtr->catomHash.begin(); catom_i != worldPtr->catomHash.end(); catom_i++) {
      
      // Get the GUID
      uint32 id = catom_i->first;
      
      // _sim version 1 ( Simple )
      assert( catom_i->second->updateFileModule( worldPtr->myModule->getContent(), 
						 worldPtr->myModule->getVersion() ) );
      
      // TODO: tell the catom codemodules to all update their StateFile::Modules
      for ( mod_i = catom_i->second->codeModules.begin(); 
	    mod_i != catom_i->second->codeModules.end(); 
	    mod_i++ ) {
	(*mod_i)->saveModule();
      }
      
      // Write the new content to the file
      worldPtr->myFile.fileWrite_Content( id );

      counter++;
      //progress bar
      while(int(counter/(s/100+1))>bar+1) {
	bar++;
	update_widget("progress_bar", "GtkProgressBar", ((float)counter)/((float)s));
      } 
    }

    update_widget("progress_bar", "GtkProgressBar", "");  
    update_widget("progress_bar", "GtkProgressBar", 0);
    
    worldPtr->myFile.fileClose();
    
    saveName = "";
    worldPtr->clearCatoms();
  }

  char* msg; //used to sprintf text to be passed to Gtk
  Point3D correction = Point3D(0,0,-CatomSim::catom_radius); //used to place stamps in correct alignment with the lattice

  msg = new char[100];
  sprintf(msg, "%lu", catomCount);
  update_widget("catomCountLabel", "GtkLabel", msg);
  delete msg;

  //draw better markers at 0,0,0
  
  glPushMatrix();
  glTranslatef ((float)0,(float)0,(float)0);
  glScaled (10, 10, 200);
  dsSetColorAlpha (1,0,0,1);
  const float k = 0.03f;
  glBegin (GL_TRIANGLE_FAN);
  glNormal3f (0,-1,1);
  glVertex3f (0,0,k);
  glVertex3f (-k,-k,0);
  glVertex3f ( k,-k,0);
  glNormal3f (1,0,1);
  glVertex3f ( k, k,0);
  glNormal3f (0,1,1);
  glVertex3f (-k, k,0);
  glNormal3f (-1,0,1);
  glVertex3f (-k,-k,0);
  glEnd();
  glPopMatrix();


  //if a guid orgranization req has been made, do it
  if (GUIDOrgReq){
    GUIDOrgReq = false;
    organizeGUIDs();
  }

  //if a clear req has been made, clear world
  if (clearReq){
    clearCatoms();

    clearReq=false;
    msg = new char[100];
    sprintf(msg, "%lu", catomCount);
    update_widget("catomCountLabel", "GtkLabel", msg);
    delete msg;
  }

  //If a model request has been made, process it
  if (!modelReq.done) {
    modelReq.done = true;

    if(modelReq.asStamp){ //prepare to make a stamp
      stamp.clear();
      //update_widget("stampFileName", "GtkEntry", "(From model)");
    }

    if(modelReq.filename != ""){ //on an empty filename, don't even bother

      //create the model using Modeler.cxx
      list<Point3D>model = (loadModel(modelReq.filename, modelReq.size, lattice, modelReq.precision, modelReq.solid, modelReq.flipYZ));
      list<Point3D>::iterator iter;
      unsigned long s = model.size();
      
      //progress bar
      int bar = 0;
      int counter =0;
      msg = new char[100];
      sprintf(msg, "Checking %lu catoms for overlaps...", s);
      update_widget("progress_bar", "GtkProgressBar", msg);
      delete msg;
      
      iter=model.begin();
      while(iter!=model.end()) { //some preprocessing
	
	if(!modelReq.asStamp){ //if it's not a stamp, check for overlaps right now
	  if(catomAt(*iter)){
	    list<Point3D>::iterator temp = iter;
	    iter++;
	    model.erase(temp);
	  }
	  else{
	    iter++;
	  }
	}
	else{ //if it is a stamp, use correction to place it correctly within our lattice
	  *iter += correction;
	  iter++;
	}
	counter++;
	
	//progress bar
	while(int(counter/(s/100+1))>bar+1) {
	  bar++;
	  update_widget("progress_bar", "GtkProgressBar", ((float)counter)/((float)s));
	} 
      } 
      update_widget("progress_bar", "GtkProgressBar", 0);
      update_widget("progress_bar", "GtkProgressBar", "");

      s = model.size();

      //progress bar
      bar = 0;
      counter = 0;

      msg = new char[100];
      sprintf(msg, "Loading %lu catoms into world", s);
      update_widget("progress_bar", "GtkProgressBar", msg);
      delete msg;

      for(iter=model.begin();iter!=model.end();iter++) { //now actually insert the catoms, either into the stamp or into the world
		
	if(!modelReq.asStamp){
	  addCatom(GUID, *iter, catomColor);
	  GUID++;
	}
	else{
	  stamp.push_back(*iter);
	}

	counter++;

	//progress bar
	while(int(counter/(s/100+1))>bar+1) {
	  bar++;
	  update_widget("progress_bar", "GtkProgressBar", ((float)counter)/((float)s));
	} 
      } 
      update_widget("progress_bar", "GtkProgressBar", 0);
      update_widget("progress_bar", "GtkProgressBar", "");
      
      msg = new char[100];
      sprintf(msg, "%lu", catomCount);
      update_widget("catomCountLabel", "GtkLabel", msg); //update the catom count
      delete msg;
    }
  }

  //If a primitive request has been made, process it
  if (!primReq.done) {
    primReq.done = true;

    if(primReq.asStamp){ //prepare to make a stamp
      stamp.clear();
      //update_widget("stampFileName", "GtkEntry", "(Catom prism)");
    }

    if(primReq.type != ""){ //if there is no type, don't even bother
      list<Point3D>model = createPrimitive(primReq.type, primReq.params, lattice, primReq.precision, primReq.solid, primReq.flipYZ);
      list<Point3D>::iterator iter;
      unsigned long s = model.size();
      
      //progress bar
      int bar = 0;
      int counter =0;
      msg = new char[100];
      sprintf(msg, "Checking %lu catoms for overlaps...", s);
      update_widget("progress_bar", "GtkProgressBar", msg);
      delete msg;
      
      iter=model.begin();
      while(iter!=model.end()) { //some preprocessing
		
	if(!primReq.asStamp){ //if it's not a stamp, check for overlaps now
	  if(catomAt(*iter)){
	    list<Point3D>::iterator temp = iter;
	    iter++;
	    model.erase(temp);
	  }
	  else{
	    iter++;
	  }
	}
	else{ //if it is a stamp, add the correction to place it correctly in our lattice
	  *iter += correction;
	  iter++;
	}
	counter++;

	//progress bar
	while(int(counter/(s/100+1))>bar+1) {
	  bar++;
	  update_widget("progress_bar", "GtkProgressBar", ((float)counter)/((float)s));
	} 
      } 
      update_widget("progress_bar", "GtkProgressBar", 0);
      update_widget("progress_bar", "GtkProgressBar", "");

      s = model.size();

      //progress bar
      bar = 0;
      counter = 0;

      msg = new char[100];
      sprintf(msg, "Loading %lu catoms into world", s);
      update_widget("progress_bar", "GtkProgressBar", msg);
      delete msg;

      for(iter=model.begin();iter!=model.end();iter++) { //actually insert the catoms, either into the stamp or into the world

	if(!primReq.asStamp){
	  addCatom(GUID, *iter, catomColor);
	  GUID++;
	}
	else{
	  stamp.push_back(*iter);
	}

	counter++;

	//progress bar
	while(int(counter/(s/100+1))>bar+1) {
	  bar++;
	  update_widget("progress_bar", "GtkProgressBar", ((float)counter)/((float)s));
	} 
      } 
      update_widget("progress_bar", "GtkProgressBar", 0);
      update_widget("progress_bar", "GtkProgressBar", "");

      msg = new char[100];
      sprintf(msg, "%lu", catomCount);
      update_widget("catomCountLabel", "GtkLabel", msg);
      delete msg;
    }
  }
  
  //If a GUID change has been requested, change it
  if(GUIDChangeReq) {
    unsigned long int target;
    GUIDChangeReq=false;
    target = catomAt(catomCursor);
    if(target != 0){ //if there is a selected catom
      
      builderCatom c(getCatom(target)); //get the catom to change
      builderCatom old(getCatom(newGUID)); //get the catom with the requested GUID
      
      if(old.GUID){ //if a catom does, in fact, already have the requested GUID
	if(display_gtk_yes_no_box("This GUID is already used. Click 'Yes' to cause the catoms to trade GUIDs; click 'No' to cancel.")){
	  swap(target, newGUID);
	}
      }
      else { //otherwise it's easy
	struct builderCatom bc = getCatom(target);
	bc.GUID = newGUID;
	addCatom(bc);

	deleteCatom(target);

	if(GUID<=newGUID){
	  GUID = newGUID+1;
	}
      }
    }
  }
  
  if(newStamp){ //if a new stamp has been requested
    newStamp = false;
    hash_set<Point3D, HashPoint3D, EqualPoint3D> stampHash;
    loadPointHash(stampName, stampHash); //load the points from the world file
    stamp.clear();
    hash_set<Point3D, HashPoint3D, EqualPoint3D>::iterator s_i;
    for(s_i=stampHash.begin(); s_i!=stampHash.end(); s_i++){
      stamp.push_back(*s_i + correction); //apply the correction and put it into the stamp
    }
  }

  //update Gtk catomCursor position readout
  if(legalCursor){
    char msg[100];
    sprintf(msg, "(%.1f, %.1f, %.1f)", catomCursor.getX(), catomCursor.getY(), catomCursor.getZ());
    update_widget("cursorPos", "GtkLabel", msg);
  }
  else{
    update_widget("cursorPos", "GtkLabel", "");
  }


  //draw builder catoms
  float r = CatomSim::catom_radius;

  {
    Point3D p;
    hash_map<unsigned long, builderCatom>::iterator b_i;
    for(b_i=builderCatoms.begin(); b_i!=builderCatoms.end(); b_i++){
      p.setX(b_i->second.pos.getX());
      p.setY(b_i->second.pos.getY()); 
      p.setZ(b_i->second.pos.getZ());
            
      glPushMatrix();
      glTranslatef(p.getX(), p.getY(), p.getZ());
      glEnable (GL_NORMALIZE);
      glShadeModel (GL_SMOOTH);
      dsSetColorAlpha(b_i->second.color.r/255, b_i->second.color.g/255, b_i->second.color.b/255, 1);

      setupDrawingMode();

      drawSphere();
      glDisable (GL_NORMALIZE);
      glPopMatrix();
      
    }
  }


  //draw catomCursor/shadow on screen
  glPushMatrix();
  glTranslatef(catomCursor.getX(), catomCursor.getY(), catomCursor.getZ());
  glEnable (GL_NORMALIZE);
  glShadeModel (GL_SMOOTH);
  
  switch (mode) {
  case -1:
    dsSetColorAlpha(0,0,0,0);
    break;
  case 0:
    dsSetColorAlpha(0, 1, 0, .35);
    break;
  case 1:
    dsSetColorAlpha(1, 0, 0, .35);
    break;
  case 2:
    dsSetColorAlpha(0, 0, 1, .35);
    break;
  case 3:
    dsSetColorAlpha(0, 0, 0, .35);
    break;
  }

  setupDrawingMode();

  if(!(stampMode&&(mode==0||mode==1))){ //if we're not in stamp mode, draw the normal cursor
    glScaled (1.1*r,1.1*r,1.1*r);
    drawSphere();
    glScaled (1/(1.1*r),1/(1.1*r),1/(1.1*r));
    glTranslatef(0, 0, -catomCursor.getZ());
    glScaled (.75,.75,.125);
    drawSphere();
  }
  glPopMatrix();


  if(stampMode&&(mode==0||mode==1)){ //stamp mode, insert or erase
    //draw additional spheres in the cursor as a preview of the catom stamp
    Point3D p;
    list<Point3D>::iterator s_i;
    for(s_i=stamp.begin(); s_i!=stamp.end(); s_i++){
      p.setX(s_i->getX());
      p.setY(s_i->getY()); 
      p.setZ(s_i->getZ());
      
      p += catomCursor;
      
      glPushMatrix();
      glTranslatef(p.getX(), p.getY(), p.getZ());
      glEnable (GL_NORMALIZE);
      glShadeModel (GL_SMOOTH);
      setupDrawingMode();
      glScaled (1.1*r,1.1*r,1.1*r);
      drawSphere();
      glDisable (GL_NORMALIZE);
      glPopMatrix();
      
    }
  }

  glDisable (GL_NORMALIZE);

}


//the next several functions are called primarily by GTK to request changes in world state
void WorldBuilder::import(string filename, int size, float precision, bool flipYZ, bool asStamp){
  //set up a model request
  //(to be processed when the main thread updates the builder)
  modelReq.done = false;
  modelReq.filename = filename;
  modelReq.size = size;
  modelReq.precision = precision;
  modelReq.solid = fillSolid;
  modelReq.flipYZ = flipYZ;
  modelReq.asStamp = asStamp;
}


void WorldBuilder::requestPrimitive(string type, vector<float> params, float precision, bool flipYZ, bool asStamp){
  //set up a primitive request
  //(to be processed when the main thread updates the builder)
  primReq.done = false;
  primReq.type = type;
  primReq.params = params;
  primReq.precision = precision;
  primReq.solid = fillSolid;
  primReq.flipYZ = flipYZ;
  primReq.asStamp = asStamp;
}


void WorldBuilder::requestGUIDChange(unsigned long int newg){
  GUIDChangeReq=true;
  newGUID=newg;
}


void WorldBuilder::requestOrganizeGUIDs(){
  GUIDOrgReq=true;
}


void WorldBuilder::clearAll(){
  clearReq = true;
}


void WorldBuilder::saveWorldAs(string filename){
  saveName = filename;
}


//the remainder of the functions are used by the builder to keep track of builderCatoms
void WorldBuilder::clearCatoms() {
  hash_map<unsigned long, builderCatom>::iterator pos;
  
  builderCatoms.clear();
  catomCount = 0;
}

void WorldBuilder::deleteCatom(unsigned long int GUID) {
  if(builderCatoms.count(GUID)){
    builderCatoms.erase(GUID);
    catomCount--;
  }
}

WorldBuilder::builderCatom WorldBuilder::getCatom(unsigned long int GUID) {
  if(builderCatoms.count(GUID))
    return builderCatoms.find(GUID)->second;
  struct builderCatom bc;
  bc.GUID=0;
  return bc;
}


bool WorldBuilder::addCatom( unsigned long GUID, Point3D pos, color c) {
  struct builderCatom bc;
  bc.GUID = GUID;
  bc.pos = pos;
  bc.color = c;
  return addCatom(bc);
}


bool WorldBuilder::addCatom(builderCatom c) { 
    
  builderCatoms[c.GUID] = c;
  // Increment the number of Catoms in the world
  catomCount++;
  return true;
}


unsigned long int WorldBuilder::catomAt(Point3D pos) {
  if(catomCount == 0 ) return 0;
  
  hash_map<unsigned long, builderCatom>::iterator c_i;
  for (c_i = builderCatoms.begin(); c_i != builderCatoms.end(); c_i++ ) {
    if(c_i->second.pos == pos) {
      return c_i->first;
    }
  }
  return 0;
}

void WorldBuilder::swap(unsigned long int GUID1, unsigned long int GUID2) {
  builderCatom temp = getCatom(GUID1);
  builderCatoms[GUID1] = getCatom(GUID2);
  builderCatoms[GUID2] = temp;
}

void WorldBuilder::organizeGUIDs(){
  hash_map<unsigned long, builderCatom> newHash;  

  hash_map<unsigned long, builderCatom>::iterator pos;  

  pos = builderCatoms.begin();
  for(unsigned long i = 1; i <= catomCount; i++){
    builderCatom c = pos->second;
    newHash[i] = c;
    pos++;
  }
    
  builderCatoms = newHash;
}
