#ifndef __WP__
#define __WP__
#include <string>
#include <vector>
#include <utility>
#include <set>
#include <deque>
#include <pthread.h>
#include "CatomSim.hxx"
#include "CatomWorld.hxx"
using namespace std;

extern pthread_mutex_t wp_debug_lock;

class WPManager;
class DistributedWatchpoint;
extern CatomWorld *worldPtr;

typedef enum {
	no,maybe,yes
} TriState;

typedef enum {
	Act_Nothing,Act_Halt,Act_Count,Act_Arrows,Act_Color
} WatchpointAction;

class WPObject {
public:
	virtual ~WPObject() {}
	virtual void propagate(string name,CatomSim *catom,WPManager *wm) {}
	virtual void propagate(string name,CatomSim *catom,DistributedWatchpoint *wp) {}
	virtual TriState isSatisfied(bool centralized = true) {return maybe;}
	virtual void tShift() {}
	virtual int minExtent(string name) {return 0;}
	virtual int maxExtent(string name) {return 0;}
	virtual WPObject *clone() {return NULL;}
	virtual set<string> watchedVariables() { return set<string>(); }
	virtual bool isValidCandidate(map<CatomSim*,set <CatomSim *> > &neighbors, 
	vector<pair<string,CatomSim*> > &names,
	CatomSim *candidate,const string &slotName) {return true;}
};

class BoolExp: public WPObject {	
public:
	virtual ~BoolExp() {}
	virtual BoolExp *clone() {return NULL;}
};

class RVal: public WPObject {
public:
	float value;
	virtual ~RVal() {}
	virtual RVal *clone() {return NULL;}
};

class CatomRef: public WPObject {
public:
	int temporalOffset;
	string name;
	CatomRef(const char *n) {
		name = n;
		temporalOffset = 0;
	}
	virtual ~CatomRef() {}
	virtual void tShift();
	virtual CatomRef *clone();
};

class CatomVar: public RVal {
	CatomRef *catomref;
	string var;
public:
	bool filled;
	CatomVar(CatomRef *c,const char *v) {
		var = v;
		catomref = c;
		filled = false;
		value = 0.0;
	}
	virtual ~CatomVar() {
		delete catomref;
	}
	virtual void propagate(string name,CatomSim *catom,WPManager *wm);
	virtual void propagate(string name,CatomSim *catom,DistributedWatchpoint *wp);
	virtual TriState isSatisfied(bool centralized = true);
	virtual void tShift();
	virtual int minExtent(string name);
	virtual int maxExtent(string name);
	virtual CatomVar *clone();
	virtual set<string> watchedVariables();
};

class Number : public RVal {
public:
	Number(float val) {value = val;}
	virtual ~Number() {}
	virtual Number *clone();
	virtual TriState isSatisfied(bool centralized = true);
};

class MathExp : public RVal {
	string op;
	RVal *lval;
	RVal *rval;
public:
	MathExp(RVal *l, const char *o,RVal *r) {
		lval = l;
		rval = r;
		op = o;
	}
	virtual ~MathExp() {
		delete lval;
		delete rval;
	}
	virtual void propagate(string name,CatomSim *catom,WPManager *wm);
	virtual void propagate(string name,CatomSim *catom,DistributedWatchpoint *wp);
	virtual TriState isSatisfied(bool centralized = true);
	virtual void tShift();
	virtual int minExtent(string name);
	virtual int maxExtent(string name);
	virtual MathExp *clone();
	virtual set<string> watchedVariables();
};

class CompExp: public BoolExp {
	string op;
	RVal *lval;
	RVal *rval;
public:
	CompExp(RVal *l,const char *o,RVal *r) {
		lval = l;
		rval = r;
		op = o;
	}
	virtual ~CompExp() {
		delete lval;
		delete rval;
	}
	virtual void propagate(string name,CatomSim *catom,WPManager *wm);
	virtual void propagate(string name,CatomSim *catom,DistributedWatchpoint *wp);
	virtual TriState isSatisfied(bool centralized = true);
	virtual void tShift();
	virtual int minExtent(string name);
	virtual int maxExtent(string name);
	virtual CompExp *clone();
	virtual set<string> watchedVariables();
};

class AndExp: public BoolExp {
	BoolExp *lval;
	BoolExp *rval;
public:
	AndExp(BoolExp *l,BoolExp *r):lval(l),rval(r) {}
	virtual ~AndExp() {delete lval; delete rval;}
	virtual void propagate(string name,CatomSim *catom,WPManager *wm);
	virtual void propagate(string name,CatomSim *catom,DistributedWatchpoint *wp);
	virtual TriState isSatisfied(bool centralized = true);
	virtual void tShift();
	virtual int minExtent(string name);
	virtual int maxExtent(string name);
	virtual AndExp *clone();
	virtual set<string> watchedVariables();
	virtual bool isValidCandidate(map<CatomSim*,set <CatomSim *> > &neighbors, 
																vector<pair<string,CatomSim*> > &names,
																CatomSim *candidate,const string &slotName);	
};

class OrExp: public BoolExp {
	BoolExp *lval;
	BoolExp *rval;
public:
	OrExp(BoolExp *l,BoolExp *r):lval(l),rval(r) {}
	virtual ~OrExp() {delete lval; delete rval;}
	virtual void propagate(string name,CatomSim *catom,WPManager *wm);
	virtual void propagate(string name,CatomSim *catom,DistributedWatchpoint *wp);
	virtual TriState isSatisfied(bool centralized = true);
	virtual void tShift();
	virtual int minExtent(string name);
	virtual int maxExtent(string name);
	virtual OrExp *clone();
	virtual set<string> watchedVariables();
	virtual bool isValidCandidate(map<CatomSim*,set <CatomSim *> > &neighbors, 
																vector<pair<string,CatomSim*> > &names,
																CatomSim *candidate,const string &slotName);
};

class NotExp: public BoolExp {
	BoolExp *val;
public:
	NotExp(BoolExp *v):val(v) {}
	virtual ~NotExp() {delete val;}
	virtual void propagate(string name,CatomSim *catom,WPManager *wm);
	virtual void propagate(string name,CatomSim *catom,DistributedWatchpoint *wp);
	virtual TriState isSatisfied(bool centralized = true);
	virtual void tShift();
	virtual int minExtent(string name);
	virtual int maxExtent(string name);
	virtual NotExp *clone();
	virtual set<string> watchedVariables();
	virtual bool isValidCandidate(map<CatomSim*,set <CatomSim *> > &neighbors, 
																vector<pair<string,CatomSim*> > &names,
																CatomSim *candidate,const string &slotName);
};

class NeighborExp: public BoolExp {
	string lval;
	string rval;
	//variables below are for distributed version only
	catomID lcatom;
	catomID rcatom;
	TriState satisfied;
public:
	NeighborExp(const char *l,const char *r) {
		lval = l;
		rval = r;
		lcatom = rcatom = 0;
		satisfied = maybe;
	}
	virtual ~NeighborExp() {}
	virtual void propagate(string name,CatomSim *catom,WPManager *wm);
	virtual void propagate(string name,CatomSim *catom,DistributedWatchpoint *wp);
	virtual TriState isSatisfied(bool centralized = true);
	virtual NeighborExp *clone();
	virtual bool isValidCandidate(map<CatomSim*,set <CatomSim *> > &neighbors, 
																vector<pair<string,CatomSim*> > &names,
																CatomSim *candidate,const string &slotName);
};

class CatomList: public WPObject {
	public:
	vector<pair<string,CatomSim*> > names;
	CatomList() {}
	virtual ~CatomList() {}
	CatomList(const char *name) {
		pair<string,CatomSim *> newCatom(name,NULL);
		names.push_back(newCatom);
	}
	void addName(const char *name) {
		pair<string,CatomSim *> newCatom(name,NULL);
		names.push_back(newCatom);
	}
	bool isFull() {
		for (unsigned i =0; i < names.size(); i++) {
			if (!names[i].second) return false;
		}
		return true;
	}
	string nextSlot() {
		for (unsigned i =0; i < names.size(); i++) {
			if (!names[i].second) return names[i].first;
		}
		return string("");
	}
	vector<string> freeSlots() {
		vector<string> freeList;
		for (unsigned i =0; i < names.size(); i++) {
			if (!names[i].second) freeList.push_back(names[i].first);
		}
		return freeList;
	}
	int filledSize() {
		int fillCount = 0;
		for (unsigned i =0; i < names.size(); i++) {
			if (names[i].second) fillCount++;
		}
		return fillCount;
	}
	virtual void propagate(string name,CatomSim *catom,WPManager *wm);
	virtual void propagate(string name,CatomSim *catom,DistributedWatchpoint *wp);
	virtual CatomList *clone();
	int slotIdx(CatomSim *c) {
		for (int i =0; i < (int) names.size(); i++) {
			if (names[i].second == c) {
				return i;
			}
		}
		return -1; //not found
	}
};

class Watchpoint: public WPObject {
public:
	CatomList *c_list;
	BoolExp *main_exp;
	WatchpointAction action;
	//multi-hop support
	bool inTransit;
	CatomSim *destination;
	char **adjacency; //2D array
	int adjSize;
	unsigned int id; //watchpoint id #
	unsigned int matches; //for Act_Count
	
		Watchpoint(CatomList *l,BoolExp *b,WatchpointAction a) {
			action = a;
			c_list = l;
			main_exp = b;
			inTransit = false;
			destination = NULL;
			//set up empty adjacency matrix
			adjSize = c_list->names.size();
			adjacency = new char*[adjSize];
			for (int i = 0; i < adjSize; ++i)
			  adjacency[i] = new char[adjSize];
			for (int i = 0; i < adjSize; ++i)
				for (int j = 0; j < adjSize; ++j)
					adjacency[i][j] = 0;
		}
		virtual ~Watchpoint() {
			delete c_list;
			delete main_exp;
			for (int i = 0; i < adjSize; ++i)
			  delete[] adjacency[i];
			delete[] adjacency;
		}
		bool isFull() {return c_list->isFull();}
		string nextSlot() {return c_list->nextSlot();}
		vector<string> freeSlots() {return c_list->freeSlots();}
		virtual int filledSize() {return c_list->filledSize();}
		virtual void propagate(string name,CatomSim *catom,WPManager *wm);
		virtual void propagate(string name,CatomSim *catom,DistributedWatchpoint *wp);
		virtual TriState isSatisfied(bool centralized = true);
		void tShift();
		virtual int minExtent(string name);
		virtual int maxExtent(string name);
		virtual Watchpoint *clone();
		set<string> watchedVariables();
		virtual bool isValidCandidate(CatomSim *candidate,const string &slotName) {
			map<CatomSim*,set <CatomSim *> > neighbors;
			//fill in neighbors from adjacency matrix
			for (unsigned i = 0; i < c_list->names.size(); i++) {
				if (c_list->names[i].second) { 
					neighbors[c_list->names[i].second] = set<CatomSim*>();
					for (unsigned j = 0; j < c_list->names.size(); j++) {
						if ( (i!=j) && adjacency[i][j] && c_list->names[j].second) {
							neighbors[c_list->names[i].second].insert(c_list->names[j].second);
						}
					}
				}
			}
				
			return main_exp->isValidCandidate(neighbors,c_list->names,candidate,slotName);
		}
		virtual bool isValidCandidate(map<CatomSim*,set <CatomSim *> > &neighbors, 
																	vector<pair<string,CatomSim*> > &names,
																	CatomSim *candidate,const string &slotName); 
		CatomSim *nextRoutingStep(CatomSim *curr);
};

Watchpoint *parseWP(string input);

#endif //__WP__
