///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright (C) 2006 by Intel Corporation and Carnegie Mellon University    //
// Contacts: bdr @ cs.cmu.edu                                                //
//           casey.j.helfrich @ intel.com                                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "VisDataTransforms.hxx"

#include "DPRSim.hxx"
#include "Historian.hxx"
#include "VisManager.hxx"

using namespace std;

// *********************
// *** VisDataInput ***
// *********************

list<string> VisDataInput::getOutputNames() {
  list<string> outputNames;
  
  outputNames.push_back("locX");
  outputNames.push_back("locY");
  outputNames.push_back("locZ");
  //inputNames.push_back("qx");
  //inputNames.push_back("qy");
  //inputNames.push_back("qz");
  //inputNames.push_back("qa");
  outputNames.push_back("r");
  outputNames.push_back("g");
  outputNames.push_back("b");
  outputNames.push_back("a");
  outputNames.push_back("catomID");

  return outputNames;
}

string VisDataInput::getOutputString(string outputName) {
  // Explicitly handle catomID
  if(outputName == "catomID") {
    char strBuf[256];
    snprintf(strBuf, 255, "%d", manager->getDisplayCatom());
    
    return string(strBuf);
  }

  // Try world state first
  try {
    return historianPtr->getWorldStateStringForCatomAndTick(outputName,
							    manager->getDisplayCatom(),
							    manager->getDisplayTick());
  }
  catch (...) {

  }

  // Then instance variables
  return historianPtr->getStringVariableForCatomAndTick(outputName,
							manager->getDisplayCatom(),
							manager->getDisplayTick());
}

double VisDataInput::getOutputDouble(string outputName) {
  // Explicitly handle catomID
  if(outputName == "catomID") {
    return manager->getDisplayCatom();
  }

  // Try world state first
  try {
    return historianPtr->getWorldStateDoubleForCatomAndTick(outputName,
							    manager->getDisplayCatom(),
							    manager->getDisplayTick());
  }
  catch (...) {
  
  }

  // Then instance variables
  return historianPtr->getDoubleVariableForCatomAndTick(outputName,
							manager->getDisplayCatom(),
							manager->getDisplayTick());
}


// *********************
// *** VisDataOutput ***
// *********************

string VisDataOutput::getOutputString(string outputName) {
  try {
    return getInputString(outputName);
  }
  catch (...) {
    return getDefaultInputString(outputName);
  }
}

double VisDataOutput::getOutputDouble(string outputName) {
  try {
    return getInputDouble(outputName);
  }
  catch (...) {
    return getDefaultInputDouble(outputName);
  }
}

list<string> VisDataOutput::getInputNames() {
  list<string> inputNames;
  
  inputNames.push_back("locx");
  inputNames.push_back("locy");
  inputNames.push_back("locz");
  //inputNames.push_back("qx");
  //inputNames.push_back("qy");
  //inputNames.push_back("qz");
  //inputNames.push_back("qa");
  inputNames.push_back("r");
  inputNames.push_back("g");
  inputNames.push_back("b");
  inputNames.push_back("a");

  inputNames.push_back("overlayText");

  return inputNames;
}

double VisDataOutput::getDefaultInputDouble(string inputName) {
  return manager->getStdInput()->getOutputDouble(inputName);
}

string VisDataOutput::getDefaultInputString(string inputName) {
  if(inputName == "overlayText") {
    string catomIDString = manager->getStdInput()->getOutputString("catomID");
    
    int start = catomIDString.length() - 2;
    int length = 2;

    if(start < 0) {
      length += start;
      start = 0;
    }

    return catomIDString.substr(start, length);
  }

  return manager->getStdInput()->getOutputString(inputName);
}

// *****************************
// *** VisDataConstantDouble ***
// *****************************

VisDataConstantDouble::VisDataConstantDouble() :
  constantValue(1.0)
{ }

list<string> VisDataConstantDouble::getOutputNames() {
  list<string> outputNames;
  outputNames.push_back("value");
  return outputNames;
}

double VisDataConstantDouble::getOutputDouble(string outputName) {
  return constantValue;
}

void VisDataConstantDouble::setValue(double num) {
  constantValue = num;
}

// **********************
// *** VisDataRainbow ***
// **********************

static double rainbowColors[7][3] =
  {
    { 255.0, 0.0, 0.0 }, // red
    { 255.0, 127.0, 0.0 }, // orange
    { 255.0, 255.0, 0.0 }, // yellow
    { 0.0, 255.0, 0.0 }, // green
    { 0.0, 0.0, 255.0 }, // blue
    { 0.0, 255.0, 255.0 }, // indigo
    { 127.0, 0.0, 127.0 }, // violet
  };

list<string> VisDataRainbow::getInputNames() {
  list<string> inputNames;
  inputNames.push_back("value");
  return inputNames;
}

list<string> VisDataRainbow::getOutputNames() {
  list<string> outputNames;
  outputNames.push_back("red");
  outputNames.push_back("green");
  outputNames.push_back("blue");
  return outputNames;
}

double VisDataRainbow::getOutputDouble(string name) {
  unsigned componentIdx;
  if(name == "red")
    componentIdx = 0;
  else if(name == "green")
    componentIdx = 1;
  else if(name == "blue")
    componentIdx = 2;
  else
    throw 0;

  unsigned colorIdx = (unsigned)getInputDouble("value");
  if(colorIdx > 6)
    throw 0;
  
  return rainbowColors[colorIdx][componentIdx];
}

// **********************
// *** VisDataCompare ***
// **********************

list<string> VisDataCompare::getInputNames() {
  list<string> inputNames;
  inputNames.push_back("valueA");
  inputNames.push_back("valueB");
  inputNames.push_back("op");
  return inputNames;
}

list<string> VisDataCompare::getOutputNames() {
  list<string> outputNames;
  outputNames.push_back("bool");
  return outputNames;
}

double VisDataCompare::getOutputDouble(string outputName) {
  VisDataCompareOp op = (VisDataCompareOp)getInputDouble("op");

  // First try double inputs
  try {
    double inputA = getInputDouble("valueA");
    double inputB = getInputDouble("valueB");

    switch(op) {
    case kOpEquals:
      return inputA==inputB;
    case kOpNotEquals:
      return inputA!=inputB;
    case kOpLessThan:
      return inputA<inputB;
    case kOpLessThanOrEqual:
      return inputA<=inputB;
    case kOpGreaterThan:
      return inputA>inputB;
    case kOpGreaterThanOrEqual:
      return inputA>=inputB;
    }
  }
  catch (...) {
    // fall through
  }

  // Then try string inputs
  try {
    string inputA = getInputString("valueA");
    string inputB = getInputString("valueB");

    switch(op) {
    case kOpEquals:
      return inputA==inputB;
    case kOpNotEquals:
      return inputA!=inputB;
    case kOpLessThan:
      return inputA<inputB;
    case kOpLessThanOrEqual:
      return inputA<=inputB;
    case kOpGreaterThan:
      return inputA>inputB;
    case kOpGreaterThanOrEqual:
      return inputA>=inputB;
    }
  }
  catch (...) {
    // fall through
  }

  // If neither, just throw
  throw 0;
}


// *********************
// *** VisDataChoose ***
// *********************

list<string> VisDataChoose::getInputNames() {
  list<string> inputNames;
  inputNames.push_back("valueT");
  inputNames.push_back("valueF");
  inputNames.push_back("bool");
  return inputNames;
}

list<string> VisDataChoose::getOutputNames() {
  list<string> outputNames;
  outputNames.push_back("value");
  return outputNames;
}

double VisDataChoose::getOutputDouble(string outputName) {
  if(getInputDouble("bool") != 0)
    return getInputDouble("valueT");
  else
    return getInputDouble("valueF");
}

string VisDataChoose::getOutputString(string outputName) {
  if(getInputDouble("bool") != 0)
    return getInputString("valueT");
  else
    return getInputString("valueF");
}
