///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright (C) 2006 by Intel Coproration and Carnegie Mellon University    //
// Contacts: casey.j.helfrich @ intel.com                                    //
//           bdr @ cs.cmu.edu                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "StatsManager.hxx"

#include "CatomWorld.hxx"
#include "DPRSim.hxx"

StatsManager::StatsManager() { }

void StatsManager::setEnabledStats(string enabledStatsList) {
  string::size_type pos;

  // insert each fragment before commas
  while(string::npos != (pos = enabledStatsList.find(",",0))) {
    enabledStats.insert(enabledStatsList.substr(0,pos));
    enabledStatsList = enabledStatsList.substr(pos+1, enabledStatsList.length()-pos-1);
  }

  // then insert whatever's left, if anything
  if(enabledStatsList.length() > 0)
    enabledStats.insert(enabledStatsList);
}

StatsManager::~StatsManager() {
  
  map<string, FILE*>::iterator filesIter = files.begin();
  while(filesIter!= files.end()) {
    if(completionStats.find(filesIter->first) != completionStats.end()) {
      // Dump the completion stats
      fprintf(filesIter->second, "COMPLETION\n%s", getStatsString(filesIter->first, completionStats[filesIter->first]).c_str());
    }

    fflush(filesIter->second);
    fclose(filesIter->second);
    
    filesIter++;
  }
}

FILE* StatsManager::fileForIdentifier(string identifier) {
  // Return existing file if one already exists
  if(files.find(identifier) != files.end())
    return files[identifier];

  // Doesn't already exist; open new file
  string filename = identifier;
  filename.append(".stats");
  FILE* newFile = fopen(filename.c_str(), "w");
  //newFile = NULL;
  if(newFile) {
    files[identifier] = newFile;
  }

  return newFile;
}

void StatsManager::log(string identifier, double data, string origin) {
  if(enabledStats.find(identifier) == enabledStats.end())
    return;
  valueHistory[identifier].push_back(data);

  FILE* fd = fileForIdentifier(identifier);
  if(fd) {
    fprintf(fd, "tick %ld val %s data %f origin %s\n",
	    worldPtr->current_time,
	    identifier.c_str(),
	    data,
	    origin.c_str());
  }
}

void StatsManager::log(string identifier, double data, catomID origin) {
  char buf[64];
  snprintf(buf, 64, "%uld", origin);
  return log(identifier, data, buf);
}

double StatsManager::counterInc(string identifier, string origin, double delta) {
  if(enabledStats.find(identifier) == enabledStats.end())
    return 0.0;

  if(counters.find(identifier) != counters.end())
    counters[identifier] += delta;
  else
    counters[identifier] = delta;

  FILE* fd = fileForIdentifier(identifier);
  if(fd) {
    fprintf(fd, "tick %ld ctr %s delta %f val %f origin %s\n",
	    worldPtr->current_time,
	    identifier.c_str(),
	    delta,
	    counters[identifier],
	    origin.c_str());
  }

  return counters[identifier];
}

double StatsManager::counterInc(string identifier, catomID origin, double delta) {
  char buf[64];
  snprintf(buf, 64, "%uld", origin);
  return counterInc(identifier, buf, delta);
}

double StatsManager::counterDec(string identifier, string origin, double delta) {
  return counterInc(identifier, origin, -1.0 * delta);
}

double StatsManager::counterDec(string identifier, catomID origin, double delta) {
  return counterInc(identifier, origin, -1.0 * delta);
}

double StatsManager::getCounter(string identifier) {
  return counters[identifier];
}

string StatsManager::getStatsString(string identifier, unsigned flags) {
  if(!flags)
    return "";

  stringstream ss;

  if(counters.find(identifier) != counters.end()) {
    // It's a counter
    if(flags & STATS_VALUE) {
      ss << "Counter value: " << counters[identifier] << endl;
    }
  } else if(valueHistory.find(identifier) != valueHistory.end()) {
    // It's a regular value
    if(flags & (STATS_MEAN | STATS_SUM)) {
      // Need to find the sum for both of these
      double sum = 0.0;
      for(vector<double>::iterator iter = valueHistory[identifier].begin();
	  iter != valueHistory[identifier].end();
	  iter++) {
	sum += *iter;
      }

      if(flags & STATS_SUM)
	ss << "Sum: " << sum << endl;
      if(flags & STATS_MEAN)
	ss << "Mean: " << (sum / valueHistory[identifier].size()) << endl;
    }

    if(flags & STATS_COUNT)
      ss << "Count: " << valueHistory[identifier].size() << endl;

    if(flags & STATS_DIST)
      ss << createDistributionString(valueHistory[identifier]);
  } else {
    ss << "Stats not found\n";
  }

  return ss.str();
}

string StatsManager::createDistributionString(vector<double> values) {
  map<double, unsigned> frequencies;

  for(vector<double>::iterator iter = values.begin();
      iter != values.end();
      iter++) {
    double value = *iter;

    if(frequencies.find(value) == frequencies.end())
      frequencies[value] = 0;

    frequencies[value]++;
  }

  stringstream output;

  output << "Distribution:\n";

  for(map<double,unsigned>::iterator iter = frequencies.begin();
      iter != frequencies.end();
      iter++) {
    output << iter->first << " -- " << iter->second << endl;
  }

  return output.str();
}

void StatsManager::setCompletionStats(string identifier, unsigned flags) {
  completionStats[identifier] = flags;
}
