///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright (C) 2006 by Intel Corporation and Carnegie Mellon University    //
// Contacts: casey.j.helfrich @ intel.com                                    //
//           bdr @ cs.cmu.edu                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "Network.hxx"

#include <iostream>
#include <errno.h>

#include "Catom.hxx"
#include "CatomSim.hxx"
#include "CatomWorld.hxx"
#include "StatsManager.hxx"

using namespace std;

extern CatomWorld *worldPtr;
extern StatsManager *statsMgrPtr;

static unsigned long long pref_bandwidthPerTick;
simtime pref_fastStartEndTime;

Message::Message(MailboxID boxID) : 
  mailboxID(boxID), arrivalContact(0) {
  // Autogen a new message ID
  static unsigned long long nextMessageID = 0;
  messageID = nextMessageID++;

  statsMgrPtr->counterInc("liveMessages", boxID);
}

Message::Message(MailboxID boxID, unsigned long long msgID) :
  messageID(msgID), mailboxID(boxID), arrivalContact(0) {
  statsMgrPtr->counterInc("liveMessages", boxID);
}

Message::~Message() {
  statsMgrPtr->counterDec("liveMessages", mailboxID);
}

NetworkAdapter::NetworkAdapter(catomID CID, featureID FID) :
  messageQueueOut(NULL),
  currentMessage(NULL),
  hostCatom(CID), hostFeature(FID) /*, stats_msgsSent(0), stats_msgsRcvd(0)*/ {
  statsMgrPtr->setCompletionStats("liveMessages", STATS_VALUE);

  pthread_mutexattr_t mutex_attrs;
  pthread_mutexattr_init(&mutex_attrs);
  pthread_mutexattr_settype(&mutex_attrs, PTHREAD_MUTEX_RECURSIVE);
  pthread_mutex_init(&objectMutex, &mutex_attrs);
  pthread_mutexattr_destroy(&mutex_attrs);
  
  // set bandwidth to max for fast start
  pref_bandwidthPerTick = 0xfffffffffffffffull;

  // set the fast start end time preference
  string faststartPrefStr = worldPtr->search_key_value_list("FASTSTARTENDTIME");
  if(faststartPrefStr != "") {
    pref_fastStartEndTime = strtoull(faststartPrefStr.c_str(), NULL, 0);
  } else {
    pref_fastStartEndTime = 0;
  }
}

NetworkAdapter::~NetworkAdapter() {
  if(messageQueueOut)
    delete messageQueueOut;
  pthread_mutex_destroy(&objectMutex);
}

void NetworkAdapter::lock() {
  int err = pthread_mutex_lock(&objectMutex);
  if(err) {
    cerr << "can't lock mutex : " << strerror(errno) << endl;
    exit(1);
  }
}

void NetworkAdapter::unlock() {
  int err = pthread_mutex_unlock(&objectMutex);
  if(err) {
    cerr << "can't unlock mutex : " << strerror(errno) << endl;
    exit(1);
  }
}

// adds the appropriate amount of bandwidth for a new tick, then
// processes any outstanding messages that it can
void NetworkAdapter::newTick() { 
  lock();

  if(worldPtr->current_time == pref_fastStartEndTime) {
    // set bandwidth pref
    string bandwidthPrefStr = worldPtr->search_key_value_list("CONTACTBANDWIDTHPERTICK");
    if(bandwidthPrefStr != "") {
      pref_bandwidthPerTick = strtoull(bandwidthPrefStr.c_str(), NULL, 0);
    }
  }
  
  remainingBandwidth = pref_bandwidthPerTick;
    
  unlock();
}

// called when a message arrives at a catom
// inserts it into the mailbox for that type of message

void NetworkAdapter::messageIntentArrived(Message* msg) {
  msg->arrivalContact = hostFeature;

  // First tell the speculation manager about it
  // -- Commented out temporarily; uncomment this when we move to SM issuing misspeculated messages --
  //SpeculationManager* specMgr = CODE_MODULE("SpeculationManager");
  //if(specMgr) {
  //  specMgr->messageIntentArrived(msg);
  //}

  if(msgTrace.is_open() && (worldPtr->current_time >= pref_fastStartEndTime)) {
    worldPtr->oStart();
    msgTrace << worldPtr->current_time << " msgTrace "
	     << msg->messageID << " "
	     << hostCatom << " " << hostFeature << " "
	     << "intentArr\n";
    worldPtr->oEnd();
  }

  // Then file it for standard message handlers
  worldPtr->catomHash[hostCatom]->C.mailboxManager.fileMessage(msg, MSGINTENT);
}

void NetworkAdapter::messageHeadersArrived(Message* msg) {
  msg->arrivalContact = hostFeature;

#ifdef HALF_DUPLEX
  lock();
  
  unsigned long long headerSize = msg->headerSize();
  if(headerSize > remainingBandwidth) {
    remainingBandwidth = 0;
  } else {
    remainingBandwidth -= headerSize;
  }
  
  unlock();
#endif

  if(msgTrace.is_open() && (worldPtr->current_time >= pref_fastStartEndTime)) {
    worldPtr->oStart();
    msgTrace << worldPtr->current_time << " msgTrace "
	     << msg->messageID << " "
	     << hostCatom << " " << hostFeature << " "
	     << "headersArr\n";
    worldPtr->oEnd();
  }

  worldPtr->catomHash[hostCatom]->C.mailboxManager.fileMessage(msg, MSGHEADER);
}

void NetworkAdapter::messagePayloadArrived(Message* msg) {
  msg->arrivalContact = hostFeature;
  
#ifdef HALF_DUPLEX
  lock();
  
  //stats_msgsRcvd++;
  
  unsigned long long payloadSize = msg->payloadSize();
  if(payloadSize > remainingBandwidth) {
    remainingBandwidth = 0;
  } else {
    remainingBandwidth -= payloadSize;
  }

  unlock();
#endif

  if(msgTrace.is_open() /*&& (worldPtr->current_time >= pref_fastStartEndTime)*/) {
    worldPtr->oStart();
    msgTrace << worldPtr->current_time << " msgTrace "
	     << msg->messageID << " "
	     << hostCatom << " " << hostFeature << " "
	     << "payloadArr\n";
    worldPtr->oEnd();
  }

  worldPtr->catomHash[hostCatom]->C.mailboxManager.fileMessage(msg, MSGPAYLOAD);
}

// send message m out contact c; (potentially) asynchronous send
bool NetworkAdapter::sendMessage(Message* m) {
  if(worldPtr->catomHash[hostCatom]->C.getFeatureMap()[hostFeature].getNumRemoteFeatures() == 0) {
    delete m;
    return false;
  }
  
  lock();
  
  if(msgTrace.is_open() && (worldPtr->current_time >= pref_fastStartEndTime)) {
    worldPtr->oStart();
    msgTrace << worldPtr->current_time << " msgTrace "
	     << m->messageID << " "
	     << hostCatom << " " << hostFeature << " "
	     << "outQ\n";
    worldPtr->oEnd();
  }

  if(!messageQueueOut)
    messageQueueOut = new queue<Message*>;
  messageQueueOut->push(m); //stats_msgsSent++;
  
  unlock();
  
  sendMessages();
  
  return true;
}


// Note: NA must already be locked before calling this method
void NetworkAdapter::workOnCurrentMessage() {
  if(!remainingBandwidth || !currentMessage)
    return;

  // Work on the header first, if needed
  if(currentMessageHeaderBytesRemaining > 0) {
    if(remainingBandwidth >= currentMessageHeaderBytesRemaining) {
      // We can complete the header
      remainingBandwidth -= currentMessageHeaderBytesRemaining;
      currentMessageHeaderBytesRemaining = 0;
      
      // Send the header to our connected features
      Feature* _f;
      for(unsigned i=0; i<worldPtr->catomHash[hostCatom]->C.getFeatureMap()[hostFeature].getNumRemoteFeatures(); i++) {
	_f = worldPtr->catomHash[hostCatom]->C.getFeatureMap()[hostFeature].getNthRemoteFeature(i);
	if(_f) // Put this check as magic move may cause remote catom to move away. - Ram
	  _f->getNetworkAdapter()->messageHeadersArrived((Message*)currentMessage->clone());
      }
    } else {
      // Just make progress on the header
      currentMessageHeaderBytesRemaining -= remainingBandwidth;
      remainingBandwidth = 0;
      // And now that we're out of bandwidth, there's nothing else to do
      return;
    }
  }
  
  // We should definitely have payload left at this point if there's no header left
  assert(currentMessagePayloadBytesRemaining > 0);

  if(remainingBandwidth >= currentMessagePayloadBytesRemaining) {
    // We can complete the payload
    remainingBandwidth -= currentMessagePayloadBytesRemaining;
    currentMessagePayloadBytesRemaining = 0;

    // Send the payload to our connected features
    Feature* _f;
    for(unsigned i=0; i<worldPtr->catomHash[hostCatom]->C.getFeatureMap()[hostFeature].getNumRemoteFeatures(); i++) {
      _f = worldPtr->catomHash[hostCatom]->C.getFeatureMap()[hostFeature].getNthRemoteFeature(i);
      if(_f) // Put this check as magic move may cause remote catom to move away. - Ram
	_f->getNetworkAdapter()->messagePayloadArrived((Message*)(currentMessage->clone()));
    }

    // And now we're done with this message; clear it out
    delete currentMessage;
    currentMessage = NULL;
  } else {
    // Just make progress on the payload
    currentMessagePayloadBytesRemaining -= remainingBandwidth;
    remainingBandwidth = 0;
    // And now that we're out of bandwidth, there's nothing else to do
    return;
  }
}

void NetworkAdapter::sendMessages() {
  lock();

  // If we don't have any bandwidth, there's nothing to do for now
  if(!remainingBandwidth) {
    unlock();
    return;
  }

  // Make progress on any current message, if there is one
  workOnCurrentMessage();

  // Then, keep trying to send any more messages we can
  while((remainingBandwidth > 0) && !currentMessage && messageQueueOut && !messageQueueOut->empty()) {
    // Get next message from queue
    currentMessage = messageQueueOut->front();
    messageQueueOut->pop();
    if(messageQueueOut->empty()) {
      delete messageQueueOut;
      messageQueueOut = NULL;
    }

    if(msgTrace.is_open() && (worldPtr->current_time >= pref_fastStartEndTime)) {
      worldPtr->oStart();
      msgTrace << worldPtr->current_time << " msgTrace "
	       << currentMessage->messageID << " "
	       << hostCatom << " " << hostFeature << " "
	       << "outBegin\n";
      worldPtr->oEnd();
    }

    // Set up size parameters
    currentMessageHeaderBytesRemaining = currentMessage->headerSize();
    currentMessagePayloadBytesRemaining = currentMessage->payloadSize();

    // Send intent to connected features
    Feature* _f;
    for(unsigned i=0; i<worldPtr->catomHash[hostCatom]->C.getFeatureMap()[hostFeature].getNumRemoteFeatures(); i++) {
      _f = worldPtr->catomHash[hostCatom]->C.getFeatureMap()[hostFeature].getNthRemoteFeature(i);
      if(_f) // Put this check as magic move may cause remote catom to move away. - Ram
	_f->getNetworkAdapter()->messageIntentArrived((Message*)currentMessage->clone());
    }

    // Then work on the rest of it
    workOnCurrentMessage();
  }
  unlock();
}
