///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright (C) 2006 by Intel Corporation and Carnegie Mellon University    //
// Contacts: bdr @ cs.cmu.edu                                                //
//           casey.j.helfrich @ intel.com                                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "Historian.hxx"

#include <pthread.h>
#include <stdio.h>
#include <unistd.h>

#include "CatomSim.hxx"
#include "CatomWorld.hxx"

using namespace std;

#define OPS_PER_TRANSACTION 100000

// Concurrency protection convenience

static pthread_mutex_t historianMutex;//PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;
static bool mutex_inited = false;
class AutoreleasingLock {
public:
  AutoreleasingLock() { 
    if (!mutex_inited) {
      mutex_inited = true;
      pthread_mutexattr_t mutex_attrs;
      pthread_mutexattr_init(&mutex_attrs);
      pthread_mutexattr_settype(&mutex_attrs, PTHREAD_MUTEX_RECURSIVE);
      pthread_mutex_init(&historianMutex, &mutex_attrs);
      pthread_mutexattr_destroy(&mutex_attrs);
    }
    pthread_mutex_lock(&historianMutex); }
  ~AutoreleasingLock() { pthread_mutex_unlock(&historianMutex); }
};


// Constructor/destructor

Historian::Historian(string _givenDBPath, string _debuggingEnabled, bool _overwrite) :
  dbPath(_givenDBPath),
  cachedWorldStateCatom(0xffffffff),
  cachedWorldStateTick(0xffffffff),
  currentSignpostNum(0),
  currentEventNum(0),
  currentValueNum(0) {
  int returnCode;
  char* errMsg = NULL;
  char* command = NULL;

  // Set debuggingEnabled flag
  if((_debuggingEnabled == "true") || (_debuggingEnabled == "1"))
    debuggingEnabled = true;
  else
    debuggingEnabled = false;

  // Create a default location, if necessary
  if(dbPath == "") {
    char tmpStr[64];
    snprintf(tmpStr, 63, "/tmp/dprdebug.%d", getpid());
    dbPath = tmpStr;
  }

  // Open the database
  if(_overwrite)
    unlink(dbPath.c_str());
  returnCode = sqlite3_open(dbPath.c_str(), &db);
  if(returnCode) {
    cerr << "Can't open SQLite database for debugging (" << returnCode << ", " << sqlite3_errmsg(db) << ")" << endl;
    exit(-1);
  }

  // Turn off DB disk synch
  command = "PRAGMA synchronous = OFF;";
  returnCode = sqlite3_exec(db, command, NULL, 0, &errMsg);
  if(SQLITE_OK != returnCode) {
    cerr << "Can't disable synchronous behavior! (" << (errMsg ? errMsg : "no error message") << ")\n"
	 << "Query: " << command << endl;
    exit(-1);
  }

  // Create tables, if we're starting a new DB
  if(_overwrite) {
    command = "CREATE TABLE 'signposts' (name, file, line, function);";
    returnCode = sqlite3_exec(db, command, NULL, 0, &errMsg);
    if(SQLITE_OK != returnCode) {
      cerr << "Can't create signposts table! (" << (errMsg ? errMsg : "no error message") << ")\n"
	   << "Query: " << command << endl;
      exit(-1);
    }
    command = "CREATE TABLE 'events' (name, hostCatom, hostCodeModule, value, tick, signpost, operation);";
    returnCode = sqlite3_exec(db, command, NULL, 0, &errMsg);
    if(SQLITE_OK != returnCode) {
      cerr << "Can't create events table! (" << (errMsg ? errMsg : "no error message") << ")\n"
	   << "Query: " << command << endl;
      exit(-1);
    }
    command = "CREATE INDEX EventsMainIndex ON 'events' (hostCatom, tick);";
    returnCode = sqlite3_exec(db, command, NULL, 0, &errMsg);
    if(SQLITE_OK != returnCode) {
      cerr << "Can't create events main index! (" << (errMsg ? errMsg : "no error message") << ")\n"
           << "Query: " << command << endl;
      exit(-1);
    }
    command = "CREATE TABLE 'values' (value, object, type, derived1, derived2);";
    returnCode = sqlite3_exec(db, command, NULL, 0, &errMsg);
    if(SQLITE_OK != returnCode) {
      cerr << "Can't create values table! (" << (errMsg ? errMsg : "no error message") << ")\n"
	   << "Query: " << command << endl;
      exit(-1);
    }
    command = "CREATE TABLE 'ws_lines' (tick, srcCatom, srcPtX, srcPtY, srcPtZ, destCatom, destPtX, destPtY, destPtZ, r, g, b, type);";
    returnCode = sqlite3_exec(db, command, NULL, 0, &errMsg);
    if(SQLITE_OK != returnCode) {
      cerr << "Can't create ws_lines table! (" << (errMsg ? errMsg : "no error message") << ")\n"
	   << "Query: " << command << endl;
      exit(-1);
    }
    #define WORLDSTATE_TICK_IDX 0
    #define WORLDSTATE_CATOMID_IDX 1
    #define WORLDSTATE_LOCX_IDX 2
    #define WORLDSTATE_LOCY_IDX 3
    #define WORLDSTATE_LOCZ_IDX 4
    #define WORLDSTATE_QX_IDX 5
    #define WORLDSTATE_QY_IDX 6
    #define WORLDSTATE_QZ_IDX 7
    #define WORLDSTATE_QA_IDX 8
    #define WORLDSTATE_RED_IDX 9
    #define WORLDSTATE_GREEN_IDX 10
    #define WORLDSTATE_BLUE_IDX 11
    #define WORLDSTATE_ALPHA_IDX 12
    command = "CREATE TABLE 'ws_catoms' (tick, catomID, locX, locY, locZ, qx, qy, qz, qa, r, g, b, a);";
    returnCode = sqlite3_exec(db, command, NULL, 0, &errMsg);
    if(SQLITE_OK != returnCode) {
      cerr << "Can't create ws_catoms table! (" << (errMsg ? errMsg : "no error message") << ")\n"
	   << "Query: " << command << endl;
      exit(-1);
    }

    command = "CREATE INDEX CatomsIndex ON 'ws_catoms' (tick, catomID);";
    returnCode = sqlite3_exec(db, command, NULL, 0, &errMsg);
    if(SQLITE_OK != returnCode) {
      cerr << "Can't create ws_catoms index! (" << (errMsg ? errMsg : "no error message") << ")\n"
           << "Query: " << command << endl;
      exit(-1);
    }
  }
 
  // Prepare compiled commands
  command = "INSERT INTO 'signposts' (name, file, line, function) VALUES (?, ?, ?, ?);";
  returnCode = sqlite3_prepare(db, command, -1, &newSignpost_stmt, NULL);
  if((SQLITE_OK != returnCode) || !newSignpost_stmt) {
    cerr << "Can't compile new signpost command! (" << sqlite3_errmsg(db) << ")\n"
	 << "Command: " << command << endl;
    exit(-1);
  }
  command = "INSERT INTO 'events' (name, hostCatom, hostCodeModule, value, tick, signpost, operation) VALUES (?, ?, ?, ?, ?, ?, ?);";
  returnCode = sqlite3_prepare(db, command, -1, &newEvent_stmt, NULL);
  if((SQLITE_OK != returnCode) || !newEvent_stmt) {
    cerr << "Can't compile new event command! (" << sqlite3_errmsg(db) << ")\n"
	 << "Command: " << command << endl;
    exit(-1);
  }
  command = "SELECT val.value FROM 'events' as evt, 'values' as val WHERE (evt.hostCatom = ?) AND (evt.tick <= ?) AND (evt.name = ?) AND (evt.value = val.rowid) ORDER BY evt.tick DESC LIMIT 1;";
  returnCode = sqlite3_prepare(db, command, -1, &findVariableVal_stmt, NULL);
  if((SQLITE_OK != returnCode) || !findVariableVal_stmt) {
    cerr << "Can't compile 'findVariableVal' command! (" << sqlite3_errmsg(db) << ")\n"
	 << "Command: " << command << endl;
    exit(-1);
  }
  command = "SELECT val.value FROM 'events' as evt, 'values' as val WHERE (evt.hostCatom = ?) AND (evt.hostCodeModule = ?) AND (evt.tick <= ?) AND (evt.name = ?) AND (evt.value = val.rowid) ORDER BY evt.tick DESC LIMIT 1;";
  returnCode = sqlite3_prepare(db, command, -1, &findVariableVal2_stmt, NULL);
  if((SQLITE_OK != returnCode) || !findVariableVal2_stmt) {
    cerr << "Can't compile 'findVariableVal2' command! (" << sqlite3_errmsg(db) << ")\n"
	 << "Command: " << command << endl;
    exit(-1);
  }
  command = "INSERT INTO 'values' (value, type, derived1, derived2) VALUES (?, ?, ?, ?);";
  returnCode = sqlite3_prepare(db, command, -1, &newValue_stmt, NULL);
  if((SQLITE_OK != returnCode) || !newValue_stmt) {
    cerr << "Can't compile new value command! (" << sqlite3_errmsg(db) << ")\n"
	 << "Command: " << command << endl;
    exit(-1);
  }
  command = "INSERT INTO 'ws_lines' (tick, srcCatom, srcPtX, srcPtY, srcPtZ, destCatom, destPtX, destPtY, destPtZ, r, g, b, type) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);";
  returnCode = sqlite3_prepare(db, command, -1, &newLine_stmt, NULL);
  if((SQLITE_OK != returnCode) || !newLine_stmt) {
    cerr << "Can't compile new line command! (" << sqlite3_errmsg(db) << ")\n"
	 << "Command: " << command << endl;
    exit(-1);
  }
  command = "INSERT INTO 'ws_catoms' (tick, catomID, locX, locY, locZ, qx, qy, qz, qa, r, g, b, a) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);";
  returnCode = sqlite3_prepare(db, command, -1, &newCatom_stmt, NULL);
  if((SQLITE_OK != returnCode) || !newCatom_stmt) {
    cerr << "Can't compile new catom command! (" << sqlite3_errmsg(db) << ")\n"
	 << "Command: " << command << endl;
    exit(-1);
  }
  command = "SELECT * FROM 'ws_catoms' WHERE (tick = ?) AND (catomID = ?) LIMIT 1;";
  returnCode = sqlite3_prepare(db, command, -1, &findCatomWorldState_stmt, NULL);
  if((SQLITE_OK != returnCode) || !findCatomWorldState_stmt) {
    cerr << "Can't compile 'findCatomWorldState' command! (" << sqlite3_errmsg(db) << ")\n"
	 << "Command: " << command << endl;
    exit(-1);
  }
  command = "SELECT DISTINCT catomID from 'ws_catoms';";
  returnCode = sqlite3_prepare(db, command, -1, &findCatomIDs_stmt, NULL);
  if((SQLITE_OK != returnCode) || !findCatomIDs_stmt) {
    cerr << "Can't compile 'findCatomIDs' command! (" << sqlite3_errmsg(db) << ")\n"
	 << "Command: " << command << endl;
    exit(-1);
  }
  command = "SELECT * from 'ws_lines' WHERE (tick = ?);";
  returnCode = sqlite3_prepare(db, command, -1, &findLinesForTick_stmt, NULL);
  if((SQLITE_OK != returnCode) || !findLinesForTick_stmt) {
    cerr << "Can't compile 'findLinesForTick' command! (" << sqlite3_errmsg(db) << ")\n"
	 << "Command: " << command << endl;
    exit(-1);
  }

  // Start initial transaction
  command = "BEGIN;";
  returnCode = sqlite3_exec(db, command, NULL, 0, &errMsg);
  if(SQLITE_OK != returnCode) {
    cerr << "Can't start first transaction! (" << (errMsg ? errMsg : "no error message") << ")\n"
	 << "Query: " << command << endl;
    exit(-1);
  }
}

Historian::~Historian() {
  historianPtr = NULL;

  if(newSignpost_stmt)
    sqlite3_finalize(newSignpost_stmt);
  if(newEvent_stmt)
    sqlite3_finalize(newEvent_stmt);
  if(newValue_stmt)
    sqlite3_finalize(newValue_stmt);
  if(newLine_stmt)
    sqlite3_finalize(newLine_stmt);
  if(newCatom_stmt)
    sqlite3_finalize(newCatom_stmt);
  if(findCatomWorldState_stmt)
    sqlite3_finalize(findCatomWorldState_stmt);
  if(findCatomIDs_stmt)
    sqlite3_finalize(findCatomIDs_stmt);
  if(findLinesForTick_stmt)
    sqlite3_finalize(findLinesForTick_stmt);
  if(findVariableVal_stmt)
    sqlite3_finalize(findVariableVal_stmt);
  if(findVariableVal2_stmt)
    sqlite3_finalize(findVariableVal2_stmt);

  // Commit final transaction
  char* errMsg = NULL;
  char* command = "COMMIT;";
  int returnCode = sqlite3_exec(db, command, NULL, 0, &errMsg);
  if(SQLITE_OK != returnCode) {
    cerr << "Can't commit final transaction! (" << (errMsg ? errMsg : "no error message") << ")\n"
	 << "Query: " << command << endl;
    exit(-1);
  }

  sqlite3_close(db);

  if(dbPath.find("/tmp/dprdebug.") == 0) {
    unlink(dbPath.c_str());
  }
}


// DB support
static unsigned opsSoFar = 0;
void Historian::maybeNewTxn() {  
  if(opsSoFar > OPS_PER_TRANSACTION) {
    newTxn();
  }
  opsSoFar ++;
}

void Historian::newTxn() {
  // End old transaction, begin new transaction
  char* errMsg = NULL;
  char* command = "COMMIT; BEGIN;";
  int returnCode = sqlite3_exec(db, command, NULL, 0, &errMsg);
  if(SQLITE_OK != returnCode) {
    cerr << "Can't start new transaction! (" << (errMsg ? errMsg : "no error message") << ")\n"
	 << "Query: " << command << endl;
    exit(-1);
  }
  opsSoFar = 0;
}


// World state support

void Historian::saveWorldStateForTick(long tick) {
  AutoreleasingLock lock;

  vector<DPRLine>::iterator linesIterator;
  for(linesIterator =  worldPtr->lines.begin();
      linesIterator != worldPtr->lines.end();
      linesIterator++) {
    saveLineForTick(*linesIterator, tick);
  }

  hash_map<const unsigned long, CatomSim *, hash<const unsigned long>, equl>::iterator catomsIterator;
  for(catomsIterator =  worldPtr->catomHash.begin();
      catomsIterator != worldPtr->catomHash.end();
      catomsIterator++) {
    pair<const unsigned long, CatomSim*> entry = *catomsIterator;
    saveCatomForTick(entry.second, tick);
  }
}

void Historian::saveLineForTick(DPRLine line, long tick) {
  int returnCode;

  maybeNewTxn();

  returnCode = sqlite3_reset(newLine_stmt);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int(newLine_stmt, 1, tick);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int64(newLine_stmt, 2, line.getSrc());
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_double(newLine_stmt, 3, line.getSrcPt().getX());
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_double(newLine_stmt, 4, line.getSrcPt().getY());
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_double(newLine_stmt, 5, line.getSrcPt().getZ());
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int64(newLine_stmt, 6, line.getDest());
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_double(newLine_stmt, 7, line.getDestPt().getX());
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_double(newLine_stmt, 8, line.getDestPt().getY());
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_double(newLine_stmt, 9, line.getDestPt().getZ());
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int(newLine_stmt, 10, line.getRed());
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int(newLine_stmt, 11, line.getGreen());
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int(newLine_stmt, 12, line.getBlue());
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int(newLine_stmt, 13, line.getType());

  if(SQLITE_OK != returnCode) {
    cerr << "Can't prepare new line command: " << sqlite3_errmsg(db) << endl;
    exit(-1);
  }

  do {
    returnCode = sqlite3_step(newLine_stmt);
  } while(SQLITE_BUSY == returnCode);
  if(SQLITE_DONE != returnCode) {
    cerr << "Can't add line to db: " << sqlite3_errmsg(db) << endl;
    exit(-1);
  }
}

void Historian::saveCatomForTick(CatomSim* sim, long tick) {
  int returnCode;
  AutoreleasingLock lock;

  decachePendingStatements();

  maybeNewTxn();

  returnCode = sqlite3_reset(newCatom_stmt);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int(newCatom_stmt, 1, tick);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int64(newCatom_stmt, 2, sim->C.getID());
  const dReal* dpos = dGeomGetPosition(sim->shape);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_double(newCatom_stmt, 3, dpos[0]);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_double(newCatom_stmt, 4, dpos[1]);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_double(newCatom_stmt, 5, dpos[2]);
  const dReal* dRot = dBodyGetQuaternion(sim->body);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_double(newCatom_stmt, 6, dRot[0]);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_double(newCatom_stmt, 7, dRot[1]);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_double(newCatom_stmt, 8, dRot[2]);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_double(newCatom_stmt, 9, dRot[3]);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int(newCatom_stmt, 10, sim->red);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int(newCatom_stmt, 11, sim->green);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int(newCatom_stmt, 12, sim->blue);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int(newCatom_stmt, 13, sim->alpha);

  if(SQLITE_OK != returnCode) {
    cerr << "Can't prepare new catom command: " << sqlite3_errmsg(db) << endl;
    exit(-1);
  }

  do {
    returnCode = sqlite3_step(newCatom_stmt);
  } while(SQLITE_BUSY == returnCode);
  if(SQLITE_DONE != returnCode) {
    cerr << "Can't add catom to db: " << sqlite3_errmsg(db) << endl;
    exit(-1);
  }
}

bool Historian::loadWorldStateForCatomAndTick(unsigned long catom, long tick) {
  if((catom == cachedWorldStateCatom) && (tick == cachedWorldStateTick))
    return true;

  int returnCode;

  returnCode = sqlite3_reset(findCatomWorldState_stmt);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int(findCatomWorldState_stmt, 1, tick);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int64(findCatomWorldState_stmt, 2, catom);
  
  if(SQLITE_OK != returnCode) {
    cerr << "Can't bind findCatomWorldState command: " << sqlite3_errmsg(db) << endl;
    exit(-1);
  }

  do {
    returnCode = sqlite3_step(findCatomWorldState_stmt);
  } while(SQLITE_BUSY == returnCode);
  
  if(SQLITE_ROW == returnCode) {
    cachedWorldStateCatom = catom;
    cachedWorldStateTick = tick;
    return true;
  } else {
    cachedWorldStateCatom = 0xffffffff;
    cachedWorldStateTick = 0xffffffff;
    return false;
  }
}

void Historian::decachePendingStatements() {
  sqlite3_reset(findCatomWorldState_stmt);
  //sqlite3_reset(findCatomIDs_stmt);
  //sqlite3_reset(findLinesForTick_stmt);
}

vector<catomID> Historian::getWorldStateCatomIDs() {
  int returnCode;
  vector<catomID> catomIDs;

  returnCode = sqlite3_reset(findCatomIDs_stmt);
  if(SQLITE_OK == returnCode) {
    while(SQLITE_ROW == (returnCode = sqlite3_step(findCatomIDs_stmt))) {
      catomIDs.push_back(sqlite3_column_int(findCatomIDs_stmt, 0));
    }
  }

  return catomIDs;
}

string Historian::getWorldStateStringForCatomAndTick(string key, unsigned long catom, long tick) {
  AutoreleasingLock lock;

  if(!loadWorldStateForCatomAndTick(catom,tick))
    throw 0;

  // No match
  throw 0;
}
double Historian::getWorldStateDoubleForCatomAndTick(string key, unsigned long catom, long tick) {
  AutoreleasingLock lock;

  if(!loadWorldStateForCatomAndTick(catom,tick))
    throw 0;

  if(key == "locx")
    return sqlite3_column_double(findCatomWorldState_stmt, WORLDSTATE_LOCX_IDX);
  if(key == "locy")
    return sqlite3_column_double(findCatomWorldState_stmt, WORLDSTATE_LOCY_IDX);
  if(key == "locz")
    return sqlite3_column_double(findCatomWorldState_stmt, WORLDSTATE_LOCZ_IDX);
  
  if(key == "qx")
    return sqlite3_column_double(findCatomWorldState_stmt, WORLDSTATE_QX_IDX);
  if(key == "qy")
    return sqlite3_column_double(findCatomWorldState_stmt, WORLDSTATE_QY_IDX);
  if(key == "qz")
    return sqlite3_column_double(findCatomWorldState_stmt, WORLDSTATE_QZ_IDX);
  if(key == "qa")
    return sqlite3_column_double(findCatomWorldState_stmt, WORLDSTATE_QA_IDX);

  if(key == "red")
    return sqlite3_column_double(findCatomWorldState_stmt, WORLDSTATE_RED_IDX);
  if(key == "blue")
    return sqlite3_column_double(findCatomWorldState_stmt, WORLDSTATE_BLUE_IDX);
  if(key == "green")
    return sqlite3_column_double(findCatomWorldState_stmt, WORLDSTATE_GREEN_IDX);
  if(key == "alpha")
    return sqlite3_column_double(findCatomWorldState_stmt, WORLDSTATE_ALPHA_IDX);

  // No match
  throw 0;
}

vector<DPRLine> Historian::getLinesForTick(long tick) {
  vector<DPRLine> returnVector;

  //command = "CREATE TABLE 'ws_lines' (tick, srcCatom, srcPtX, srcPtY, srcPtZ, destCatom, destPtX, destPtY, destPtZ, r, g, b, type);";
  int returnCode = sqlite3_reset(findLinesForTick_stmt);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int(findLinesForTick_stmt, 1, tick);
  if(SQLITE_OK != returnCode){
    cerr << "Can't bind line-getting command: " << sqlite3_errmsg(db) << endl;
    exit(-1);
  }

  do {
    returnCode = sqlite3_step(findLinesForTick_stmt);
    switch(returnCode) {
    case SQLITE_DONE:
    case SQLITE_BUSY:
      // don't have to do anything
      break;
    case SQLITE_ROW:
      {
	// Add an entry to the lines vector
	Point3D srcPt (sqlite3_column_double(findLinesForTick_stmt, 2),
		       sqlite3_column_double(findLinesForTick_stmt, 3),
		       sqlite3_column_double(findLinesForTick_stmt, 4));
	Point3D destPt(sqlite3_column_double(findLinesForTick_stmt, 6),
		       sqlite3_column_double(findLinesForTick_stmt, 7),
		       sqlite3_column_double(findLinesForTick_stmt, 8));
	DPRLine theLine(sqlite3_column_int(findLinesForTick_stmt, 1), // srcCatom
			srcPt,
			sqlite3_column_int(findLinesForTick_stmt, 5), // destCatom
			destPt,
			sqlite3_column_int(findLinesForTick_stmt, 9),  // r
			sqlite3_column_int(findLinesForTick_stmt, 10), // g
			sqlite3_column_int(findLinesForTick_stmt, 11), // b
			(DPRLINETYPE)sqlite3_column_int(findLinesForTick_stmt, 12));// type
	returnVector.push_back(theLine);
      }
      break;
    default:
      // some kind of error
      cerr << "Error getting lines from db: " << sqlite3_errmsg(db) << endl;
      exit(-1);
    }
  } while(SQLITE_DONE != returnCode);
  
  return returnVector;
}

// Debugging support

string Historian::getStringVariableForCatomAndTick(string name, unsigned long catom, long tick) {
  int returnCode = sqlite3_reset(findVariableVal_stmt);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int64(findVariableVal_stmt, 1, catom);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int  (findVariableVal_stmt, 2, tick);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_text (findVariableVal_stmt, 3, name.c_str(), -1, SQLITE_TRANSIENT);
  if(SQLITE_OK != returnCode) {
    cerr << "Can't prepare find variable val command: " << sqlite3_errmsg(db) << endl;
    exit(-1);
  }

  do {
    returnCode = sqlite3_step(findVariableVal_stmt);
  } while(SQLITE_BUSY == returnCode);
  if(SQLITE_ROW != returnCode) {
    cerr << "Can't find variable value: " << returnCode << " " << sqlite3_errmsg(db) << endl;
    cerr << name << " " << catom << " " << tick << endl;
    throw 0;
  }

  string retVal = (const char *)sqlite3_column_text(findVariableVal_stmt, 0);
  sqlite3_reset(findVariableVal_stmt);

  return retVal;
}

double Historian::getDoubleVariableForCatomCodeModuleAndTick(string name, unsigned long catom, string codeModule, long tick) {
  newTxn();

  int returnCode = sqlite3_reset(findVariableVal2_stmt);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int64(findVariableVal2_stmt, 1, catom);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_text (findVariableVal2_stmt, 2, codeModule.c_str(), -1, SQLITE_TRANSIENT);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int  (findVariableVal2_stmt, 3, tick);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_text (findVariableVal2_stmt, 4, name.c_str(), -1, SQLITE_TRANSIENT);
  if(SQLITE_OK != returnCode) {
    cerr << "Can't prepare find variable val command: " << sqlite3_errmsg(db) << endl;
    exit(-1);
  }

  do {
    returnCode = sqlite3_step(findVariableVal2_stmt);
  } while(SQLITE_BUSY == returnCode);
  if(SQLITE_ROW != returnCode) {
    cerr << "Can't find variable value: " << returnCode << " " << sqlite3_errmsg(db) << endl;
    cerr << name << " " << catom << " " << codeModule << " " << tick << endl;
    throw 0;
  }

  double retVal = sqlite3_column_double(findVariableVal2_stmt, 0);
  sqlite3_reset(findVariableVal2_stmt);

  return retVal;
}

double Historian::getDoubleVariableForCatomAndTick(string name, unsigned long catom, long tick) {
  string::size_type periodLoc = name.find('.');
  if(periodLoc != string::npos) {
    // We have a class name!
    return getDoubleVariableForCatomCodeModuleAndTick(name.substr(periodLoc+1, name.size()-periodLoc-1),
						      catom,
						      name.substr(0, periodLoc),
						      tick);
  }
  // Otherwise, it's a generic one

  int returnCode = sqlite3_reset(findVariableVal_stmt);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int64(findVariableVal_stmt, 1, catom);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int  (findVariableVal_stmt, 2, tick);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_text (findVariableVal_stmt, 3, name.c_str(), -1, SQLITE_TRANSIENT);
  if(SQLITE_OK != returnCode) {
    cerr << "Can't prepare find variable val command: " << sqlite3_errmsg(db) << endl;
    exit(-1);
  }

  do {
    returnCode = sqlite3_step(findVariableVal_stmt);
  } while(SQLITE_BUSY == returnCode);
  if(SQLITE_ROW != returnCode) {
    cerr << "Can't find variable value: " << returnCode << " " << sqlite3_errmsg(db) << endl;
    cerr << name << " " << catom << " " << tick << endl;
    throw 0;
  }

  double retVal = sqlite3_column_double(findVariableVal_stmt, 0);
  sqlite3_reset(findVariableVal_stmt);

  return retVal;
}

void Historian::addSignpost(string name, string file, uint32_t line, string function) {
  if(!debuggingEnabled)
    return;
  AutoreleasingLock lock;

  decachePendingStatements();

  maybeNewTxn();

  int returnCode = sqlite3_reset(newSignpost_stmt);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_text(newSignpost_stmt, 1, name.c_str(),     -1, SQLITE_TRANSIENT);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_text(newSignpost_stmt, 2, file.c_str(),     -1, SQLITE_TRANSIENT);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int (newSignpost_stmt, 3, line                                  );
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_text(newSignpost_stmt, 4, function.c_str(), -1, SQLITE_TRANSIENT);
  if(SQLITE_OK != returnCode) {
    cerr << "Can't prepare new signpost command: " << sqlite3_errmsg(db) << endl;
    exit(-1);
  }

  do {
    returnCode = sqlite3_step(newSignpost_stmt);
  } while(SQLITE_BUSY == returnCode);
  if(SQLITE_DONE != returnCode) {
    cerr << "Can't add signpost to debugging db!:" << sqlite3_errmsg(db) << endl;
    exit(-1);
  }

  currentSignpostNum = sqlite3_last_insert_rowid(db);
}

void Historian::addEvent(string name, void* object, uint64_t value, uint8_t operation) {
  if(!debuggingEnabled)
    return;
  AutoreleasingLock lock;

  decachePendingStatements();

  maybeNewTxn();

  int returnCode = sqlite3_reset(newEvent_stmt);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_text (newEvent_stmt, 1, name.c_str(), -1, SQLITE_TRANSIENT);
  void* ownerObject = findOwnerOf(object).first;// ownerObject = NULL;
  uint64_t ownerNum = (ownerObject != NULL) ? ((CodeModule*)ownerObject)->getHostCatom() : 0;
  string ownerCodeModule = (ownerObject != NULL) ? ((CodeModule*)ownerObject)->getName() : "";
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int64(newEvent_stmt, 2, ownerNum );
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_text (newEvent_stmt, 3, ownerCodeModule.c_str(), -1, SQLITE_TRANSIENT);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int64(newEvent_stmt, 4, value);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int  (newEvent_stmt, 5, worldPtr->current_time);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int64(newEvent_stmt, 6, currentSignpostNum);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int  (newEvent_stmt, 7, operation);
  if(SQLITE_OK != returnCode) {
    cerr << "Can't prepare new event command: " << sqlite3_errmsg(db) << endl;
    exit(-1);
  }

  do {
    returnCode = sqlite3_step(newEvent_stmt);
  } while(SQLITE_BUSY == returnCode);
  if(SQLITE_DONE != returnCode) {
    cerr << "Can't add event to debugging db!:" << sqlite3_errmsg(db) << endl;
    exit(-1);
  }
}

uint64_t Historian::addValue(uint64_t value, string type, uint64_t derived1, uint64_t derived2) {
  if(!debuggingEnabled)
    return 0;

  AutoreleasingLock lock;

  decachePendingStatements();

  maybeNewTxn();

  int returnCode = sqlite3_reset(newValue_stmt);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int64(newValue_stmt, 1, value);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_text (newValue_stmt, 2, type.c_str(), -1, SQLITE_TRANSIENT);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int64(newValue_stmt, 3, derived1);
  if(SQLITE_OK == returnCode)
    returnCode = sqlite3_bind_int64(newValue_stmt, 4, derived2);
  if(SQLITE_OK != returnCode) {
    cerr << "Can't prepare new value command: " << sqlite3_errmsg(db) << endl;
    exit(-1);
  }

  do {
    returnCode = sqlite3_step(newValue_stmt);
  } while(SQLITE_BUSY == returnCode);
  if(SQLITE_DONE != returnCode) {
    cerr << "Can't add value to debugging db!:" << sqlite3_errmsg(db) << endl;
    exit(-1);
  }

  uint64_t valueNum = sqlite3_last_insert_rowid(db);

  return valueNum;
}

void Historian::insertMemoryItem(void* address, memItemType type) {
  if(debuggingEnabled) {
    AutoreleasingLock lock;
    memoryLocations[address] = type;
  }
}

void Historian::removeMemoryItem(void* address) {
  if(debuggingEnabled) {
    AutoreleasingLock lock;
    memoryLocations.erase(address);
  }
}

pair<void*,void*> Historian::findOwnerOf(void* address) {
  if(debuggingEnabled) {
    AutoreleasingLock lock;
    void* classInfo = NULL;

    map<void*,memItemType>::iterator iter = memoryLocations.find(address);
    if(iter == memoryLocations.end())
      return pair<void*,void*>(NULL,NULL);
    
    iter--;
    while (iter != memoryLocations.begin() ) {
      switch((*iter).second) {
      case dprdebug_ivar_type:
	break;
      case dprdebug_classinfo_type:
	if(!classInfo)
	  classInfo = (*iter).first;
	break;
      case dprdebug_class_type:
	return pair<void*,void*>((*iter).first,classInfo);
      default:
	assert(false);
      }

      iter--;
    }
    
    if(((*iter).first < address) && ((*iter).second == dprdebug_class_type))
      return pair<void*,void*>((*iter).first,classInfo);
    else
      return pair<void*,void*>(NULL, classInfo);
  } else {
    return pair<void*,void*>(NULL,NULL);
  }
}
