#ifndef __DCOND_PARSE__
#define __DCOND_PARSE__
#include <string>
#include <vector>
#include <utility>
#include <set>
#include <deque>
#include <iostream>
#include <pthread.h>
#include "Primitives.hxx"
#include "boost/multi_array.hpp"

#define IMM_OFFSET INT_MIN

using namespace std;

typedef enum {
	no,maybe,yes
} TriState;

typedef enum {
	plus_op,minus_op,times_op,div_op,mod_op
} MathOp;

typedef enum {
	gt_op,lt_op,gte_op,lte_op,eq_op,neq_op
} CompOp;

typedef boost::multi_array<bool,2> adj_array;
typedef adj_array::index adj_idx;

class DCondition;
class ModuleRef;
class ModuleVar;
class Number;
class MathExp;
class CompExp;
class AndExp;
class OrExp;
class NotExp;
class NeighborExp;
class ModuleDecl;
class ModuleList;
class SetVarAction;
class ChangeTopologyAction;
class CallFcnAction;
class ActionList;
class SetRExp;
class SetBExp;
class SetSExp;
class SetVar;
class CountExp;
class IdxVar;
class IdxModuleRef;
class KleeneParent;
class DCMessage;
class DCondManager;

typedef catomID moduleID;
#define INVALID_ID 0

typedef int VarCode;

class ModuleWrapper {
	public:
		moduleID id;
		virtual set<ModuleWrapper *> neighbors() {return set<ModuleWrapper *>();}
		virtual void send(DCMessage* msg, ModuleWrapper *dest) {}
		virtual VarCode codeForVarname(string varname) {cout << "abstract module wrapper (get code):" << varname << endl; return -1;}
		virtual float getRealVar(VarCode v) {cout << "abstract module wrapper (get real):" << v << endl; return 0.0;}
		virtual void setRealVar(VarCode v, float val) {cout << "abstract module wrapper (set real):" << v << endl;}
		virtual set<float> getSetVar(VarCode v) {cout << "abstract module wrapper:" << v << endl; return set<float>();}
		virtual void setSetVar(VarCode v, set<float> val) {cout << "abstract module wrapper:" << v << endl;}
		virtual void callFcn(string fname, float arg) {cout << "abstract module wrapper:" << fname << endl;}
		virtual ~ModuleWrapper() {}
};

class DCMessage {
	public:
		DCondition *matcher;
		bool isMultihop;
		bool isAction;
		ModuleWrapper *dest;
		DCMessage(DCondition *c,bool isMulti,bool isAct,ModuleWrapper *d):matcher(c),
			isMultihop(isMulti),isAction(isAct),dest(d) {}
};

//start traversal by calling v.visitDCondition(c);
class Visitor {
	public:
		virtual void visitDCondition(DCondition *o);
		virtual void visitModuleRef(ModuleRef *o);
		virtual void visitModuleVar(ModuleVar *o);
		virtual void visitNumber(Number *o);
		virtual void visitMathExp(MathExp *o);
		virtual void visitCompExp(CompExp *o);
		virtual void visitAndExp(AndExp *o);
		virtual void visitOrExp(OrExp *o);
		virtual void visitNotExp(NotExp *o);
		virtual void visitNeighborExp(NeighborExp *o);
		virtual void visitModuleDecl(ModuleDecl *o);
		virtual void visitModuleList(ModuleList *o);
		virtual void visitSetVarAction(SetVarAction *o);
		virtual void visitChangeTopologyAction(ChangeTopologyAction *o);
		virtual void visitCallFcnAction(CallFcnAction *o);
		virtual void visitActionList(ActionList *o);
		virtual void visitSetRExp(SetRExp *o);
		virtual void visitSetBExp(SetBExp *o);
		virtual void visitSetSExp(SetSExp *o);
		virtual void visitSetVar(SetVar *o);
		virtual void visitCountExp(CountExp *o);
		virtual void visitIdxVar(IdxVar *o);
		virtual void visitIdxModuleRef(IdxModuleRef *o);
		virtual void visitKleeneParent(KleeneParent *o);
		virtual ~Visitor() {};
};

class DCObject {
	public:
		DCObject *parent; //weak ref
		virtual ~DCObject() {}
		virtual TriState isSatisfied(bool centralized = true) {return maybe;}
		virtual DCObject *clone() {return NULL;}
		virtual void accept(Visitor *v) {}
};

class BoolExp: public DCObject {	
	public:
		virtual ~BoolExp() {}
		virtual BoolExp *clone() {return NULL;}
};

class RVal: public DCObject {
	public:
		float value;
		bool filled;
		virtual ~RVal() {}
		virtual RVal *clone() {return NULL;}
};

class ModuleRef: public DCObject {
	public:
		int temporalOffset;
		string name;
		ModuleRef(const char *n);
		virtual ~ModuleRef() {}
		virtual ModuleRef *clone();
		virtual void accept(Visitor *v) {v->visitModuleRef(this);}
		virtual TriState isSatisfied(bool centralized = true);
		virtual bool matchesModule(string n, int idx) {return ((name == n) && (idx == 0));}
		virtual int idx() {return 0;}
};

class ModuleVar: public RVal {
	public:
		ModuleRef *mref;
		string var;
		VarCode code;
		ModuleVar(ModuleRef *m,const char *v);
		virtual ~ModuleVar() { delete mref; }
		virtual TriState isSatisfied(bool centralized = true);
		virtual ModuleVar *clone();
		virtual void accept(Visitor *v) {v->visitModuleVar(this);}
};

class Number : public RVal {
	public:
		Number(float val) {value = val; filled = true;}
		virtual ~Number() {}
		virtual Number *clone();
		virtual TriState isSatisfied(bool centralized = true);
		virtual void accept(Visitor *v) {v->visitNumber(this);}
};

class MathExp : public RVal {
	public:
		RVal *lval;
		MathOp op;
		RVal *rval;
		MathExp(RVal *l, MathOp o,RVal *r):lval(l),op(o),rval(r) {}
		virtual ~MathExp() { delete lval; delete rval;}
		virtual TriState isSatisfied(bool centralized = true);
		virtual MathExp *clone();
		virtual void accept(Visitor *v) {v->visitMathExp(this);}
};

class CompExp: public BoolExp {
	public:
		RVal *lval;
		CompOp op;
		RVal *rval;
		CompExp(RVal *l,const CompOp o,RVal *r):lval(l),op(o),rval(r) {}
		virtual ~CompExp() { delete lval; delete rval;}
		virtual TriState isSatisfied(bool centralized = true);
		virtual CompExp *clone();
		virtual void accept(Visitor *v) {v->visitCompExp(this);}
};

class AndExp: public BoolExp {
	public:
		BoolExp *lval;
		BoolExp *rval;
		AndExp(BoolExp *l,BoolExp *r):lval(l),rval(r) {}
		virtual ~AndExp() {delete lval; delete rval;}
		virtual TriState isSatisfied(bool centralized = true);
		virtual AndExp *clone();
		virtual void accept(Visitor *v) {v->visitAndExp(this);}
};

class OrExp: public BoolExp {
	public:
		BoolExp *lval;
		BoolExp *rval;
		OrExp(BoolExp *l,BoolExp *r):lval(l),rval(r) {}
		virtual ~OrExp() {delete lval; delete rval;}
		virtual TriState isSatisfied(bool centralized = true);
		virtual OrExp *clone();
		virtual void accept(Visitor *v) {v->visitOrExp(this);}
};

class NotExp: public BoolExp {
	public:
		BoolExp *val;
		NotExp(BoolExp *v):val(v) {}
		virtual ~NotExp() {delete val;}
		virtual TriState isSatisfied(bool centralized = true);
		virtual NotExp *clone();
		virtual void accept(Visitor *v) {v->visitNotExp(this);}
};

class NeighborExp: public BoolExp {
	public:
		ModuleRef* lval;
		ModuleRef* rval;
		//variables below are for distributed version only
		moduleID lmodule;
		moduleID rmodule;
		TriState satisfied;
		NeighborExp(ModuleRef *l,ModuleRef *r):lval(l),rval(r),satisfied(maybe) {lmodule = rmodule = INVALID_ID;}
		virtual ~NeighborExp() {delete lval; delete rval;}
		virtual TriState isSatisfied(bool centralized = true);
		virtual NeighborExp *clone();
		virtual void accept(Visitor *v) {v->visitNeighborExp(this);}
};

class ModuleDecl: public DCObject {
	public:
		string name;
		unsigned minSize;
		unsigned maxSize;
		vector<ModuleWrapper*> modules;
		ModuleDecl(const char* n):name(n),minSize(1),maxSize(1) {}
		ModuleDecl(const char* n,int min,int max):name(n),minSize(min),maxSize(max) {}
		virtual ModuleDecl *clone();
		virtual void accept(Visitor *v) {v->visitModuleDecl(this);}
		virtual TriState isSatisfied(bool centralized = true);
};

class ModuleList: public DCObject {
	public:
		vector<ModuleDecl*> modules;
		ModuleList() {}
		virtual ~ModuleList();
		void addName(ModuleDecl* m) {
			modules.push_back(m);
		}
		virtual ModuleList *clone();
		virtual void accept(Visitor *v) {v->visitModuleList(this);}
		virtual TriState isSatisfied(bool centralized = true);
};

class Action: public DCObject {
	public:
		virtual void trigger(ModuleWrapper *module,DCondManager *manager) {}
		virtual set<pair<string, int> > triggerLocs() { return set<pair<string, int> >();}
		virtual Action *clone() {return NULL;}
};

class SetVarAction: public Action {
	public:
		ModuleRef *module;
		string varname;
		VarCode code;
		RVal *value;
		SetVarAction(ModuleRef *m,const char *n, RVal *v):module(m),varname(n),value(v) {}
		virtual ~SetVarAction() {delete module; delete value;}
		virtual SetVarAction *clone();
		virtual void accept(Visitor *v) {v->visitSetVarAction(this);}
		virtual TriState isSatisfied(bool centralized = true);
		virtual set<pair<string, int> > triggerLocs();
		virtual void trigger(ModuleWrapper *module,DCondManager *manager);
};

class ChangeTopologyAction: public Action {
	public:
		ModuleRef *moduleA;
		ModuleRef *moduleB;
		bool connectLink;
		ChangeTopologyAction(ModuleRef *a,ModuleRef *b,bool conn):moduleA(a),moduleB(b),connectLink(conn) {}
		virtual ~ChangeTopologyAction() {delete moduleA; delete moduleB;}
		virtual ChangeTopologyAction *clone();
		virtual void accept(Visitor *v) {v->visitChangeTopologyAction(this);}
		virtual TriState isSatisfied(bool centralized = true);
		virtual set<pair<string, int> > triggerLocs();
		virtual void trigger(ModuleWrapper *module,DCondManager *manager) {cout << "changing topology not implemented yet" << endl;}
};

class CallFcnAction: public Action {
	public:
		ModuleRef *module;
		string fname;
		RVal *value;
		CallFcnAction(ModuleRef *m,const char *f, RVal *v):module(m),fname(f),value(v) {}
		virtual ~CallFcnAction() {delete module; delete value;}
		virtual CallFcnAction *clone();
		virtual void accept(Visitor *v) {v->visitCallFcnAction(this);}
		virtual TriState isSatisfied(bool centralized = true);
		virtual set<pair<string, int> > triggerLocs();
		virtual void trigger(ModuleWrapper *module,DCondManager *manager) {module->callFcn(fname,value->value);}
};

class ActionList: public DCObject {
	public:
		vector<Action *> actions;
		virtual ~ActionList();
		virtual ActionList *clone();
		virtual void accept(Visitor *v) {v->visitActionList(this);}
		virtual TriState isSatisfied(bool centralized = true);
		pair<string,int> triggerLoc();
};

class DCondition: public DCObject {
	public:
		ModuleList *c_list;
		BoolExp *main_exp;
		ActionList *a_list;
		adj_array adjacency;
		
		unsigned int id; //watchpoint id #
		static int nextId;
	
		DCondition(ModuleList *l,BoolExp *b,ActionList *a):c_list(l),main_exp(b),a_list(a), id(nextId++) {
			adj_array::extent_gen newExtent;
			adjacency.resize(newExtent[0][0]);
		}
		virtual ~DCondition() {delete c_list; delete main_exp; delete a_list;}
		virtual DCondition *clone();
		virtual void accept(Visitor *v) {v->visitDCondition(this);}
		virtual TriState isSatisfied(bool centralized = true);
		set<pair<string, int> > nextSlots();
		int slotIdx(ModuleWrapper *m);
		pair<string, int> slotName(ModuleWrapper *m);
		ModuleWrapper *moduleAtIndex(int idx);
		ModuleWrapper *moduleForName(string name, int idx);
		ModuleWrapper *nextRoutingStep(ModuleWrapper *start,ModuleWrapper *dest);
		bool isFull();
		int fillCount();
		int maxSizeForName(string n);
		int minSizeForName(string n);
};

class SetExp: public DCObject {
	public:
		set<float> value;
		bool filled;
		virtual SetExp *clone() {return NULL;}
};

class SetRExp: public RVal {
	public:
		SetExp *set;
		string op;
		SetRExp(SetExp* s,const char* o):set(s),op(o) {}
		virtual ~SetRExp() {delete set;}
		virtual SetRExp *clone();
		virtual void accept(Visitor *v) {v->visitSetRExp(this);}
		virtual TriState isSatisfied(bool centralized = true);
};

class SetBExp: public BoolExp {
	public:
		SetExp *set;
		string op;
		TriState satisfied;
		SetBExp(SetExp* s,const char* o):set(s),op(o),satisfied(maybe) {}
		virtual ~SetBExp() {delete set;}
		virtual SetBExp *clone();
		virtual void accept(Visitor *v) {v->visitSetBExp(this);}
		virtual TriState isSatisfied(bool centralized = true);
};

class SetSExp: public SetExp {
	public:	
		SetExp *lval;
		string op;
		SetExp *rval;
		SetSExp(SetExp *l,const char *o, SetExp *r):lval(l),op(o),rval(r) {}
		virtual ~SetSExp() {delete lval; delete rval;}
		virtual SetSExp *clone();
		virtual void accept(Visitor *v) {v->visitSetSExp(this);}
		virtual TriState isSatisfied(bool centralized = true);
};

class SetVar: public SetExp {
	public:
		ModuleRef *mref;
		string var;
		VarCode code;
		SetVar(ModuleRef *m,const char *v):mref(m),var(v) {var = var; filled = false;}
		virtual ~SetVar() {delete mref;}
		virtual SetVar *clone();
		virtual void accept(Visitor *v) {v->visitSetVar(this);}
		virtual TriState isSatisfied(bool centralized = true);
};

class CountExp: public RVal {
	public:
		vector<BoolExp*> required; //non-quantified, and quantified w/ required indices
		vector<BoolExp*> unexpanded; //quantified templates that haven't been instantiated
		vector<BoolExp*> optional; //instantiated quantifieds that are optional
		virtual ~CountExp();
		virtual CountExp *clone();
		virtual void accept(Visitor *v) {v->visitCountExp(this);}
		virtual TriState isSatisfied(bool centralized = true);
		void expand(BoolExp *quantExp);
};

class IdxVar: public RVal {
	public:
		string name;
		IdxVar(const char *n):name(n) {filled = false;}
		virtual IdxVar *clone();
		virtual void accept(Visitor *v) {v->visitIdxVar(this);}
		virtual TriState isSatisfied(bool centralized = true);
};

class IdxModuleRef: public ModuleRef {
	public:
		RVal *idxExp;
		IdxModuleRef(string n,RVal *exp):ModuleRef(n.c_str()),idxExp(exp) {}
		virtual ~IdxModuleRef() {delete idxExp;}
		virtual IdxModuleRef *clone();
		virtual void accept(Visitor *v) {v->visitIdxModuleRef(this);}
		virtual TriState isSatisfied(bool centralized = true);
		virtual bool matchesModule(string n, int idx) {
			//cout <<"idx exp valid: " << idxExp->filled << endl;
			//cout << "trying to match " << n << " " << idx << " against " << name << " " << idxExp->value << endl;
			bool match =  (((idxExp->filled) && (name == n) && (idx == idxExp->value)) ||
					    ((idxExp->filled) && (name == n) && (idxExp->value == -1)));
			//cout << "match status: " << match << endl;
			return match;
		} //idx of -1 always gets filled
		virtual int idx() {return (int) idxExp->value;}
};

class KleeneParent: public BoolExp {
	public:
		BoolExp *origExp;
		vector<BoolExp*>required;
		vector<BoolExp*>optional;
		KleeneParent(BoolExp *o):origExp(o){}
		virtual ~KleeneParent() { delete origExp;}
		virtual TriState isSatisfied(bool centralized = true);
		virtual KleeneParent *clone();
		virtual void accept(Visitor *v) {v->visitKleeneParent(this);}
		void expand();
};

class SetParentVisitor: public Visitor {
	public:
		virtual void visitDCondition(DCondition *o) {
			o->c_list->parent = o; o->main_exp->parent = o; o->parent = o;
			o->c_list->accept(this); o->main_exp->accept(this); o->a_list->accept(this);
			o->parent = NULL;
		}
		virtual void visitModuleVar(ModuleVar *o) {
			o->mref->parent = o;
			o->mref->accept(this);
		}
		virtual void visitMathExp(MathExp *o) {
			o->lval->parent = o; o->rval->parent = o;
			o->lval->accept(this); o->rval->accept(this);
		}
		virtual void visitCompExp(CompExp *o) {
			o->lval->parent = o; o->rval->parent = o;
			o->lval->accept(this); o->rval->accept(this);
		}
		virtual void visitAndExp(AndExp *o) {
			o->lval->parent = o; o->rval->parent = o;
			o->lval->accept(this); o->rval->accept(this);
		}
		virtual void visitOrExp(OrExp *o) {
			o->lval->parent = o; o->rval->parent = o;
			o->lval->accept(this); o->rval->accept(this);
		}
		virtual void visitNotExp(NotExp *o) {
			o->val->parent = o;
			o->val->accept(this);
		}
		virtual void visitNeighborExp(NeighborExp *o) {
			o->lval->parent = o; o->rval->parent = o;
			o->lval->accept(this); o->rval->accept(this);
		}
		virtual void visitModuleList(ModuleList *o) {
			for(unsigned i = 0; i < o->modules.size(); i++) {o->modules[i]->parent = o; o->modules[i]->accept(this);}
		}
		virtual void visitSetVarAction(SetVarAction *o) {
			o->module->parent = o; o->value->parent = o;
			o->module->accept(this); o->value->accept(this);
		}
		virtual void visitChangeTopologyAction(ChangeTopologyAction *o) {
			o->moduleA->parent = o; o->moduleB->parent = o;
			o->moduleA->accept(this); o->moduleB->accept(this);
		}
		virtual void visitCallFcnAction(CallFcnAction *o) {
			o->module->parent = o; o->value->parent = o;
			o->module->accept(this); o->value->accept(this);
		}
		virtual void visitActionList(ActionList *o) {
			for(unsigned i = 0; i < o->actions.size(); i++) {o->actions[i]->parent = o; o->actions[i]->accept(this);}
		}
		virtual void visitSetRExp(SetRExp *o) {
			o->set->parent = o;
			o->set->accept(this);
		}
		virtual void visitSetBExp(SetBExp *o) {
			o->set->parent = o;
			o->set->accept(this);
		}
		virtual void visitSetSExp(SetSExp *o) {
			o->lval->parent = o; o->rval->parent = o;
			o->lval->accept(this); o->rval->accept(this);
		}
		virtual void visitSetVar(SetVar *o) {
			o->mref->parent = o;
			o->mref->accept(this);
		}
		virtual void visitCountExp(CountExp *o) {
			for(unsigned i = 0; i < o->required.size(); i++) {o->required[i]->parent = o; o->required[i]->accept(this);}
			for(unsigned i = 0; i < o->optional.size(); i++) {o->optional[i]->parent = o; o->optional[i]->accept(this);}
			for(unsigned i = 0; i < o->unexpanded.size(); i++) {o->unexpanded[i]->parent = o; o->unexpanded[i]->accept(this);}
		}
		virtual void visitIdxModuleRef(IdxModuleRef *o) {
			o->idxExp->parent = o;
			o->idxExp->accept(this);
		}
		virtual void visitKleeneParent(KleeneParent *o) {
			o->origExp->parent = o;
			o->origExp->accept(this);
		}
};

class ReverseVisitor: public Visitor {
	public:
	virtual void visitDCondition(DCondition *o) {}
	virtual void visitModuleRef(ModuleRef *o) {o->parent->accept(this);}
	virtual void visitModuleVar(ModuleVar *o) {o->parent->accept(this);}
	virtual void visitNumber(Number *o) {o->parent->accept(this);}
	virtual void visitMathExp(MathExp *o) {o->parent->accept(this);}
	virtual void visitCompExp(CompExp *o) {o->parent->accept(this);}
	virtual void visitAndExp(AndExp *o) {o->parent->accept(this);}
	virtual void visitOrExp(OrExp *o) {o->parent->accept(this);}
	virtual void visitNotExp(NotExp *o) {o->parent->accept(this);}
	virtual void visitNeighborExp(NeighborExp *o) {o->parent->accept(this);}
	virtual void visitModuleDecl(ModuleDecl *o) {o->parent->accept(this);}
	virtual void visitModuleList(ModuleList *o) {o->parent->accept(this);}
	virtual void visitSetVarAction(SetVarAction *o) {o->parent->accept(this);}
	virtual void visitChangeTopologyAction(ChangeTopologyAction *o) {o->parent->accept(this);}
	virtual void visitCallFcnAction(CallFcnAction *o) {o->parent->accept(this);}
	virtual void visitActionList(ActionList *o) {o->parent->accept(this);}
	virtual void visitSetRExp(SetRExp *o) {o->parent->accept(this);}
	virtual void visitSetBExp(SetBExp *o) {o->parent->accept(this);}
	virtual void visitSetSExp(SetSExp *o) {o->parent->accept(this);}
	virtual void visitSetVar(SetVar *o) {o->parent->accept(this);}
	virtual void visitCountExp(CountExp *o) {o->parent->accept(this);}
	virtual void visitIdxVar(IdxVar *o) {o->parent->accept(this);}
	virtual void visitIdxModuleRef(IdxModuleRef *o) {o->parent->accept(this);}
	virtual void visitKleeneParent(KleeneParent *o) {o->parent->accept(this);}
};

DCondition *parseCondition(string input);
vector<DCondition *> parseFile(string filename);

#endif //__DCOND_PARSE__
