//#define BOOST_SPIRIT_DEBUG 
#define BOOST_SPIRIT_DEBUG_FLAGS BOOST_SPIRIT_DEBUG_FLAGS_NODES
#include <algorithm>
#include <boost/spirit/core.hpp>
#include <boost/spirit/tree/ast.hpp>
#include <boost/spirit/tree/tree_to_xml.hpp>
#include <algorithm>
#include <string>
#include <iostream>
#include <fstream>
#include <map>
#include <boost/any.hpp>
#include <stdlib.h>
#include <typeinfo>
#include <vector>
#include "DCondParse.hxx"
#include "DCondManager.hxx"

#define EQUALS_THRESH 0.001 //threshold for equality comparisons 

using namespace std;
using namespace boost::spirit;
using namespace boost;

typedef char const*         iterator_t;
typedef tree_match<iterator_t> parse_tree_match_t;
typedef parse_tree_match_t::tree_iterator iter_t;
any evaluate(parse_tree_match_t hit);
any eval_expression(iter_t const& i);

int DCondition::nextId = 0;

////////////////////////////////////////////////////////////////////////////
//
//  Our distributed condition grammar
//
////////////////////////////////////////////////////////////////////////////
struct dcond_grammar : public grammar<dcond_grammar>
{
	static const int conditionID = 1;
	static const int identifierID = 2;
	static const int numericID = 3;
	static const int varnameID = 4;
	static const int moduleID = 5;
	static const int moduleDeclID = 6;
	static const int tShiftID = 7;
	static const int moduleListID = 8;
	static const int realExpID = 9;
	static const int realTermID = 10;
	static const int realFactorID = 11;
	static const int stateVarID = 12;
	static const int countExpID = 13;
	static const int setRID = 14;
	static const int boolExpID = 15;
	static const int boolExpSimpleID = 16;
	static const int compExpID = 17;
	static const int compOpID = 18;
	static const int setBID = 19;
	static const int setVarID = 20;
	static const int setExpID = 21;
	static const int actionID = 22;
	static const int actionSimpleID = 23;

		template <typename ScannerT>
		struct definition
		{
		definition(dcond_grammar const& /*self*/) {
			//Start grammar definition
						
			//top level
			condition = infix_node_d[moduleList >> ';' >> boolExp >> ';' >> action];
			
			//common types
			identifier = lexeme_d[token_node_d[alpha_p >> *('_'|alnum_p)]];
			numeric = real_p;
			varname = (identifier >> '.' >> identifier) | identifier;
			module = (identifier >> '[' >> realExp >> ']') | identifier;
			moduleDecl = (identifier >> '[' >> int_p >> ',' >> int_p >> ']') | identifier;
			tShift = ((str_p("next(") | str_p("prev(")) >> int_p >> ')') | str_p("curr");
			
			moduleList = ch_p('(') >> *(moduleDecl % ',') >> ch_p(')');
			
			//real-valued expressions
			realExp = realTerm >> *((ch_p('+') >> realTerm) | (ch_p('-') >> realTerm));
			realTerm = realFactor >> *((ch_p('*') >> realFactor) | (ch_p('/') >> realFactor) | (ch_p('%') >> realFactor));
			realFactor = numeric 
								| stateVar
								| countExp
								| setR
								| identifier
								| ('(' >> realExp >> ')')
								| ('-' >> realFactor);
			stateVar = module >> '.' >> !(tShift >> '.') >> varname;
			countExp = str_p("count(") >> *(boolExp % ',') >> ch_p(')');
			setR = setExp >> (str_p("->size") | str_p("->any") | str_p("->random"));
			
			//boolean-valued expresions
			boolExp = (boolExpSimple >> (ch_p('&') | ch_p('|')) >> boolExp) 
							| boolExpSimple;
			boolExpSimple = (ch_p('!') >> boolExpSimple)
					| ("n(" >> module >> ',' >> module >> ch_p(')'))
							| ('(' >> compExp >> ')')
							| ('(' >> boolExp >> ')')
							| setB;
			compExp = realExp >> compOp >> realExp;
			compOp = str_p(">=") | str_p(">") | str_p("==") | str_p("<=") | str_p("<") | str_p("!=");
			setB = setExp >> "->empty?";
			
			//set-valued expressions
			setExp = (setVar >> (str_p("->union(") | str_p("->intersect(")) >> setExp >> ')') | setVar;
			setVar = module >> !('.' >> tShift) >> str_p(".$") >> varname;
			
			//actions
			action = *(actionSimple % '&') >> ';';
			actionSimple = (module >> '.' >> varname >> '=' >> realExp)
					| (!ch_p('!') >> "n(" >> module >> ',' >> module >> ch_p(')'))
					| (module >> '.' >> identifier >> '(' >> realExp >> ch_p(')'));
			
			BOOST_SPIRIT_DEBUG_NODE(condition);
			BOOST_SPIRIT_DEBUG_NODE(identifier);
			BOOST_SPIRIT_DEBUG_NODE(numeric);
			BOOST_SPIRIT_DEBUG_NODE(varname);
			BOOST_SPIRIT_DEBUG_NODE(module);
			BOOST_SPIRIT_DEBUG_NODE(moduleDecl);
			BOOST_SPIRIT_DEBUG_NODE(tShift);
			BOOST_SPIRIT_DEBUG_NODE(moduleList);
			BOOST_SPIRIT_DEBUG_NODE(realExp);
			BOOST_SPIRIT_DEBUG_NODE(realTerm);
			BOOST_SPIRIT_DEBUG_NODE(realFactor);
			BOOST_SPIRIT_DEBUG_NODE(stateVar);
			BOOST_SPIRIT_DEBUG_NODE(countExp);
			BOOST_SPIRIT_DEBUG_NODE(setR);
			BOOST_SPIRIT_DEBUG_NODE(boolExp);
			BOOST_SPIRIT_DEBUG_NODE(boolExpSimple);
			BOOST_SPIRIT_DEBUG_NODE(compExp);
			BOOST_SPIRIT_DEBUG_NODE(compOp);
			BOOST_SPIRIT_DEBUG_NODE(setB);
			BOOST_SPIRIT_DEBUG_NODE(setVar);
			BOOST_SPIRIT_DEBUG_NODE(setExp);
			BOOST_SPIRIT_DEBUG_NODE(action);
			BOOST_SPIRIT_DEBUG_NODE(actionSimple);
		}
		
		rule<ScannerT, parser_context<>, parser_tag<conditionID> >  condition;
		rule<ScannerT, parser_context<>, parser_tag<identifierID> >  identifier;
		rule<ScannerT, parser_context<>, parser_tag<numericID> >  numeric;
		rule<ScannerT, parser_context<>, parser_tag<varnameID> >  varname;
		rule<ScannerT, parser_context<>, parser_tag<moduleID> >  module;
		rule<ScannerT, parser_context<>, parser_tag<moduleDeclID> >  moduleDecl;
		rule<ScannerT, parser_context<>, parser_tag<tShiftID> >  tShift;
		rule<ScannerT, parser_context<>, parser_tag<moduleListID> >  moduleList;
		rule<ScannerT, parser_context<>, parser_tag<realExpID> >  realExp;
		rule<ScannerT, parser_context<>, parser_tag<realTermID> >  realTerm;
		rule<ScannerT, parser_context<>, parser_tag<realFactorID> >  realFactor;
		rule<ScannerT, parser_context<>, parser_tag<stateVarID> >  stateVar;
		rule<ScannerT, parser_context<>, parser_tag<countExpID> >  countExp;
		rule<ScannerT, parser_context<>, parser_tag<setRID> >  setR;
		rule<ScannerT, parser_context<>, parser_tag<boolExpID> >  boolExp;
		rule<ScannerT, parser_context<>, parser_tag<boolExpSimpleID> >  boolExpSimple;
		rule<ScannerT, parser_context<>, parser_tag<compExpID> >  compExp;
		rule<ScannerT, parser_context<>, parser_tag<compOpID> >  compOp;
		rule<ScannerT, parser_context<>, parser_tag<setBID> >  setB;
		rule<ScannerT, parser_context<>, parser_tag<setVarID> >  setVar;
		rule<ScannerT, parser_context<>, parser_tag<setExpID> >  setExp;
		rule<ScannerT, parser_context<>, parser_tag<actionID> >  action;
		rule<ScannerT, parser_context<>, parser_tag<actionSimpleID> >  actionSimple;

		rule<ScannerT, parser_context<>, parser_tag<conditionID> > const& start() const { return condition; }
		};
};

//needed if we put a derived class in, but want a base class out
BoolExp *decodeBool(any a) {
	if (BoolExp **e = any_cast<BoolExp*>(&a)) return *e;
	else if (AndExp **e = any_cast<AndExp*>(&a)) return *e;
	else if (OrExp **e = any_cast<OrExp*>(&a)) return *e;
	else if (NotExp **e = any_cast<NotExp*>(&a)) return *e;
	else if (CompExp **e = any_cast<CompExp*>(&a)) return *e;
	else if (NeighborExp **e = any_cast<NeighborExp*>(&a)) return *e;
	else if (SetBExp **e = any_cast<SetBExp*>(&a)) return *e;
	cout << "no match in decodeBool:" << a.type().name() << endl;
	return NULL;
}

RVal *decodeReal(any a) {
	if (RVal **e = any_cast<RVal*>(&a)) return *e;
	else if (Number **e = any_cast<Number*>(&a)) return *e;
	else if (ModuleVar **e = any_cast<ModuleVar*>(&a)) return *e;
	else if (MathExp **e = any_cast<MathExp*>(&a)) return *e;
	else if (SetRExp **e = any_cast<SetRExp*>(&a)) return *e;
	else if (CountExp **e = any_cast<CountExp*>(&a)) return *e;
	else if (IdxVar **e = any_cast<IdxVar*>(&a)) return *e;
	else if (string *s = any_cast<string>(&a)) return (RVal*) new IdxVar(s->c_str()); //converts bare identifier to IdxVar
	cout << "no match in decodeReal:" << a.type().name() << endl;
	return NULL;
}

SetExp *decodeSet(any a) {
	if (SetExp **e = any_cast<SetExp*>(&a)) return *e;
	else if (SetSExp **e = any_cast<SetSExp*>(&a)) return *e;
	else if (SetVar **e = any_cast<SetVar*>(&a)) return *e;
	cout << "no match in decodeSet:" << a.type().name() << endl;
	return NULL;
}

ModuleRef *decodeRef(any a) {
	if (ModuleRef **e = any_cast<ModuleRef*>(&a)) return *e;
	else if (IdxModuleRef **e = any_cast<IdxModuleRef*>(&a)) return *e;
	else if (string *s = any_cast<string>(&a)) return new ModuleRef(s->c_str()); //converts bare identifier to ModuleRef
	cout << "no match in decodeRef:" << a.type().name() << endl;
	return NULL;
}

Action *decodeAction(any a) {
	if (Action **e = any_cast<Action*>(&a)) return *e;
	else if (SetVarAction **e = any_cast<SetVarAction*>(&a)) return *e;
	else if (ChangeTopologyAction **e = any_cast<ChangeTopologyAction*>(&a)) return *e;
	else if (CallFcnAction **e = any_cast<CallFcnAction*>(&a)) return *e;
	cout << "no match in decodeAction:" << a.type().name() << endl;
	return NULL;
}

any evaluate(tree_parse_info<> info)
{
	return eval_expression(info.trees.begin());
}

MathOp mathOpFromString(string s) {
	if (s == "*") return times_op;
	else if (s == "/") return div_op;
	else if (s == "+") return plus_op;
	else if (s == "-") return minus_op;
	else if (s == "%") return mod_op;
	else {
		cout << "Unknown operator: " << s << endl;
		return plus_op;
	}
}

CompOp compOpFromString(string s) {
	if (s == "<") return lt_op;
	else if (s == ">") return gt_op;
	else if (s == "<=") return lte_op;
	else if (s == ">=") return gte_op;
	else if (s == "==") return eq_op;
	else if (s == "!=") return neq_op;
	else {
		cout << "Unknown operator: " << s << endl;
		return lt_op;
	}
}

//most of the work goes on here: this fcn converts an AST subtree into the equivalent class-based structure
any eval_expression(iter_t const& i) {
	//cout << i->value.id().to_long() << endl;
	//do construction of matcher object here
	switch (i->value.id().to_long()) {
		case dcond_grammar::conditionID: 
			assert(i->children.size() == 3);
			return any(new DCondition(
					any_cast<ModuleList*>(eval_expression(i->children.begin())),
					decodeBool(eval_expression(i->children.begin()+1)),
					any_cast<ActionList*>(eval_expression(i->children.begin()+2))));
		case dcond_grammar::identifierID:
			return any(string(i->value.begin(), i->value.end()));
		case dcond_grammar::numericID:
			{
				string s(i->value.begin(), i->value.end());
				return any(new Number(atof(s.c_str())));
			}
		case dcond_grammar::varnameID:
			if (i->children.size() == 3) {
				return any(string(i->children[0].value.begin(), i->children[0].value.end()) + "." +
						 string(i->children[2].value.begin(), i->children[2].value.end()));
			}
			else return any(string(i->value.begin(), i->value.end()));
		case dcond_grammar::moduleID:
			if (i->children.size() == 4) {
				return any(new IdxModuleRef(string(i->children[0].value.begin(), i->children[0].value.end()),
						 decodeReal(eval_expression(i->children.begin()+2))));
			}
			else return any(new ModuleRef(string(i->value.begin(),i->value.end()).c_str()));
		case dcond_grammar::moduleDeclID:
			if (i->children.size() == 6) {
				return any(new ModuleDecl(string(i->children[0].value.begin(), i->children[0].value.end()).c_str(),
							atoi(string(i->children[2].value.begin(), i->children[2].value.end()).c_str()),
							atoi(string(i->children[4].value.begin(), i->children[4].value.end()).c_str())));
			}
			else return any(new ModuleDecl(string(i->value.begin(),i->value.end()).c_str()));
		case dcond_grammar::tShiftID:
			{
				if (i->children.size() > 1) {
					string dir(i->children[0].value.begin(),i->children[0].value.end());
					int shift = atoi(string(i->children[1].value.begin(), i->children[1].value.end()).c_str());
					if (dir == "next(") return any(shift*-1);
					else return any(shift);
				}
				else return any(IMM_OFFSET);
			}
		case dcond_grammar::moduleListID: {
				ModuleList *ml = new ModuleList();
				for (unsigned j = 1; j < i->children.size() -1; j+=2) {
					if (i->children[j].value.id() == dcond_grammar::identifierID)
						ml->addName(new ModuleDecl(string(i->children[j].value.begin(),i->children[j].value.end()).c_str()));
					else ml->addName(any_cast<ModuleDecl*>(eval_expression(i->children.begin()+j)));
				}
				return any(ml);
			}
		case dcond_grammar::realExpID: 
		case dcond_grammar::realTermID: {
				RVal *topExp = decodeReal(eval_expression(i->children.begin()));
				for (unsigned j = 1; j < i->children.size(); j+=2) {
					topExp = new MathExp(topExp,
							mathOpFromString(string(i->children[j].value.begin(),i->children[j].value.end()).c_str()),
							 decodeReal(eval_expression(i->children.begin()+j+1)));
				}
				return any(topExp);
			}
		case dcond_grammar::realFactorID:
			if (i->children.size() == 1) {
				if (i->children[0].value.id() == dcond_grammar::identifierID) //index variable
					return any(new IdxVar(string(i->children[0].value.begin(),i->children[0].value.end()).c_str()));
				else return eval_expression(i->children.begin());
			}
			else if (i->children.size() == 2) { //negation
				return any(new MathExp(new Number(-1),mathOpFromString("*"),decodeReal(eval_expression(i->children.begin()+1))));
			}
			else { //parens
				return eval_expression(i->children.begin()+1);
			}
		case dcond_grammar::stateVarID: {
			ModuleRef *mref = decodeRef(eval_expression(i->children.begin()));
			string varname;
			if (i->children.size() == 3) {
				varname = any_cast<string>(eval_expression(i->children.begin()+2));
			}
			else {
				mref->temporalOffset = any_cast<int>(eval_expression(i->children.begin()+2));
				varname = any_cast<string>(eval_expression(i->children.begin()+4));
			}
			return any(new ModuleVar(mref,varname.c_str()));
		}
		case dcond_grammar::countExpID: {
			CountExp *ce = new CountExp();
			for (unsigned j = 1; j < i->children.size() -1; j+=2) {
				ce->required.push_back(decodeBool(eval_expression(i->children.begin()+j)));
			}
			return any(ce);
		}
		case dcond_grammar::setRID:
			return any(new SetRExp(decodeSet(eval_expression(i->children.begin())),
									string(i->children[1].value.begin(),i->children[1].value.end()).c_str()));
		case dcond_grammar::boolExpSimpleID: {
			BoolExp *head;
			if (*i->children[0].value.begin() == '!')
			{
				head = new NotExp(decodeBool(eval_expression(i->children.begin()+1)));
			}
			else if (*i->children[0].value.begin() == 'n')
			{
				head = new NeighborExp((decodeRef(eval_expression(i->children.begin()+1))),
					(decodeRef(eval_expression(i->children.begin()+3))));
			}
			else if (i->children[1].value.id() == dcond_grammar::boolExpID) {
				head = decodeBool(eval_expression(i->children.begin()+1));
			}
			else if (i->children[1].value.id() == dcond_grammar::compExpID) {
				head = any_cast<CompExp*>(eval_expression(i->children.begin()+1));
			}
			else { //boolean set expression
				head = decodeBool(eval_expression(i->children.begin()));
			}
			return any(head);
		}
		case dcond_grammar::boolExpID:
			if (i->children.size() == 3) {
				if (*i->children[1].value.begin() == '&')
				{
					return any(new AndExp(decodeBool(eval_expression(i->children.begin())),
											decodeBool(eval_expression(i->children.begin()+2)))); 
				}
				else {
					return any(new OrExp(decodeBool(eval_expression(i->children.begin())),
											decodeBool(eval_expression(i->children.begin()+2))));
				}
			}
			else {
				return eval_expression(i->children.begin());
			}
			case dcond_grammar::compExpID: {
			return any(new CompExp(decodeReal(eval_expression(i->children.begin())),
				   compOpFromString((any_cast<string>(eval_expression(i->children.begin()+1))).c_str()),
								decodeReal(eval_expression(i->children.begin()+2))));
		}
		case dcond_grammar::compOpID:
			return any(string(i->value.begin(), i->value.end()));
		case dcond_grammar::setBID:
			return any(new SetBExp(decodeSet(eval_expression(i->children.begin())),
								 string(i->children[1].value.begin(),i->children[1].value.end()).c_str()));
		case dcond_grammar::setExpID:
			if (i->children.size() == 1)
					return eval_expression(i->children.begin());
			else return any(new SetSExp(decodeSet(eval_expression(i->children.begin())),
						string(i->children[1].value.begin(),i->children[1].value.end()).c_str(),
						decodeSet(eval_expression(i->children.begin()+2))));
		case dcond_grammar::setVarID: {
				ModuleRef *mref = decodeRef(eval_expression(i->children.begin()));
				string varname;
				if (i->children.size() == 3) {
					varname = any_cast<string>(eval_expression(i->children.begin()+2));
				}
				else {
					mref->temporalOffset = any_cast<int>(eval_expression(i->children.begin()+2));
					varname = any_cast<string>(eval_expression(i->children.begin()+4));
				}
				return any(new SetVar(mref,("$" + varname).c_str()));
			}
		case dcond_grammar::actionID: {
				ActionList *al = new ActionList();
				for (unsigned j = 0; j < i->children.size()-1; j+=2) {
					al->actions.push_back(decodeAction(eval_expression(i->children.begin()+j)));
				}
				return any(al);
			}
		case dcond_grammar::actionSimpleID:
			if ((i->children.size() == 5) && (*i->children[3].value.begin() == '=')) {
				return any(new SetVarAction(decodeRef(eval_expression(i->children.begin())),
						any_cast<string>(eval_expression(i->children.begin()+2)).c_str(),
						decodeReal(eval_expression(i->children.begin()+4))));
			}
			else if ((i->children.size() == 6) && (*i->children[1].value.begin() == '.')) {
				return any(new CallFcnAction(decodeRef(eval_expression(i->children.begin())),
								any_cast<string>(eval_expression(i->children.begin()+2)).c_str(),
								decodeReal(eval_expression(i->children.begin()+4))));
			}
			else {
				if (i->children.size() == 6) { // !n(a,b)
					return any(new ChangeTopologyAction(decodeRef(eval_expression(i->children.begin()+2)),
										 decodeRef(eval_expression(i->children.begin()+4)),false));
				}
				else { // n(a,b)
					return any(new ChangeTopologyAction(decodeRef(eval_expression(i->children.begin()+1)),
										 decodeRef(eval_expression(i->children.begin()+3)),true));
				}
			}
		default:
			cout << "default: empty " << i->value.id().to_long() << endl;
			return any();
	}
}

//get the set of slot/index pairs that we can fill currently
set<pair<string, int> > DCondition::nextSlots()  {
	set<pair<string, int> > validSlots;
	if (isFull()) return validSlots;
	
	for(unsigned i = 0; i < c_list->modules.size() ; i++) {
		//name that still needs to be filled
		if (c_list->modules[i]->modules.size() < c_list->modules[i]->minSize) {
			validSlots.insert(pair<string,int>(c_list->modules[i]->name,c_list->modules[i]->modules.size()));
			break;
		}
		//name that can optionally be filled
		if (c_list->modules[i]->modules.size() < c_list->modules[i]->maxSize) {
			validSlots.insert(pair<string,int>(c_list->modules[i]->name,c_list->modules[i]->modules.size()));
		}
	}
	return validSlots;
}

int DCondition::slotIdx(ModuleWrapper *m) {
	int slot = 0;
	for (unsigned i = 0; i < c_list->modules.size(); i++) {
		for (unsigned j = 0; j < c_list->modules[i]->modules.size(); j++) {
			if (c_list->modules[i]->modules[j]->id == m->id) return slot;
			slot++;
		}
	}
	return -1;
}

pair<string, int> DCondition::slotName(ModuleWrapper *m) {
	pair<string,int> n;
	for (unsigned i = 0; i < c_list->modules.size(); i++) {
		for (unsigned j = 0; j < c_list->modules[i]->modules.size(); j++) {
			if (c_list->modules[i]->modules[j]->id == m->id) {
				n.first = c_list->modules[i]->name;
				n.second = j;
				return n;
			}
		}
	}
	return n;
}

ModuleWrapper* DCondition::moduleAtIndex(int idx) {
	int slot = 0;
	for (unsigned i = 0; i < c_list->modules.size(); i++) {
		for (unsigned j = 0; j < c_list->modules[i]->modules.size(); j++) {
			if (slot++ == idx) return c_list->modules[i]->modules[j];
		}
	}
	return NULL;
}

ModuleWrapper *DCondition::moduleForName(string name, int idx) {
	for (unsigned i = 0; i < c_list->modules.size(); i++) {
		if ((c_list->modules[i]->name == name) & ((int)c_list->modules[i]->modules.size() > idx)) return c_list->modules[i]->modules[idx];
	}
	//cout << "no module for " << name << " idx: " << idx << endl;
	return NULL;
}

bool DCondition::isFull() {
	for (unsigned i = 0; i < c_list->modules.size(); i++) {
		if (c_list->modules[i]->modules.size() < c_list->modules[i]->maxSize) return false;
	}
	return true;
}

int DCondition::fillCount() {
	int count = 0;
	for (unsigned i = 0; i < c_list->modules.size(); i++) {
		count += c_list->modules[i]->modules.size();
	}
	return count;
}

int DCondition::maxSizeForName(string n) {
	for (unsigned i = 0; i < c_list->modules.size(); i++) {
		if (c_list->modules[i]->name == n) return c_list->modules[i]->maxSize;
	}
	return -1;
}

int DCondition::minSizeForName(string n) {
	for (unsigned i = 0; i < c_list->modules.size(); i++) {
		if (c_list->modules[i]->name == n) return c_list->modules[i]->minSize;
	}
	return INT_MAX;
}

ModuleWrapper *DCondition::nextRoutingStep(ModuleWrapper *src,ModuleWrapper *dest) {
	//calculate the next module to send to, given adjacency matrix, destination, & curr location
	//get source and dest #'s
	int sourceIdx = slotIdx(src);
	int destIdx = slotIdx(dest);
	//queue of actives, set of visited
	deque<int> active;
	set<int> visited;
	//push dest on to queue
	active.push_back(destIdx);
	//pop currIdx off of queue...
	while (active.size()) {
		int currIdx = active.front();
		active.pop_front();
		//add to currIdx to visited set
		visited.insert(currIdx);
		//for each entry j in adjacency matrix for currIdx...
		for (int j = 0; j < (int)adjacency.shape()[0]; j++) {
			//if j is adjacent...
			if (adjacency[currIdx][j]) {
				//if j is source, return modulewrapper for currIdx
				if (j == sourceIdx) {
					return moduleAtIndex(currIdx);
				}
				//else, if j not in queue or visited set, push j on to queue
				deque<int>::iterator result = find(active.begin(), active.end(), j);
				if ((result == active.end()) && (visited.find(j) == visited.end())) active.push_back(j);
			}
		}
	}
	cout << "No route found from slot " << sourceIdx << " to " << destIdx <<  ". Badness will ensue." << endl;
	cout << "src id: " << src->id << " dest id: " << dest->id << " for rule " << this->id << endl;
	//print routing table
	for (unsigned i = 0; i < adjacency.shape()[0]; i++) {
		for (unsigned j = 0; j < adjacency.shape()[1]; j++) {
			cout << adjacency[i][j] << " ";
		}
		cout << endl;
	}
	
	return NULL;
}
 
 //constructors/destructors----------------------------------------------------
ModuleRef::ModuleRef(const char *n) {
	name = n;
	temporalOffset = 0;
}
 
ModuleVar::ModuleVar(ModuleRef *m,const char *v) {
	mref = m;
	var = v;
	filled = false;
	value = 0.0;
}

ModuleList::~ModuleList() {
	for (unsigned i = 0; i < modules.size(); i++) {
		delete modules[i];
	}
}

ActionList::~ActionList() {
	for (unsigned i = 0; i < actions.size(); i++) {
		delete actions[i];
	}
}

CountExp::~CountExp() {
	for (unsigned i = 0; i < required.size(); i++) {delete required[i];}
	for (unsigned i = 0; i < unexpanded.size(); i++) {delete unexpanded[i];}
	for (unsigned i = 0; i < optional.size(); i++) {delete optional[i];}
}

//clone()----------------------------------------------------------------------
NotExp *NotExp::clone() {return new NotExp(val->clone());}
OrExp *OrExp::clone() {return new OrExp(lval->clone(),rval->clone());}	
AndExp *AndExp::clone() {return new AndExp(lval->clone(),rval->clone());}
CompExp *CompExp::clone() {return new CompExp(lval->clone(),op,rval->clone());}
Number *Number::clone() {return new Number(value);}
ChangeTopologyAction *ChangeTopologyAction::clone() {return new ChangeTopologyAction(moduleA->clone(),moduleB->clone(),connectLink);}
CallFcnAction *CallFcnAction::clone() {return new CallFcnAction(module->clone(),fname.c_str(),value->clone());}
SetBExp *SetBExp::clone() {return new SetBExp(set->clone(),op.c_str());}

SetVarAction *SetVarAction::clone() {
	 SetVarAction * sva = new SetVarAction(module->clone(),varname.c_str(),value->clone());
	 sva->code = code;
	 return sva;
}

ModuleDecl *ModuleDecl::clone() {
	ModuleDecl *md = new ModuleDecl(name.c_str(),minSize,maxSize);
	md->modules = modules;
	return md;
}

MathExp *MathExp::clone() {
	 MathExp *me = new MathExp(lval->clone(),op,rval->clone());
	 me->filled = filled;
	 me->value = value;
	 return me;
}

SetRExp *SetRExp::clone() {
	SetRExp *se = new SetRExp(set->clone(),op.c_str());
	se->filled = filled;
	se->value = value;
	return se;
}

ModuleVar *ModuleVar::clone() {
	ModuleVar *mv = new ModuleVar(mref->clone(),var.c_str());
	mv->code = code;
	mv->filled = filled;
	mv->value = value;
	return mv;
}
ModuleRef *ModuleRef::clone() {
	ModuleRef *newRef = new ModuleRef(name.c_str());
	newRef->temporalOffset = temporalOffset;
	return newRef;
}
NeighborExp *NeighborExp::clone() {
	NeighborExp *myClone = new NeighborExp(lval->clone(),rval->clone());
	myClone->lmodule = this->lmodule;
	myClone->rmodule = this->rmodule;
	myClone->satisfied = this->satisfied;
	return myClone;
}
ModuleList *ModuleList::clone() {
	ModuleList *newML = new ModuleList();
	for (unsigned i = 0; i < modules.size(); i++) {
		newML->modules.push_back(modules[i]->clone());
	}
	return newML;
}
ActionList *ActionList::clone() {
	ActionList *newAL = new ActionList();
	for (unsigned i = 0; i < actions.size(); i++) {
		newAL->actions.push_back(actions[i]->clone());
	}
	return newAL;
}
SetSExp *SetSExp::clone() {
	SetSExp *ss = new SetSExp(lval->clone(),op.c_str(),rval->clone());
	ss->value = value;
	ss->filled = filled;
	return ss;
}
SetVar *SetVar::clone() {
	SetVar *sv = new SetVar(mref->clone(),var.c_str());
	sv->code = code;
	sv->filled = filled;
	sv->value = value;
	return sv;
}
CountExp *CountExp::clone() {
	CountExp *newCE = new CountExp();
	for (unsigned i = 0; i < required.size(); i++) {
		newCE->required.push_back(required[i]->clone());
	}
	for (unsigned i = 0; i < unexpanded.size(); i++) {
		newCE->unexpanded.push_back(unexpanded[i]->clone());
	}
	for (unsigned i = 0; i < optional.size(); i++) {
		newCE->optional.push_back(optional[i]->clone());
	}
	newCE->filled = filled;
	newCE->value = value;
	return newCE;
}
IdxVar *IdxVar::clone() {
	IdxVar *newIV = new IdxVar(name.c_str());
	newIV->value = value;
	newIV->filled = filled;
	return newIV;
}
IdxModuleRef *IdxModuleRef::clone() {
	IdxModuleRef *newRef = new IdxModuleRef(name.c_str(),idxExp->clone());
	newRef->temporalOffset = temporalOffset;
	return newRef;
}
KleeneParent *KleeneParent::clone() {
	KleeneParent *newKP = new KleeneParent(origExp->clone());
	for (unsigned i = 0; i < required.size(); i++) {
		newKP->required.push_back(required[i]->clone());
	}
	for (unsigned i = 0; i < optional.size(); i++) {
		newKP->optional.push_back(optional[i]->clone());
	}
	return newKP;
}
DCondition *DCondition::clone() {
	DCondition *dc = new DCondition(c_list->clone(),main_exp->clone(),a_list->clone());
	dc->id = id;
	adj_array::extent_gen newExtent;
	dc->adjacency.resize(newExtent[adjacency.shape()[0]][adjacency.shape()[1]]);
	dc->adjacency = adjacency;
	nextId--;
	return dc;
}

//isSatisfied()----------------------------------------------------------------------
TriState NotExp::isSatisfied(bool centralized) {if (val->isSatisfied(centralized) == yes) return no;
	if (val->isSatisfied(centralized) == no) return yes;
	else return maybe;
}

TriState OrExp::isSatisfied(bool centralized) {
	if ((lval->isSatisfied(centralized) == no) && (rval->isSatisfied(centralized) == no)) return no;
	if ((lval->isSatisfied(centralized) == yes) || (rval->isSatisfied(centralized) == yes)) return yes;
	else return maybe;
}	

TriState AndExp::isSatisfied(bool centralized) {
	if ((lval->isSatisfied(centralized) == yes) && (rval->isSatisfied(centralized) == yes)) return yes;
	else if ((lval->isSatisfied(centralized) == no) || (rval->isSatisfied(centralized) == no)) return no;
	else return maybe;
}

TriState CompExp::isSatisfied(bool centralized) {
	if ((lval->isSatisfied(centralized) == yes) && (rval->isSatisfied(centralized) == yes)) {
		//cout << lval->value << op << rval->value << endl;
		switch(op) {
			case eq_op:
				if (abs(lval->value - rval->value) < EQUALS_THRESH) return yes;
				else return no;
				break;
			case neq_op:
				if (abs(lval->value - rval->value) > EQUALS_THRESH) return yes;
				else return no;
				break;
			case gte_op:
				if (lval->value >= rval->value) return yes;
				else return no;
				break;
			case lte_op:
				if (lval->value <= rval->value) return yes;
				else return no;
				break;
			case gt_op:
				if (lval->value > rval->value) return yes;
				else return no;
				break;
			case lt_op:
				if (lval->value < rval->value) return yes;
				else return no;
				break;
		}
		cout << "Unknown comparison operator" << endl;
		return maybe;
	}
	else return maybe;
}

TriState MathExp::isSatisfied(bool centralized) {
	if ((lval->isSatisfied(centralized) == yes) && (rval->isSatisfied(centralized) == yes)) return yes;
	else return maybe;
}

TriState Number::isSatisfied(bool centralized) {return yes;}


TriState ModuleDecl::isSatisfied(bool centralized) {
	if (modules.size() >= minSize) return yes;
	else return maybe;
}
TriState SetVarAction::isSatisfied(bool centralized) {
	if ((module->isSatisfied(centralized) == yes) && (value->isSatisfied(centralized) == yes)) return yes;
	return maybe;
}

TriState ChangeTopologyAction::isSatisfied(bool centralized) {	
	if ((moduleA->isSatisfied(centralized) == yes) && (moduleB->isSatisfied(centralized) == yes)) return yes;
	return maybe;
}

TriState CallFcnAction::isSatisfied(bool centralized) {
	if ((module->isSatisfied(centralized) == yes) && (value->isSatisfied(centralized) == yes)) return yes;
	return maybe;
}
TriState SetRExp::isSatisfied(bool centralized) {
	if (set->isSatisfied(centralized) == yes) return yes;
	return maybe;
}

TriState SetBExp::isSatisfied(bool centralized) {return satisfied;}

TriState ModuleVar::isSatisfied(bool centralized) {
	if (filled) return yes;
	else return maybe;
}

TriState ModuleRef::isSatisfied(bool centralized) {return yes;}

TriState NeighborExp::isSatisfied(bool centralized) {return satisfied;}

TriState ModuleList::isSatisfied(bool centralized) {
	for (unsigned i = 0; i < modules.size(); i++) {
		if (modules[i]->isSatisfied(centralized) == maybe) return maybe;
	}
	return yes;
}

TriState ActionList::isSatisfied(bool centralized) {
	for (unsigned i = 0; i < actions.size(); i++) {
		if (actions[i]->isSatisfied(centralized) == maybe) return maybe;
	}
	return yes;
}

TriState SetSExp::isSatisfied(bool centralized) {
	if ((lval->isSatisfied(centralized) == yes) && (rval->isSatisfied(centralized) == yes)) return yes;
	return maybe;
}

TriState SetVar::isSatisfied(bool centralized) {
	if (filled) return yes;
	return maybe;
}

TriState CountExp::isSatisfied(bool centralized) {
	for (unsigned i = 0; i < required.size(); i++) {
		if (required[i]->isSatisfied(centralized) == maybe) return maybe;
	}
	return yes;
}

TriState IdxVar::isSatisfied(bool centralized) {
	if (filled) return yes;
	else return no;
}

TriState IdxModuleRef::isSatisfied(bool centralized) {return idxExp->isSatisfied(centralized);}

TriState DCondition::isSatisfied(bool centralized) {
	TriState actState = a_list->isSatisfied(centralized);
	TriState expState = main_exp->isSatisfied(centralized);
	if ((actState == no) || (expState == no)) return no;
	if ((actState == maybe) || (expState == maybe)) return maybe;
	return yes;
}

TriState KleeneParent::isSatisfied(bool centralized) {
	int maybeCount = 0;
	for (unsigned i = 0; i < required.size(); i++) {
		if (required[i]->isSatisfied() == no) return no;
		if (required[i]->isSatisfied() == maybe) maybeCount++;
	}
	for (unsigned i = 0; i < optional.size(); i++) {
		if (optional[i]->isSatisfied() == no) return no;
	}
	return maybeCount ? maybe : yes;
}

//basic Visitor-------------------------------------------------------
void Visitor::visitDCondition(DCondition *o) {o->c_list->accept(this); o->main_exp->accept(this); o->a_list->accept(this);}
void Visitor::visitModuleRef(ModuleRef *o) {}
void Visitor::visitModuleVar(ModuleVar *o) {o->mref->accept(this);}
void Visitor::visitNumber(Number *o) {}
void Visitor::visitMathExp(MathExp *o) {o->lval->accept(this); o->rval->accept(this);}
void Visitor::visitCompExp(CompExp *o) {o->lval->accept(this); o->rval->accept(this);}
void Visitor::visitAndExp(AndExp *o) {o->lval->accept(this); o->rval->accept(this);}
void Visitor::visitOrExp(OrExp *o) {o->lval->accept(this); o->rval->accept(this);}
void Visitor::visitNotExp(NotExp *o) {o->val->accept(this);}
void Visitor::visitNeighborExp(NeighborExp *o) {o->lval->accept(this); o->rval->accept(this);}
void Visitor::visitModuleDecl(ModuleDecl *o) {}
void Visitor::visitModuleList(ModuleList *o) {for(unsigned i = 0; i < o->modules.size(); i++) o->modules[i]->accept(this);}
void Visitor::visitSetVarAction(SetVarAction *o) {o->module->accept(this); o->value->accept(this);}
void Visitor::visitChangeTopologyAction(ChangeTopologyAction *o) {o->moduleA->accept(this); o->moduleB->accept(this);}
void Visitor::visitCallFcnAction(CallFcnAction *o) {o->module->accept(this); o->value->accept(this);}
void Visitor::visitActionList(ActionList *o) {for(unsigned i = 0; i < o->actions.size(); i++) o->actions[i]->accept(this);}
void Visitor::visitSetRExp(SetRExp *o) {o->set->accept(this);}
void Visitor::visitSetBExp(SetBExp *o) {o->set->accept(this);}
void Visitor::visitSetSExp(SetSExp *o) {o->lval->accept(this); o->rval->accept(this);}
void Visitor::visitSetVar(SetVar *o) {o->mref->accept(this);}
void Visitor::visitCountExp(CountExp *o) {
	for(unsigned i = 0; i < o->required.size(); i++) o->required[i]->accept(this);
	for(unsigned i = 0; i < o->optional.size(); i++) o->optional[i]->accept(this);
	for(unsigned i = 0; i < o->unexpanded.size(); i++) o->unexpanded[i]->accept(this);
}
void Visitor::visitIdxVar(IdxVar *o) {}
void Visitor::visitIdxModuleRef(IdxModuleRef *o) {o->idxExp->accept(this);}
void Visitor::visitKleeneParent(KleeneParent *o) {o->origExp->accept(this);}

//action support-----------------------------------------------------
set<pair<string, int> > SetVarAction::triggerLocs() {
	set<pair<string, int> > out;
	out.insert(pair<string,int>(module->name,module->idx()));
	return out;
}

set<pair<string, int> > ChangeTopologyAction::triggerLocs() {
	set<pair<string, int> > out;
	out.insert(pair<string,int>(moduleA->name,moduleA->idx()));
	out.insert(pair<string,int>(moduleB->name,moduleB->idx()));
	return out;
}

set<pair<string, int> > CallFcnAction::triggerLocs() {
	set<pair<string, int> > out;
	out.insert(pair<string,int>(module->name,module->idx()));
	return out;
}

pair<string, int> ActionList::triggerLoc() {
	set<pair<string, int> > intersect;
	set<pair<string, int> > tempSet;
	if (actions.size()) intersect = actions[0]->triggerLocs();
	for (unsigned i = 1; i < actions.size(); i++) {
		tempSet.clear();
		set<pair<string, int> > currSet = actions[i]->triggerLocs();
		set_intersection(intersect.begin(),intersect.end(),currSet.begin(),currSet.end(),inserter(tempSet,tempSet.end()));
		intersect = tempSet;
	}
	if (intersect.size()) return * intersect.begin();
	else return pair<string, int>("",-1); //no valid trigger location
}

void SetVarAction::trigger(ModuleWrapper *module,DCondManager *manager) {
	module->setRealVar(code,value->value);
}
		
// Kleene-* expansion support-----------------------

//3rd visitor: does no traversal, but only inserts KleeneParent object above specified child, then calls expand
class MungeTreeVisitor: public Visitor {
	public:
		BoolExp *quantExp;
		KleeneParent *munge(DCObject *parent,BoolExp *child) {
			KleeneParent *kp = new KleeneParent(child);
			kp->parent = parent;
			child->parent = kp;
			return kp;
		}
		
		virtual void visitAndExp(AndExp *o) {
			if (o->lval == quantExp) {o->lval = munge(o,o->lval); ((KleeneParent*)o->lval)->expand();}
			if (o->rval == quantExp) {o->rval = munge(o,o->rval); ((KleeneParent*)o->rval)->expand();}
		}
		virtual void visitOrExp(OrExp *o) {
			if (o->lval == quantExp) {o->lval = munge(o,o->lval); ((KleeneParent*)o->lval)->expand();}
			if (o->rval == quantExp) {o->rval = munge(o,o->rval); ((KleeneParent*)o->rval)->expand();}
		}
		virtual void visitNotExp(NotExp *o) {
			if (o->val == quantExp) {o->val = munge(o,o->val); ((KleeneParent*)o->val)->expand();}
		}
		virtual void visitDCondition(DCondition *o) {
			if (o->main_exp == quantExp) {o->main_exp = munge(o,o->main_exp); ((KleeneParent*)o->main_exp)->expand();}
		}
		virtual void visitCountExp(CountExp *o) {
			o->required.erase(find(o->required.begin(),o->required.end(),quantExp));
			o->unexpanded.push_back(quantExp);
			o->expand(quantExp);
		}
};

//2nd visitor: traverses up to find closest parent p (neighbor,comp,setB), adds it to the set
class FindContainingIdxParent: public ReverseVisitor {
	public:
		set<BoolExp*> parentExps;
		virtual void visitCompExp(CompExp *o) {parentExps.insert(o);}
		virtual void visitNeighborExp(NeighborExp *o) {parentExps.insert(o);}
		virtual void visitSetBExp(SetBExp *o) {parentExps.insert(o);}
};

//1st visitor: finds each idx module ref w/ non-constant index, then calls visitor 2 on each one
class LocateIdxExpressions: public Visitor {
	public:
		FindContainingIdxParent stage2;
		virtual void visitIdxModuleRef(IdxModuleRef *o) {
			if (o->idxExp->isSatisfied(false) != yes) stage2.visitIdxModuleRef(o);
		}
};

class SlotIdxPairVisitor: public Visitor {// find all <moduleName,idxVarName> usage pairs in subexpression
	public: 
		set<pair<string, string> > usedNames; //pair of <moduleName,idxVarName>
		string currName;
		virtual void visitIdxModuleRef(IdxModuleRef *o) {
			currName = o->name;
			o->idxExp->accept(this);
		}
		virtual void visitIdxVar(IdxVar *o) {usedNames.insert(pair<string,string>(currName,o->name)); }
};

class SetIdxValuesVisitor: public Visitor { //set values for all idx variables & expressions
	public: 
		vector<pair<string, int> > assignments; //pair of <idxVarName,idxValue>
		set <pair<string,int> > idxUses; // set of output values for idx exp uses
		virtual void visitIdxModuleRef(IdxModuleRef *o) {
			o->idxExp->accept(this); //fill in idx expression
			if (o->idxExp->filled)
				idxUses.insert(pair<string,int>(o->name,(int)o->idxExp->value)); //save output value
		}
		
		virtual void visitMathExp(MathExp *o) {
			o->lval->accept(this); o->rval->accept(this);
			if (o->lval-> filled && o->rval->filled) {
				switch(o->op) {
					case plus_op:
						o->value = o->lval->value + o->rval->value;
						break;
					case minus_op:
						o->value = o->lval->value - o->rval->value;
						break;
					case times_op:
						o->value = o->lval->value * o->rval->value;
						break;
					case div_op:
						o->value = o->lval->value / o->rval->value;
						break;
					case mod_op:
						o->value = fmod(o->lval->value,o->rval->value);
						break;
				}
				o->filled = true;
			}
		}
		virtual void visitIdxVar(IdxVar *o) {
			for (unsigned i = 0; i < assignments.size(); i++) {
				if (assignments[i].first == o->name) {
					o->value = assignments[i].second;
					o->filled = true;
				}
			}
		}
};

void KleeneParent::expand() { //NOTE: CountExp's expand method is exactly the same. Be sure to mirror any changes
	//get all <moduleName,idxVarName> usage pairs
	SlotIdxPairVisitor slotPairs;
	origExp->accept(&slotPairs);
	
	//get DCondition ancestor
	DCObject *p = this;
	while (p->parent != NULL) {p = p->parent;}
	DCondition *cond = (DCondition*)p;
	
	//calculate max index for each idxVar
	map<string,int> idxVarMax;
	for(set<pair<string, string> >::iterator i = slotPairs.usedNames.begin(); i != slotPairs.usedNames.end(); i++) {
		int slotCount = cond->maxSizeForName((*i).first);
		if (idxVarMax.count((*i).first)) {
			if (idxVarMax[(*i).second] < slotCount) idxVarMax[(*i).second] = slotCount;
		}
		else idxVarMax[(*i).second] = slotCount;
		//cout << "name: " << (*i).first << " var: " << (*i).second << " max: " << idxVarMax[(*i).second] << endl;
	}
	
	//get all valid idx var assignments
	vector<pair<string,int> >currAssn;
	currAssn.insert(currAssn.begin(),idxVarMax.begin(),idxVarMax.end());
	for(unsigned i = 0; i < currAssn.size(); i++) {currAssn[i].second = 0;}
	//iterate through assignments
	unsigned lastVarIdx = currAssn.size() - 1;
	bool done = false;
	while(!done) {
		//for(unsigned i = 0; i < currAssn.size(); i++) {cout << currAssn[i].first << ":" << currAssn[i].second << ",";}
		//cout << endl;
		//clone current exp
		BoolExp *newExp = origExp->clone();
		//set idxVar values
		SetIdxValuesVisitor assigner;
		assigner.assignments = currAssn;
		newExp->accept(&assigner);
		//get idx exp output values
		bool allRequired = true;
		for (set<pair<string,int> >::iterator j = assigner.idxUses.begin(); j != assigner.idxUses.end(); j++) {
			//cout << (*j).first << " used " << (*j).second << " max " << cond->minSizeForName((*j).first) << endl;
			if ((*j).second >=  cond->minSizeForName((*j).first)) { //see if used index falls in required range
				allRequired = false;
				break;
			}
		}
		//add to either required or optional set
		if (allRequired) required.push_back(newExp);
		else optional.push_back(newExp);
		
		//iterate to next assignment
		unsigned j = 0;
		while((j <= lastVarIdx) && (++currAssn[j].second >= idxVarMax[currAssn[j].first])) {
			currAssn[j].second = 0;
			j++;
		}
		done = (j > lastVarIdx);
	}
	//cout << required.size() << " " <<optional.size() << endl;
}

void CountExp::expand(BoolExp *origExp) {
	//get all <moduleName,idxVarName> usage pairs
	SlotIdxPairVisitor slotPairs;
	origExp->accept(&slotPairs);
	
	//get DCondition ancestor
	DCObject *p = this;
	while (p->parent != NULL) {p = p->parent;}
	DCondition *cond = (DCondition*)p;
	
	//calculate max index for each idxVar
	map<string,int> idxVarMax;
	for(set<pair<string, string> >::iterator i = slotPairs.usedNames.begin(); i != slotPairs.usedNames.end(); i++) {
		int slotCount = cond->maxSizeForName((*i).first);
		if (idxVarMax.count((*i).first)) {
			if (idxVarMax[(*i).second] < slotCount) idxVarMax[(*i).second] = slotCount;
		}
		else idxVarMax[(*i).second] = slotCount;
		//cout << "name: " << (*i).first << " var: " << (*i).second << " max: " << idxVarMax[(*i).second] << endl;
	}
	
	//get all valid idx var assignments
	vector<pair<string,int> >currAssn;
	currAssn.insert(currAssn.begin(),idxVarMax.begin(),idxVarMax.end());
	for(unsigned i = 0; i < currAssn.size(); i++) {currAssn[i].second = 0;}
	//iterate through assignments
	unsigned lastVarIdx = currAssn.size() - 1;
	bool done = false;
	while(!done) {
		//for(unsigned i = 0; i < currAssn.size(); i++) {cout << currAssn[i].first << ":" << currAssn[i].second << ",";}
		//cout << endl;
		//clone current exp
		BoolExp *newExp = origExp->clone();
		//set idxVar values
		SetIdxValuesVisitor assigner;
		assigner.assignments = currAssn;
		newExp->accept(&assigner);
		//get idx exp output values
		bool allRequired = true;
		for (set<pair<string,int> >::iterator j = assigner.idxUses.begin(); j != assigner.idxUses.end(); j++) {
			//cout << (*j).first << " used " << (*j).second << " max " << cond->minSizeForName((*j).first) << endl;
			if ((*j).second >=  cond->minSizeForName((*j).first)) { //see if used index falls in required range
				allRequired = false;
				break;
			}
		}
		//add to either required or optional set
		if (allRequired) required.push_back(newExp);
		else optional.push_back(newExp);
		
		//iterate to next assignment
		unsigned j = 0;
		while((j <= lastVarIdx) && (++currAssn[j].second >= idxVarMax[currAssn[j].first])) {
			currAssn[j].second = 0;
			j++;
		}
		done = (j > lastVarIdx);
	}
	//cout << required.size() << " " <<optional.size() << endl;
}

vector<DCondition*> parseFile(string filename) {
	vector<DCondition*> condOut;
	
	//grab the file's contents
	vector<string> src;
	fstream fs_src;
	string line;
	fs_src.open(filename.c_str(), ios::in);
	if(!fs_src.good()) {
		cerr << "ERROR: File not found: " << filename << endl;
		exit(1);
	}
	while(fs_src.good()) {
		getline(fs_src, line);
		src.push_back(line);
	}
	fs_src.close();
	
	//strip full-line comments and blank lines
	vector<string> noComments;
	for (unsigned i = 0; i < src.size(); i++) {
		if ((src[i][0] != '#') && (src[i].find_first_not_of(" \n\t") < src[i].length()))
			noComments.push_back(src[i]);
	}
	//parse out conditions
	for (unsigned i = 0; i < noComments.size(); i++) {
		DCondition *cond = parseCondition(noComments[i]);
		if (cond) condOut.push_back(cond);
		cout << "^";
	}
	
	return condOut;
}

DCondition* parseCondition(string input) {
	dcond_grammar cond;
	BOOST_SPIRIT_DEBUG_NODE(cond);
	//cout << "Parser input:" << input << endl;
	tree_parse_info<> info = ast_parse(input.c_str(), cond,space_p);	
	if (info.full) {
		// dump parse tree as XML		
		std::map<parser_id, std::string> rule_names;
		rule_names[dcond_grammar::conditionID] = "condition";
		rule_names[dcond_grammar::identifierID] = "id";
		rule_names[dcond_grammar::numericID] = "float";
		rule_names[dcond_grammar::varnameID] = "variable";
		rule_names[dcond_grammar::moduleID] = "module";
		rule_names[dcond_grammar::moduleDeclID] = "module decl.";
		rule_names[dcond_grammar::tShiftID] = "t shift";
		rule_names[dcond_grammar::moduleListID] = "catom list";
		rule_names[dcond_grammar::realExpID] = "real exp";
		rule_names[dcond_grammar::realTermID] = "real term";
		rule_names[dcond_grammar::realFactorID] = "real factor";
		rule_names[dcond_grammar::stateVarID] = "state var";
		rule_names[dcond_grammar::countExpID] = "count()";
		rule_names[dcond_grammar::setRID] = "set(r)";
		rule_names[dcond_grammar::boolExpID] = "bool";
		rule_names[dcond_grammar::boolExpSimpleID] = "bool simple";
		rule_names[dcond_grammar::compExpID] = "comparison";
		rule_names[dcond_grammar::compOpID] = "comp. op";
		rule_names[dcond_grammar::setBID] = "set(b)";
		rule_names[dcond_grammar::setVarID] = "set var";
		rule_names[dcond_grammar::setExpID] = "set(s)";
		rule_names[dcond_grammar::actionID] = "action";
		rule_names[dcond_grammar::actionSimpleID] = "action simple";

		// print the result
		//cout << "parsing succeeded\n";
		//tree_to_xml(cout, info.trees, input.c_str(), rule_names);
		DCondition *rawTree = any_cast<DCondition*>(evaluate(info));
		//set parents
		SetParentVisitor spv;
		spv.visitDCondition(rawTree);
		//munge tree to handle kleene-*
		LocateIdxExpressions idxExps;
		//find locations to munge
		idxExps.visitDCondition(rawTree);
		for (set<BoolExp*>::const_iterator p = idxExps.stage2.parentExps.begin(); p != idxExps.stage2.parentExps.end(); p++) {
			//insert new node
			MungeTreeVisitor mtv;
			mtv.quantExp = *p;
			(*p)->parent->accept(&mtv);
		}
		
		return rawTree;
	}
	else {
		cout << "condition parsing failed:" << input << endl;
		return NULL;
	}
}
