// TransactionManager.hxx
// Nels Beckman
// March 1, 2007
//
// The transaction manager should contain methods and fields that
//   purely pertain to the management of transactions of one catom,
//   for instance who we have to ping regularly and whether or not
//   a given transaction is still operational.

#ifndef __TRANSACTIONMANAGER_H
#define __TRANSACTIONMANAGER_H

#include <list>
#include <set>
#include <map>
#include <cassert>
#include <utility>
#include <pthread.h>
#include <cstdlib>

#include <pthread.h>

//#include "Tid.hxx"
#include "CompManager.hxx"
//#include "CatomToPing.hxx"
#include "FBlockMsgs.hxx"

#include "CatomWorld.hxx"
#include "Primitives.hxx"
#include "Debugging/signpost.hxx"

#define Tid int

const int TIMEOUT = 10;

using namespace std;

extern CatomWorld *worldPtr;

enum TidStatus { INIT, ACTIVE, REPORTED, FAILED_DISPOSED, ENDED };

class CollabInfo { 
public:
  Tid tid;
  featureID fid;
  int timeout;
};

class Compensation;

class TidFailure { 
public:
  TidFailure(Tid _tid = 0, catomID _localCatom = 0, catomID _remoteCatom = 0) :
    tid(_tid),
    localCatom(_localCatom),
    remoteCatom(_remoteCatom) {}
  
  Tid tid; catomID localCatom; catomID remoteCatom; 
};

class TransactionManager {

protected:
  //unusable!
  TransactionManager() {}
  
  // A list of abort messages that the admin thread should send.
  list< pair<featureID, ErrorMsg*> > abortMsgsToSend;

  // Holds the transactions and the status for those transactions.
  //   This is static because a thread can't return home until its
  //   home has recognized the failure.
  static map<catomID, map<Tid, TidStatus> > transactions;

  // For a given transaction, the Catoms we must ping.
  // Also the catoms we expect to be pinged by.
  // Inv: Tid should be in activeTids.
  list<CollabInfo> collabs;

  // The compensation manager. Holds compensations, and we delegate
  //   the running of compensations to it as well.
  CompManager compMngr;

  // This is a very important data structure.
  //   It allows a given thread
  //   to know in which transaction it is currently operating.
  //   If there is no entry for a given thread in this map, it is not
  //   currently in a transaction, otherwise its
  //   thread id maps to its
  //   transaction id. It is static because even as you jump to other
  //   catoms you are still in the same transaction. 
  static map<pthread_t, Tid> inTransactionMap;

  // On which catom are we located. 
  // Inv: This Must equal FailBlock.mngr
  catomID myCatom;
  
  // Transaction Manager lock. Protects all per-object state.
  pthread_mutex_t tmStateMutex;

  // Static Transaction Manager lock. Protects all per-class state.
  static pthread_mutex_t tmStaticStateMutex;

  // Goes through the list of collaborators, finds ones with the given tid,
  //   removes them and adds them to abortMsgsToSend.
  void findCollabsToAbort(Tid);
  //  
  //  foreach CollabInfo(fid, _) in removeAndReturnCollabs(tid)
  //  do
  //    abortMessagesToSend.add(new ErrorMsg(fid, tid));
  //  od

  // Sends all collaborators the END msg and removes them from
  //   the collabs list. 
  void sendEndToCollabs(Tid);
  //
  // foreach CollabInfo(fid, _) in removeAndReturnCollabs(tid)
  // do
  //   send( fid, new EndMsg(tid) );
  // od

  // Resets the ping to the MAX in the given map of pings.
  void resetPingTimeout(Tid, catomID, featureID);
  //
  // foreach CallabInfo(_tid, _fid, _timeout) in collabs 
  //     where tid = _tid & fid = _fid
  // do
  //   _timeout = MAX_TIMEOUT;
  // od

  // if sendMsg fails, we care and we take action.
  void sendMsg(featureID, Message*);
  //
  // sendMsg(fid, msg)

  void sendPing(Tid, featureID);
  // 
  // sendMsg(fid, new Ping(tid));

public:
  // Requires the catomID, RPC condition variable and that condition
  //   variable's mutex to create a transaction manager.
  TransactionManager(catomID);
  
  ~TransactionManager();
  
  // Adds a collaborator to the list of collabs that we ping each
  //   tick.
  void addCollab(Tid, featureID);
  //
  // collabs.add( new CollabInfo(tid, fid) );
  
  // Atomically, with respect to both local TM lock,
  // remove all of the collaborators for the given Tid and add in
  // the new ones from the list. The remaining bool parameters tell
  // us whether we should acquire and release (respectively) the
  // TM static state lock.
  void atomicRemoveAndAddCollabs( Tid, list<featureID>, bool, bool );

  // Removes a collaborator at the given fid.
  void removeCollab(featureID);

  void removeAllCollabs();

  void resetAllPings();

  // Might be called by f_block() or by handleSendMsg()
  void beginTid(Tid);
  //
  // transactions.get(thisCatom).add( tid );

  static void clearCurrentTid();
  // 
  // getThisThread().setTid(NONE);

  // Goes through all of the catoms who have contacted us and
  //   decrements the timout counter.
  //   It is the responsibility of the RPC caller to to ping
  //   but the callee expects to get pinged.
  void decrAllExpectedPings();
  //
  // foreach CollabInfo(_, _timeout) in collabs
  // do
  //   _timeout := MAX_TIMEOUT;
  // od

  // Ends a failure block
  // This is the normal primitive that is called by the app programmer.
  void end_f_block();
  //
  // let tid = getThread().getTid() in
  //   getThread().setTid(NONE);
  //   endTid(tid);
  // tel

  // When a transaction has ended in the normal way, this
  //   method will be called. It removes the tid from the
  //   active Tids list, tells our collabs, removes those
  //   collabs and removes all compensations.
  //   This should be called by other threads not in the TID
  //   when an end message is received.
  void endTid(Tid);
  //
  // transactions.get(thisCatom).set(tid, ENDED);
  // sendEndToCollabs(tid);

  // The f_block primitive.
  Tid f_block();
  // let tid = uniqueTid() in
  //   getThread().setTid(tid);
  //   begin(tid);
  //   tid
  // tel

  // Returns the transaction id that this thread is currently in.
  // 0 if this thread is not currently in a transaction.
  static Tid getCurrentTid();
  //
  // return getThread().getTid();

  // What is the status of the given Tid?
  static TidStatus getTidStatus(catomID, Tid);
  //
  // return transactions.get(catom).get(tid);

  // Returns true if this thread is in a transaction.
  static bool inTransaction(); 
  // 
  // return getThread().getTid() != NONE;
 
  // The admin thread can call the new tick method.
  //   Performs all the tasks that must be done on a
  //   regular basis. This includes sending abort messages
  //   running outstanding compensations
  //   sending out ping messages and decrementing expected
  //   pings.
  void newTick();
  //
  // sendAllAbortMsgs();
  // compMngr.runOutstandingComps();
  // sendAllPings();
  // decrAllExpectedPings();

  void printCollabs();

  // the push_comp primitive.
  void push_comp(Compensation*);
  //
  // compMngr.push_comp( getThread().getTid(), comp );

  // Does the job of putting this tid into a failed state, but does
  //   not actually begin the process of alerting collabs, etc.
  void reportFailure(Tid);
  //
  // transaction.get(thisCatom).set(tid, REPORTED);
  // compMngr.addOutstandingComp(tid);
  // findCollabsToAbort(tid);

  // Sends all of the abort messages listed in abortMsgsToSend, then
  //   removes them.
  void sendAllAbortMsgs();
  // 
  // foreach (fid, msg) in abortMsgsToSend
  // do
  //   sendMsg( fid, msg );
  // od

  // Send a new error message to the given catom.
  void sendError(Tid, featureID);
  //
  // sendMsg( sendMsg(fid, new ErrorMsg(fid) ) );

  // This methods pings all of the catoms
  //   that we are supposed to be pinging.
  void sendAllPings();
  //
  // foreach CollabInfo(_fid, _tid) in collabs
  // do
  //   sendPing(_fid, _tid);
  // od

   // This catom was pinged by the given catom.
  void wasPinged(Tid, catomID, featureID); 
  // 
  // resetPingTimeout(tid, c, fid);
};

#endif
