// TransactionManager.cxx
// Nels Beckman
// March 1, 2007

#include "TransactionManager.hxx"
#include "RPCCodeModuleStatic.hxx"


// Static initialization
map<pthread_t, Tid> TransactionManager::inTransactionMap = map<pthread_t, Tid>();

map<catomID, map<Tid, TidStatus> > TransactionManager::transactions = map<catomID, map<Tid, TidStatus> >();

pthread_mutex_t TransactionManager::tmStaticStateMutex =
  PTHREAD_MUTEX_INITIALIZER;

TransactionManager::TransactionManager(catomID _id) :
  myCatom(_id)
{
  pthread_mutexattr_t mta;
  pthread_mutexattr_init(&mta);
  pthread_mutexattr_settype(&mta, PTHREAD_MUTEX_RECURSIVE);

  pthread_mutex_init( &tmStateMutex, &mta );

  pthread_mutex_lock( &tmStaticStateMutex );
  inTransactionMap[pthread_self()] = 0;
  pthread_mutex_unlock( &tmStaticStateMutex );
}

TransactionManager::~TransactionManager() {
  pthread_mutex_destroy( &tmStateMutex );
}

bool containsTid( list<CollabInfo> cs, Tid tid ) {
  for( list<CollabInfo>::iterator iter = cs.begin();
       iter != cs.end();
       iter++ ) {
    if( iter->tid == tid )
      return true;
  }
  return false;
}

void removeCollab( list<CollabInfo> &cs, Tid tid, featureID fid ) {

  list<list<CollabInfo>::iterator> to_erase;
  
  for( list<CollabInfo>::iterator iter = cs.begin();
       iter != cs.end();
       iter++ ) {
    if( iter->tid == tid &&
	iter->fid == fid ) {
  
      to_erase.push_back( iter );
    }
  }

  // This is ugmo because C++/STL is dumb.
  for( list<list<CollabInfo>::iterator>::iterator iter = to_erase.begin();
       iter != to_erase.end();
       iter++ ) {
    cs.erase( (*iter) );
  } 
}

void removeTidCollabs( list<CollabInfo> &cs, Tid tid ) {
  list<list<CollabInfo>::iterator> to_erase;
  
  for( list<CollabInfo>::iterator iter = cs.begin();
       iter != cs.end();
       iter++ ) {
    if( iter->tid == tid ) {
  
      to_erase.push_back( iter );
    }
  }

  // This is ugmo because C++/STL is dumb.
  for( list<list<CollabInfo>::iterator>::iterator iter = to_erase.begin();
       iter != to_erase.end();
       iter++ ) {
    cs.erase( (*iter) );
  }   
}

void removeThisCollab( list<CollabInfo> &cs, featureID fid ) {
  
  list<list<CollabInfo>::iterator> to_erase;

  for( list<CollabInfo>::iterator iter = cs.begin();
       iter != cs.end();
       iter++ ) {
    if( iter->fid == fid ) { 
      to_erase.push_back( iter );
    } 
  }

  // This is ugmo because C++/STL is dumb.
  for( list<list<CollabInfo>::iterator>::iterator iter = to_erase.begin();
       iter != to_erase.end();
       iter++ ) {
    cs.erase( (*iter) );
  }
}

list<CollabInfo> removeAndReturnCollabs( list<CollabInfo> &cs, Tid tid ) {
  
  list<CollabInfo> to_return;
  list<list<CollabInfo>::iterator> to_erase;

  for( list<CollabInfo>::iterator iter = cs.begin();
       iter != cs.end();
       iter++ ) {
    if( iter->tid == tid ) {
      CollabInfo c; 
      c.tid = iter->tid; 
      c.fid = iter->fid; 
      c.timeout = iter->timeout;
      to_return.push_back( c );
      
      to_erase.push_back( iter );
    }
  }
  
  // This is ugmo because C++/STL is dumb.
  for( list<list<CollabInfo>::iterator>::iterator iter = to_erase.begin();
       iter != to_erase.end();
       iter++ ) {
    cs.erase( (*iter) );
  }

  return to_return;
}

//static
void addCollabToList( list<CollabInfo> &cs, Tid tid, featureID fid ) {
  CollabInfo c; c.tid = tid; c.fid = fid; c.timeout = TIMEOUT;
  cs.push_back(c);
}

void TransactionManager::addCollab(Tid tid, featureID fid) {
  //printCollabs();
  
  //worldPtr->oStart();
  //cout << "MyCatom: " << myCatom <<" "
  //     << this << " Tid: " << tid << " Fid: " << fid << endl;
  //worldPtr->oEnd();

  //assert( isActive(tid) );

  //worldPtr->oStart();
  //cout << "Catom " << this->myCatom << ": adding new collab in tid "
  //     << tid << endl;
  //worldPtr->oEnd();

  pthread_mutex_lock( &tmStateMutex );
  addCollabToList(collabs, tid, fid);
  pthread_mutex_unlock( &tmStateMutex );
}

void TransactionManager::removeCollab(featureID fid) {
  pthread_mutex_lock( &tmStateMutex );
  removeThisCollab( collabs, fid );
  pthread_mutex_unlock( &tmStateMutex );
}

void TransactionManager::removeAllCollabs() {
  pthread_mutex_lock( &tmStateMutex );
  collabs = list<CollabInfo>();
  pthread_mutex_unlock( &tmStateMutex );
}

void TransactionManager::atomicRemoveAndAddCollabs( Tid trans, 
						   list<featureID> new_collabs,
						    bool get_static,
						    bool release_static )
{
  if( get_static )
    pthread_mutex_lock( &tmStaticStateMutex );

  pthread_mutex_lock( &tmStateMutex );
  
  removeTidCollabs( collabs, trans );

  for( list<featureID>::iterator iter = new_collabs.begin();
       iter != new_collabs.end(); iter++ ) {
    addCollab( trans, (*iter) );
  }

  pthread_mutex_unlock( &tmStateMutex );

  if( release_static )
    pthread_mutex_unlock( &tmStaticStateMutex );
}

// ALWAYS REMOVES COLLABORATORS!!
void TransactionManager::sendEndToCollabs(Tid tid) {
 
  // Alert all of your collaborators, if they exist.
  //   If this tid has already been cancelled, nothing happens.
  
  //this->printCollabs();
  pthread_mutex_lock( &tmStateMutex ); 
  list<CollabInfo> fed = removeAndReturnCollabs( collabs, tid );
  pthread_mutex_unlock( &tmStateMutex );
  //this->printCollabs();

  for( list<CollabInfo>::iterator iter = fed.begin();
       iter != fed.end();
       iter ++ ) {
    
    EndMsg* msg = new EndMsg(tid); 

    // We are done. If it fails, we don't care!
    try {
      sendMsg( iter->fid, msg );
    } catch (SendFail s ) {}
  }
}

void TransactionManager::printCollabs() {
  
  worldPtr->oStart();
  cout << "Catom " << myCatom << ": printing all collabs. " <<endl;
  worldPtr->oEnd();

  pthread_mutex_lock( &tmStateMutex );
  for( list<CollabInfo>::iterator iter = collabs.begin();
       iter != collabs.end();
       iter++ ) {
    cout << "Catom " << myCatom << "," << this << ": (tid: " << iter->tid
	 << " fid: " << iter->fid << " t/o: " << iter->timeout 
	 << ")" << endl;
  }
  pthread_mutex_unlock( &tmStateMutex );

 
}


void TransactionManager::reportFailure(Tid tid) {
  worldPtr->oStart();
  cout << "Catom " << myCatom << ": reportingFailure. Tid: " << tid <<endl;
  worldPtr->oEnd();

  pthread_mutex_lock( &tmStateMutex );
  compMngr.addOutstandingComp( tid );
  pthread_mutex_unlock( &tmStateMutex );

  findCollabsToAbort(tid);
  
  pthread_mutex_lock( &tmStaticStateMutex );
  transactions[myCatom][tid] = REPORTED;
  pthread_mutex_unlock( &tmStaticStateMutex );  
}

void TransactionManager::decrAllExpectedPings() {
  //  worldPtr->oStart();
  //cout << "Catom " << myCatom << ": decrAllExpectedPings(). " <<endl;
  //worldPtr->oEnd();  
  
  list<Tid> fed;

  pthread_mutex_lock( &tmStateMutex );
  for( list<CollabInfo>::iterator iter = collabs.begin();
       iter != collabs.end();
       iter ++ ) {
    
    iter->timeout = iter->timeout - 1;
    
    if( iter->timeout == 0 ) {
      worldPtr->oStart();
      cout << "Catom " << myCatom << ": found a failure. Tid: " 
	   << iter->tid << " fid " << iter->fid <<endl;
      worldPtr->oEnd();

      fed.push_back( iter->tid );
    }
  }
  pthread_mutex_unlock( &tmStateMutex );

  for( list<Tid>::iterator iter = fed.begin();
       iter != fed.end();
       iter++ ) {
    reportFailure( (*iter) );
  }
}

void TransactionManager::beginTid(Tid tid) {
  pthread_mutex_lock( &tmStaticStateMutex );
  transactions[myCatom][tid] = ACTIVE;  
  pthread_mutex_unlock( &tmStaticStateMutex );

  //worldPtr->oStart();
  //cout << "Catom: " << myCatom << " begins " << tid 
  //     << " and status is " << transactions[myCatom][tid] << endl;
  //worldPtr->oEnd();
}

void TransactionManager::end_f_block() {
  assert(inTransaction());
  Tid tid = getCurrentTid();

  pthread_mutex_lock( &tmStaticStateMutex );
  inTransactionMap[pthread_self()] = 0;
  pthread_mutex_unlock( &tmStaticStateMutex );

  endTid(tid);
}

void TransactionManager::endTid(Tid tid) {
  //worldPtr->oStart();
  //cout << "Catom " << myCatom << ": leaving transaction " << tid << endl;
  //worldPtr->oEnd();

  pthread_mutex_lock( &tmStaticStateMutex );
  transactions[myCatom][tid] = ENDED;
  pthread_mutex_unlock( &tmStaticStateMutex );
  
  sendEndToCollabs(tid);
}

Tid TransactionManager::f_block() {
  assert( getCurrentTid() == 0 );
  
  //Tid tid = Tid( myCatom );
  Tid tid;
  do {
    tid = rand();
  } while( tid == 0 );
  

  pthread_mutex_lock( &tmStaticStateMutex );
  inTransactionMap[pthread_self()] = tid;
  pthread_mutex_unlock( &tmStaticStateMutex );

  beginTid(tid);
  
  //worldPtr->oStart();
  //cout << "Catom " << myCatom << ":  creating transaction " 
  //     << tid << endl;
  //worldPtr->oEnd();  

  return tid;
}

bool TransactionManager::inTransaction() {
  
  pthread_mutex_lock( &tmStaticStateMutex );
  pthread_t cur_thread = pthread_self();
  bool ret = inTransactionMap.count(cur_thread) > 0;
  pthread_mutex_unlock( &tmStaticStateMutex );
  
  return ret;
}

void TransactionManager::push_comp(Compensation* comp) {
  assert(inTransaction());
  
  Tid tid = getCurrentTid();
  compMngr.pushComp(comp, tid);
}

void TransactionManager::resetPingTimeout(Tid tid, catomID c, featureID f) {

  //assert( getTidStatus(c, tid) == ACTIVE ); //Is done by the admin thread
  //assert( containsTid( collabs, tid ) );

  //worldPtr->oStart();
  //cout << "Catom " << myCatom << ": re-seting ping on " << tid << endl;
  //cout << "c: " << c << " f: " << f << endl;
  //worldPtr->oEnd();

  pthread_mutex_lock( &tmStateMutex );
  for( list<CollabInfo>::iterator iter = collabs.begin();
       iter != collabs.end();
       iter ++ ) {
    
    if( iter->tid == tid && iter->fid == f )
      iter->timeout = TIMEOUT;
	
  }
  pthread_mutex_unlock( &tmStateMutex );
}

void TransactionManager::resetAllPings() {
   pthread_mutex_lock( &tmStateMutex );
  for( list<CollabInfo>::iterator iter = collabs.begin();
       iter != collabs.end();
       iter ++ ) {
    iter->timeout = TIMEOUT;
  }
  pthread_mutex_unlock( &tmStateMutex ); 
}

// Send error has to be separate, because if it fails we don't care!
void TransactionManager::sendError(Tid tid, featureID fid) {
  ErrorMsg* msg = new ErrorMsg(tid);
  RPCCodeModuleStatic::sendMsg( myCatom, fid, msg );
}

void TransactionManager::sendMsg( featureID fid, Message* msg) {
  RPCCodeModuleStatic::sendMsg( myCatom, fid, msg );
}

void TransactionManager::sendPing(Tid tid, featureID fid) {
  // create the ping message
  PingMsg* msg = new PingMsg(tid, myCatom);
  sendMsg(fid, msg);

  //worldPtr->oStart();
  //cout << "Catom " << myCatom << ": sent ping." <<endl;
  //worldPtr->oEnd();
}

void TransactionManager::sendAllPings() {
  
  //worldPtr->oStart();
  //cout << "Catom " << myCatom << ": sendPings()." <<endl;
  //worldPtr->oEnd();  
  
  list<CollabInfo> copy;
  
  // Alert all of your collaborators, if they exist.
  //   If this tid has already been cancelled, nothing happens.
  pthread_mutex_lock( &tmStateMutex ); 
  for( list<CollabInfo>::iterator iter = collabs.begin();
       iter != collabs.end();
       iter++ ) {
    copy.push_back( (*iter) );
  }
  pthread_mutex_unlock( &tmStateMutex );
  
  for( list<CollabInfo>::iterator iter = copy.begin();
       iter != copy.end();
       iter ++ ) {
    
    //worldPtr->oStart();
    //cout << "Catom " << myCatom << ":  sending ping " 
    //	 << iter->tid << " to " << iter->fid << endl;
    //worldPtr->oEnd();  
    sendPing( iter->tid, iter->fid );
  }
}

void TransactionManager::wasPinged(Tid tid, catomID c, featureID f) {

  //worldPtr->oStart();
  //cout << "Catom " << myCatom << ":  was pinged. tid: " 
  //     << tid << " on fid " << f << endl;
  //worldPtr->oEnd();  

  resetPingTimeout(tid, c, f);
}


TidStatus TransactionManager::getTidStatus(catomID c, Tid t) {
  pthread_mutex_lock( &tmStaticStateMutex );
  TidStatus ret = INIT;
  if ( transactions.count(c) == 0 ) {
  }
  else if ( transactions[c].count(t) == 0 ) {
  }
  else {
    ret = transactions[c][t];
  }
  pthread_mutex_unlock( &tmStaticStateMutex );
  return ret;
}


// Methods introduced with the new design.
Tid TransactionManager::getCurrentTid() {
  pthread_mutex_lock( &tmStaticStateMutex );
  Tid t = inTransactionMap[pthread_self()];
  pthread_mutex_unlock( &tmStaticStateMutex );

  return t;
}

void TransactionManager::clearCurrentTid() {
  pthread_mutex_lock( &tmStaticStateMutex );
  inTransactionMap[pthread_self()] = 0;
  pthread_mutex_unlock( &tmStaticStateMutex );
}

void TransactionManager::sendAllAbortMsgs() {
  list<Tid> changeStatus;

  pthread_mutex_lock( &tmStateMutex );
  for( list< pair<featureID, ErrorMsg*> >::iterator iter = 
	 abortMsgsToSend.begin();
       iter != abortMsgsToSend.end();
       iter++ ) {
    
    changeStatus.push_back( iter->second->tid );

    // We don't care if it fails.
    try {
      RPCCodeModuleStatic::sendMsg( myCatom, iter->first, iter->second );
    } catch( SendFail s ) {}
  }
  abortMsgsToSend.clear();
  pthread_mutex_unlock( &tmStateMutex );

  
  pthread_mutex_lock( &tmStaticStateMutex );
  for( list<Tid>::iterator iter = changeStatus.begin();
       iter != changeStatus.end();
       iter++ ) {
    transactions[myCatom][(*iter)] = FAILED_DISPOSED;
  }
  pthread_mutex_unlock( &tmStaticStateMutex );
}

void TransactionManager::findCollabsToAbort(Tid tid) {
  pthread_mutex_lock( &tmStateMutex ); 
  list<CollabInfo> cs = removeAndReturnCollabs( this->collabs, tid ); 
  pthread_mutex_unlock( &tmStateMutex );

  for( list<CollabInfo>::iterator iter = cs.begin();
       iter != cs.end();
       iter ++ ) {
    
    ErrorMsg* msg = new ErrorMsg(tid); 
    abortMsgsToSend.push_back( pair<featureID, ErrorMsg*>( iter->fid, msg ) );
  }
}

void TransactionManager::newTick() {

  // At this point some tids are in the `reported failed' state
  sendAllAbortMsgs();
  compMngr.runOutstandingComps();
  // At this point, no tids should be in the reported failed state.
  //   They should have all migrated to the Failed & Disposed

  sendAllPings();
  decrAllExpectedPings();
  // And now there may again be some tids in the `reported failed' state.
}
