// RPCTemplates.hxx
// Nels Beckman
// March 18th, 2007
//
// Contains ALL of the template code for the RPC system.

#ifndef __RPC_TEMPLATE
#define __RPC_TEMPLATE

#include <pthread.h>
#include <cassert>
#include <map>
#include <stack>

#include <string>
#include <sstream>
#include <ctime>
#include <cstdlib>


#include "RPCCodeModuleStatic.hxx"
#include "FailureSimulator.hxx"
#include "Network.hxx"
#include "CodeModule.hxx"
#include "CatomWorld.hxx"

#define HOSTCATOM (worldPtr->catomHash[hostCatom]->C)

// Let the hacks begin! We really need a compiler...
#define ENTER_RPC_METHOD enterRpcMethod()
#define RPC_RETURN rpcReturn(); return

using namespace std;

template<class T> class RPCCodeModule;

template<class N>
class RPCNeighbor {
  
protected:

  // protects data local to this object
  pthread_mutex_t neighborLock;
  
  // Set by RPCCodeModule->getNeighbor
  RPCCodeModule<N>* c_source;
  
  // Set by RPCCodeModule->handleSendMsg
  N* c_dest;
  
  // Set by RPCCodeModule->getNeighbor()
  featureID f_source;
  featureID f_return;
 
public:

  RPCNeighbor( RPCCodeModule<N>* src_catom, featureID send_to_feature);

  virtual ~RPCNeighbor();

  // Probably the coolest method in this class. When a method is called
  //   on an RPC neighbor, this constitutes an RPC call. This message
  //   does the following:
  //   1 - Send an RPC message to the destination with the source convar
  //       as its payload.
  //   2 - Call yield on the same convar
  //   3 - Upon awakening, send RPC message back to the source with the
  //       source convar as its payload.
  //   4 - Call yield on the source convar
  //   5 - Return the RPCCodeModule pointer from the destination, which as it
  //       is returned will simulate the actual remote behavior.
  virtual N* get(Message* msg = NULL);

  featureID getReturnFeat();

  RPCCodeModule<N>* getSourceCatom();

  void setDestCatom(N* dest);
  
  void setReturnFeat(featureID);
};


template <class M>
class RPCMessage : public Message {
  
public:
  static const MailboxID RPC_SEND_BOX;
  static const MailboxID RPC_REPLY_BOX;
  
  // I hope this isn't too confusing. If you call this
  //   constructor, you have created an RPC send message.
  RPCMessage(catomID, RPCNeighbor<M>*);
  
  // If you call this constructor, you have created an RPC
  //   reply message.
  RPCMessage(catomID);
  
  // Copy constructor!
  RPCMessage(const RPCMessage<M> &m);

  // clone
  // This method is abstract in message, so I assume our message
  //   is getting copied somewhere in the network code.
  virtual Message* clone();

  // getNeighbor
  // Returns the RPCNeighbor object that was created as part of this
  //   RPC communication.
  RPCNeighbor<M>* getNeighbor();
  
  // Returns the thread that sent this RPC message.
  pthread_t getSendThread();

  // When a RPC_SEND message arrives at the destination catom, that catom
  //   should be stored in the dest RPCNeighbor object.
  void setDestCatom(RPCCodeModule<M>* dest);

  catomID sourceCatom;
  
private:
  RPCNeighbor<M>* rpcNeighbor;
};


template<class T>
class RPCCodeModule : public RPCCodeModuleStatic {
protected:
  
  // To be called by a macro function when entering an RPC method.
  virtual void enterRpcMethod();

  // To be called by a macro function when returning from an RPC
  //   method.
  virtual void rpcReturn();

  // Uh oh, statics in our not-static thing.
  static map<pthread_t, stack<RPCNeighbor<T>*> > rpcNeighborStacks;

public:
  RPCCodeModule(catomID _hostCatom);
  
  // Gets the neighbor located at the given feature. If
  //   no neighbor is present, returns NULL. Keep in mind
  //   that even if there is a neighbor when this method is
  //   called, that neigbhor could move before a RPC call is
  //   attempted, and then an exception will be thrown.
  virtual RPCNeighbor<T>* getNeighbor(featureID feat); 

  // Get the top neighbor object, corresponding to the most
  //   recent RPC call for the given thread.
  RPCNeighbor<T>* getTopNeighbor(pthread_t);
  
  // Handle new message. Does nothing but wake the thread to whom the
  //   message was actually intended. This message is the type sent
  //   back from the destination catom as a response.
  bool handleReplyMsg(RPCMessage<T>* msg);
  
  // A message sent immediately in response to an RPC message.
  bool handleRPCAckMsg(RPCAckMsg* msg);

  // Handle send message. In addition to waking the thread that is
  //   sleeping on the conditioin variable, it also sets the incoming
  //   feature upon which the message arrived and this catom so that
  //   the RPCNeighbor has all of this information.
  bool handleSendMsg(RPCMessage<T>* msg);

  // Called once per tick. Lets RPC threads free.
  // There are only two threads going per code module; the one that
  //   enters newTick and the one that runs the thread() code. These
  //   two threads alternate strictly. The thread() code thread is only
  //   run if wakeRPCThread is set to true. This should be the case at
  //   the beginning of the simulator.
  virtual void newTick();

  // Pop the top neighbor object on our stack, corresponding to the most
  //   recent thread call for the given thread.
  void popTopNeighbor(pthread_t);
  
  static void pushNeighbor(RPCNeighbor<T>*, pthread_t);

  // Actually registers the handler, which is pretty simple
  virtual void registerHandler();
  
  // At the beginning of a simulation, we have to register the
  //   message handler, of which there is only one.
  virtual void simulationStart();

};

pthread_mutex_t RPCCodeModuleStatic::rpcGlobalMTX =
  PTHREAD_MUTEX_INITIALIZER;

//map<pthread_t, int> RPCCodeModuleStatic::rpcMethodStacks =
//map<pthread_t, int>();

map<catomID, RPCCodeModuleStatic*> RPCCodeModuleStatic::wakeFlags =
  map<catomID, RPCCodeModuleStatic*>();

map<pthread_t, catomID> RPCCodeModuleStatic::threadHomeMap =
  map<pthread_t, catomID>();

map<pthread_t, catomID> RPCCodeModuleStatic::threadLocationMap =
  map<pthread_t, catomID>();

map<catomID, set<RPCMsgID> > RPCCodeModuleStatic::arrivedMsgs =
  map<catomID, set<RPCMsgID> >();

template<class T>
map< pthread_t, stack<RPCNeighbor<T>* > > RPCCodeModule<T>::rpcNeighborStacks =
  map< pthread_t, stack<RPCNeighbor<T>* > >();

RPCCodeModuleStatic::RPCCodeModuleStatic(catomID _hostCatom): 
  CodeModule(_hostCatom),
  rpcConVar(),
  rpcMutex(),
  wakeRPCThread(false),
  failureSim(_hostCatom),
  readyMove(),
  moveCountdown(0)//,
  //  rpcMethodStacks()
{ 
  pthread_cond_init( &rpcConVar, NULL );
  pthread_mutex_init( &rpcMutex, NULL );

  pthread_mutex_lock( &rpcGlobalMTX );
  wakeFlags[_hostCatom] = this;
  pthread_mutex_unlock( &rpcGlobalMTX );

  seedRandomizer();
}

bool RPCCodeModuleStatic::hasMsgArrived(catomID c, RPCMsgID id) {
  pthread_mutex_lock( &rpcGlobalMTX );
  bool ret = ( arrivedMsgs[c].count(id) > 0 );
  pthread_mutex_unlock( &rpcGlobalMTX );
  return ret;
}

void RPCCodeModuleStatic::seedRandomizer() {
  unsigned int i;

  std::string val = worldPtr->search_key_value_list("SRAND");

  std::istringstream iss(val);

  if( (iss >> std::dec >> i).fail() ) {
    srand( (unsigned)time(NULL) );
  }
  else {
    srand(i);
  }
}

void RPCCodeModuleStatic::setMsgHasArrived(catomID c, RPCMsgID id) {
  pthread_mutex_lock( &rpcGlobalMTX );
  arrivedMsgs[c].insert(id);
  pthread_mutex_unlock( &rpcGlobalMTX );
}

// THIS IS TOTALLY BOGUS AND SPECIFIC TO THE BELT APPLICATION.
bool RPCCodeModuleStatic::moveReady() {

  // COUNT UP THE NUMBER OF THREADS ON THIS LOCAL HOST.
  //pthread_mutex_lock( &rpcGlobalMTX );
  //int num = 0;
  //for( map<pthread_t, catomID>::iterator iter = 
  //	 threadLocationMap.begin();
  //    iter != threadLocationMap.end();
  //    iter ++ ) {
  // if( hostCatom == iter->second ) num++;
  //}
  //pthread_mutex_unlock( &rpcGlobalMTX );

  bool ret = false;

  pthread_mutex_lock( &rpcMutex );
//  if( moveCountdown == 1 )//&& 
//    //      num == 0 ) {
//    {
//      ret = true;
//      moveCountdown--;
//  }
//  else if( moveCountdown > 0 ) {
//    moveCountdown--;
//  }
//
  if( moveCountdown == 1 ) ret = true;
  if( moveCountdown > 0 ) moveCountdown--;
  pthread_mutex_unlock( &rpcMutex );
  
  return ret;
}

void RPCCodeModuleStatic::makeReadyMove() {
  pthread_mutex_lock( &rpcMutex );
  HOSTCATOM.moveTo( readyMove.first, readyMove.second );
  pthread_mutex_unlock( &rpcMutex );

  int failed = failureSim.getNumberFailed();

  worldPtr->oStart();
  cout << "MOVE WITH " << failed << " DEAD CATOMS, CATOM " 
       << hostCatom << endl;
  worldPtr->oEnd();
}

void RPCCodeModuleStatic::sched_move( catomID c, featureID f ) {
  pthread_mutex_lock( &rpcMutex );
  moveCountdown = 6;
  readyMove.first = c; readyMove.second = f;
  pthread_mutex_unlock( &rpcMutex );
}

template<class T>
RPCCodeModule<T>::RPCCodeModule(catomID _hostCatom) :
  RPCCodeModuleStatic(_hostCatom)
{ 
  name_ = "RPCCodeModule";
}

// IF ANY OF THIS CHANGES, FBTemplates.hxx MUST BE MODIFIED.
template<class T>
RPCNeighbor<T>* RPCCodeModule<T>::getNeighbor(featureID feat) {
  catomID neighbor = HOSTCATOM.getNeighbor(feat);

  if( HOSTCATOM.getNeighbor(feat) == 0 )
    return NULL;
  
  // Check if neighbor is DEAD!!!
  if( FailureSimulator::hasCatomDied(neighbor) )
    return NULL;

  RPCNeighbor<T>* to_return = new RPCNeighbor<T>( this, feat );

  return to_return;
}

template<class T>
bool RPCCodeModule<T>::handleReplyMsg(RPCMessage<T>* msg) {
  assert( msg != NULL );
  
  if( this->failureSim.isCatomDead() ) {
    return false;
  }

  sendMsg( hostCatom, msg->arrivalContact, new RPCAckMsg(msg->messageID) );

  setThreadLocation( hostCatom );

  return false;
}

template<class T>
bool RPCCodeModule<T>::handleRPCAckMsg(RPCAckMsg* msg) {
  if( this->failureSim.isCatomDead() ) {
    return false;
  }
  
  setMsgHasArrived( hostCatom, msg->messageID );
  
  return false;
}

template<class T>
RPCNeighbor<T>* RPCCodeModule<T>::getTopNeighbor(pthread_t thread) {
  pthread_mutex_lock( &rpcGlobalMTX );

  //assert( rpcNeighborStacks.count(thread) > 0 );
  // Not true if we have just sent the message.
  assert( !rpcNeighborStacks[thread].empty() );

  RPCNeighbor<T>* neighbor = rpcNeighborStacks[thread].top();
  pthread_mutex_unlock( &rpcGlobalMTX );
  return neighbor;
}

template<class T>
void RPCCodeModule<T>::popTopNeighbor(pthread_t thread) {
  pthread_mutex_lock( &rpcGlobalMTX );

  //assert( rpcNeighborStacks.count(thread) > 0 );
  // Not true if we have jsut sent the message.
  assert( !rpcNeighborStacks[thread].empty() );

  rpcNeighborStacks[thread].pop();
  pthread_mutex_unlock( &rpcGlobalMTX );
}

template<class T>
void RPCCodeModule<T>::pushNeighbor(RPCNeighbor<T>* n, pthread_t t) {
  pthread_mutex_lock( &rpcGlobalMTX );
  rpcNeighborStacks[t].push( n );
  pthread_mutex_unlock( &rpcGlobalMTX );
}

// MAKE SURE CHANGES GO TO FTEMPLATE IF NEED BE
template<class T>
bool RPCCodeModule<T>::handleSendMsg(RPCMessage<T>* msg) {
  assert( msg != NULL );
  
  if( this->failureSim.isCatomDead() ) {
    return false;
  }
  
  // Have we been scheduled to move?
  bool waiting_to_move = false;
  pthread_mutex_lock( &rpcMutex );
  waiting_to_move = moveCountdown > 0;
  pthread_mutex_unlock( &rpcMutex );
  if( waiting_to_move > 0 ) return false;

  // set incoming feature
  msg->getNeighbor()->setReturnFeat(msg->arrivalContact);

  // set destination catom, so that it can be returned from ->
  msg->setDestCatom( this );
  
  sendMsg( hostCatom, msg->arrivalContact, new RPCAckMsg(msg->messageID) );

  setThreadLocation( hostCatom );
  return false;
}

void RPCCodeModuleStatic::setThreadWake( catomID c ) {
  pthread_mutex_lock( &rpcGlobalMTX );
  RPCCodeModuleStatic* cm = wakeFlags[c];
  pthread_mutex_unlock( &rpcGlobalMTX );

  cm->wakeRPCThreadLazy();
}

void RPCCodeModuleStatic::clearThreadWake( catomID c ) {
  pthread_mutex_lock( &rpcGlobalMTX );
  RPCCodeModuleStatic* cm = wakeFlags[c];
  pthread_mutex_unlock( &rpcGlobalMTX );

  cm->clearRPCThreadWake();
}

void RPCCodeModuleStatic::doRPCYield() {
  catomID thread_home = RPCCodeModuleStatic::getThreadHome();

  pthread_mutex_lock( &rpcGlobalMTX );
  RPCCodeModuleStatic* mod = wakeFlags[thread_home];
  pthread_mutex_unlock( &rpcGlobalMTX );

  pthread_mutex_t& the_mutex = mod->getMutex();
  pthread_cond_t& the_convar = mod->getConVar();

  pthread_mutex_lock( &the_mutex );
  do {
    pthread_cond_broadcast( &the_convar );
    pthread_cond_wait( &the_convar, &the_mutex );  
  } while( !mod->wakeRPCThread );
  pthread_mutex_unlock( &the_mutex );

  RPCCodeModuleStatic::clearThreadWake(thread_home);

}

catomID RPCCodeModuleStatic::getThreadHome() {
  pthread_mutex_lock( &rpcGlobalMTX );
  catomID c = threadHomeMap[pthread_self()];
  pthread_mutex_unlock( &rpcGlobalMTX );  

  assert( c != 0 );

  return c;
}

catomID RPCCodeModuleStatic::getThreadLocation() {
  pthread_mutex_lock( &rpcGlobalMTX );
  catomID c = threadLocationMap[pthread_self()];
  pthread_mutex_unlock( &rpcGlobalMTX );  

  return c;  
}

void RPCCodeModuleStatic::setThreadLocation(catomID c) {
  pthread_mutex_lock( &rpcGlobalMTX );
  threadLocationMap[pthread_self()] = c;
  pthread_mutex_unlock( &rpcGlobalMTX );   
}

// So we don't have to do all the annoying work every time we want
//   to send a message. Also does extra stuff. Drops w/ non-zero
//   prob.
bool RPCCodeModuleStatic::sendMsg(catomID c, featureID f, Message* msg) {
  
  if( !(FailureSimulator::canSendReceive()) )
    return true;

  pthread_mutex_lock( &RPCCodeModuleStatic::rpcGlobalMTX );
  Feature* fmap =
    worldPtr->catomHash[c]->C.getFeatureMap();
  pthread_mutex_unlock( &RPCCodeModuleStatic::rpcGlobalMTX );
  
  return fmap[f].getNetworkAdapter()->sendMessage(msg);
}

template<class T>
void RPCCodeModule<T>::enterRpcMethod() {
}

// IF THIS METHOD CHANGES MAKE SURE FBTemplate::newTick IS OKAY.
template<class T>
void RPCCodeModule<T>::newTick() {  
  pthread_mutex_unlock( &rpcMutex );
  if( !this->failureSim.isCatomDead() ) {
    // If there is a move available, we can take it (and return).
    if( moveReady() ) {
      makeReadyMove();
      return;
    }
  }
  else {
    HOSTCATOM.setColor( 255, 255, 0, 0 );
    worldPtr->oStart();
    cout << "Catom " << this->hostCatom << ": ---------DEAD---------" <<endl;
    worldPtr->oEnd();
  }

  pthread_mutex_lock( &this->rpcMutex );
  if( this->wakeRPCThread ) {
    pthread_cond_broadcast( &this->rpcConVar );
    pthread_cond_wait( &this->rpcConVar, &this->rpcMutex );
  }
  pthread_mutex_unlock( &this->rpcMutex );
}

template<class T>
void RPCCodeModule<T>::registerHandler() {

  HOSTCATOM.mailboxManager.registerHandler(RPCMessage<T>::RPC_SEND_BOX,
					   this,
					   (msgHandlerPtr)&RPCCodeModule<T>::handleSendMsg);
  
  HOSTCATOM.mailboxManager.registerHandler(RPCMessage<T>::RPC_REPLY_BOX,
					   this,
					   (msgHandlerPtr)&RPCCodeModule<T>::handleReplyMsg);

  HOSTCATOM.mailboxManager.registerHandler(RPCAckMsg::RPC_ACK_BOX,
					   this,
					   (msgHandlerPtr)&RPCCodeModule<T>::handleRPCAckMsg);
}

template<class T>
void RPCCodeModule<T>::rpcReturn() {
    RPCNeighbor<T>* neighbor = getTopNeighbor( pthread_self() );
    popTopNeighbor( pthread_self() );
    assert( neighbor != 0 );

    featureID send_f = neighbor->getReturnFeat();
    
    RPCMsgID m_id = rand();

    for( int attempts = 0;
	 !RPCCodeModuleStatic::hasMsgArrived( hostCatom, m_id );
	 attempts++ )
    {
      //try {
	if( attempts % 2 == 0 ) {
	  RPCMessage<T>* msg = new RPCMessage<T>( getThreadHome() );
	  msg->messageID = m_id;
	  if( RPCCodeModuleStatic::sendMsg( hostCatom, send_f, msg ) == 0 ) {
	    worldPtr->oStart();
	    cout << "Catom " << hostCatom << " failing to return. Feat: " 
		 << send_f << endl;
	    worldPtr->oEnd();
	  }
	}
	else if( attempts == 9 ) {
	  // SHOULD BE PUT BACK IN FOR THE FB VERSION
	  //throw SendFail();
	}
	
	assert( ! (attempts > 30 ) );

	setThreadWake( getThreadHome() );
	doRPCYield();      
	//} catch( SendFail sf ) {}
    }
}


void RPCCodeModuleStatic::rpcThreadBegin() {
  pthread_mutex_lock( &rpcMutex );
  wakeRPCThread = true;
  pthread_mutex_unlock( &rpcMutex );

  pthread_mutex_lock( &rpcGlobalMTX );

  threadHomeMap[pthread_self()] = hostCatom;
  threadLocationMap[pthread_self()] = hostCatom;
  
  pthread_mutex_unlock( &rpcGlobalMTX );

  doRPCYield(); //TODO Race?
  this->thread();

  //once it's all done, we want this thread to yield forever.
  //  TODO: It might be better to actually make this more 
  //  permanent somehow, just in case messages are being received
  //  after we are over.
  doRPCYield(); 
}

void* RPCCodeModuleStatic::rpcThreadMain(void* m) {
  assert( m != NULL );
  
  RPCCodeModuleStatic* cm = (RPCCodeModuleStatic*)m;
  cm->rpcThreadBegin();
  
  return NULL;
}

void RPCCodeModuleStatic::simulationStart() {
  startRPCThread();
}

template<class T>
void RPCCodeModule<T>::simulationStart() {
  RPCCodeModuleStatic::simulationStart();
  this->registerHandler();
}


void RPCCodeModuleStatic::startRPCThread() {
  pthread_t new_thread;

  // Setting stack size
  pthread_attr_t attrs;
  pthread_attr_init(&attrs);
  pthread_attr_setstacksize(&attrs, (1<<17));

  // Create a thread for this code module.
  {    
    int ret;

    if( (ret = pthread_create(&new_thread, &attrs, 
			      rpcThreadMain, this)) != 0 ) {
      cerr << "Error: unable to create thread for " 
	   << "one catom per thread functionality."<< " :: " 
	   << strerror(ret) << endl;
      exit(0);
    }
  }
}

void RPCCodeModuleStatic::thread() {}

void RPCCodeModuleStatic::wakeRPCThreadLazy() {
  pthread_mutex_lock( &rpcMutex );
  wakeRPCThread = true;
  pthread_mutex_unlock( &rpcMutex );

  //worldPtr->oStart();
  //cout << "Catom " << hostCatom << ": Just set wakeRPCThread to true." << endl;
  //worldPtr->oEnd();
}

void RPCCodeModuleStatic::clearRPCThreadWake() {
  pthread_mutex_lock( &rpcMutex );
  wakeRPCThread = false;
  pthread_mutex_unlock( &rpcMutex );
}

pthread_mutex_t& RPCCodeModuleStatic::getMutex() {
  return rpcMutex;
}

pthread_cond_t& RPCCodeModuleStatic::getConVar() {
  return rpcConVar;
}

template<class M>
const MailboxID RPCMessage<M>::RPC_SEND_BOX = "RPC_SEND_BOX";

template<class M>
const MailboxID RPCMessage<M>::RPC_REPLY_BOX = "RPC_REPLY_BOX";

const MailboxID RPCAckMsg::RPC_ACK_BOX = "RPC_ACK_BOX";

template<class M>
RPCMessage<M>::RPCMessage(catomID c) :
  Message(RPC_REPLY_BOX),
  sourceCatom(c)
{}

template<class M>
RPCMessage<M>::RPCMessage(catomID c, RPCNeighbor<M>* neighbor) :
  Message(RPC_SEND_BOX),
  sourceCatom(c),
  rpcNeighbor(neighbor)
{
  assert( neighbor != NULL );
}

template<class M>
RPCMessage<M>::RPCMessage(const RPCMessage<M> &m) : 
  Message(m.mailboxID, m.messageID)
{
  rpcNeighbor = m.rpcNeighbor;
  sourceCatom = m.sourceCatom;
}

template<class M>
Message* RPCMessage<M>::clone() { 
  RPCMessage<M>* to_return = new RPCMessage<M>(*this);
  return to_return;
}

template<class M>
RPCNeighbor<M>* RPCMessage<M>::getNeighbor() {
  return rpcNeighbor;
}

template<class M>
void RPCMessage<M>::setDestCatom(RPCCodeModule<M>* dest) {
  // This cast is required by the compiler, even though M <= RPCCodeMod<M>
  M* _dest = (M*)dest;
  rpcNeighbor->setDestCatom(_dest);
}

RPCAckMsg::RPCAckMsg() :
  Message(RPC_ACK_BOX)
{}

RPCAckMsg::RPCAckMsg(unsigned long long id) :
  Message(RPC_ACK_BOX, id)
{}

Message* RPCAckMsg::clone() {
  return new RPCAckMsg(messageID);
}

template<class N>
RPCNeighbor<N>::RPCNeighbor( RPCCodeModule<N>* src_catom,
			     featureID send_to_feature) :
  c_source(src_catom),
  f_source(send_to_feature)
  
{
  pthread_mutex_init(&neighborLock, NULL);
  assert( src_catom != NULL );
}

template<class N>
RPCNeighbor<N>::~RPCNeighbor() {
  pthread_mutex_destroy(&neighborLock);
}

// This is a serious un-realistic thing about this experiment.
//   We don't Migrate the thread until the host knows that the
//   RPC request has been received.
template<class N>
N* RPCNeighbor<N>::get(Message* msg) {
  
  RPCMsgID m_id = rand();
  
  for( int attempts = 0; 
       !RPCCodeModuleStatic::hasMsgArrived(c_source->getHostCatom(),
					  m_id); 
       attempts++ )
    { 
      // Only send every other time...
      if( attempts % 2 == 0 ) {
	// New message
	if ( msg == NULL ) {
	  msg = new RPCMessage<N>( c_source->getHostCatom(), this);
	}
	msg->messageID = m_id;
	
	// Send it, if no host throw exn
	if( RPCCodeModuleStatic::sendMsg( c_source->getHostCatom(), 
					  f_source, 
					  msg->clone() ) == 0 ) {
	  //throw SendFail();
	}
	
      }
      else if( attempts == 9 ) {
	throw SendFail();
      }
  
      // We should wake up every single cycle.
      RPCCodeModuleStatic::setThreadWake(RPCCodeModuleStatic::getThreadHome());
      // I HAVE to have some c_source, even though doRPCYield is static  
      c_source->doRPCYield();
    }
  
  // Putting neighbor into the stack so that we can return properly.
  RPCCodeModule<N>::pushNeighbor(this, pthread_self());
  
  // 2nd phase.
  // We're done and we just return the destination catom.
  // The final send occurs elsewhere.
  pthread_mutex_lock(&neighborLock);
  N* ret = c_dest;
  pthread_mutex_unlock(&neighborLock);
  
  return ret;
}

template<class N>
featureID RPCNeighbor<N>::getReturnFeat() {
  pthread_mutex_lock(&neighborLock);
  featureID ret = f_return;
  pthread_mutex_unlock(&neighborLock);

  return ret;
}

template<class N>
RPCCodeModule<N>* RPCNeighbor<N>::getSourceCatom() {
  return c_source;
}

template<class N>
void RPCNeighbor<N>::setDestCatom(N* dest) {
  assert( dest != NULL );
  
  pthread_mutex_lock(&neighborLock);
  c_dest = dest;
  pthread_mutex_unlock(&neighborLock);
}

template<class N>
void RPCNeighbor<N>::setReturnFeat(featureID feat) {
  pthread_mutex_lock(&neighborLock);
  f_return = feat;
  pthread_mutex_unlock(&neighborLock);
}

#endif

