// RPCHoleMotion.cxx
// Nels Beckman
// August 28th, 2007
//
// 2D hole motion with no failure handling.

#include "RPCHoleMotion.hxx"

CODE_MODULE_DECLARATION( RPCHoleMotion, RPCHoleMotion );



// The probability that a hole will collapse in a region that is supposed to
// grow.
const int GROW_REGION_PROB = 100000;
// The probability that a hole will be created in a region that is supposed
// to shrink.
const int SHRINK_REGION_PROB = 100000;

const pair<Point3D, Point3D> SHRINK_REGION_1 = 
  pair<Point3D, Point3D>( Point3D( 0.0, -10.0, 1.0 ), 
			  Point3D( 25.0, 25.0, 1.0 ) );

const pair<Point3D, Point3D> SHRINK_REGION_2 =
  pair<Point3D, Point3D>( Point3D( 25.0, 25.0, 1.0 ),
			  Point3D( 50.0, 60.0, 1.0 ) );

const pair<Point3D, Point3D> GROW_REGION_1 = 
  pair<Point3D, Point3D>( Point3D( 0.0, 40.0, 1.0 ), 
			  Point3D( 25.0, 62.5, 1.0 ) );

const pair<Point3D, Point3D> GROW_REGION_2 = 
  pair<Point3D, Point3D>( Point3D( 25.0, -12.5, 1.0 ),
			  Point3D( 50.0, 10.0, 1.0 ) );

RPCHoleMotion::RPCHoleMotion(catomID _hostCatom) : 
  RPCCodeModule<RPCHoleMotion>(_hostCatom),
  myState(IDLE), myOwner(0), myWaitPeriod(400, 400), busyTicks(0)
{
  // This is busted because DPRSim is.
  if( _hostCatom != 0 )
    catomPtr = &(worldPtr->catomHash[_hostCatom]->C);

  pthread_mutex_init( &localLock, NULL );
}

//void RPCHoleMotion::simulationStart()
//{
//  worldPtr->oStart();
//  cout << "Catom " << hostCatom << " location: " 
//       << catomPtr->getLocation() << endl;
//  worldPtr->oEnd();  
//}

bool inShape( Point3D loc )
{
  return ( loc.getX() > 25.0 && loc.getY() > 25.0 ) ||
    ( loc.getX() < 25.0 && loc.getY() < 25.0 );
}

void RPCHoleMotion::simulationEnd()
{
  RPCCodeModule<RPCHoleMotion>::simulationEnd();

  static int in_compliance = 0;

  worldPtr->oStart();
  if( inShape( catomPtr->getLocation() ) ) {
    in_compliance++;
  }
  cout << in_compliance << " / 700 catoms are wack." << endl;
  worldPtr->oEnd(); 
}

bool inGrowRegion(Point3D loc)
{
  return ( GROW_REGION_1.first.getX() <= loc.getX() && 
	   loc.getX() <= GROW_REGION_1.second.getX() &&
	   GROW_REGION_1.first.getY() <= loc.getY() && 
	   loc.getY() <= GROW_REGION_1.second.getY() ) ||
    ( GROW_REGION_2.first.getX() <= loc.getX() && 
      loc.getX() <= GROW_REGION_2.second.getX() &&
      GROW_REGION_2.first.getY() <= loc.getY() && 
      loc.getY() <= GROW_REGION_2.second.getY() );
}

bool inShrinkRegion(Point3D loc)
{
  return ( SHRINK_REGION_1.first.getX() <= loc.getX() && 
	   loc.getX() <= SHRINK_REGION_1.second.getX() && 
	   SHRINK_REGION_1.first.getY() <= loc.getY() && 
	   loc.getY() <= SHRINK_REGION_1.second.getY() ) ||
    ( SHRINK_REGION_2.first.getX() <= loc.getX() && 
      loc.getX() <= SHRINK_REGION_2.second.getX() && 
      SHRINK_REGION_2.first.getY() <= loc.getY() && 
      loc.getY() <= SHRINK_REGION_2.second.getY() );
}

void wakeMe() {
  catomID c = RPCCodeModuleStatic::getThreadHome();
  RPCCodeModuleStatic::setThreadWake(c);
}

void RPCHoleMotion::thread() {
  for( myWaitPeriod.first = rand() % myWaitPeriod.second;
       myWaitPeriod.first != 0; myWaitPeriod.first-- ){
    wakeMe();
    doRPCYield();
  }
  
  do {
    
    start();
    
//    worldPtr->oStart();
//    cout << "Catom " << hostCatom << " wait period: " 
//	 << myWaitPeriod.first << " / " << myWaitPeriod.second << endl;
//    worldPtr->oEnd();

    if( myWaitPeriod.second > 400 ) 
      myWaitPeriod.second = 400;

    while( myWaitPeriod.first > 0 ){
      wakeMe();
      doRPCYield();
      myWaitPeriod.first--;
    }

  } while(true);
}

featureID rotateFeature( featureID fid, RotDir dir, int places )
{
  int adj_fid = fid - 1;
  places = places % 6;

  switch( dir ) {
  case CCW:
    return (( adj_fid + places ) % 6) + 1;
  case CW:
    places = 6 - places;
    return (( adj_fid + places ) % 6) + 1;
  }

  throw "BUG.";
}

// Getting featureID information from letter information.
featureID letterToFID(Letter my_letter, CatomState cur_state, 
		      Letter letter, 
		      pair<featureID, featureID> empty_side)
{
  switch( my_letter ) {
  case A:
    switch( letter ) {
    case B: return rotateFeature( empty_side.first, CCW, 1 );
    case C: return rotateFeature( empty_side.second, CW, 1 );
    case D: return rotateFeature( empty_side.first, CCW, 2 );
    case E: return rotateFeature( empty_side.second, CW, 2 );
    default:
      throw "Invariant Violated";
    }
  case B:
    switch( letter ) {
    case F: return rotateFeature( empty_side.first, CCW, 1 );
    case G: return rotateFeature( empty_side.first, CCW, 2 );
    case D: return rotateFeature( empty_side.first, CCW, 3 );
    default:
      throw "Invariant Violated";
    }
  case C:
    switch( letter ) {
    case E: return rotateFeature(empty_side.second, CW, 3);
    case K: return rotateFeature(empty_side.second, CW, 2);
    case L: return rotateFeature(empty_side.second, CW, 1);
    default:
      throw "Invariant Violated";
    }
  case D:
    switch( letter ) {
    case H: return rotateFeature( empty_side.first, CCW, 2 );
    case I: return rotateFeature( empty_side.second, CW, 2 );
    default:
      throw "Invariant Violated";
    }
  case E:
    switch( letter ) {
    case J: return rotateFeature( empty_side.second, CW, 2 );
    default:
      throw "Invariant Violated";
    }
  default:
    throw "Invariant Violated";
  }
}

pair<featureID, featureID> RPCHoleMotion::findFullEmptyEmptyFull(void) {
  for(int start_i = 1; start_i <= 6; start_i++ ) {
    int pos_1 = start_i;
    int pos_2 = rotateFeature(start_i, CW, 1);
    int pos_3 = rotateFeature(start_i, CW, 2);
    int pos_4 = rotateFeature(start_i, CW, 3);
    int pos_5 = rotateFeature(start_i, CW, 4);
    int pos_6 = rotateFeature(start_i, CW, 5);

    if( getNeighbor(pos_1) != NULL && getNeighbor(pos_4) != NULL &&
	getNeighbor(pos_2) == NULL && getNeighbor(pos_3) == NULL &&
	getNeighbor(pos_5) != NULL && getNeighbor(pos_6) != NULL    )
      {
	return pair<featureID, featureID>(pos_2, pos_3);
      }
  }

  return pair<featureID, featureID>(0,0);
}

// When in the basic shepherd group shape, this function returns the
//   next feature for going around the circle in a CW direction.
//   It also returns the correct name for the next catom.
//   Make sure you are looking at fig. 2.6 for the right letters.
pair<featureID, Letter> getCWFeature(Letter my_pos, 
				     pair<featureID, featureID> empty_side)
{
  switch( my_pos ) {
  case A:
    return pair<featureID, Letter>(rotateFeature( empty_side.second, CW, 1 ),
				   D);
  case B:
    return pair<featureID, Letter>(empty_side.second, A);
  case C:
    return pair<featureID, Letter>(rotateFeature( empty_side.second, CW, 2 ),
				   L);
  case D:
    return pair<featureID, Letter>(rotateFeature( empty_side.second, CW, 1 ),
				   E);
  case E:
    return pair<featureID, Letter>(rotateFeature( empty_side.second, CW, 2 ),
				   C);
  case F:
    return pair<featureID, Letter>(empty_side.second, B);
  case G:
    return pair<featureID, Letter>(empty_side.first, F);
  case H:
    return pair<featureID, Letter>(empty_side.first, G);
  case I:
    return pair<featureID, Letter>(rotateFeature( empty_side.first, CCW, 1 ),
				   H);
  case J:
    return pair<featureID, Letter>(rotateFeature( empty_side.first, CCW, 1 ),
				   I);
  case K:
    return pair<featureID, Letter>(rotateFeature( empty_side.first, CCW, 2 ),
				   J);
  case L:
    return pair<featureID, Letter>(rotateFeature( empty_side.first, CCW, 2 ),
				   K);
  }

  throw "BUG";
}

// When in the basic shepherd group shape, this function returns the
//   next feature for going around the circle in a CCW direction.
//   Make sure you are looking at fig. 2.6 for the right letters.
pair<featureID, Letter> getCCWFeature(Letter my_pos, pair<featureID, featureID> empty_side)
{
  switch( my_pos ) {
  case A:
    return pair<featureID, Letter>(rotateFeature( empty_side.first, CCW, 2 ),
				   B);
  case B:
    return pair<featureID, Letter>(rotateFeature( empty_side.first, CCW, 2 ),
				   F);
  case C:
    return pair<featureID, Letter>(empty_side.first, E);
  case D:
    return pair<featureID, Letter>(rotateFeature( empty_side.first, CCW, 1 ),
				   A);
  case E:
    return pair<featureID, Letter>(rotateFeature( empty_side.first, CCW, 1 ),
				   D);
  case F:
    return pair<featureID, Letter>(rotateFeature( empty_side.second, CW, 2 ),
				   G);
  case G:
    return pair<featureID, Letter>(rotateFeature( empty_side.second, CW, 2 ),
				   H);
  case H:
    return pair<featureID, Letter>(rotateFeature( empty_side.second, CW, 1 ),
				   I);
  case I:
    return pair<featureID, Letter>(rotateFeature( empty_side.second, CW, 1 ),
				   J);
  case J:
    return pair<featureID, Letter>(empty_side.second, K);
  case K:
    return pair<featureID, Letter>(empty_side.second, L);
  case L:
    return pair<featureID, Letter>(empty_side.first, C);
  }

  throw "BUG";
}

// Given a direction of travel, and a empty_side (which translates
//   specifies a feature to letter mapping) return a list of the
//   leading edge catoms.
bool onLeadingEdge( Letter my_pos, featureID dir, 
		    pair<featureID, featureID> empty_side ) {

  // This is a manner of calculating the offset...
  if( dir == empty_side.first ) {
    //f b a d e
    switch( my_pos ) {
    case F: case B: case A: case D: case E: return true;
    default: return false;
    }
  }
  else if( rotateFeature( dir, CW, 1 ) == empty_side.first ) {
    //h g f b a
    switch( my_pos ) {
    case H: case G: case F: case B: case A: return true;
    default: return false;
    }
  }
  else if( rotateFeature( dir, CW, 2 ) == empty_side.first ) {
    //j i h g f
    switch( my_pos ) {
    case J: case I: case H: case G: case F: return true;
    default: return false;
    }
  }
  else if( rotateFeature( dir, CW, 3 ) == empty_side.first ) {
    //l k j i h
    switch( my_pos ) {
    case L: case K: case J: case I: case H: return true;
    default: return false;
    }
  }
  else if( rotateFeature( dir, CW, 4 ) == empty_side.first ) {
    //e c l k j
    switch( my_pos ) {
    case E: case C: case L: case K: case J: return true;
    default: return false;
    }
  }
  else if( rotateFeature( dir, CW, 5 ) == empty_side.first ) {
    //a d e c l
    switch( my_pos ) {
    case A: case D: case E: case C: case L: return true;
    default: return false;
    }
  }
  
  throw "BUG";
}

// Get the position of the nodes that are immediately adjacent to
// the three catoms that are going to be deleted.
bool isDeletionEndpoint( Letter owner_pos, Letter my_pos, RotDir msg_dir )
{
  switch( owner_pos ) {
  case A:
    return (msg_dir == CW ?
	    my_pos == C || my_pos == I :
	    my_pos == G || my_pos == K);
  case E:
    return (msg_dir == CW ?
	    my_pos == G || my_pos == K :
	    my_pos == B || my_pos == I);
  case F:
    return (msg_dir == CW ? 
	    my_pos == D || my_pos == K :
	    my_pos == I || my_pos == C);
      
  case H:
    return (msg_dir == CW ?
	    my_pos == C || my_pos == B :
	    my_pos == K || my_pos == D);
  case J:
    return (msg_dir == CW ?
	    my_pos == G || my_pos == D :
	    my_pos == B || my_pos == C);
  case L:
    return (msg_dir == CW ?
	    my_pos == B || my_pos == I :
	    my_pos == G || my_pos == D);
  default:
    return false;
  }

  throw "BUG.";
}
				 
// Get the three nodes that need to be on the empty side for the
// hole to collapse. And the features that need to be empty for
// them to succeed.
map<Letter, pair<featureID, featureID> >
getDeletionEmptySide( Letter owner_pos, RotDir msg_dir,
		      pair<featureID, featureID> empty_side )
{
  map<Letter, pair<featureID, featureID> > result;
  
  switch( owner_pos ) {
  case A: 
    if( msg_dir == CW ) {
      featureID f1 = rotateFeature( empty_side.first, CW, 1 );
      featureID f2 = rotateFeature( empty_side.second, CW, 2 );
      pair<featureID, featureID> delete_side(f1,f2);
      result[J] = delete_side;
      result[K] = delete_side;
      result[L] = delete_side;
    }
    else {
      featureID f1 = rotateFeature( empty_side.first, CCW, 2 );
      featureID f2 = rotateFeature( empty_side.second, CW, 2 );
      pair<featureID, featureID> delete_side(f1,f2);
      result[H] = delete_side;
      result[I] = delete_side;
      result[J] = delete_side;      
    }
    break;
  case E:
    if( msg_dir == CW ) {
      featureID f1 = rotateFeature( empty_side.first, CCW, 2 );
      featureID f2 = rotateFeature( empty_side.second, CW, 2 );      
      pair<featureID, featureID> delete_side(f1,f2);
      result[H] = delete_side;
      result[I] = delete_side;
      result[J] = delete_side;            
    }
    else {
      featureID f1 = rotateFeature( empty_side.first, CCW, 1 );
      featureID f2 = rotateFeature( empty_side.second, CCW, 2 );      
      pair<featureID, featureID> delete_side(f1,f2);
      result[F] = delete_side;
      result[G] = delete_side;
      result[H] = delete_side;                  
    }
    break;
  case F:
    if( msg_dir == CW ) {
      featureID f1 = empty_side.second;
      featureID f2 = rotateFeature( empty_side.second, CW, 1 );      
      pair<featureID, featureID> delete_side(f1,f2);
      result[E] = delete_side;
      result[C] = delete_side;
      result[L] = delete_side;     
    }
    else { 
      featureID f1 = rotateFeature( empty_side.first, CW, 1 );
      featureID f2 = rotateFeature( empty_side.second, CW, 2 );     
      pair<featureID, featureID> delete_side(f1,f2);
      result[J] = delete_side;
      result[K] = delete_side;
      result[L] = delete_side;           
    }
    break;
  case H:
    if( msg_dir == CW ) {
      featureID f1 = empty_side.first;
      featureID f2 = empty_side.second;
      pair<featureID, featureID> delete_side(f1,f2);
      result[A] = delete_side;
      result[D] = delete_side;
      result[E] = delete_side;        
    }
    else {
      featureID f1 = empty_side.second;
      featureID f2 = rotateFeature( empty_side.second, CW, 1 );      
      pair<featureID, featureID> delete_side(f1,f2);
      result[E] = delete_side;
      result[C] = delete_side;
      result[L] = delete_side; 
    }
    break;
  case J:
    if( msg_dir == CW ) {
      featureID f1 = empty_side.first;
      featureID f2 = rotateFeature( empty_side.first, CCW, 1 );      
      pair<featureID, featureID> delete_side(f1,f2);
      result[F] = delete_side;
      result[B] = delete_side;
      result[A] = delete_side;      
    }
    else {
      featureID f1 = empty_side.first;
      featureID f2 = empty_side.second;
      pair<featureID, featureID> delete_side(f1,f2);
      result[A] = delete_side;
      result[D] = delete_side;
      result[E] = delete_side; 
    }
    break;
  case L:
    if( msg_dir == CW ) {
      featureID f1 = rotateFeature( empty_side.first, CCW, 1 );
      featureID f2 = rotateFeature( empty_side.second, CCW, 2 );      
      pair<featureID, featureID> delete_side(f1,f2);
      result[F] = delete_side;
      result[G] = delete_side;
      result[H] = delete_side;        
    }
    else {
      featureID f1 = empty_side.first;
      featureID f2 = rotateFeature( empty_side.first, CCW, 1 );      
      pair<featureID, featureID> delete_side(f1,f2);
      result[F] = delete_side;
      result[B] = delete_side;
      result[A] = delete_side;  
    }
    break;
  default:
    assert(false);
  }

  return result;
}

pair<featureID, Letter> getNextCircularFeat( RotDir dir, 
					     Letter my_pos, 
					     pair<featureID, featureID> empty_side )
{
  switch( dir ) {
  case CW:
    return getCWFeature( my_pos, empty_side );
  case CCW:
    return getCCWFeature( my_pos, empty_side );
  }

  throw "BUG.";
}

void RPCHoleMotion::start() {
  
  pthread_mutex_lock( &localLock );
  if( myState == IDLE ) {
    myOwner = hostCatom;
    myState = BUSY;
    pthread_mutex_unlock( &localLock );
    //worldPtr->oStart();
    //cout << "Catom " << hostCatom << " set to BUSY. start() 1" << endl;
    //worldPtr->oEnd();    
  }
  else if( myState == NEW_LEADER ) {
    //worldPtr->oStart();
    //cout << "Catom " << hostCatom << " set to BUSY. start() 2" << endl;
    //worldPtr->oEnd();

    myState = BUSY;
    
    // This is just going here because it's inside the lock.
    myWaitPeriod.first = rand() % myWaitPeriod.second;
    myWaitPeriod.second = 20;

    pthread_mutex_unlock( &localLock );
    // tell your crew you own this piece.
    pair<featureID, Letter> next = 
      getCWFeature( newLeaderInfo.Pos, newLeaderInfo.EmptySide);
    getNeighbor( next.first )->get()->setNewLeaderCircle( next.second,
							  hostCatom,
							  CW,
							  newLeaderInfo.EmptySide);

    // Now we can go on to move another step.
    beginMovement( newLeaderInfo.Pos, newLeaderInfo.Dir, 
		   newLeaderInfo.EmptySide, 5 );
    return;
  }
  else if( myState == SMOOTHING ) {
    pthread_mutex_unlock( &localLock );
    smooth( newLeaderInfo.Pos, newLeaderInfo.EmptySide );

    pthread_mutex_lock( &localLock );
    if( myState != SMOOTHING ) {
      // We actually finished! Back off so others can smooth.
      myWaitPeriod.second = 400;
      myWaitPeriod.first = rand() % myWaitPeriod.second;
    }
    else {
      myWaitPeriod.second = 20;
      myWaitPeriod.first = rand() % myWaitPeriod.second;
    }
    pthread_mutex_unlock( &localLock );
    return;
  }
  else {
    // Should be busy...
    busyTicks++;
    myWaitPeriod.first = rand() % myWaitPeriod.second;
    myWaitPeriod.second = myWaitPeriod.second * 2;
    pthread_mutex_unlock( &localLock );
    if( busyTicks > 20 ) {
      worldPtr->oStart();
      cout << "Catom " << hostCatom << " has been busy for > 20 turns." <<endl;
      worldPtr->oEnd();     
    }
    return;
  }

  // Look for the tell-tale, full, empty, empty, full.
  pair<featureID, featureID> empty_side = findFullEmptyEmptyFull();
  if(empty_side.first == 0) {
    pthread_mutex_lock( &localLock );
    myState = IDLE;
    myOwner = 0;
    myWaitPeriod.first = rand() % myWaitPeriod.second;
    myWaitPeriod.second = myWaitPeriod.second * 2;
    pthread_mutex_unlock( &localLock );    
    return;
  }

  // If we've gotten this far, we are good to create a hole, but let's
  // check to make sure we're not in a shrink region.
  if( !inGrowRegion( catomPtr->getLocation() ) && 
      rand() % SHRINK_REGION_PROB != 0 ) {
    pthread_mutex_lock( &localLock );
    myState = IDLE;
    myOwner = 0;
    myWaitPeriod.first = rand() % myWaitPeriod.second;
    myWaitPeriod.second = myWaitPeriod.second * 2;
    pthread_mutex_unlock( &localLock );    
    return;
  }

  HOSTCATOM.setColor( 255, 0, 0, 0 );

  //Now we go about asking our neighbors.
  featureID b_fid = letterToFID(A, BUSY, B, empty_side);
  featureID c_fid = letterToFID(A, BUSY, C, empty_side);
  featureID d_fid = letterToFID(A, BUSY, D, empty_side);
  featureID e_fid = letterToFID(A, BUSY, E, empty_side);

  list<RPCHoleMotion*> objs;
  list<pair<Letter, featureID> > neighbors_to_do;
  list<pair<Letter, featureID> > neighbors_to_clear;
  
  neighbors_to_do.push_back( pair<Letter, featureID>(B, b_fid) );
  neighbors_to_do.push_back( pair<Letter, featureID>(C, c_fid) );
  neighbors_to_do.push_back( pair<Letter, featureID>(D, d_fid) );
  neighbors_to_do.push_back( pair<Letter, featureID>(E, e_fid) );
  
  while( !neighbors_to_do.empty() ) {
    Letter pos = neighbors_to_do.front().first;
    featureID fid = neighbors_to_do.front().second;
    neighbors_to_do.pop_front();

    RPCHoleMotion* obj = getNeighbor(fid)->get();
    bool success = obj->canCreateHoleTAS( pos, empty_side, hostCatom );

    if( success ) {
      objs.push_back( obj );
      neighbors_to_clear.push_back( pair<Letter, featureID>(pos, fid) );
    }
    else {
      // Failure. Clear all the people we contacted.
      while( !neighbors_to_clear.empty() ) {
	Letter pos = neighbors_to_clear.front().first;
	featureID fid = neighbors_to_clear.front().second;	
	neighbors_to_clear.pop_front();

	getNeighbor(fid)->get()->resetStates(hostCatom, pos, empty_side);
      }

      // Now clear and return
      myWaitPeriod.first = rand() % myWaitPeriod.second;
      myWaitPeriod.second = myWaitPeriod.second * 2;
      reset();
      return;
    }
  }

  assert( neighbors_to_clear.size() == 4 && objs.size() == 4 );

  // If we've made it here we are free to move!
  worldPtr->oStart();
  cout << "Wow. Catom " << hostCatom << " actually got a response." 
       << endl;
  worldPtr->oEnd();

  // Magic move them right now.
  Point3D first_empty_vec = 
    catomPtr->getNeighborCenter(empty_side.first) - 
    catomPtr->getLocation();
  featureID fid_to_left = rotateFeature( empty_side.first, CCW, 1 );
  Point3D second_empty_vec = 
    catomPtr->getNeighborCenter(empty_side.second) - 
    catomPtr->getLocation();
  featureID fid_to_right = rotateFeature( empty_side.second, CW, 1 );
  Point3D new_a_pos( 2*first_empty_vec + catomPtr->getLocation() );
  Point3D new_e_pos( 2*second_empty_vec + catomPtr->getLocation() );
  Point3D new_b_pos( first_empty_vec + 
		     catomPtr->getNeighborCenter(fid_to_left));
  Point3D new_c_pos( second_empty_vec + 
		     catomPtr->getNeighborCenter(fid_to_right));
  Point3D new_d_pos( 2*second_empty_vec + 
		     catomPtr->getNeighborCenter(fid_to_left));

  list<Point3D> poss;
  poss.push_back(new_e_pos);
  poss.push_back(new_b_pos);
  poss.push_back(new_c_pos);
  poss.push_back(new_d_pos);

  // Move them all
  catomPtr->moveTo( new_a_pos, 0, true, false );
  for( ; !poss.empty() && !objs.empty(); poss.pop_front(), objs.pop_front() )
    {
      RPCHoleMotion* obj = objs.front();
      Point3D pos = poss.front();
      
      obj->catomPtr->moveTo( pos, 0, false, objs.size() == 1 );
    }

  // We should choose a random leader for the movement direction.
  if( rand() % 2 == 0 )
    beginMovement( A, rotateFeature( empty_side.second, CW, 2 ),
		   empty_side, 5 );
  else {
    // Chose the E catom to be the leader.
    pair<featureID, Letter> next = getNextCircularFeat( CW, A, empty_side);
    featureID new_dir = rotateFeature( empty_side.first, CCW, 2 );
    catomID new_owner = 
      getNeighbor( next.first )->get()->setNewLeaderCircle2( next.second,
							     E, 0, CW,
							     new_dir,
							     empty_side,
							     10 );
    // Cease being the leader.
    pthread_mutex_lock( &localLock );
    myOwner = new_owner;
    myState = BUSY;
    HOSTCATOM.setColor( 0, 128, 0, 0 );
    pthread_mutex_unlock( &localLock );
    
    //worldPtr->oStart();
    //cout << "Catom " << hostCatom << " set to BUSY. start() 3" << endl;
    //worldPtr->oEnd();
  }

  myWaitPeriod.first = rand() % myWaitPeriod.second;
  myWaitPeriod.second = 20;
  return;

}

void RPCHoleMotion::beginMovement( Letter my_pos, 
				   featureID movement_dir,
				   pair<featureID, featureID> empty_side,
				   int attempts )
{
  RotDir dir;  {
    int _dir = rand() % 2;
    dir = _dir == 0 ? CW : CCW;
  }
  
  pair<featureID, Letter> next =
    getNextCircularFeat( dir, my_pos, empty_side );

  pair<CanMove, list<pair<Letter, Catom*> > > can_move = 
    getNeighbor( next.first )->get()->setLeadingEdge( next.second,
						      movement_dir,
						      dir,
						      empty_side );
						      
  // Assuming all goes well, let's move this piece!
  switch( can_move.first ) {
  case Yes: {
    worldPtr->oStart();
    cout << "Leader catom " << hostCatom 
	 << " about to move hole one step." << endl;
    worldPtr->oEnd();     
    
    // Now we can do our magic move on the resulting objects returned.
    // The middle catom in the list moves three diameters in the opposite
    // direction of the movement vector. The other two move two diameters
    // in the opposite direrection of the movement vector.
    assert( can_move.second.size() == 5 );
    can_move.second.pop_front();
    Catom* c_1 = can_move.second.front().second; can_move.second.pop_front();
    Catom* c_2 = can_move.second.front().second; can_move.second.pop_front();
    Catom* c_3 = can_move.second.front().second; can_move.second.pop_front();
    can_move.second.pop_front();
    assert( can_move.second.size() == 0 );

    Point3D movement_vec = catomPtr->getLocation() -
      catomPtr->getNeighborCenter( movement_dir );

    Point3D new_1_pos( 2*movement_vec + c_1->getLocation() );
    Point3D new_2_pos( 3*movement_vec + c_2->getLocation() );
    Point3D new_3_pos( 2*movement_vec + c_3->getLocation() );

    c_1->moveTo( new_1_pos, 0, true, false );
    c_2->moveTo( new_2_pos, 0, false, false );
    c_3->moveTo( new_3_pos, 0, false, true );

    c_3->setColor( 255, 255, 0, 0 );
    c_2->setColor( 255, 255, 0, 0 );
    c_1->setColor( 255, 255, 0, 0 );

    handOffMovement( my_pos, movement_dir, empty_side );
  }
    // Later make this transfer ownership to the next guy.
    break;
  case Empty:
    // We may still want to bounce if we are in a growing region.
    if( !inShrinkRegion( catomPtr->getLocation() ) &&
	rand() % GROW_REGION_PROB != 0 ) {
      bounce( my_pos, movement_dir, empty_side );
    }
    else {
      collapseHole( my_pos, movement_dir, empty_side );
    }
    break;
  case Bounce:
    if( attempts > 0 )
      beginMovement( my_pos, movement_dir, empty_side, attempts-1 );
    else 
      bounce( my_pos, movement_dir, empty_side );
    break;
  }
  
  return;
}

list<Catom*>
RPCHoleMotion::findThreeEmpty( Letter my_pos, RotDir msg_dir,
			       Letter owner_pos, 
			       pair<featureID, featureID> empty_side,
			       list<Catom*> empty_nodes )
{
  ENTER_RPC_METHOD;
  if( isDeletionEndpoint( owner_pos, my_pos, msg_dir ) ) {
    // We are either the first or second endpoint.

    empty_nodes.push_back( catomPtr );
    if( empty_nodes.size() != 5 ) {
      // We are the first endpoint.
      pair<featureID, Letter> next = getNextCircularFeat( msg_dir, my_pos, 
							  empty_side );
      empty_nodes = 
	getNeighbor( next.first )->get()->findThreeEmpty( next.second,
							  msg_dir,
							  owner_pos,
							  empty_side,
							  empty_nodes );
    }
    // Else, we are the last endpoint.
  }
  else if( getDeletionEmptySide(owner_pos,
				msg_dir,
				empty_side).count( my_pos ) > 0 ) {
    // We are actually one of the middle guys.
    pair<featureID, featureID> check_side = 
      getDeletionEmptySide( owner_pos, msg_dir, empty_side )[my_pos];
    if( (getNeighbor( check_side.first ) == NULL && 
	 getNeighbor( check_side.second ) == NULL && empty_nodes.size() == 2)
	|| (empty_nodes.size() != 2) ) {

      // Now we only make sure check_side is empty if we are middle node.

      empty_nodes.push_back( catomPtr );

      // The collapse might work!
      pair<featureID, Letter> next = getNextCircularFeat( msg_dir, my_pos, 
							  empty_side );
      empty_nodes = 
	getNeighbor( next.first )->get()->findThreeEmpty( next.second,
							  msg_dir,
							  owner_pos,
							  empty_side,
							  empty_nodes );      
    }
    else {
      // Send back an empty list; it won't work.
      empty_nodes = list<Catom*>();
    }
  }
  else {
    // We are just one node on the path towards the side ones.
    pair<featureID, Letter> next = getNextCircularFeat( msg_dir, my_pos, 
							empty_side );
    empty_nodes = 
      getNeighbor( next.first )->get()->findThreeEmpty( next.second,
							msg_dir,
							owner_pos,
							empty_side,
							empty_nodes );
  }

  RPC_RETURN empty_nodes;
}

void RPCHoleMotion::collapseHole( Letter my_pos, featureID move_dir,
				  pair<featureID, featureID> empty_side )
{
  // We want to choose the side randomly.
  RotDir dir = rand() % 2 == 0 ? CW : CCW;

  // Just means try CW first...
  pair<featureID, Letter> next = getNextCircularFeat( dir, my_pos,
						      empty_side );
  list<Catom*> nodes =
    getNeighbor( next.first )->get()->findThreeEmpty( next.second, dir,
						      my_pos,
						      empty_side,
						      list<Catom*>() );
    
  if( nodes.size() == 0 ) {
    dir = (dir == CW ? CCW : CW);
    next = getNextCircularFeat( dir, my_pos,
				empty_side );
    nodes =
      getNeighbor( next.first )->get()->findThreeEmpty( next.second, dir,
							my_pos,
							empty_side,
							list<Catom*>() );
  }

  if( nodes.size() == 0 ) {
    // We can't collapse that way. Bounce for now...
  }
  else { 
    assert( nodes.size() == 5 );

    worldPtr->oStart();
    cout << "Catom " << hostCatom << " collapsing hole." << endl;
    worldPtr->oEnd();

    // Time to move. First figure out the 5 positions.
    Point3D leader_pos = catomPtr->getLocation();
    Point3D md_vec = catomPtr->getNeighborCenter( move_dir ) - leader_pos;
    Point3D mr_vec = catomPtr->getNeighborCenter( rotateFeature( move_dir,
								 CCW, 1 ) ) -
      leader_pos;
    Point3D ml_vec = catomPtr->getNeighborCenter( rotateFeature( move_dir,
								 CW, 1 ) ) -
      leader_pos;
    
    Point3D pos_1 = leader_pos + md_vec;
    Point3D pos_2 = leader_pos + md_vec + mr_vec;
    Point3D pos_3 = leader_pos + md_vec + ml_vec;
    Point3D pos_4 = leader_pos + (2*md_vec);
    Point3D pos_5 = leader_pos + (2*md_vec) + (dir == CCW ? mr_vec : ml_vec);

    nodes.front()->moveTo( pos_1, 0, true, false ); nodes.pop_front();
    nodes.front()->moveTo( pos_2, 0, false, false ); nodes.pop_front();
    nodes.front()->moveTo( pos_3, 0, false, false ); nodes.pop_front();
    nodes.front()->moveTo( pos_4, 0, false, false ); nodes.pop_front();
    nodes.front()->moveTo( pos_5, 0, false, true ); nodes.pop_front();

    // Now we have to release these guys...
    pair<featureID, Letter> next_cw = 
      getNextCircularFeat( CW, my_pos, empty_side );
    pair<featureID, Letter> next_ccw = 
      getNextCircularFeat( CCW, my_pos, empty_side );
    getNeighbor( move_dir )->get()->resetCollapsed( hostCatom, G,
						    move_dir, dir, true,
						    empty_side );
    getNeighbor( next_cw.first )->get()->resetCollapsed( hostCatom, 
							 next_cw.second,
							 move_dir, CW, false,
							 empty_side );
    getNeighbor( next_ccw.first )->get()->resetCollapsed( hostCatom,
							  next_ccw.second,
							  move_dir, CCW, false,
							  empty_side );
    // And we have to release ourself.
    reset();
    return;
  }
  
  bounce( my_pos, move_dir, empty_side );
}

void RPCHoleMotion::setNewLeaderCircle( Letter my_pos, catomID owner, 
					RotDir msg_dir,
					pair<featureID, featureID> empty_side)
{
  ENTER_RPC_METHOD;
  pthread_mutex_lock( &localLock );
  if( myOwner == owner || hostCatom == owner ) {
    // we have made it around the circle.
    pthread_mutex_unlock( &localLock );
  }
  else {
    myOwner = owner;
    HOSTCATOM.setColor( 0, 128, 0, 0 );
    pthread_mutex_unlock( &localLock );
    pair<featureID, Letter> next = 
      getNextCircularFeat( msg_dir, my_pos, empty_side );
    getNeighbor( next.first )->get()->setNewLeaderCircle( next.second,
							  owner,
							  msg_dir,
							  empty_side );
  }
  
  RPC_RETURN;
}

// Okay, this method will go around in a circle setting a new leader,
// but it will only stop when nodes_left == 0.
// If you are the leader, then we will also set our state to NEW_LEADER.
catomID RPCHoleMotion::setNewLeaderCircle2( Letter my_pos, Letter owner_pos,
					    catomID owner_id,
					    RotDir msg_dir, featureID move_dir,
					    pair<featureID, featureID> empty_side,
					    int nodes_left )
{
  ENTER_RPC_METHOD;
  
  pthread_mutex_lock( &localLock );
  HOSTCATOM.setColor( 0, 128, 0, 0 );
  if( my_pos == owner_pos ) {
    owner_id = hostCatom;
  }
  pthread_mutex_unlock( &localLock );

  if( nodes_left > 0 ) {
    pair<featureID, Letter> next = getNextCircularFeat( msg_dir, my_pos,
							empty_side );
    owner_id = 
      getNeighbor( next.first )->get()->setNewLeaderCircle2( next.second,
							     owner_pos,
							     owner_id,
							     msg_dir,
							     move_dir,
							     empty_side,
							     nodes_left-1 );
  }

  pthread_mutex_lock( &localLock );
  myOwner = owner_id;
  
  if( owner_id == hostCatom ) {
    myOwner = hostCatom;
    myState = NEW_LEADER;
    HOSTCATOM.setColor( 255, 0, 0, 0 );
    newLeaderInfo.Pos = my_pos;
    newLeaderInfo.Dir = move_dir;
    newLeaderInfo.EmptySide = empty_side;
    myWaitPeriod.first = 20; // set to 30 so the entire circle can know first.
    myWaitPeriod.second = 20;
  }
  pthread_mutex_unlock( &localLock );


  RPC_RETURN owner_id;
}

void RPCHoleMotion::resetEdge(catomID owner, featureID fid_to_tell, int count)
{
  ENTER_RPC_METHOD;

  pthread_mutex_lock( &localLock );
  if( owner == myOwner ) {
    pthread_mutex_unlock( &localLock );

    if( count > 0 && getNeighbor( fid_to_tell ) != NULL ) {
      getNeighbor( fid_to_tell )->get()->resetEdge( owner, 
						    fid_to_tell,
						    count-1 );
    }

    reset();
  }
  else {
    pthread_mutex_unlock( &localLock );
  }

  RPC_RETURN;
}

void RPCHoleMotion::handOffMovement(Letter my_pos, featureID dir,
				    pair<featureID, featureID> empty_side)
{
  // First we need to clear the back edge.
  featureID left_side = rotateFeature( dir, CW, 1 );
  featureID right_side = rotateFeature( dir, CCW, 1 );
  getNeighbor( left_side )->get()->resetEdge( hostCatom, left_side, 1 );
  getNeighbor( right_side )->get()->resetEdge( hostCatom, right_side, 1 );

  // Then we need to set the "New Leader" flag on the next catom.
  if( getNeighbor( dir ) == NULL ) {
    throw "Invariant violated.";
  }
  else {
    getNeighbor( dir )->get()->setNewLeader( my_pos, dir, empty_side );
  }
  
  // If we just started the hole, we want to smooth the edge.
  if( !getNeighbor(empty_side.first) && !getNeighbor(empty_side.second) ) {
    pthread_mutex_lock( &localLock );
    myState = SMOOTHING;
    newLeaderInfo.Pos = my_pos; 
    newLeaderInfo.EmptySide = empty_side;
    pthread_mutex_unlock( &localLock );    
  }
  else {
    // Finally clear ourselves, and return.
    reset();
  }
}

void RPCHoleMotion::smooth( Letter my_pos, 
			    pair<featureID, featureID> empty_side )
{
  pair<list<Point3D>, pair<list<Catom*>, list<pair<featureID, featureID> > > >
    move_info;

  worldPtr->oStart();
  cout << "Catom " << hostCatom << " called smooth." << endl;
  worldPtr->oEnd();

  if( my_pos == A ) {
    featureID mover_fid = rotateFeature( empty_side.second, CW, 1 );
    featureID ph_fid = rotateFeature( empty_side.first, CCW, 2 );

    // Check to make sure we should still even be smoothing.
    // There was a problem earlier with the leader of the SECOND
    // hole trying to smooth!
    featureID left = rotateFeature( empty_side.first, CCW, 1 );
    if( getNeighbor( mover_fid ) && getNeighbor( ph_fid ) &&
	!getNeighbor(left) && !getNeighbor(empty_side.first) &&
	!getNeighbor(empty_side.second) ) {
      
      list<Catom*> movers = 
	getNeighbor( mover_fid )->get()->smoothEdgeMover( 1, hostCatom, 
							 mover_fid );
      if( movers.size() == 0 )
	return;
      
      move_info = 
	getNeighbor( ph_fid )->get()->smoothEdgePlaceholder(0, 3, 0, hostCatom,
							    empty_side);
      move_info.second.first = movers;
      if( move_info.first.size() == 0 ) {
	// We have to cancel the first one and return.
	getNeighbor( mover_fid )->get()->resetFailedSEMover( 1, hostCatom,
							     mover_fid);
	return;
      }
    }
    else {
      // Strange configuration... We'll cancel because we're not in the
      // right position any more.
      worldPtr->oStart();
      cout << "Smoother exiting early. " << endl;
      worldPtr->oEnd();
      reset();
      return;
    }
  }
  else {
    // We are in the E position.
    featureID ph_fid = rotateFeature( empty_side.first, CCW, 1 );
    featureID right = rotateFeature( empty_side.second, CW, 1 );
    featureID down_right = rotateFeature( empty_side.second, CW, 2 );
    featureID down_left = rotateFeature( empty_side.first, CCW, 2 );

    if( getNeighbor( ph_fid ) && getNeighbor(down_right) &&
	getNeighbor(down_left) && !getNeighbor(right) &&
	!getNeighbor(empty_side.first) && !getNeighbor(empty_side.second) ) {
      move_info = 
	getNeighbor(ph_fid)->get()->smoothEdgePlaceholder(-1, 3, 2, hostCatom,
							  empty_side);
      if( move_info.first.size() == 0 ) {
	// Everything has been cancelled.
	return;
      }
    }
    else {
      worldPtr->oStart();
      cout << "Smoother exiting early. " << endl;
      worldPtr->oEnd();
      reset();
      return;
    }
  }
  
  // So at this point, we know that it was a success, and everything is in
  // move_info. Just move, and clear.
  assert( move_info.first.size() == 3 );
  assert( move_info.second.first.size() == 2 );

  worldPtr->oStart();
  cout << "Catom smoothing." << endl;
  worldPtr->oEnd();

  // Just move two of the catoms.
  for( int i = 0; i < 2; i++ ) {
    Catom* ptr = move_info.second.first.front(); 
    move_info.second.first.pop_front();
    Point3D loc = move_info.first.front();
    move_info.first.pop_front();

    ptr->moveTo( loc, 0, (i==0), (my_pos == A ? (i==1) : false) );
  }

  // Now clear other catoms, and clear yourself.
  if( my_pos == A ) {
    getNeighbor( rotateFeature( empty_side.first, CCW, 2 ) )->get()
      ->resetSmoothCatoms( hostCatom, 3, move_info.second.second );
  }
  else {
    // Oh man this is gross. We've already moved the other two. So there
    // is a disconnected path. We kind of have to move two places over now.
    featureID left = rotateFeature( empty_side.first, CCW, 1 );
    featureID down_left = rotateFeature( empty_side.first, CCW, 2 );
    Point3D loc = catomPtr->getNeighborCenter( left );
    featureID next = down_left;

    catomPtr->moveTo( loc, 0, false, false );
    assert( move_info.second.second.back().second == 0 );
    move_info.second.second.pop_back();
    
    loc = catomPtr->getNeighborCenter( left );
    catomPtr->moveTo( loc, 0, false, true );
    assert( move_info.second.second.back().second == 0 );
    next = move_info.second.second.back().first;
    move_info.second.second.pop_back();
    
    getNeighbor( next )->get()
      ->resetSmoothCatoms( hostCatom, 3, move_info.second.second );
  }

  worldPtr->oStart();
  cout << "Catom smoothing; finished resetSmoothCatoms." << endl;
  worldPtr->oEnd();

  // Um, now move. This is an obvious race condition / shortcut.
  catomPtr->moveTo( move_info.first.front(), 0, true, true );

  reset();
}

list<Catom*> RPCHoleMotion::smoothEdgeMover( int remains, catomID owner,
					     featureID next )
{
  ENTER_RPC_METHOD;
  list<Catom*> result;

  worldPtr->oStart();
  cout << "smoothEdgeMover." << endl;
  worldPtr->oEnd();

  pthread_mutex_lock( &localLock );
  if( myState == IDLE ) {
    myState = BUSY;
    myOwner = owner;
    HOSTCATOM.setColor( 128, 0, 128, 0 );
    pthread_mutex_unlock( &localLock );    

    //worldPtr->oStart();
    //cout << "Catom " << hostCatom << " set to BUSY. smoothEdgeMover" << endl;
    //worldPtr->oEnd();
  }
  else {
    pthread_mutex_unlock( &localLock );
    RPC_RETURN result;
  }
  
  if( remains > 0 ) {
    if( getNeighbor( next ) ) {
      result = getNeighbor(next)->get()
	->smoothEdgeMover(remains-1, owner, next);
      if( result.size() > 0 )
	result.push_back( catomPtr );
    }
  }
  else {
    result.push_back( catomPtr );
  }

  if( result.size() == 0 ) {
    // It didn't work...
    reset();
  }
  
  RPC_RETURN result;
}

featureID getArrivalFid( featureID prev_out ) {
  featureID result = 0;

  switch( prev_out ) {
  case 1: 
    result = 4; break;
  case 2: 
    result = 5; break;
  case 3: 
    result = 6; break;
  case 4: 
    result = 1; break;
  case 5:
    result = 2; break;
  case 6: 
    result = 3; break;
  default:
    assert(false);
  }
  
  return result;
}

// There are uglier methods in this program, and that really scares me.
pair<list<Point3D>, pair<list<Catom*>, list<pair<featureID, featureID> > > >
RPCHoleMotion::smoothEdgePlaceholder( int steps_b_g, int holes_needed,
				      int need_movers,
				      catomID owner,
				      pair<featureID,featureID> empty_side )
{
  ENTER_RPC_METHOD;

  worldPtr->oStart();
  cout << "smoothEdgePlaceholder." << endl;
  worldPtr->oEnd();

  featureID left = rotateFeature( empty_side.first, CCW, 1 );
  featureID right = rotateFeature( empty_side.second, CW, 1 );
  featureID down_left = rotateFeature( empty_side.first, CCW, 2 );
  featureID down_right = rotateFeature( empty_side.first, CCW, 3 );
  featureID up_left = empty_side.first;
  featureID up_right = empty_side.second;

  pair<list<Point3D>, pair<list<Catom*>, list<pair<featureID, featureID> > > >
    result;
  pair<featureID, featureID> trail(0,0);

  pthread_mutex_lock( &localLock );
  if( myState == IDLE ) {
    myState = BUSY;
    myOwner = owner; 
    HOSTCATOM.setColor( 128, 0, 128, 0 );
    pthread_mutex_unlock( &localLock );

    //worldPtr->oStart();
    //cout << "Catom " << hostCatom << " set to BUSY. smoothEdgePH" << endl;
    //worldPtr->oEnd();
  }
  else {
    pthread_mutex_unlock( &localLock );
    RPC_RETURN result;
  }

  // Sometimes we need movers, that changes everything we do.
  // Grab them and move on to the next home.
  if( need_movers == 2 ) {
    if( getNeighbor( left ) ) {
      result = getNeighbor( left )->get()
	->smoothEdgePlaceholder( steps_b_g,
				 holes_needed,
				 need_movers-1,
				 owner, empty_side );
      if( result.first.size() == 0 ) {
	reset();
	RPC_RETURN result;	
      }
      else {
	result.second.first.push_back( catomPtr );
	result.second.second.push_back( pair<featureID, featureID>(left, 0) );
	RPC_RETURN result;
      }
    }
    else {
      assert(false);
    }
  }
  else if( need_movers == 1 ) {
    featureID next;
    if( getNeighbor( left ) ) {
      next = left;
      steps_b_g = steps_b_g;
    }
    else if( getNeighbor( down_left ) ) {
      next = down_left;
      steps_b_g = steps_b_g + 1;
    }
    else {
      assert( false );
    }

    result = getNeighbor(next)->get()->smoothEdgePlaceholder( steps_b_g,
							      holes_needed,
							      need_movers-1,
							      owner, 
							      empty_side );
    if( result.first.size() == 0 ) {
      reset();
      RPC_RETURN result;	
    }
    else {
      result.second.first.push_back( catomPtr );
      result.second.second.push_back( pair<featureID, featureID>(next, 0) );
      RPC_RETURN result;
    }
    
  }

  if( !getNeighbor( up_right ) && !getNeighbor( up_left ) &&
      getNeighbor( left ) && getNeighbor( right ) ) {
    // STRAIGHT LINE
    worldPtr->oStart();
    cout << "STRAIGHT LINE" << endl;
    worldPtr->oEnd();
    trail.first = left;
    if( steps_b_g > 0 ) 
      trail.second = up_right;
  }
  else if( !getNeighbor( up_right ) && !getNeighbor( up_left ) &&
	   getNeighbor( right ) && !getNeighbor(left) &&
	   getNeighbor( down_left ) ) {
    // CURVING DOWN
    worldPtr->oStart();
    cout << "CURVING DOWN" << endl;
    worldPtr->oEnd();
    trail.first = down_left;
    if( steps_b_g > 0 )
      trail.second = up_right;
  }
  else if( !getNeighbor( left ) && !getNeighbor( up_left ) &&
	   getNeighbor( down_left ) && getNeighbor( up_right ) ) {
    // DIAG LINE GOING DOWN AND LEFT
    worldPtr->oStart();
    cout << "DIAG D & L" << endl;
    worldPtr->oEnd();
    trail.first = down_left;
    if( steps_b_g > 0 )
      trail.second = up_left;
  }
  else if( !getNeighbor( left ) && !getNeighbor( down_left ) &&
	   getNeighbor( up_left ) && getNeighbor( up_right) &&
	   getNeighbor( right ) && getNeighbor( down_right ) ) {
    // DIAG LINE GOING DOWN AND RIGHT.
    trail.first = down_right;
    if( steps_b_g > 0 )
      trail.second = left;
  }
  else if( !getNeighbor( up_left ) && !getNeighbor( left ) && 
	   !getNeighbor( down_left ) && !getNeighbor( down_right ) &&
	   getNeighbor( right ) && getNeighbor( up_right ) ) {

    trail.first = right;
    if( steps_b_g > 0 )
      trail.second= up_left;
  }
  else if( !getNeighbor( up_left ) && getNeighbor( up_right ) &&
	   getNeighbor( left ) ) {
    // CORNER 1
    worldPtr->oStart();
    cout << "CORNER 1" << endl;
    worldPtr->oEnd();
    trail.first = left;
  }
  else if( !getNeighbor( up_right ) && getNeighbor( right ) &&
	   getNeighbor( up_left ) ) {
    // CORNER 2 
    worldPtr->oStart();
    cout << "CORNER 2" << endl;
    worldPtr->oEnd();
    trail.first = up_left;
  }
  else if( !getNeighbor( right ) && getNeighbor( down_right ) ) {
    // CORNER 3
    worldPtr->oStart();
    cout << "CORNER 3" << endl;
    worldPtr->oEnd();
    if( steps_b_g > 0 )
      trail.second = right;

    if( getNeighbor( up_left ) ) {
      trail.first = up_left;
    }
    else if( getNeighbor( left ) ) {
      trail.first = left;
    }
    else if( getNeighbor( down_left ) ) {
      trail.first = down_left;
    }
    else {
      worldPtr->oStart();
      cout << "Unrecognized shape: " << (getNeighbor(left) ? "LEFT " : "____ ")
	   << (getNeighbor(up_left) ? "UP_LEFT " : "____ ")
	   << (getNeighbor(up_right) ? "UP_RIGHT " : "____ ")
	   << (getNeighbor(right) ? "RIGHT " : "____ ")
	   << (getNeighbor(down_right) ? "D_RIGHT " : "____ ")
	   << (getNeighbor(down_left) ? "D_LEFT " : "____ ") << endl;
      worldPtr->oEnd();  
      reset();
      RPC_RETURN result;
    }
  }
  else if( !getNeighbor( up_right ) && !getNeighbor( up_left ) &&
	   !getNeighbor( left ) && !getNeighbor( down_left ) &&
	   getNeighbor( right ) && getNeighbor( down_right ) ) {
    // SIDE CORNER THING
    worldPtr->oStart();
    cout << "SIDE CORNER THING" << endl;
    worldPtr->oEnd();    

    trail.first = down_right;
    if( steps_b_g > 0 )
      trail.second = up_right;    
  }
//  else if( !getNeighbor( left ) && !getNeighbor( down_left ) &&
//	   getNeighbor( down_right ) && getNeighbor( right ) ) {
//    //SIDE 1
//    worldPtr->oStart();
//    cout << "SIDE 1" << endl;
//    worldPtr->oEnd();
//    trail.first = down_right;
//  }
  else if( !getNeighbor( left ) && getNeighbor( up_left ) &&
	   getNeighbor( down_left ) ) {
    //SIDE 2
    worldPtr->oStart();
    cout << "SIDE 2" << endl;
    worldPtr->oEnd();
    trail.first = down_left;
  }
  else if( !getNeighbor( up_left ) && getNeighbor( up_right ) &&
	   getNeighbor( down_right ) ) {
    //SIDE 3
    worldPtr->oStart();
    cout << "SIDE 3" << endl;
    worldPtr->oEnd();
    trail.first = down_right;

    if( steps_b_g > 0 )
      trail.second = up_left;
  }
  else {
    worldPtr->oStart();
    cout << "Unrecognized shape: " << (getNeighbor(left) ? "LEFT " : "____ ")
	 << (getNeighbor(up_left) ? "UP_LEFT " : "____ ")
	 << (getNeighbor(up_right) ? "UP_RIGHT " : "____ ")
	 << (getNeighbor(right) ? "RIGHT " : "____ ")
	 << (getNeighbor(down_right) ? "D_RIGHT " : "____ ")
	 << (getNeighbor(down_left) ? "D_LEFT " : "____ ") << endl;
    worldPtr->oEnd();
    reset();
    RPC_RETURN result;
  }
  
  // Calculate new steps_b_g
  steps_b_g = ( (trail.first == left || trail.first == right) ?
		steps_b_g : ( (trail.first == down_left || 
			       trail.first == down_right) ?
			      steps_b_g + 1 : steps_b_g - 1) );

  holes_needed =  holes_needed - (trail.second != 0 ? 1 : 0);
  
  // Make the next call if we still need more holes.
  if( holes_needed > 0 ) {
    result =
      getNeighbor( trail.first )->get()->smoothEdgePlaceholder(steps_b_g,
							       holes_needed,
							       need_movers,
							       owner,
							       empty_side);
    // If we needed more holes but didn't get them, failure...
    // This could only happen if we run into a BUSY catom.
    if( result.first.size() == 0 ) {
      reset();
      RPC_RETURN result;
    }
  }
  
  // Must be pushed back no matter what.
  result.second.second.push_back( trail );

  // We reserved a space...
  if( trail.second != 0 ) 
    result.first.push_back( catomPtr->getNeighborCenter( trail.second ) );


  worldPtr->oStart();
  cout << "Finished smoothEdgePlaceholder." << endl;
  worldPtr->oEnd();
  
  RPC_RETURN result;
}

void RPCHoleMotion::reset()
{
  pthread_mutex_lock( &localLock );
  myState = IDLE;
  myOwner = 0;
  HOSTCATOM.setColor( 128, 128, 128, 0 );
  busyTicks = 0;
  pthread_mutex_unlock( &localLock ); 
}

void RPCHoleMotion::resetSmoothCatoms( catomID owner, int remains,
				       list<pair<featureID, featureID> > trail)
{
  ENTER_RPC_METHOD;

  worldPtr->oStart();
  cout << "resetSmoothCatoms." << endl;
  worldPtr->oEnd();

  pthread_mutex_lock( &localLock );
  assert( owner == myOwner );
  pthread_mutex_unlock( &localLock );

  assert(!trail.empty());
  pair<featureID, featureID> t = trail.back(); trail.pop_back();
  if( t.second != 0 ) {
    remains--;
    // Reset moved catom, if it was not the leader.
    if( getNeighbor( t.second ) )
      getNeighbor(t.second)->get()->resetState( owner );
    else {
      // This could indicate an error, but the space that the leader goes
      // should remain empty.
      worldPtr->oStart();
      cout << "Catom missing. Hope it's the leader's spot." << endl;
      worldPtr->oEnd();     
    }
  }

  if( remains > 0 ) {
    // Reset along the trail.
    assert( t.first != 0 );
    getNeighbor( t.first )->get()->resetSmoothCatoms( owner, remains,
						      trail );
  }
  
  reset();

  RPC_RETURN;
}

void RPCHoleMotion::resetFailedSEMover( int remains, catomID owner, 
					featureID next )
{
  ENTER_RPC_METHOD;

  worldPtr->oStart();
  cout << "resetFailedSEMover." << endl;
  worldPtr->oEnd();

  pthread_mutex_lock( &localLock );
  if( myOwner != owner ) {
    pthread_mutex_unlock( &localLock );
    RPC_RETURN;
  }
  else {
    pthread_mutex_unlock( &localLock );
  }
  
  if( remains > 0 && getNeighbor( next ) != NULL ) 
    getNeighbor(next)->get()->resetFailedSEMover( remains-1, owner, next );
  
  reset();
  RPC_RETURN;
}

// Given the current movement direction, and the empty_side for alignment,
// this function returns a new random direction and the letter of the node
// who would be the new leader.
pair<featureID, Letter> newRandomDir( featureID move_dir,
				      pair<featureID, featureID> empty_side )
{
  // With equal prob, rotate move_dir 2, 3, or 4 spaces. 
  int chance = rand() % 5;
  featureID new_dir = 0;

  switch( chance ) {
  case 0: new_dir = rotateFeature( move_dir, CW, 2 ); break;
  case 1: new_dir = rotateFeature( move_dir, CW, 3 ); break;
  case 2: new_dir = rotateFeature( move_dir, CW, 4 ); break;
  case 3: new_dir = rotateFeature( move_dir, CW, 5 ); break;
  case 4: new_dir = rotateFeature( move_dir, CW, 1 ); break;
  }
  
  // We can determine who the leader should be based on how many rotations
  // off from the empty side the new feature is.
  Letter new_leader = A;
  if( new_dir == empty_side.first ) 
    new_leader = J;
  else if( new_dir == empty_side.second ) 
    new_leader = H;
  else if( new_dir == rotateFeature( empty_side.second, CW, 1 ) ) 
    new_leader = F;
  else if( new_dir == rotateFeature( empty_side.second, CW, 2 ) )
    new_leader = A;
  else if( new_dir == rotateFeature( empty_side.second, CW, 3 ) )
    new_leader = E;
  else if( new_dir == rotateFeature( empty_side.second, CW, 4 ) )
    new_leader = L;

  return pair<featureID, Letter>(new_dir, new_leader);
}

void RPCHoleMotion::bounce( Letter my_pos, featureID move_dir, 
			    pair<featureID, featureID> empty_side )
{

  worldPtr->oStart();
  cout << "Whoa dude. Catom " << hostCatom << " is bouncing." << endl;
  worldPtr->oEnd();

  // Randomly choose one of the three opposing directions.
  pair<featureID, Letter> new_leader = newRandomDir( move_dir, empty_side );
  
  // Hand off new leader.
  pair<featureID, Letter> next = getNextCircularFeat( CW, my_pos, empty_side );

  catomID new_owner =
  getNeighbor( next.first )->get()->setNewLeaderCircle2( next.second,
							 new_leader.second,
							 0, CW, 
							 new_leader.first,
							 empty_side,
							 10 );

  // Cease being the leader.
  pthread_mutex_lock( &localLock );
  myOwner = new_owner;
  myState = BUSY;
  HOSTCATOM.setColor( 0, 128, 0, 0 );
  pthread_mutex_unlock( &localLock );

  //worldPtr->oStart();
  //cout << "Catom " << hostCatom << " set to BUSY. bounce()" << endl;
  //worldPtr->oEnd();
}

void RPCHoleMotion::setNewLeader( Letter my_pos, featureID move_dir,
				  pair<featureID, featureID> empty_side )
{
  ENTER_RPC_METHOD;

  pthread_mutex_lock( &localLock );
  myState = NEW_LEADER;
  myOwner = hostCatom;
  HOSTCATOM.setColor( 255, 0, 0, 0 );
  newLeaderInfo.Pos = my_pos;
  newLeaderInfo.Dir = move_dir;
  newLeaderInfo.EmptySide = empty_side;
  myWaitPeriod.first = 4;
  myWaitPeriod.second = 20;
  pthread_mutex_unlock( &localLock );

  RPC_RETURN;
}

bool RPCHoleMotion::canCreateHoleTAS( Letter my_pos, 
				      pair<featureID, featureID> empty_side,
				      catomID owner) 
{
  ENTER_RPC_METHOD;

  pthread_mutex_lock( &localLock );
  if( myState == IDLE ) {
    myOwner = owner;
    myState = BUSY;
    pthread_mutex_unlock( &localLock );

    //worldPtr->oStart();
    //cout << "Catom " << hostCatom << " set to BUSY. canCreateHoleTAS" << endl;
    //worldPtr->oEnd();
  }
  else {
    pthread_mutex_unlock( &localLock );
    RPC_RETURN false;
  }

  bool to_return = true;
  switch( my_pos ) {
  case A: throw "Invariant violated.";
  case B: {
    featureID f_fid = letterToFID(B, BUSY, F, empty_side);
    featureID g_fid = letterToFID(B, BUSY, G, empty_side);
    to_return = 
      !getNeighbor( empty_side.first ) && 
      !getNeighbor( empty_side.second ) &&
      getNeighbor( f_fid ) && getNeighbor( g_fid );

    if( to_return ) {
      // We need to clear right away if things go sour.
      to_return =
	getNeighbor( f_fid )->get()->canCreateHoleTAS(F, empty_side, owner);
      
      if( to_return ) {
	to_return =
	  getNeighbor( g_fid )->get()->canCreateHoleTAS(G, empty_side, owner);
	
	if( !to_return ) {
	  getNeighbor( f_fid )->get()->resetStates( owner, F, empty_side );
	}
      }
    } 
  }
    break;

  case C: {
    featureID k_fid = letterToFID(C, BUSY, K, empty_side);
    featureID l_fid = letterToFID(C, BUSY, L, empty_side);
    to_return =
      !getNeighbor( empty_side.first ) && 
      !getNeighbor( empty_side.second ) &&
      getNeighbor( k_fid ) && getNeighbor( l_fid );

    if( to_return ) {
      // We need to clear right away if things go sour.
      to_return =
	getNeighbor( k_fid )->get()->canCreateHoleTAS(K, empty_side, owner);
      
      if( to_return ) {
	to_return =
	  getNeighbor( l_fid )->get()->canCreateHoleTAS(L, empty_side, owner);
	
	if( !to_return ) {
	  getNeighbor( k_fid )->get()->resetStates( owner, K, empty_side );
	}
      }
    }
  }
    break;

  case D: {
    featureID h_fid = letterToFID(D, BUSY, H, empty_side);
    featureID i_fid = letterToFID(D, BUSY, I, empty_side);
    to_return =
      getNeighbor( h_fid ) && getNeighbor( i_fid );

    if( to_return ) {
      // We need to clear right away if things go sour.
      to_return =
	getNeighbor( h_fid )->get()->canCreateHoleTAS(H, empty_side, owner);
      
      if( to_return ) {
	to_return =
	  getNeighbor( i_fid )->get()->canCreateHoleTAS(I, empty_side, owner);
	
	if( !to_return ) {
	  getNeighbor( h_fid )->get()->resetStates( owner, H, empty_side );
	}
      }
    }
  }
    break;

  case E: {
    featureID j_fid = letterToFID(E, BUSY, J, empty_side);
    to_return =
      getNeighbor( j_fid ) &&
      getNeighbor( j_fid )->get()->canCreateHoleTAS(J, empty_side, owner);

  }
    break;

  case F:
    to_return = !getNeighbor( empty_side.second );
    break;
  case L:
    to_return = !getNeighbor( empty_side.first );
    break;

  case G: case H: case I: case J: case K: to_return = true; break;

  }

  if( !to_return ) {
    reset();
  }
  else {
    HOSTCATOM.setColor( 0, 128, 0, 0 );
  }

  RPC_RETURN to_return;
}

bool amILastEdgeCatom( Letter my_pos, RotDir msg_dir, featureID move_dir,
		       pair<featureID, featureID> empty_side )
{
  // This is a manner of calculating the offset...
  if( move_dir == empty_side.first ) {
    //f b a d e
    switch( my_pos ) {
    case F: return msg_dir == CCW;
    case E: return msg_dir == CW;
    default: return false;
    }
  }
  else if( rotateFeature( move_dir, CW, 1 ) == empty_side.first ) {
    //h g f b a
    switch( my_pos ) {
    case H: return msg_dir == CCW;
    case A: return msg_dir == CW;
    default: return false;
    }
  }
  else if( rotateFeature( move_dir, CW, 2 ) == empty_side.first ) {
    //j i h g f
    switch( my_pos ) {
    case J: return msg_dir == CCW;
    case F: return msg_dir == CW;
    default: return false;
    }
  }
  else if( rotateFeature( move_dir, CW, 3 ) == empty_side.first ) {
    //l k j i h
    switch( my_pos ) {
    case L: return msg_dir == CCW;
    case H: return msg_dir == CW;
    default: return false;
    }
  }
  else if( rotateFeature( move_dir, CW, 4 ) == empty_side.first ) {
    //e c l k j
    switch( my_pos ) {
    case E: return msg_dir == CCW;
    case J: return msg_dir == CW;
    default: return false;
    }
  }
  else if( rotateFeature( move_dir, CW, 5 ) == empty_side.first ) {
    //a d e c l
    switch( my_pos ) {
    case A: return msg_dir == CCW;
    case L: return msg_dir == CW;
    default: return false;
    }
  }
  
  throw "BUG";  
}

pair<CanMove, list<pair<Letter, Catom*> > > 
RPCHoleMotion::setLeadingEdge( Letter my_pos, featureID move_dir, 
			       RotDir msg_dir, 
			       pair<featureID, featureID> empty_side )
{
  ENTER_RPC_METHOD;
  
  if( !onLeadingEdge(my_pos, move_dir, empty_side) ) {
    // All we have to do is pass the message on.
    pair<featureID, Letter> next = 
      getNextCircularFeat( msg_dir, my_pos, empty_side);
    RPC_RETURN getNeighbor( next.first )->get()->setLeadingEdge( next.second,
								 move_dir,
								 msg_dir,
								 empty_side );
  }
  else {
    // First we ask the new leading edge.
    if( getNeighbor( move_dir ) == NULL ) {
      // There was no one there... Empty.
      list<pair<Letter, Catom*> > empty_list;
      RPC_RETURN pair<CanMove, list<pair<Letter, Catom*> > >( Empty, 
							      empty_list );
    }
    else if( !getNeighbor( move_dir )->get()->canBeNewEdge( myOwner ) ) {
      // There was someone there, but he couldn't join... Bounce.
      list<pair<Letter, Catom*> > empty_list;
      RPC_RETURN pair<CanMove, list<pair<Letter, Catom*> > >( Bounce, 
							      empty_list );
    }
    
    // Then we ask the next guy in the circle, if we are not done.
    if( amILastEdgeCatom( my_pos, msg_dir, move_dir, empty_side ) ) {
      list<pair<Letter, Catom*> > l; 
      l.push_back( pair<Letter, Catom*>(my_pos, catomPtr) );
      RPC_RETURN pair<CanMove, list<pair<Letter, Catom*> > >(Yes, l);
    }
      
    // Then we check to see what the response was, cancelling the new
    // edge if necessary.
    pair<featureID, Letter> next = 
      getNextCircularFeat( msg_dir, my_pos, empty_side );
    pair<CanMove, list<pair<Letter, Catom*> > > result =
      getNeighbor( next.first )->get()->setLeadingEdge( next.second,
							move_dir,
							msg_dir,
							empty_side );
    if( result.first == Yes ) {
      // Add yourself to the list, return.
      result.second.push_back( pair<Letter, Catom*>(my_pos, catomPtr) );
      RPC_RETURN result;
    }
    else {
      // Cancel new edge.
      getNeighbor( move_dir )->get()->resetState(myOwner);
      RPC_RETURN result;
    }
  }
}

bool RPCHoleMotion::canBeNewEdge(catomID owner)
{
  ENTER_RPC_METHOD;

  bool result = false;

  pthread_mutex_lock( &localLock );
  if( myState == IDLE ) {
    myState = BUSY;
    myOwner = owner;
    result = true;
    HOSTCATOM.setColor( 0, 128, 0, 0 );

    //worldPtr->oStart();
    //cout << "Catom " << hostCatom << " set to BUSY: canBeNewEdge" << endl;
    //worldPtr->oEnd();
  }
  else {
    result = false;
  }
  pthread_mutex_unlock( &localLock );

  RPC_RETURN result;
}

void RPCHoleMotion::resetCollapsed( catomID owner, Letter my_pos, 
				    featureID move_dir, 
				    RotDir msg_dir,
				    bool in_center, 
				    pair<featureID, featureID> empty_side )
{
  ENTER_RPC_METHOD;
  
  pthread_mutex_lock( &localLock );
  if( myOwner == owner ) {
    pthread_mutex_unlock( &localLock );
  }
  else {
    pthread_mutex_unlock( &localLock );
    RPC_RETURN;
  }

  if( in_center ) {
    // All the positions in the center have different names...
    switch( my_pos ) {
    case G: 
      {
	featureID h_fid = rotateFeature( move_dir, CW, 1 );
	featureID i_fid = move_dir;
	featureID k_fid = rotateFeature( move_dir, CCW, 1 );
	getNeighbor( h_fid )->get()->resetCollapsed( owner, H, move_dir,
						     msg_dir,
						     in_center, empty_side );
	getNeighbor( i_fid )->get()->resetCollapsed( owner, I, move_dir,
						     msg_dir,
						     in_center, empty_side );
	getNeighbor( k_fid )->get()->resetCollapsed( owner, K, move_dir,
						     msg_dir,
						     in_center, empty_side );
      }
      break;
    case K:
      if( msg_dir == CCW ) {
	featureID j_fid = move_dir;
	getNeighbor( j_fid )->get()->resetCollapsed( owner, J, move_dir,
						     msg_dir,
						     in_center, empty_side );
      }
      break;
    case H:
      if( msg_dir == CW ) {
	featureID j_fid = move_dir;
	getNeighbor( j_fid )->get()->resetCollapsed( owner, J, move_dir,
						     msg_dir,
						     in_center, empty_side );
      }
      break;
    case I:
    case J:
      // They are done.
      break; 
    default:
      assert(false);
    }
  }
  else {
    // Just do normal around the circle clearing.
    pair<featureID, Letter> next =
      getNextCircularFeat( msg_dir, my_pos, empty_side );
    
    if( getNeighbor( next.first ) != NULL )
      getNeighbor( next.first )->get()->resetCollapsed( owner, next.second,
							move_dir,
							msg_dir, in_center,
							empty_side );
  }
    
  reset();
  RPC_RETURN;
}

void RPCHoleMotion::resetState(catomID owner)
{
  ENTER_RPC_METHOD;

  pthread_mutex_lock( &localLock );
  if( myOwner == owner ) {
    myState = IDLE;
    myOwner = 0;
    HOSTCATOM.setColor( 128, 128, 128, 0 );
  }
  pthread_mutex_unlock( &localLock );

  RPC_RETURN;
}

void RPCHoleMotion::resetStates(catomID owner, Letter my_pos,
				pair<featureID, featureID> empty_side)
{
  ENTER_RPC_METHOD;

  pthread_mutex_lock( &localLock );
  if( myOwner != owner ) {
    pthread_mutex_unlock( &localLock );
    RPC_RETURN;
  }
  else {
    pthread_mutex_unlock( &localLock );
  }
  

  switch( my_pos ) {
  case B: {
    featureID f_fid = letterToFID(B, BUSY, F, empty_side);
    featureID g_fid = letterToFID(B, BUSY, G, empty_side);
    if( getNeighbor(f_fid) != NULL )
      getNeighbor(f_fid)->get()->resetStates(owner, F, empty_side);
    if( getNeighbor(g_fid) != NULL )
      getNeighbor(g_fid)->get()->resetStates(owner, G, empty_side);
  }
    break;

  case C: {
    featureID k_fid = letterToFID(C, BUSY, K, empty_side);
    featureID l_fid = letterToFID(C, BUSY, L, empty_side);
    if( getNeighbor(k_fid) != NULL )
      getNeighbor(k_fid)->get()->resetStates(owner, K, empty_side);
    if( getNeighbor(l_fid) != NULL )
      getNeighbor(l_fid)->get()->resetStates(owner, L, empty_side);
  }
    break;    

  case D: {
    featureID h_fid = letterToFID(D, BUSY, H, empty_side);
    featureID i_fid = letterToFID(D, BUSY, I, empty_side);
    if( getNeighbor(h_fid) != NULL )
      getNeighbor(h_fid)->get()->resetStates(owner, H, empty_side);
    if( getNeighbor(i_fid) != NULL )
      getNeighbor(i_fid)->get()->resetStates(owner, I, empty_side);
  }
    break;        

  case E: {
    featureID j_fid = letterToFID(E, BUSY, J, empty_side);
    if( getNeighbor(j_fid) != NULL )
      getNeighbor(j_fid)->get()->resetStates(owner, J, empty_side);
  }
    break;
  case A: 
    throw "Invariant violated!";
  case F: case G: case H: case I: case J: case K: case L:
    break;
  }

  reset();
  RPC_RETURN;
}
