///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright (C) 2006 by Intel Corporation and Carnegie Mellon University    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <iostream>
#include <errno.h>
#include <string.h>

#include "Localize2.hxx"
#include "CatomWorld.hxx"
#include "CatomSim.hxx"

#include "newmat.h"
#include "newmatap.h"
#include "newmatio.h"

CODE_MODULE_DECLARATION( Localize2, Localize2 );

using namespace std;
using namespace NEWMAT;

// Modes

//#define set_color( r, g, b, a ) do { char* foo = (char*) &rootID; if (color_by_root) HOSTCATOM.setColor( foo[0], foo[1], foo[2], 0 ); else HOSTCATOM.setColor( r, g, b, a ); } while (0)
#define set_color( r, g, b, a ) HOSTCATOM.setColor( r, g, b, a )
#define COLOR_INIT set_color( 128, 128, 128,   0 )
#define COLOR_NONE set_color( 255, 255,   0,   0 )
#define COLOR_PART set_color(   0, 255,   0,   0 )
#define COLOR_FULL set_color( 255,   0,   0,   0 )
#define COLOR_ROOT set_color( 128,   0, 255,   0 )
#define COLOR_DONE set_color(   0,   0, 200,   0 )
#define COLOR_DONE_ROOT set_color( 0, 200, 200, 0 )

// Messages


Localize2::mode_type Localize2::mode = Localize2::MODE_NULL;
bool Localize2::debug = false;
bool Localize2::color_by_root = false;
bool Localize2::move_catoms = false;
bool Localize2::use_rk4 = false;
bool Localize2::randtheta = false;
bool Localize2::usevel = false;
bool Localize2::msgs_pending = false;
unsigned int Localize2::prop_delay = 0;
double Localize2::veldamp=0.95;
double Localize2::init_scale=1.0;
double Localize2::root_P = 1.0;
catomID Localize2::root_node = 0;
pthread_mutex_t Localize2::lock = PTHREAD_MUTEX_INITIALIZER;
int Localize2::num_msgs=0;
int Localize2::num_msgs_this_tick=0;
set<catomID> Localize2::pieces;
ofstream Localize2::binout;
bool Localize2::use_hier_s = false;
edgemap Localize2::all_edges;
unsigned int Localize2::binout_last_m=0;
int Localize2::grad_steps=0;
bool Localize2::binout_allgrad=false;
double Localize2::group_k=0.0;

// returns true if angle v is between a and b, stores value in vout; 
// note: will adjust a and b as well
static bool aint_contains( double &a, double &b, double v, double &vout ) {
  double eps = 1e-10;  // is this really necessary?
  if (b-a < 2 *M_PI) {
    double tmp = 2*M_PI*floor( a/(2*M_PI) );
    a -= tmp;
    b -= tmp;
    v -= 2*M_PI*floor( v/(2*M_PI) );   // true modulo operation
    if ( v>=a-eps && v<=b+eps ) {
	vout = v;
	return true;
    }
    if ( v+2*M_PI>=a-eps && v+2*M_PI<=b+eps ) {
	vout = v+2*M_PI;
	return true;
    }
    return false;
  } else return true;
}

static inline bool aint_contains( double &a, double &b, double v) {
  double tmp;
  return aint_contains( a, b, v, tmp );
}


#define MIN( a, b ) ((a)<(b)?(a):(b))
#define MAX( a, b ) ((a)>(b)?(a):(b))
#define EQL( a, b ) (((a)<(b)+1e-8)&&((a)>(b)-1e-8))

static void angle_int(double amin, double amax, double bmin, double bmax, double &cmin, double &cmax) {
  int n = 0;
  double smin, smax, lmin, lmax;  // smaller and larger theta intervals
  if ( amax-amin > bmax-bmin ) {
	smin = bmin; smax = bmax;
	lmin = amin; lmax = amax;
  } else {
	smin = amin; smax = amax;
	lmin = bmin; lmax = bmax;
  }
  if ( aint_contains( lmin, lmax, smin, cmin ) ) { n=1; cmax = lmax; }
  if ( aint_contains( lmin, lmax, smax, cmax ) ) { n++; cmin = lmin; }
  if (n==0) {
	cerr << "XXXXX\n";
	cmin=0;
	cmax=0;
	// rough inconsistency
  }
  if (n==2) {
	// subset or double pacman -- use smaller
	cmin = smin;
	cmax = smax;
  }
}


static box box_int( const box &a, const box &b ) {
  box c;
  c.min.x = MAX(a.min.x, b.min.x);
  c.min.y = MAX(a.min.y, b.min.y);
  c.max.x = MIN(a.max.x, b.max.x);
  c.max.y = MIN(a.max.y, b.max.y);
  if (c.min.x > c.max.x) {
	c.min.x = (c.min.x+c.max.x)/2.0;
	c.max.x = c.min.x;
	// inconsistency
  }
  if (c.min.y > c.max.y) {
	c.min.y = (c.min.y+c.max.y)/2.0;
	c.max.y = c.min.y;
	// inconsistency
  }
/*  int n = 0;
  double smin, smax, lmin, lmax;  // smaller and larger theta intervals
  if ( a.max.theta-a.min.theta > b.max.theta-b.min.theta ) {
	smin = b.min.theta; smax = b.max.theta;
	lmin = a.min.theta; lmax = a.max.theta;
  } else {
	smin = a.min.theta; smax = a.max.theta;
	lmin = b.min.theta; lmax = b.max.theta;
  }
  if ( aint_contains( lmin, lmax, smin, c.min.theta ) ) { n=1; c.max.theta = lmax; }
  if ( aint_contains( lmin, lmax, smax, c.max.theta ) ) { n++; c.min.theta = lmin; }
  if (n==0) {
	c.min.theta=0;
	c.max.theta=0;
	// rough inconsistency
  }
  if (n==2) {
	// subset or double pacman -- use smaller
	c.min.theta = smin;
	c.max.theta = smax;
  }
*/
  angle_int( a.min.theta,  a.max.theta, b.min.theta, b.max.theta, c.min.theta, c.max.theta );
  return c;
}


static double box_size( const box &a ) {
  if ( a.max.theta-a.min.theta > 2*M_PI ) return (a.max.x-a.min.x)*(a.max.y-a.min.y);
  return (a.max.x - a.min.x) * (a.max.y - a.min.y) * (a.max.theta - a.min.theta);
}

static bool is_point( const box &a ) {
	return a.min.x==a.max.x && a.min.y==a.max.y;
}


static box obs2box( const obslist &obsl, const box &f1 ) {
  double dir_eps = M_PI/4;  // NOTE: should be max of rx/tx angle
  double utheta = 2*dir_eps; // NOTE: should be rx angle + tx angle
  obslist::const_iterator it;
  box cbest;
  for ( it=obsl.begin(); it!=obsl.end(); it++ ) {
	double o2 = it->loc;  // NOTE: obslist is target (j in matlab code) catom's perspective
				// so local and remote are switched
	double o1 = it->rem;
	double dir_min = f1.min.theta + o1 - dir_eps;
	double dir_max = f1.max.theta + o1 + dir_eps;
	double x0 = cos (dir_min);
	double x1 = cos (dir_max);
	double y0 = sin (dir_min);
	double y1 = sin (dir_max);
	double xmax, xmin, ymax, ymin;
	if ( aint_contains( dir_min, dir_max, 0 ) ) {
	  xmax = 1;
	} else {
	  xmax = MAX( x0, x1 );
	}
	if ( aint_contains( dir_min, dir_max, M_PI/2 ) ) {
	  ymax = 1;
	} else {
	  ymax = MAX( y0, y1 );
	}
	if ( aint_contains( dir_min, dir_max, M_PI ) ) {
	  xmin = -1;
	} else {
	  xmin = MIN( x0, x1 );
	}
	if ( aint_contains( dir_min, dir_max, 3*M_PI/2 ) ) {
	  ymin = -1;
	} else {
	  ymin = MIN( y0, y1 );
	}
	box c;
	double D = 2 * CatomSim::catom_radius;
	c.min.x=D*xmin; c.min.y=D*ymin; c.max.x=D*xmax; c.max.y=D*ymax;
	//c.min.x = MIN( D*xmin, D*xmin*1.25 ); c.max.x = MAX( D*xmax, D*xmax*1.25 );
	//c.min.y = MIN( D*ymin, D*ymin*1.25 ); c.max.y = MAX( D*ymax, D*ymax*1.25 );
	double dtheta = o1-o2+M_PI;
	c.min.theta = dtheta-utheta; c.max.theta = dtheta+utheta;
	if ( it == obsl.begin() ) cbest = c;
	else cbest = box_int( c, cbest );
  }
  // NOTE: original matlab code does this outside of this function
  cbest.min.x += f1.min.x; cbest.min.y += f1.min.y; cbest.min.theta += f1.min.theta;
  cbest.max.x += f1.max.x; cbest.max.y += f1.max.y; cbest.max.theta += f1.max.theta;
  return cbest;
};

/* old version
static box obs2box( const obslist &obsl, const box &f1 ) {
  double dir_eps = M_PI/8;  // NOTE: should be max of rx/tx angle
  double utheta = 2*dir_eps; // NOTE: should be rx angle + tx angle
  obslist::const_iterator it;
  box cbest;
  for ( it=obsl.begin(); it!=obsl.end(); it++ ) {
	double o2 = it->loc;  // NOTE: obslist is target (j in matlab code) catom's perspective
				// so local and remote are switched
	double o1 = it->rem;
	double dir_min = f1.min.theta + o1 - dir_eps;
	double dir_max = f1.max.theta + o1 + dir_eps;
	double x0 = cos (dir_min);
	double x1 = cos (dir_max);
	double y0 = sin (dir_min);
	double y1 = sin (dir_max);
	double xmax, xmin, ymax, ymin;
	if ( aint_contains( dir_min, dir_max, 0 ) ) {
	  xmax = 1;
	} else {
	  xmax = MAX( x0, x1 );
	}
	if ( aint_contains( dir_min, dir_max, M_PI/2 ) ) {
	  ymax = 1;
	} else {
	  ymax = MAX( y0, y1 );
	}
	if ( aint_contains( dir_min, dir_max, M_PI ) ) {
	  xmin = -1;
	} else {
	  xmin = MIN( x0, x1 );
	}
	if ( aint_contains( dir_min, dir_max, 3*M_PI/2 ) ) {
	  ymin = -1;
	} else {
	  ymin = MIN( y0, y1 );
	}
	box c;
	double D = 2 * CatomSim::catom_radius;
	c.min.x=D*xmin; c.min.y=D*ymin; c.max.x=D*xmax; c.max.y=D*ymax;
	//c.min.x = MIN( D*xmin, D*xmin*1.25 ); c.max.x = MAX( D*xmax, D*xmax*1.25 );
	//c.min.y = MIN( D*ymin, D*ymin*1.25 ); c.max.y = MAX( D*ymax, D*ymax*1.25 );
	double dtheta = o1-o2+M_PI;
	c.min.theta = dtheta-utheta; c.max.theta = dtheta+utheta;
	if ( it == obsl.begin() ) cbest = c;
	else cbest = box_int( c, cbest );
  }
  // NOTE: original matlab code does this outside of this function
  cbest.min.x += f1.min.x; cbest.min.y += f1.min.y; cbest.min.theta += f1.min.theta;
  cbest.max.x += f1.max.x; cbest.max.y += f1.max.y; cbest.max.theta += f1.max.theta;
  return cbest;
};
*/

static void obs2theta( const obslist &obsl, const position &n, const position &m, box &b ) {
  double dir_eps = M_PI/4;  // NOTE: should be max of rx/tx angle
  obslist::const_iterator it;
  double dir = atan2( n.y-m.y, n.x-m.x );
  double tmin = 0;
  double tmax = 2*M_PI;
  for ( it=obsl.begin(); it!=obsl.end(); it++ ) {
	double o2 = it->loc;
	double tmp = dir - o2 - dir_eps;
	//tmp -= floor( tmp / (2*M_PI) ) * 2 * M_PI;
	//tmin = MAX( tmin, tmp );
	//tmax = MIN( tmax, tmp+2*dir_eps ); 
	//cerr << " HERE " << tmin << " " << tmax << " " << dir << " " << o2 <<"\n";
	angle_int( tmin, tmax, tmp, tmp+2*dir_eps, tmin, tmax );
	//cerr << " HERE2 " << tmin << " " << tmax << " \n";
  }
  b.min.theta = tmin;
  b.max.theta = tmax;
}

static inline bool point_in_box( const position &a, const box &b ) {
  return a.x>=b.min.x && a.x<=b.max.x && a.y>=b.min.y && a.y<=b.max.y;
}

static double point_depth_box( const position &a, const box &b ) {
  double xdepth = MIN( b.max.x - a.x, a.x - b.min.x );
  double ydepth = MIN( b.max.y - a.y, a.y - b.min.y );
  if ( xdepth < 0 && ydepth < 0 ) return -sqrt( xdepth*xdepth + ydepth*ydepth );
  return MIN( xdepth, ydepth );
}

static position circle_int ( const position &a, const position &b, const box &c ) {
  double dx = b.x - a.x;
  double dy = b.y - a.y;
  double d = sqrt( dx*dx + dy*dy );
  double w = sqrt( 4*CatomSim::catom_radius*CatomSim::catom_radius - d*d/4 );
  if (d==0) cerr << "HERE -- both neighbors are at same center!\n";
  if (d>4*CatomSim::catom_radius) {
    cerr << "HERE2 -- neighbors more than 4 radii apart\n";
    w = 0;
  }
  position h1, h2;
  h1.x = a.x+dx/2 - w*dy/d;
  h1.y = a.y+dy/2 + w*dx/d;
  h2.x = a.x+dx/2 + w*dy/d;
  h2.y = a.y+dy/2 - w*dx/d;
  //if (point_in_box( h1, c ) ) return h1;
  //if (point_in_box( h2, c ) ) return h2;
  if ( point_depth_box( h1, c ) > point_depth_box( h2, c ) ) return h1;
  return h2;
}

static position circle_int2 ( const position &a, const position &b, const box &c, const obslist &obsl ) {
  double dx = b.x - a.x;
  double dy = b.y - a.y;
  double d = sqrt( dx*dx + dy*dy );
  double w = sqrt( 4*CatomSim::catom_radius*CatomSim::catom_radius - d*d/4 );
  if (d==0) cerr << "HERE -- both neighbors are at same center!\n";
  if (d>4*CatomSim::catom_radius) {
    cerr << "HERE2 -- neighbors more than 4 radii apart\n";
    w = 0;
  }
  position h1, h2;
  h1.x = a.x+dx/2 - w*dy/d;
  h1.y = a.y+dy/2 + w*dx/d;
  h2.x = a.x+dx/2 + w*dy/d;
  h2.y = a.y+dy/2 - w*dx/d;
  double dir_eps = M_PI/3;  // NOTE: should be max of rx/tx angle
  obslist::const_iterator it;
  double tmin = 0;
  double tmax = 2*M_PI;
  for ( it=obsl.begin(); it!=obsl.end(); it++ ) {
	double o1 = it->rem;
	double tmp = a.theta + o1 - dir_eps;
	angle_int( tmin, tmax, tmp, tmp+2*dir_eps, tmin, tmax );
  }
  double dir = atan2( h1.y-a.y, h1.x-a.x );
  if ( aint_contains( tmin, tmax, dir ) ) return h1;
  dir = atan2( h2.y-a.y, h2.x-a.x );
  if ( aint_contains( tmin, tmax, dir ) ) return h2;
  cerr << "HERE3: neither option seems to work well\n";
  return circle_int( a, b, c );
}

static double contact_objective(position x, double xt, position y, double yt) {
  xt += x.theta;
  position xc;
  xc.x = x.x + CatomSim::catom_radius*cos(xt);
  xc.y = x.y + CatomSim::catom_radius*sin(xt);
  yt += y.theta;
  position yc;
  yc.x = y.x + CatomSim::catom_radius*cos(yt);
  yc.y = y.y + CatomSim::catom_radius*sin(yt);
  double dotp = (xc.x-x.x)*(yc.x-y.x) + (xc.y-x.y)*(yc.y-y.y);
  double dist2 = (x.x-y.x)*(x.x-y.x) + (x.y-y.y)*(x.y-y.y);
  return (yc.x-xc.x)*(yc.x-xc.x) + (yc.y-xc.y)*(yc.y-xc.y) + 1*(dotp+1)*(dotp+1);
		//+ 0.5*MAX(dotp, 0 ) * MIN(4*CatomSim::catom_radius*CatomSim::catom_radius, dist2);
}


static double local_objective( const position &x, const map<catomID,position> &nei, const map<catomID,obslist> &obsl ) {
  double v = 0;
  map<catomID,position>::const_iterator nit;
  for ( nit=nei.begin(); nit!=nei.end(); nit++ ) {
	catomID n = nit->first;
	position y = nit->second;
	double v2 = 0;
	const obslist &nobs = obsl.find(n)->second;
	obslist::const_iterator oit;
	for ( oit=nobs.begin(); oit!=nobs.end(); oit++ ) {
		v2 += contact_objective( x, oit->loc, y, oit->rem );
	}
	v += v2 / nobs.size();
  }
  return v;
}

static position local_gradient( position x, const map<catomID,position> &nei, const map<catomID,obslist> &obsl ) {
  position dv;
  double deps = 0.000001;
  double v = local_objective( x, nei, obsl );
  x.x += deps;
  dv.x = -(local_objective( x, nei, obsl ) - v) / deps;
  x.x -= deps;
  x.y += deps;
  dv.y = -(local_objective( x, nei, obsl ) - v) / deps;
  x.y -= deps;
  x.theta += deps;
  dv.theta = -(local_objective( x, nei, obsl ) - v) / deps;
  x.theta -= deps;
  return dv;
}

static unsigned int num_fixed = 0;
static pthread_mutex_t fpl = PTHREAD_MUTEX_INITIALIZER;

void Localize2::fix_position( double x, double y, double theta ) {
  pthread_mutex_lock(&fpl);
  num_fixed++;
  pthread_mutex_unlock(&fpl);
  my_x.x = x;
  my_x.y = y;
  my_x.theta = theta;
  my_box.min = my_x;
  my_box.max = my_x;
  prop_constraints = true;
  box_done = true;
  COLOR_FULL;
  move_catom();
  if (debug) cerr << "FIX: fixing position of " << hostCatom << "\n";
}

#define USE_2_NEI_FIX 0

void Localize2::fix_position() {
  if ( USE_2_NEI_FIX && my_nei.size() > 1 ) {
	map<catomID,position>::const_iterator it = my_nei.begin();
	catomID n1 = it->first;
	position np1 = it->second;
	it++;
	catomID n2 = it->first;
	position np2 = it->second;
	//my_x = circle_int( np1, np2, my_box );
	my_x = circle_int2( np1, np2, my_box, my_obs[n1] );
	obs2theta( my_obs[n1], np1, my_x, my_box );
	obs2theta( my_obs[n2], np2, my_x, my_box );
	fix_position( my_x.x, my_x.y, (my_box.min.theta + my_box.max.theta)/2 );
  } else if ( 1 && mode==MODE_BOXPROP && my_nei.size()==1 ) {
	catomID n = my_nei.begin()->first;
	position np = my_nei.begin()->second;
	double dx = (my_box.min.x+my_box.max.x)/2 - np.x;
	double dy = (my_box.min.y+my_box.max.y)/2 - np.y;
	double dist = sqrt( dx*dx + dy*dy );
	my_x.x = np.x + 2*CatomSim::catom_radius*dx/dist;
	my_x.y = np.y + 2*CatomSim::catom_radius*dy/dist;
	obs2theta( my_obs[n], np, my_x, my_box );
	fix_position( my_x.x, my_x.y, (my_box.min.theta + my_box.max.theta)/2 );
  } else if ( 1 && my_nei.size()==1 ) {
	catomID n = my_nei.begin()->first;
	position np = my_nei.begin()->second;
	double dir_eps = M_PI/4;  // NOTE: should be max of rx/tx angle
	obslist::const_iterator it;
	double tmin = my_box.min.theta = 0;
	double tmax = my_box.max.theta = 2*M_PI;
	for ( it=my_obs[n].begin(); it!=my_obs[n].end(); it++ ) {
		double o1 = it->rem;
		double tmp = np.theta + o1 - dir_eps;
		angle_int( tmin, tmax, tmp, tmp+2*dir_eps, tmin, tmax );
	}
	my_x.x = np.x + 2*CatomSim::catom_radius * cos( (tmin+tmax)/2 );
	my_x.y = np.y + 2*CatomSim::catom_radius * sin( (tmin+tmax)/2 );
	obs2theta( my_obs[n], np, my_x, my_box );
	fix_position( my_x.x, my_x.y, (my_box.min.theta + my_box.max.theta)/2 );
  } else fix_position( (my_box.min.x+my_box.max.x)/2,
		(my_box.min.y+my_box.max.y)/2,
		(my_box.min.theta+my_box.max.theta)/2 );
}

void Localize2::move_catom() {
  if ( move_catoms ) {
	dBodySetPosition( HOSTCATOMSIM->body, my_x.x, my_x.y, 1 );
	dQuaternion q;
	dQFromAxisAndAngle( q, 0, 0, 1, my_x.theta );
	dBodySetQuaternion( HOSTCATOMSIM->body, q );
  }
}


void Localize2::simulationStart() {
  my_x.x = my_x.y = my_x.theta = INFINITY;
  my_box.min.x = my_box.min.y = my_box.min.theta = -INFINITY;
  my_box.max = my_x;
  my_vel.x = my_vel.y = my_vel.theta = 0;
  my_root = 0;
  prop_constraints = false;
  prop_delay_count = 0;
  max_ttl = 0;
  box_done = false;
  pthread_mutex_lock( &lock);
  list<CatomSim::sensor_val>::const_iterator it;
  for ( it = HOSTCATOMSIM->sensor_vals.begin(); it != HOSTCATOMSIM->sensor_vals.end(); it++ ) {
	if ( it-> siglevel < 0.2 ) continue;
	catomID rid = it->rcat->C.getID();
	obs tmpobs;
	Point3D p = Feature::locations[it->lfid];
	tmpobs.loc = atan2( p.getY(), p.getX() );
	p = Feature::locations[it->rfid];
	tmpobs.rem = atan2( p.getY(), p.getX() );
	my_obs[ rid ].push_back( tmpobs );
	// my_nei[ rid ] = my_x;  // don't do this
	if (debug) cerr << "OBS: " << hostCatom << " sees " << rid << " at " 
		<< tmpobs.loc << " " << tmpobs.rem << " " << it-> siglevel << "\n";
	// add edges
	my_edges.insert( rid );
	all_edges[ pair<catomID,catomID>(hostCatom,rid) ].insert( pair<catomID,catomID>(hostCatom,rid) );
  }
  COLOR_NONE;
  // save real position, orientation
  const dReal *rot = dBodyGetQuaternion( HOSTCATOMSIM->body );
  double angle = 2 * acos ( rot[0] );
  if (rot[3]<0) angle = -angle;
  Point3D pos = HOSTCATOM.getLocation();

  savedpos.x=pos.getX(); savedpos.y = pos.getY(); savedpos.theta = angle;
  if (hostCatom == root_node || mode==MODE_INIT_NULL ||
		mode==MODE_INIT_REAL || mode==MODE_HIER ) {
	if ( mode == MODE_INIT_NULL ) {
	  fix_position( 0, 0, 0 );
	} else if ( mode==MODE_HIER ) {
	  fix_position( init_scale*savedpos.x, init_scale*savedpos.y, 0 );
	  my_root = hostCatom;
	  pieces.insert( hostCatom );
	  members.insert( hostCatom );
	} else {
	  fix_position( savedpos.x, savedpos.y, savedpos.theta );
	  if (hostCatom == root_node) COLOR_ROOT;
	}
  }
  pthread_mutex_unlock( &lock);
}

#define send_message( id, msg ) mail.send( REMOTE_CODE_MODULE( (id), Localize2 )->mail, (msg) );

#define PTSTR( p ) \
  "(" << (p).x << "," << (p).y << "," << (p).theta << ")"

#define BOXSTR( b ) \
  PTSTR( (b).min ) << "-" << PTSTR( (b).max )

#define PRINT_MSG( m ) \
do { if (!debug) break;\
  cout << hostCatom << ": got type=" << (int) (m).type << ", root=" << (m).rootID << " from " << (m).sender; \
  cout << " my " << PTSTR( (m).myPos ) << ", your " << PTSTR( (m).yourPos ); \
  cout << ", other " << PTSTR( (m).otherPos ) << " dist " << (m).otherDist << endl; \
} while(0)

#define PRINT_STATE \
do { if (!debug) break;\
  cout << hostCatom << ": state=" << mode << " root=" << rootID << " pos ="; \
  Point3D pt = Point3D( dBodyGetPosition( body ) ); \
  cout << PTSTR( pt ) << endl; \
} while(0)

void Localize2::newTick() {
  mymsg msg;
  while ( (msg = mail.read()).type != mymsg::M_EMPTY ) {
	if (msg.type == mymsg::M_BOX && box_done==false) {
		max_ttl = MAX( max_ttl, msg.boxmsg.ttl-1 );
		box cij = obs2box( my_obs[ msg.boxmsg.i ], msg.boxmsg.box_i );
		cij = box_int( my_box, cij );
		if ( is_point( msg.boxmsg.box_i ) ) {
			if (mode==MODE_TREE_GRADIENT) my_nei.clear();
			my_nei[msg.boxmsg.i] = msg.boxmsg.box_i.min;
			if (mode==MODE_TREE || mode==MODE_TREE_GRADIENT) {
				my_box = cij;
				fix_position();
			}
		}
		if ( box_size( cij ) < box_size( my_box ) ) {
			my_box = cij;
			prop_constraints = true;
			COLOR_PART;
			if (is_point( my_box )) {
				//fix_position();
			}
		}
		if ( my_nei.size() > 1 && USE_2_NEI_FIX  ) {
			fix_position();
		}
	}
	if (msg.type == mymsg::M_GRAD && my_root==msg.gradmsg.root) {
		my_nei[msg.gradmsg.i] = msg.gradmsg.x_i;
	}
  }
  if ( box_done ) max_ttl = 5;  // max box prop radius
  if ( prop_constraints && max_ttl>0 && mode!=MODE_HIER && mode!=MODE_GRADIENT) {
	msgs_pending = true;
	if (prop_delay_count==0) prop_delay_count = prop_delay;
	else prop_delay_count--;
	if (prop_delay_count==0) {
		prop_constraints = false;
		msg.type = mymsg::M_BOX;
		msg.boxmsg.i = hostCatom;
		msg.boxmsg.ttl = max_ttl;
		msg.boxmsg.box_i = my_box;
		max_ttl = 0;
		if (debug) cerr << "BOX: " << hostCatom << " sends " << BOXSTR(my_box) 
			<< " ttl " << max_ttl << " to";
		map<catomID,obslist>::const_iterator it;
		for ( it=my_obs.begin(); it!=my_obs.end(); it++) {
			send_message( it->first, msg );
			if (debug) cerr << " " << it->first;
		}
		if (debug) cerr << "\n";
		pthread_mutex_lock( &lock );
		num_msgs_this_tick += my_obs.size();
		pthread_mutex_unlock( &lock );
	}
  }
  if ( box_done && (mode==MODE_GRADIENT || mode==MODE_TREE_GRADIENT || mode==MODE_HIER) ) {

	if ( use_rk4 ) { // RK4
		double h = 0.05;
		position k1 = local_gradient( my_x, my_nei, my_obs );
		position k2 = local_gradient( my_x + k1*(h/2), my_nei, my_obs );
		position k3 = local_gradient( my_x + k2*(h/2), my_nei, my_obs );
		position k4 = local_gradient( my_x + k3*h, my_nei, my_obs );
		position dx = (k1 + k2*2 + k3*2 + k4)/6;
		my_x += dx*h;
	} else if ( usevel ) {
		position dv = local_gradient( my_x, my_nei, my_obs );
		double h = 0.05;
		my_vel = my_vel*veldamp + dv;
		my_x += my_vel*h;
	} else { // euler integration
		position dx = local_gradient( my_x, my_nei, my_obs );
		double h = 0.05;
		my_x.x += dx.x * h;
		my_x.y += dx.y * h;
		my_x.theta += dx.theta * h;
	}
	my_box.min = my_x;
	my_box.max = my_x;
	msg.type = mymsg::M_GRAD;
	msg.gradmsg.i = hostCatom;
	msg.gradmsg.x_i = my_x;
	msg.gradmsg.root = my_root;
	if (debug) cerr << "GRAD: " << hostCatom << " sends " << PTSTR(my_x) << " to";
	map<catomID,obslist>::const_iterator it;
	for ( it=my_obs.begin(); it!=my_obs.end(); it++) {
		send_message( it->first, msg );
		if (debug) cerr << " " << it->first;
	}
	if (debug) cerr << "\n";
	pthread_mutex_lock( &lock );
	num_msgs_this_tick += my_obs.size();
	pthread_mutex_unlock( &lock );
	move_catom();
  }
}

void Localize2::endTick() {
  mail.done_this_tick();
//	cerr << "HERE\n";
}


static void ralign( const Matrix &X, const Matrix &Y, Matrix &R, ColumnVector &t2, ColumnVector &t3, bool warn=false ) {
  int m = X.Nrows();
  int n = X.Ncols();
  int i;
  ColumnVector mx(m);
  mx=0;
  ColumnVector my(m);
  my=0;
  for (i=1; i<=n; i++) {
    mx += X.Column(i);
    my += Y.Column(i);
  }
  mx /= n;
  my /= n;
  Matrix Xc(m,n);
  Matrix Yc(m,n);
  for ( i=1; i<=n; i++) {
    Xc.Column(i) = X.Column(i)-mx;
    Yc.Column(i) = Y.Column(i)-my;
  }
  Matrix Sxy;
  Sxy = (Yc/n) * Xc.t();
  Matrix U;
  DiagonalMatrix D;
  Matrix V;
  SVD( Sxy, D, U, V );
  double d = Sxy.Determinant();
  int r = 0;
  double max_s=0;
  for (i=1; i<=m; i++ ) {
    if (D(i,i) > max_s ) max_s=D(i,i);
    if (D(i,i) < -max_s ) max_s=-D(i,i);
  }
  max_s *= 1e-8;
  for (i=1; i<=m; i++ ) {
    if (D(i,i)>=max_s || D(i,i)<=-max_s) r++;
  }
	//cout << max_s << endl;
  Matrix S;
  S  = IdentityMatrix(m);
  if (r>m-1) {
    if (Sxy.Determinant()<0) { S(m,m) = -1; cerr<<"ralign: here1\n"; }
  } else if ( r==m-1) {
    if ( U.Determinant() * V.Determinant() < 0) { S(m,m)=-1; cerr << "ralign: here2\n"; }
  } else {
    R = IdentityMatrix(m);
    t2 = 0;
    cerr<< "WARN: insufficient rank to determine rigid transform\n";
    return;
  }
  //cerr << U << D << S << V << r << endl;
  R = U * S * V.t();
  t2 = my - R*mx;
  t3 = my - (-R)*mx;
}


#define L2(a) (REMOTE_CODE_MODULE( (a), Localize2 ))

void Localize2::do_binout() {
    static int num_rec = 0;
    binout.seekp(16);
    num_rec++;
    binout.write((char*)&num_rec, 4 );  //num records
    binout.seekp(0,ofstream::end);
    double tmpdouble = worldPtr->current_time;
    binout.write((char*)&tmpdouble, 8 );
    tmpdouble=0;
    binout.write((char*)&tmpdouble, 8 );
    binout.write((char*)&tmpdouble, 8 );
    hash_map<const unsigned long, CatomSim *, hash<const unsigned long>, equl>::iterator cit = worldPtr->catomHash.begin();
    while ( cit!=worldPtr->catomHash.end() ) {
	catomID tmpid = cit->second->C.getID();
	tmpdouble = tmpid;
	binout.write((char*)&tmpdouble, 8 );
	binout.write((char*)&(L2(tmpid)->my_x.x),8);
	binout.write((char*)&(L2(tmpid)->my_x.y),8);
	binout.write((char*)&(L2(tmpid)->my_x.theta),8);
	cit++;
    }
    binout.flush();
}

void Localize2::oracle() {
  //cout << "Oracle: running concurrently with simulationStart\n";
  oracleYield();
  //cout << "Oracle: running with other oracles only\n";
  num_msgs = 0;
  num_msgs_this_tick = 0;
  int hier_delay = 0;
  if ( mode==MODE_INIT_NULL || mode==MODE_INIT_REAL ) mode=MODE_GRADIENT;
  if ( root_node==0 ) {
	catomID r = worldPtr->catomHash.begin()->second->C.getID();
	L2(r)->fix_position( L2(r)->savedpos.x, L2(r)->savedpos.y, L2(r)->savedpos.theta );
  }
  // write binary header: LOCS, ver, ndim, ncatoms, nrecords
  binout.write("LOCS", 4);
  int tmpint = 2;
  binout.write((char*)&tmpint, 4 );  //ver 2
  binout.write((char*)&tmpint, 4 );  //2 dim
  tmpint = worldPtr->catomHash.size();
  cerr << "NUMCATOMS: " << tmpint << endl;
  binout.write((char*)&tmpint, 4 );  //num catoms
  tmpint = 0;
  binout.write((char*)&tmpint, 4 );  //num records --- placeholder
  oracleYield();
  static double best_prob_prev = 1.0;
//try {
  while (1) {
    hash_map<const unsigned long, CatomSim *, hash<const unsigned long>, equl>::iterator cit;
    //cout << "Oracle: running concurrently with newTick, endTick\n";
    oracleYield();
    //cout << "Oracle: running with other oracles only\n";
	//cout << hier_delay << endl;
    if ( ( binout_allgrad && (worldPtr->timesteps - worldPtr->current_time <= grad_steps ) )
	|| ( worldPtr->timesteps - worldPtr->current_time == grad_steps-1 ) 
	|| ( worldPtr->timesteps - worldPtr->current_time <= 1 ) ) 
		do_binout();
    num_msgs += num_msgs_this_tick;
    if (debug) cerr << "MSGS: " << num_msgs_this_tick << " " << num_msgs << "\n";
    if ( msgs_pending==false && num_msgs_this_tick==0 ) {
	catomID next_catom = 0;
	double smallest_box = INFINITY;
	cit = worldPtr->catomHash.begin();
        if ( mode==MODE_BOXPROP ) while (cit!=worldPtr->catomHash.end()) {
	  Localize2 *cl = REMOTE_CODE_MODULE( cit->first, Localize2 );
	  double bs = box_size( cl->my_box );
	  if ( cl->box_done==false && ( bs < smallest_box ) ) {
		smallest_box = bs;
		next_catom = cit->first;
	  }
	  cit++;
	}
	if ( next_catom ) {
	  REMOTE_CODE_MODULE( next_catom, Localize2 )->fix_position();
	} else {
	  if (debug) cerr << "DONE INIT after " << num_msgs << " MSGS\n";
	if (grad_steps) worldPtr->timesteps = worldPtr->current_time+grad_steps;  
	  mode = MODE_GRADIENT;
	  for ( cit = worldPtr->catomHash.begin(); cit != worldPtr->catomHash.end(); cit++ ) {
		Localize2 *cl = REMOTE_CODE_MODULE( cit->first, Localize2 );
		cl->my_x.x *= init_scale;
		cl->my_x.y *= init_scale;
		cl->move_catom();
	  }
	}
    }
    if ( num_fixed == worldPtr->num_catoms ) {
	static bool done=false;
	if (!done) {
	  done=true;
	  cerr << "DONE \n";
	  if (grad_steps) worldPtr->timesteps = worldPtr->current_time+grad_steps;
	}
    }
    if ( 0 && mode==MODE_HIER ) do {  // old code
	if ( hier_delay ) { hier_delay--; continue; }
	hier_delay = prop_delay;
	set<catomID> new_pieces;
	ostringstream s1, s2;
	double best_prob_next=0.0;
	if (debug) s1 << "HIER: " << hier_delay;
	while ( pieces.size() ) {
	  catomID newroot = *(pieces.begin());
	  pieces.erase( pieces.begin() );
	  typedef map< catomID, map< pair< catomID, catomID >, const obslist* > > cpo_t; 
	  cpo_t child_piece_obs;
	  if (debug) s1 << " " << newroot;

	  // get all pieces touching newroot by scanning all member catom observations
	  set<catomID>::const_iterator mit = L2(newroot)->members.begin();
	  while ( mit!=L2(newroot)->members.end() ) {
	    catomID m = *mit;
	    map<catomID,obslist>::const_iterator nit = L2(m)->my_obs.begin();
	    while( nit != L2(m)->my_obs.end() ) {
		catomID n = nit->first;
		if (L2(n)->my_root != L2(m)->my_root) {
		  (child_piece_obs[L2(n)->my_root])[pair<catomID,catomID>(m,n)] = &(nit->second);
		//cerr << "add obs: " << newroot << " -> " << L2(n)->my_root << " " << m << ":" << n << " size " <<  (child_piece_obs[L2(n)->my_root]).size() << endl;
		}
		nit++;
	    }
	    mit++;
	  }

	  // cull any neighboring pieces that have already been combined this timestep
	  cpo_t::iterator pit = child_piece_obs.begin();
	  bool got_match=false;  // use to restrict to pairwise matching only
	  while ( pit != child_piece_obs.end() ) {
	    catomID p = pit->first;
	    if ( (!use_hier_s && got_match) || pieces.find(p) == pieces.end() ) {
		pit++;
		child_piece_obs.erase( p );
	    }
	    else {
		set<catomID> a, b;
		map<pair<catomID,catomID>,const obslist*>::const_iterator pobsit = pit->second.begin();
		while ( pobsit != pit->second.end() ) {
		    a.insert( pobsit->first.first );
		    b.insert( pobsit->first.second );
		//cerr << pobsit->first.first << " " <<  pobsit->first.second << endl;
		    pobsit++;
		}
		pit++;
		//double prob = (double) (a.size() * b.size()) / 
		//		(double) (L2(newroot)->members.size() * L2(p)->members.size());
		double prob = MAX( (double) a.size() / L2(newroot)->members.size(),
					(double) b.size() / L2(p)->members.size() );
		if (prob > best_prob_next) best_prob_next = prob;
		prob /= best_prob_prev;
		if ( use_hier_s || (((double)rand())/RAND_MAX)<prob ) {
		  got_match=true;
		  pieces.erase( p );
		  cerr << "match edge: " << a.size() << " " << L2(newroot)->members.size() 
			<< " " << b.size() << " " << L2(p)->members.size() << " " << prob << endl; 
		} else {
		  child_piece_obs.erase( p );
		  cerr << "cull edge: " << a.size() << " " << L2(newroot)->members.size() 
			<< " " << b.size() << " " << L2(p)->members.size() << " " << prob << endl; 
		}
	    }
	  }

	  // for each piece, find best transform, apply to all member catoms, join them to newroot
	  pit = child_piece_obs.begin();
	  while ( pit != child_piece_obs.end() ) {
	    catomID p = pit->first;
	    // find best transform
	    map<pair<catomID,catomID>,const obslist*>::const_iterator pobsit = pit->second.begin();
	    Matrix R(2,2);
	    ColumnVector t2(2), t3(2);
	    Matrix X(2,0), Y(2,0);
	    Matrix mdx(2,0), mdy(2,0);
	    while ( pobsit != pit->second.end() ) {
		catomID a = pobsit->first.first;
		catomID b = pobsit->first.second;
		obslist::const_iterator ccobsit = pobsit->second->begin();
		while ( ccobsit != pobsit->second->end() ) {
		  // get mean pos
		  ColumnVector tmpx(2);
		  tmpx << ( L2(a)->my_x.x + CatomSim::catom_radius * cos(L2(a)->my_x.theta+ccobsit->loc) )
		  	<< ( L2(a)->my_x.y + CatomSim::catom_radius * sin(L2(a)->my_x.theta+ccobsit->loc) );
		  X |= tmpx;
		  tmpx << cos(L2(a)->my_x.theta+ccobsit->loc) << sin(L2(a)->my_x.theta+ccobsit->loc);
		  mdx |= tmpx;
		  ColumnVector tmpy(2);
		  tmpy << ( L2(b)->my_x.x + CatomSim::catom_radius * cos(L2(b)->my_x.theta+ccobsit->rem) )
		  	<< ( L2(b)->my_x.y + CatomSim::catom_radius * sin(L2(b)->my_x.theta+ccobsit->rem) );
		  Y |= tmpy;
		  tmpy << cos(L2(b)->my_x.theta+ccobsit->rem) << sin(L2(b)->my_x.theta+ccobsit->rem);
		  mdy |= tmpy;
		  ccobsit++;
		}
		pobsit++;
	    }
	    ralign( Y, X, R, t2, t3 );
	    if ( !EQL(R(1,1),R(2,2)) || !EQL(R(1,2),-R(2,1)) ) cout << "HERE -- bad R:\n" << R;
	    // sanity check Rotation
	    //mdx /= X.Ncols();
	    //mdy /= Y.Ncols();
	    double dotp = DotProduct( mdx, R*mdy );
	    if ( dotp>=0 ) {
		// 180 degree rotation
		cerr << "HERE -- need to reverse, dotp = " << dotp << endl;
		R = -R;
		t2 = t3;
	    }
	    double r = atan2( R(2,1), R(1,1) );
	    // apply transform, given by R and t2, and merge piece members to newroot
	    set<catomID>::const_iterator mit = L2(p)->members.begin();
	    while ( mit != L2(p)->members.end() ) {
		catomID m = *mit;
		ColumnVector CV(2);
		CV << L2(m)->my_x.x << L2(m)->my_x.y;
		CV = t2 + R*CV;
		L2(m)->my_x.x = CV(1);
		L2(m)->my_x.y = CV(2);
		L2(m)->my_x.theta += r;
		L2(m)->my_root*=0;
		L2(m)->my_nei.clear();
		L2(m)->my_root = newroot;
		L2(newroot)->members.insert( m );
		mit++;
		L2(p)->members.erase( m );
	    }
	    if (debug) s1 << "+" << p;
	    pit++;
	  }

	  new_pieces.insert( newroot );
	}
	pieces = new_pieces;
	if (debug) cerr << (s1.str());
	if (debug) cerr << "\nHIER_NUM_PIECES: " << pieces.size() << "\n";
	//best_prob_prev = best_prob_next;
    } while (0);

    if ( group_k==0.0 && mode==MODE_HIER ) do {
 	if ( hier_delay ) { hier_delay--; continue; }
	hier_delay = prop_delay;
	//set<catomID> new_pieces;
	//ostringstream s1, s2;
	// go through all edges, find best to merge
int nthistick;
for ( nthistick=50; nthistick; nthistick-- ) {
	set< pair<catomID,catomID> >::const_iterator ecit;
	edgemap::const_iterator eit = all_edges.begin();
	double bestval=-1;
	catomID bestA, bestB;
	while ( eit!=all_edges.end() ) {
	  catomID a,b;
	  a = eit->first.first;
	  b = eit->first.second;
	  double value=0;
	  // sum heuristic: use num obs on edge / size a+b
	  ecit = eit->second.begin();
	  while ( ecit != eit->second.end() ) {
	     value += L2(ecit->first)->my_obs[ecit->second].size();
	     ecit++;
	  }
	  value /= L2(a)->members.size() + L2(b)->members.size();
	  if (value>bestval) {
	     bestval=value;
	     bestA=a;
	     bestB=b;
	  }
	  eit++;
	}
	if (bestval<0) break;
	cerr << "HIER: " << bestA << "+" << bestB << " val " << bestval << endl;
	// get vectors of point positions on the edges
	Matrix X(2,0), Y(2,0);
	Matrix mdx(2,0), mdy(2,0);
	ecit = all_edges[ pair<catomID,catomID>(bestA,bestB) ].begin();
	while ( ecit != all_edges[ pair<catomID,catomID>(bestA,bestB) ].end() ) {
	  catomID a,b;
	  a = ecit->first;
	  b = ecit->second;
	  obslist::const_iterator ccobsit = L2(a)->my_obs[b].begin();
	  while ( ccobsit != L2(a)->my_obs[b].end() ) {
	      // get mean pos
	      ColumnVector tmpx(2);
	      tmpx << ( L2(a)->my_x.x + CatomSim::catom_radius * cos(L2(a)->my_x.theta+ccobsit->loc) )
		   << ( L2(a)->my_x.y + CatomSim::catom_radius * sin(L2(a)->my_x.theta+ccobsit->loc) );
	      X |= tmpx;
	      tmpx << cos(L2(a)->my_x.theta+ccobsit->loc) << sin(L2(a)->my_x.theta+ccobsit->loc);
	      mdx |= tmpx;
	      ColumnVector tmpy(2);
	      tmpy << ( L2(b)->my_x.x + CatomSim::catom_radius * cos(L2(b)->my_x.theta+ccobsit->rem) )
		   << ( L2(b)->my_x.y + CatomSim::catom_radius * sin(L2(b)->my_x.theta+ccobsit->rem) );
	      Y |= tmpy;
	      tmpy << cos(L2(b)->my_x.theta+ccobsit->rem) << sin(L2(b)->my_x.theta+ccobsit->rem);
	      mdy |= tmpy;
	      ccobsit++;
	  }
	  ecit++;
	}
	// get transform
	Matrix R(2,2);
	ColumnVector t2(2), t3(2);
	ralign( Y, X, R, t2, t3 );
	if ( !EQL(R(1,1),R(2,2)) || !EQL(R(1,2),-R(2,1)) ) cout << "HERE -- bad R:\n" << R;
	// sanity check Rotation
	//mdx /= X.Ncols();
	//mdy /= Y.Ncols();
	double dotp = DotProduct( mdx, R*mdy );
	if ( dotp>=0 ) {
	    // 180 degree rotation
	    cerr << "HERE -- need to reverse, dotp = " << dotp << endl;
	    R = -R;
	    t2 = t3;
	}
	double r = atan2( R(2,1), R(1,1) );
	// apply transform, given by R and t2, and merge piece members to newroot
	set<catomID>::const_iterator mit = L2(bestB)->members.begin();
	while ( mit != L2(bestB)->members.end() ) {
		catomID m = *mit;
		ColumnVector CV(2);
		CV << L2(m)->my_x.x << L2(m)->my_x.y;
		CV = t2 + R*CV;
		L2(m)->my_x.x = CV(1);
		L2(m)->my_x.y = CV(2);
		L2(m)->my_x.theta += r;
		L2(m)->my_root*=0;
		L2(m)->my_nei.clear();
		L2(m)->my_root = bestA;
		L2(bestA)->members.insert( m );
		mit++;
		L2(bestB)->members.erase( m );
	}
	// update edges
	mit = L2(bestB)->my_edges.begin();
	while ( mit != L2(bestB)->my_edges.end() ) {
		catomID c = *mit;
		if ( c!=bestA ) {
		    // copy edges and catoms on edges to bestA
		    ecit = all_edges[ pair<catomID,catomID>(bestB,c) ].begin();
		    while ( ecit != all_edges[ pair<catomID,catomID>(bestB,c) ].end() ) {
			pair<catomID,catomID> tmp = *ecit;
			all_edges[ pair<catomID,catomID>(bestA,c) ].insert( tmp );
			ecit++;
		    }
		    ecit = all_edges[ pair<catomID,catomID>(c,bestB) ].begin();
		    while ( ecit != all_edges[ pair<catomID,catomID>(c,bestB) ].end() ) {
			pair<catomID,catomID> tmp = *ecit;
			all_edges[ pair<catomID,catomID>(c,bestA) ].insert( tmp );
			ecit++;
		    }
		    L2(bestA)->my_edges.insert(c);
		    L2(c)->my_edges.insert(bestA);	
		}
		all_edges.erase( pair<catomID,catomID>(bestB,c) );
		all_edges.erase( pair<catomID,catomID>(c,bestB) );
		L2(c)->my_edges.erase(bestB);
		mit++;
		L2(bestB)->my_edges.erase(c);
	}
	pieces.erase(bestB);
	if (pieces.size() <= binout_last_m) do_binout();
	if (pieces.size()==1 && grad_steps) worldPtr->timesteps = worldPtr->current_time+grad_steps;
}
    } while (0);

    if ( group_k!=0.0 && mode==MODE_HIER ) do {
	if ( hier_delay ) { hier_delay--; continue; }
	hier_delay = prop_delay;

	// generate partition 
	// random group leaders
	set<catomID>::const_iterator pit = pieces.begin();
	int ngroups = 0;
	list<catomID> remaining;
	while( pit != pieces.end() ) {
	    if ( (((double)rand())/RAND_MAX) < (1.0 / group_k) ) {
		ngroups++;
		L2(*pit)->pgroup = ngroups;
	    } else {
		L2(*pit)->pgroup = 0;
		remaining.push_back( *pit );
	    }
	    pit++;
	}
	if (ngroups==0) continue;
	// put all remaining pieces into groups
	while ( remaining.size() ) {
	    catomID p = *(remaining.begin());
	    remaining.pop_front();
	    set<catomID>::const_iterator pnit = L2(p)->my_edges.begin();
	    while (pnit != L2(p)->my_edges.end() ) {
		int tmp = L2( *pnit )->pgroup;
		if (tmp) {
		    L2(p)->pgroup = tmp;
		    break;
		}
		pnit++;
	    }
	    if ( L2(p)->pgroup==0 ) remaining.push_back(p);
	}
	for ( ; ngroups; ngroups-- ) {
	    // go through all edges, find best to merge in this group
	    set< pair<catomID,catomID> >::const_iterator ecit;
	    edgemap::const_iterator eit;
	    double bestval=-1;
	    catomID bestA, bestB;
	    for ( eit=all_edges.begin(); eit!=all_edges.end(); eit++ ) {
		catomID a,b;
		a = eit->first.first;
		b = eit->first.second;
		if ( L2(a)->pgroup != ngroups || L2(b)->pgroup != ngroups ) continue;
		double value=0;
		// sum heuristic: use num obs on edge / size a+b
		ecit = eit->second.begin();
		while ( ecit != eit->second.end() ) {
		   value += L2(ecit->first)->my_obs[ecit->second].size();
		   ecit++;
		}
		value /= L2(a)->members.size() + L2(b)->members.size();
		if (value>bestval) {
		   bestval=value;
		   bestA=a;
		   bestB=b;
		}
	    }
	    if (bestval<0) continue;
	    cerr << "HIER: " << bestA << "+" << bestB << " val " << bestval << " group " << ngroups << endl;
	    // get vectors of point positions on the edges
	    Matrix X(2,0), Y(2,0);
	    Matrix mdx(2,0), mdy(2,0);
	    ecit = all_edges[ pair<catomID,catomID>(bestA,bestB) ].begin();
	    while ( ecit != all_edges[ pair<catomID,catomID>(bestA,bestB) ].end() ) {
		catomID a,b;
		a = ecit->first;
		b = ecit->second;
		obslist::const_iterator ccobsit = L2(a)->my_obs[b].begin();
		while ( ccobsit != L2(a)->my_obs[b].end() ) {
		    // get mean pos
		    ColumnVector tmpx(2);
		    tmpx << ( L2(a)->my_x.x + CatomSim::catom_radius * cos(L2(a)->my_x.theta+ccobsit->loc) )
			<< ( L2(a)->my_x.y + CatomSim::catom_radius * sin(L2(a)->my_x.theta+ccobsit->loc) );
		    X |= tmpx;
		    tmpx << cos(L2(a)->my_x.theta+ccobsit->loc) << sin(L2(a)->my_x.theta+ccobsit->loc);
		    mdx |= tmpx;
		    ColumnVector tmpy(2);
		    tmpy << ( L2(b)->my_x.x + CatomSim::catom_radius * cos(L2(b)->my_x.theta+ccobsit->rem) )
			<< ( L2(b)->my_x.y + CatomSim::catom_radius * sin(L2(b)->my_x.theta+ccobsit->rem) );
		    Y |= tmpy;
		    tmpy << cos(L2(b)->my_x.theta+ccobsit->rem) << sin(L2(b)->my_x.theta+ccobsit->rem);
		    mdy |= tmpy;
		    ccobsit++;
		}
		ecit++;
	    }
	    // get transform
	    Matrix R(2,2);
	    ColumnVector t2(2), t3(2);
	    ralign( Y, X, R, t2, t3 );
	    if ( !EQL(R(1,1),R(2,2)) || !EQL(R(1,2),-R(2,1)) ) cout << "HERE -- bad R:\n" << R;
	    // sanity check Rotation
	    //mdx /= X.Ncols();
	    //mdy /= Y.Ncols();
	    double dotp = DotProduct( mdx, R*mdy );
	    if ( dotp>=0 ) {
		// 180 degree rotation
		cerr << "HERE -- need to reverse, dotp = " << dotp << endl;
		R = -R;
		t2 = t3;
	    }
	    double r = atan2( R(2,1), R(1,1) );
	    // apply transform, given by R and t2, and merge piece members to bestA
	    set<catomID>::const_iterator mit = L2(bestB)->members.begin();
	    while ( mit != L2(bestB)->members.end() ) {
		catomID m = *mit;
		ColumnVector CV(2);
		CV << L2(m)->my_x.x << L2(m)->my_x.y;
		CV = t2 + R*CV;
		L2(m)->my_x.x = CV(1);
		L2(m)->my_x.y = CV(2);
		L2(m)->my_x.theta += r;
		L2(m)->my_root*=0;
		L2(m)->my_nei.clear();
		L2(m)->my_root = bestA;
		L2(bestA)->members.insert( m );
		mit++;
		L2(bestB)->members.erase( m );
	    }
	    // update edges
	    mit = L2(bestB)->my_edges.begin();
	    while ( mit != L2(bestB)->my_edges.end() ) {
		catomID c = *mit;
		if ( c!=bestA ) {
		    // copy edges and catoms on edges to bestA
		    ecit = all_edges[ pair<catomID,catomID>(bestB,c) ].begin();
		    while ( ecit != all_edges[ pair<catomID,catomID>(bestB,c) ].end() ) {
			pair<catomID,catomID> tmp = *ecit;
			all_edges[ pair<catomID,catomID>(bestA,c) ].insert( tmp );
			ecit++;
		    }
		    ecit = all_edges[ pair<catomID,catomID>(c,bestB) ].begin();
		    while ( ecit != all_edges[ pair<catomID,catomID>(c,bestB) ].end() ) {
			pair<catomID,catomID> tmp = *ecit;
			all_edges[ pair<catomID,catomID>(c,bestA) ].insert( tmp );
			ecit++;
		    }
		    L2(bestA)->my_edges.insert(c);
		    L2(c)->my_edges.insert(bestA);	
		}
		all_edges.erase( pair<catomID,catomID>(bestB,c) );
		all_edges.erase( pair<catomID,catomID>(c,bestB) );
		L2(c)->my_edges.erase(bestB);
		mit++;
		L2(bestB)->my_edges.erase(c);
	    }
	    pieces.erase(bestB);
	    if (pieces.size() <= binout_last_m) do_binout();
	    if (pieces.size()==1 && grad_steps) worldPtr->timesteps = worldPtr->current_time+grad_steps;
	}
    } while (0);

    num_msgs_this_tick = 0;
    msgs_pending = false;
    oracleYield();
  }
//} 
//catch(Exception) { cerr<<Exception::what()<<endl; }
//catch(...) { cerr << "Oracle died\n"; }
}

