///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright (C) 2006 by Intel Corporation and Carnegie Mellon University    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <iostream>
#include <errno.h>
#include <string.h>

#include "Localize.hxx"
#include "CatomWorld.hxx"
#include "CatomSim.hxx"

CODE_MODULE_DECLARATION( Localize, Localize );

using namespace std;


// Modes
#define M_ROOT 1
#define M_PART 2
#define M_FULL 3
#define M_NONE 4

#define set_color( r, g, b, a ) do { char* foo = (char*) &rootID; if (color_by_root) HOSTCATOM.setColor( foo[0], foo[1], foo[2], 0 ); else HOSTCATOM.setColor( r, g, b, a ); } while (0)
#define COLOR_INIT set_color( 128, 128, 128,   0 )
#define COLOR_NONE set_color( 255, 255,   0,   0 )
#define COLOR_PART set_color(   0, 255,   0,   0 )
#define COLOR_FULL set_color( 255,   0,   0,   0 )
#define COLOR_ROOT set_color( 128,   0, 255,   0 )
#define COLOR_DONE set_color(   0,   0, 200,   0 )
#define COLOR_DONE_ROOT set_color( 0, 200, 200, 0 )

// Messages
#define M_EMPTY    0
#define M_TREEACK  1
#define M_TREENACK 2
#define M_YOURPOS  3
#define M_MYPOS    4
#define M_OTHERPOS 5

dWorldID Localize::world = 0;
bool Localize::debug = false;
bool Localize::color_by_root = false;
bool Localize::move_catoms = false;
double Localize::root_P = 1.0;
catomID Localize::root_node = 0;

void Localize::simulationStart() {
  rootID = 0;
  parentID = 0;
  mode = M_NONE;
  changed = false;
  subtree_acked = false;
  unsigned int i;
  if (move_catoms) body = HOSTCATOMSIM->body;
  for (i=0; i<NUM_NEIGHBORS; i++) {
    featureID fid = HOSTCATOM.getNthFeature( i );
    if (fid) {
      info tmp;
      tmp.fid = fid;
      tmp.nid = HOSTCATOM.getNthNeighbor( i );
      tmp.rootID = 0;
      tmp.known_pos = false;
//      tmp.has_replied = false;
//      tmp.has_been_acked = false;
      neigh.push_back(tmp);
    } else break;
  }
  num_neighbors = i;
  num_replied = 0;
  COLOR_INIT;
}

#define PTSTR( p ) \
  "(" << (p).getX() << "," << (p).getY() << "," << (p).getZ() << ")"

#define PRINT_MSG( m ) \
do { if (!debug) break;\
  cout << hostCatom << ": got type=" << (int) (m).type << ", root=" << (m).rootID << " from " << (m).sender; \
  cout << " my " << PTSTR( (m).myPos ) << ", your " << PTSTR( (m).yourPos ); \
  cout << ", other " << PTSTR( (m).otherPos ) << " dist " << (m).otherDist << endl; \
} while(0)

#define PRINT_STATE \
do { if (!debug) break;\
  cout << hostCatom << ": state=" << mode << " root=" << rootID << " pos ="; \
  Point3D pt = Point3D( dBodyGetPosition( body ) ); \
  cout << PTSTR( pt ) << endl; \
} while(0)

void Localize::newTick() {
  						// first go through our messages
  mymsg msg;
  while (1) {
    msg = mail.read();
    if ( msg.type == M_EMPTY ) break;		// no more messages
    if ( msg.rootID < rootID ) continue;	// ignore if id is less
    PRINT_MSG (msg);
    if ( msg.rootID > rootID ) {		// become part of new coordinate frame
      rootID = msg.rootID;
      parentID = msg.sender;
      children.clear();
      num_replied = 0;
      mode = M_NONE;
      L.m = 0;
      subtree_acked = false;
      changed = false; //may not have the right info to send any kind of update yet
      PRINT_STATE;
      COLOR_NONE;
    }
    
    int i;					// find entry for message sender
    for (i=0; i<num_neighbors && neigh[i].nid != msg.sender; i++);
    if (i>= num_neighbors) continue;		// this ought not happen
    
    
    if(neigh[i].rootID < rootID) {		// 1st message from this neighbor, with this rootID
      neigh[i].rootID = rootID;
      neigh[i].known_pos = false;
//      neigh[i].has_replied = false;
//      neigh[i].has_been_acked = false;
      if (neigh[i].nid != parentID) {		// NACK if we are part of this coordinate frame already
        mymsg newmsg;
	newmsg.rootID = rootID;
	newmsg.sender = hostCatom;
	newmsg.type = M_TREENACK;
	mail.send( REMOTE_CODE_MODULE( neigh[i].nid, Localize )->mail, newmsg );
      }
    }
    
    						// handle tree ack & nack messages
    if ( (msg.type == M_TREEACK || msg.type == M_TREENACK) /*&& neigh[i].has_replied==false*/ ) {
//      neigh[i].has_replied = true;
      num_replied++;
      if (msg.type == M_TREEACK) children.insert( msg.sender );
    }
    
    						// update location info for neighbor
    if ( msg.type == M_MYPOS || msg.type == M_YOURPOS ) {
      neigh[i].pos = msg.myPos;
      neigh[i].known_pos = true;
      if (mode==M_NONE) changed=true; // may be able to send M_OTHER updates
    }
    
    						// switch from unconfigured to partially configured
    if ( msg.type == M_YOURPOS && mode == M_NONE ) {
      mode = M_PART;
      changed = true;
      dBodySetPosition( body, msg.yourPos.getX(), msg.yourPos.getY(), msg.yourPos.getZ() );
      PRINT_STATE;
      COLOR_PART;
    }
    
    						// partially configured from distance info
    if ( (msg.type==M_MYPOS || msg.type==M_OTHERPOS) && mode==M_NONE ) {
      int loopcount = 0;
      if ( msg.type == M_OTHERPOS ) loopcount = 1;
      while (loopcount < 2 ) {
        Point3D npt; 
	double ndist;
        if (loopcount) { npt = msg.otherPos; ndist = msg.otherDist; }
	else { npt = msg.myPos; ndist = 2; }
	switch ( L.m ) {
	case 0:					// unconstrained -> sphere
	  L.a = npt;
	  L.r = ndist;
	  if (L.r==0) L.m = 5;
	  else L.m = 1;
	  break;
	case 1:					// sphere + sphere -> circle
	{
	  L.b = npt - L.a;
	  double d = L.b.norm();
	  if (d==0 || d>L.r+ndist || d<fabs(L.r-ndist)) break;
	  L.b /= d;
	  double x = ( d*d - ndist*ndist + L.r*L.r ) * 0.5 / d;
	  L.a += L.b*x;
	  L.r = sqrt( L.r*L.r - x*x );
	  if (L.r == 0) L.m = 5;
	  else L.m = 2;
	  break;
	}
	case 2:					// circle + sphere -> two points
	{
	  double x = (npt - L.a).dot( L.b );	// project sphere into circle's plane
	  if (x>ndist || x<-ndist) break;
	  npt -= x*L.b;
	  ndist = sqrt( ndist*ndist - x*x );
	  npt -= L.a;				// circle intersection
	  double d = npt.norm();
	  if (d==0 || d>L.r+ndist || d<fabs(L.r-ndist)) break;
	  npt /= d;
	  x = ( d*d - ndist*ndist + L.r*L.r ) *0.5 / d;
	  L.a += npt*x;
	  L.r = sqrt( L.r*L.r - x*x );
	  npt = npt.cross( L.b );
	  L.b = L.a + npt*L.r;
	  L.a -= npt*L.r;
	  if (L.r == 0) L.m = 5;
	  else L.m = 3;
	  break;
	}
	case 4:					// two points + sphere -> one point
	{
	  double d1 = (npt - L.a).norm();
	  double d2 = (npt - L.b).norm();
	  if ( d1==d2 || ( d1 != ndist && d2 !=ndist ) ) break;
	  if ( d2==ndist ) L.a=L.b;
	  L.m = 5;
	  break;
	}
	default:
	  break;
	}
	if ( L.m==5 ) {				// one point
	  mode = M_PART;
	  changed = true;
	  dBodySetPosition( body, L.a.getX(), L.a.getY(), L.a.getZ() );
	  PRINT_STATE;
	  COLOR_PART;
	  break;
	}
	loopcount++;
      }
    }
    
    						// partially configured to fully configured
    if ( mode==M_PART ) {
      Point3D axis;
      dQuaternion q1, q2, q3;
      int j;
      for ( j=0; j<num_neighbors; j++) {
        if (neigh[j].rootID == rootID && neigh[j].known_pos) break;
      }
      if (j<num_neighbors) {
        Point3D a = HOSTCATOM.getFeature( neigh[j].fid )->getLocation();
	Point3D b = neigh[j].pos - Point3D( dBodyGetPosition( body ) );
	a /= a.norm();  // this should not be needed
	b /= b.norm();
	axis = a.cross( b );
	double angle = 2*asin( (a-b).norm() / 2 );
	dQFromAxisAndAngle( q1, axis.getX(), axis.getY(), axis.getZ(), angle );
	dBodySetQuaternion( body, q1 );
	axis = b;
      }
tryagain:
      for ( j++ ; j<num_neighbors; j++ ) {
        if (neigh[j].rootID == rootID && neigh[j].known_pos) break;
      }
      if (j<num_neighbors) {
        Point3D a = HOSTCATOM.getFeature( neigh[j].fid )->getLocation();
	dVector3 tmp;
	dBodyGetRelPointPos( body, a.getX(), a.getY(), a.getZ(), tmp );
	a = Point3D( tmp ) - Point3D( dBodyGetPosition( body ) );
	Point3D b = neigh[j].pos - Point3D( dBodyGetPosition( body ) );
	a -= a.dot(axis) * axis;
	b -= b.dot(axis) * axis;
	if ( a!=Point3D(0,0,0) && b!=Point3D(0,0,0) ) {
	  a /= a.norm();
	  b /= b.norm();
	  double angle = 2*asin( (a-b).norm() / 2 );
	  if ( a.cross(b).dot(axis) < 0 ) angle = -angle;
	  dQFromAxisAndAngle( q2, axis.getX(), axis.getY(), axis.getZ(), angle );
	  dQMultiply0( q3, q2 ,q1 );
	  dBodySetQuaternion( body, q3 );
          mode = M_FULL;
	  changed = true;
	  PRINT_STATE;
	  COLOR_FULL;
	} else goto tryagain;
      }
    }
    
    						// subtree completion
    if (subtree_acked == false && num_replied == num_neighbors) {
      if ( mode==M_FULL ) {			// send ack to parent, but only after localized
        mymsg newmsg;
	newmsg.rootID = rootID;
	newmsg.sender = hostCatom;
	newmsg.type = M_TREEACK;
	mail.send( REMOTE_CODE_MODULE( parentID, Localize )->mail, newmsg );
	subtree_acked = true;
	COLOR_DONE;
      }
      if ( mode==M_ROOT ){			// intial localization complete
        subtree_acked = true;
	COLOR_DONE_ROOT;
      }
    }
    
  }
  
  
  						// become root if needed
  if ( mode==M_NONE && rootID==0 ) {
    if (
    //hostCatom ==  2146410762
    //hostCatom ==  8285608|| 0<1 
      hostCatom == root_node || rand() < ((double)RAND_MAX+1)*root_P
    ) {
      rootID = hostCatom;
      parentID = 0;
      mode = M_ROOT;
      num_replied = 0;
      dBodySetPosition (body, 100, 100, 100);
      subtree_acked = false;
      changed = true;
      PRINT_STATE;
      COLOR_ROOT;
    }
  }
  
  
  						// send updates
  if (changed) {
    mymsg newmsg;
    newmsg.sender = hostCatom;
    newmsg.rootID = rootID;
    newmsg.myPos = Point3D( dBodyGetPosition( body ) );
    int j,k;
    for ( j=0; j<num_neighbors; j++ ) {
      if ( neigh[j].known_pos && neigh[j].rootID==rootID ) break;
    }
    for ( k=0; k<num_neighbors; k++ ) {
      if ( mode==M_FULL || mode==M_ROOT ) {
        newmsg.type = M_YOURPOS;
	Point3D tmp1 = HOSTCATOM.getFeature( neigh[k].fid )->getLocation() * 2;
	dVector3 tmp2;
	dBodyGetRelPointPos( body, tmp1.getX(), tmp1.getY(), tmp1.getZ(), tmp2 );
	newmsg.yourPos = Point3D( tmp2 );
      } else {
        if ( j<num_neighbors ) {
	  newmsg.otherPos = neigh[j].pos;
	  newmsg.otherDist = 2*( HOSTCATOM.getFeature( neigh[k].fid )->getLocation() - HOSTCATOM.getFeature( neigh[j].fid )->getLocation() ).norm();
	} else {
	  newmsg.otherPos = newmsg.myPos;
	  newmsg.otherDist = 2;
	}
	if ( mode==M_PART ) newmsg.type = M_MYPOS;
	else newmsg.type = M_OTHERPOS;
      }
      mail.send( REMOTE_CODE_MODULE( neigh[k].nid, Localize )->mail, newmsg );
    }
  }
  changed = false;
}

void Localize::endTick() {
  mail.done_this_tick();
}

