///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright (C) 2006 by Intel Coproration and Carnegie Mellon University    //
// Contacts: casey.j.helfrich @ intel.com                                    //
//           bdr @ cs.cmu.edu                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "GradientHierarchy.hxx"

#include "CatomSim.hxx"
#include "CatomWorld.hxx"

using namespace std;

GradientMsg::GradientMsg(GradientHierarchy* _hierarchy, unsigned _depth) :
  Message(_hierarchy->gradientBox),
  depth(_depth) { }

GradientMsg::GradientMsg(MailboxID _mailboxID, unsigned _depth) :
  Message(_mailboxID),
  depth(_depth) { }
GradientMsg::GradientMsg(GradientHierarchy* _hierarchy, unsigned long long _msgID, unsigned _depth) :
  Message(_hierarchy->gradientBox, _msgID),
  depth(_depth) { }

GradientMsg::GradientMsg(MailboxID _mailboxID, unsigned long long _msgID, unsigned _depth) :
  Message(_mailboxID, _msgID),
  depth(_depth) { }

GradientHierarchy::GradientHierarchy(catomID _hostCatom, string _hierarchyName) :
  Hierarchy(_hostCatom, _hierarchyName),
  gradientBox("_gradient_" + _hierarchyName),
  upstreamFeature(0),
  isRoot(false) { }

GradientHierarchy::~GradientHierarchy() { }

void GradientHierarchy::simulationStart() {
  Hierarchy::simulationStart();

  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(gradientBox,
								   this,
								   (msgHandlerPtr)&GradientHierarchy::gradientHandler);
}

void GradientHierarchy::endTick() {
  if(worldPtr->current_time == 0) {
    string rootIDString = worldPtr->search_key_value_list("GRADIENTROOT");
    long long rootID = strtoll(rootIDString.c_str(), NULL, 10);
    if(!rootID) {
      // Wasn't directly a number--try it as a path
      FILE* f = fopen(rootIDString.c_str(), "r");
      if(!f) {
	cerr << "GRADIENTROOT wasn't an ID or an openable path\n";
	exit(-1);
      }

      char inBuf[32];
      if(!fgets(inBuf, 32, f)) {
	cerr << "Couldn't read from file specified as GRADIENTROOT\n";
	exit(-1);
      }

      fclose(f);

      rootID = strtoll(inBuf, NULL, 10);
      if(!rootID && (errno == EINVAL)) {
	cerr << "Couldn't interpret GRADIENTROOT value: " << rootIDString << endl;
	exit(-1);
      }
    }
    
    if(rootID == hostCatom)
      emitGradient();
  }

  Hierarchy::endTick();
}

bool GradientHierarchy::gradientHandler(Message* _msg) {
  GradientMsg* msg = (GradientMsg*)_msg;

  if(!isRoot && (upstreamFeature == 0)) {
    upstreamFeature = msg->arrivalContact;
    catomID upstreamCatom = worldPtr->catomHash[hostCatom]->C.getFeatureMap()[upstreamFeature].getNthRemoteFeature(0)->getHostcatomID();
    worldPtr->oStart();
    cerr << worldPtr->current_time << " beacon heard by " << hostCatom
	 << " (depth " << msg->depth << ") from " << upstreamCatom << endl;
    worldPtr->oEnd();

    setParent(upstreamCatom);

    // Propagate the gradient
    for(unsigned int i=1; i<=NUM_FEATURES; i++) {
      Feature* contact = &((worldPtr->catomHash[hostCatom]->C.getFeatureMap())[i]);
      contact->getNetworkAdapter()->sendMessage(new GradientMsg(this, (msg->depth + 1)));
    }
  }

  return false;
}

void GradientHierarchy::emitGradient() {
#if 1
  worldPtr->oStart();
  cerr << "Catom " << hostCatom << " knows it's the root; emitting gradient...\n";
  worldPtr->oEnd();
#endif
  isRoot = true;
  for(unsigned int i=1; i<=NUM_FEATURES; i++) {
    Feature* contact = &((worldPtr->catomHash[hostCatom]->C.getFeatureMap())[i]);
    contact->getNetworkAdapter()->sendMessage(new GradientMsg(this,0));
  }
}

bool GradientHierarchy::sendMsgToParent(Message* msg) {
  if(isRoot) {
    // Root doesn't need to send anything to the parent
    delete msg;
    return false; // this is *not* the same memory management contract as NetworkAdapter::sendMessage()
  } else if (upstreamFeature == 0) {
    delete msg;
    return false; // this is *not* the same memory management contract as NetworkAdapter::sendMessage()
  } else {
    Feature* contact = &((worldPtr->catomHash[hostCatom]->C.getFeatureMap())[upstreamFeature]);
    return contact->getNetworkAdapter()->sendMessage(msg);
  }
}
