// FBHoleMotion.hxx
// Nels Beckman
// September 11th, 2007
// Hilariously close to the ICSE deadline!
//
// 2D hole motion with failure handling.

#ifndef __FBHM_H
#define __FBHM_H

#include <pthread.h>
#include <list>

#include "FBTemplates.hxx"

CODE_MODULE_PROTOTYPE(FBHoleMotion,FBHoleMotion);

using namespace std;

enum CatomState { IDLE, BUSY, NEW_LEADER, SMOOTHING };

enum Letter { A, B, C, D, E, F, G, H, I, J, K, L };

enum CanMove { Empty, Bounce, Yes };

enum RotDir { CW, CCW };

enum Geom { Circle, BackEdge, Mover, FrontEdgeEP };

struct SEResult { 
  list<Point3D> points;
  list<FBHoleMotion*> movers;
  list<FBHoleMotion*> other_nodes;
  list<pair<featureID, featureID> > trail;
};

class FBHoleMotion : public FBCodeModule<FBHoleMotion> {
  
protected:
  Catom* catomPtr;

public:
  CatomState myState;
  catomID myOwner; 
  pthread_mutex_t localLock;

protected:
  // How long to wait until trying again with the main thread.
  pair<int,int> myWaitPeriod;

  // We are trying to avoid using heap state, so this will only
  // be up to date immediately after transfer of control to a new
  // leader. Please don't use otherwise...
  struct { Letter Pos; featureID Dir;
    pair<featureID, featureID> EmptySide; } newLeaderInfo;

  pair<featureID, featureID> findFullEmptyEmptyFull(void);

public:
  FBHoleMotion(catomID _hostCatom);
  virtual void thread();
  virtual void simulationEnd();

  // FAILURE BLOCK METHODS

  // Defines a compensating action that resets the owner of the given host.
  void addUndoComp(catomID);

  // When called, I find the two new collabs (or more, I guess depending
  // on the Geom) and then call atomicRemoveAndAddCollabs on my own
  // transaction manager.
  void changeCollabs( Letter, Tid, Geom, pair<featureID, featureID>,
		      bool, bool, featureID=0);

  // NON RPC METHODS
  void start();

  void reset();

  // After we know we have a hole ready to move, calling this method
  // signifies a state transition into the movement phase. You pass it
  // the featureID that specifies our direction of movement.
  // The last argument is the number of times we should try.
  void beginMovement(Letter, featureID, pair<featureID, featureID>, int);

  // After we have moved one step, we have to hand off the whole thing
  // to a new leader. This method does exactly that. This involves
  // clearing the back edge, then setting a "you're the leader" flag
  // on the next catom in the direction of movement.
  // The arguments are, 'my current position,' 'dir of movement,' and
  // 'empty side,' for alignment.
  void handOffMovement(Letter, featureID, pair<featureID, featureID>);

  // This method should transition our hole to a bouncing state.
  // We must choose a new direction, and then hand off to the
  // appropriate new leader.
  void bounce(Letter, featureID, pair<featureID, featureID>);
  
  // Calling this method on the hole leader indicates that we have detected
  // and empty edge, and we will try to collapse the circle.
  void collapseHole(Letter, featureID, pair<featureID, featureID> );

  // We know that we just created the hole, and moved one step, so try to 
  // smooth it over.
  void smooth( Letter, pair<featureID, featureID> );

  // RPC METHODS

  // to be called on remote catoms. If they are able to create
  // a hole as the given member of the shephard group, it changes
  // its state, adds a leader, and returns true, else false.
  list<pair<FBHoleMotion*, Letter> >
  canCreateHoleTAS(Letter, pair<featureID, featureID>, catomID);

  // This is the function that moves ensures we can take one step forward.
  // It has a lot of arguments. We pass the current position of the receiver
  // catom, the direction in which the hole wants to move, the direction
  // around the circle we are passing the message, and the empty side for
  // alignment. If the receiving catom is willing to move the circle, 
  // it will return Yes and the list of the three catoms that will actually
  // be moved.
  pair<CanMove, list<pair<Letter, FBHoleMotion*> > > 
  setLeadingEdge( Letter, featureID, RotDir, pair<featureID, featureID> );

  // This function is called on the catoms that would be the next edge
  // in the shepherd group. Returns true if basically if they are not
  // part of another group.
  bool canBeNewEdge(catomID);

  // Find three empty is an RPC method that attempts to find an empty
  // edge on the moving hole that can be collapsed.
  list<Catom*> findThreeEmpty( Letter, RotDir, Letter, 
			       pair<featureID, featureID>, list<Catom*> );

  // Remotely called on the catom that will become the next leader.
  void setNewLeader( Letter, featureID, pair<featureID, featureID> );

  // Should reset a new leader the entire way around a circle. Stops
  // when it gets to an element of the circle that already has the new
  // leader.
  void setNewLeaderCircle( Letter, catomID, RotDir, 
			   pair<featureID, featureID>, int );

  // This method tells the entire circle that we have a new leader,
  // just like the above method. However, this method differs in that
  // it is meant to be used in situations where the new leader does not
  // yet know that he is meant to be the new leader, and therefore we
  // must find out his catomID.
  catomID setNewLeaderCircle2( Letter, Letter, catomID, RotDir, featureID,
			       pair<featureID, featureID>, int );

  // Not always called! Only called if the catom that used to be the
  // leader was in the A position. It will need those catom ptrs.
  list<FBHoleMotion*> smoothEdgeMover( int, catomID, featureID );
			       
  // Whoa I will never again be able to figure out this method.
  // The tuple returned holds a.) the list of empty spots to which we can
  // move, b.) a list of catoms that can move there, if necessary (ie. we
  // are in position A), and c.) a list of the next feature back along the
  // route that we followed, and another feature if we saved a spot.
  //pair<list<Point3D>, pair<list<Catom*>,
  // list<pair<featureID, featureID> > > >
  SEResult smoothEdgePlaceholder( int, int, int, catomID,
				  pair<featureID,featureID> );

  // If we are trying to smooth a section, and it doesn't work, we can call
  // this to clear the mover catoms.
  void resetFailedSEMover(int, catomID, featureID);

  // Resets a successful smoothing operation.
  // This method follows a trail of feature IDs, setting each one to have
  // no leader & IDLE state upon its return.
  void resetSmoothCatoms( catomID, int, list<pair<featureID, featureID> > );

  // Method used remotely to clear an edge. You pass it the owner, and it
  // will clear itself if that is its owner. Moreover, if count > 0, it
  // will recursively call itself on the same feature, sending the clear
  // message out in a straight line, count more times.
  void resetEdge(catomID, featureID, int);

  // Releases the nodes after a hole has collapsed. Keep in mind that
  // in this method of all the methods, the 'my_pos' argument has a
  // different meaning. What is normally A-L may not be here. In
  // particular, the outside is cleared like a normal circle, while the
  // inside is cleared with completely new letters.
  // THIS METHOD IS NASTY!!
  void resetCollapsed( catomID, Letter, featureID, RotDir, bool, 
		       pair<featureID, featureID> );

  // This function resets the state to IDLE, if owner matches his owner.
  void resetState(catomID);

  // If the given owner matches the owner of this catom, we set
  // our state to IDLE. If this is true, the state recursively calls
  // resetStates on other group catoms, depending on what state is was in.
  void resetStates(catomID, Letter, pair<featureID, featureID>);
};

#endif 
