///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright (C) 2006 by Intel Coproration and Carnegie Mellon University    //
// Contacts: mderosa@cs.cmu.edu						                         //
//           									                             //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <algorithm>
#include <iostream>
#include <errno.h>
#include <string.h>
#include <stdlib.h>

#include "DistributedWatchpoint.hxx"
#include "WPManager.hxx"

//#define TEMPORAL_SHIFT //activate temporal shift on each hop

CODE_MODULE_DECLARATION( DistributedWatchpoint, DistributedWatchpoint );

using namespace std;

MailboxID watchpointMessagingBox = "watchpointMessagingBox";
static Watchpoint *wpMaster;
//global settings/values
Watchpoint *DistributedWatchpoint::wpMaster = NULL;
set<string> DistributedWatchpoint::wpNames;
int DistributedWatchpoint::maxTemporal = 0;
map<string,int> DistributedWatchpoint::minExtent;
map<string,int> DistributedWatchpoint::maxExtent;
unsigned DistributedWatchpoint::liveCount[10];
unsigned DistributedWatchpoint::deadCount[10];
unsigned long DistributedWatchpoint::matches = 0;
bool DistributedWatchpoint::allEnabled = true;
bool DistributedWatchpoint::linearOnly = false;
bool DistributedWatchpoint::enableEarlyTerm = true;
bool DistributedWatchpoint::enableNeighborSelect = true;

void DistributedWatchpoint::simulationStart() {
	steps = 0;
	
	//register for messages
	worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(watchpointMessagingBox,
		this,
		(msgHandlerPtr)&DistributedWatchpoint::watchpointMsgHandler);

	// prepare watchpoint support 
	pthread_mutex_lock(&wp_debug_lock);	//only one thread should build the master object/prep global settings
	if (!wpMaster && allEnabled) {
		//linear wps only?
		string linear_str =  worldPtr->search_key_value_list("WPD_LINEAR"); 
		if (linear_str != "") {
			linearOnly = (bool) atoi(linear_str.c_str());
		}
		//enable early termination?
		string term_str =  worldPtr->search_key_value_list("WPD_EARLY_TERMINATION"); 
		if (term_str != "") {
			enableEarlyTerm = (bool) atoi(term_str.c_str());
		}
		//neighbor candidate culling?
		string cull_str =  worldPtr->search_key_value_list("WPD_NEIGHBOR_CULLING"); 
		if (cull_str != "") {
			enableNeighborSelect = (bool) atoi(linear_str.c_str());
		}
		string watchpoint_str = worldPtr->search_key_value_list("WPD");
		if (watchpoint_str != "") {
			wpMaster =  parseWP(watchpoint_str);
			//determine temporal extent for each variable, and full-catom-state
			wpNames = wpMaster->watchedVariables();
			maxTemporal = 0;
			for (set<string>::const_iterator i = wpNames.begin(); i != wpNames.end(); i++) {
				#ifdef TEMPORAL_SHIFT
					if (!linearOnly) 
						minExtent[*i] = wpMaster->minExtent(*i) - (wpNames.size() * wpNames.size());
					else
						minExtent[*i] = wpMaster->minExtent(*i) - wpNames.size(); 
				#else
					minExtent[*i] = wpMaster->minExtent(*i);
				#endif
				maxExtent[*i] = wpMaster->maxExtent(*i);
				if (maxExtent[*i] > maxTemporal) maxTemporal = maxExtent[*i];
			}
		}
		else allEnabled = false;
	}
	pthread_mutex_unlock(&wp_debug_lock);	
}

//oracle used only for Act-Count processing
void DistributedWatchpoint::oracle() {
  //running concurrently with simulationStart
  oracleYield();
  //running with other oracles only
  oracleYield();
  while (1) {
    //running concurrently with newTick, endTick
    oracleYield();
    //running with other oracles only
		if (allEnabled && (wpMaster->action == Act_Count)) {
			//print out counters
			cout << "live:";
			for (unsigned i = 0; i < 10; i++) {cout << liveCount[i] << " ";}
			cout << endl;

			cout << "dead:";
			for (unsigned i = 0; i < 10; i++) {cout << deadCount[i] << " ";}
			cout << endl;
			
			cout << "wp step end, " << matches << " matches" << endl;

			//reset master counters
			for (unsigned i = 0; i < 10; i++) {
				liveCount[i] = deadCount[i] = 0;
			}
			matches = 0;
		}
    oracleYield();
  }
}

void DistributedWatchpoint::newTick() {
	if (!allEnabled) return;
	
	CatomSim *currCatom = worldPtr->catomHash[hostCatom];
	if (steps == 0) {
		//initialize local tracking structures
		for (set<string>::const_iterator i = wpNames.begin(); i != wpNames.end(); i++) {
				savedState[*i] = deque<float>();
		}
	}
	steps++;
	//save state
	unsigned maxVSize = 1;
	for (set<string>::const_iterator i = wpNames.begin(); i != wpNames.end(); i++) {
		unsigned vSize = maxExtent[*i] - minExtent[*i] + 1;
		if (vSize > maxVSize) maxVSize = vSize;
		//get value of i for catom j
		savedState[*i].push_front(getState(*i));
		//trim vector to correct length if too long
		while (savedState[*i].size() > vSize) savedState[*i].pop_back();
	}
	
	//MDR-TODO:store full state of every catom (if needed)
	if (maxTemporal > 0) {
	
	}
	
	//don't try to match if we don't have enough state
	if (steps < maxVSize) return;
	
	//generate new matcher
	Watchpoint *newWP = wpMaster->clone();
	activeMatchers.insert(newWP);
	
	//fill next slot of all matchers in set
	for (set<Watchpoint*>::const_iterator i = activeMatchers.begin(); i != activeMatchers.end(); i++) {
		(*i)->propagate((*i)->nextSlot(),currCatom,this);
	}
	
	//test & spread existing matchers
	Feature* featureMap = currCatom->C.getFeatureMap();
	while (activeMatchers.size()) {
		//for each matcher...
		set<Watchpoint*> tempSet;
		for (set<Watchpoint*>::const_iterator i = activeMatchers.begin(); i != activeMatchers.end(); i++) {
			//check for termination/success
			TriState sat = (*i)->isSatisfied(false);
			switch (sat) {
				case yes:
					switch((*i)->action) {
						case Act_Halt:
							cout << "DPRSim:watchpoint triggered. Simulator will now halt." << endl;
							exit(0);
							break;
						case Act_Count:
							liveCount[(*i)->filledSize()]++;
							matches++;
							break;
						case Act_Arrows:
							cout << "DPRSim:arrows not yet supported on distributed watchpoints." << endl;
							break;
					}
					break;
				case no:
					if ((*i)->action == Act_Count) deadCount[(*i)->filledSize()]++;
					if (enableEarlyTerm || (*i)->isFull()) break;//if early termination disabled, non-full always propagates
				case maybe:
					if ((*i)->action == Act_Count) liveCount[(*i)->filledSize()]++;
					//if not terminated, get uniqued list of possible neighbors,spread,and propagate values
					if ((*i)->isFull()) 
						cout << "DPRSim:watchpoint full, but not verified. Uncool." << endl;
					else {
						//send to 1-hop neighbors here
						set<catomID> possible = possibleNeighbors(*i);
						for (unsigned k = 0; k < NUM_FEATURES; k++) {
							catomID kthFeatureNeighbor = currCatom->C.getNeighbor(k);
							if (kthFeatureNeighbor && (possible.find(kthFeatureNeighbor) != possible.end())) {
									//clone & send
									Watchpoint *newWP = (*i)->clone();
									WatchpointMessage *msg = new WatchpointMessage(watchpointMessagingBox, newWP);
									featureMap[k].getNetworkAdapter()->sendMessage(msg);
								}
							}
						
						//start routing to all distant neighbors
						possible = possibleDistantNeighbors(*i);
						//for each remote candidate...
						for (set<catomID>::const_iterator j = possible.begin(); j != possible.end(); j++) {
							//find the first hop
							Watchpoint *newWP = (*i)->clone();
							newWP->inTransit = true;
							newWP->destination = worldPtr->catomHash[(*j)];
							catomID firstHop = newWP->nextRoutingStep(currCatom)->C.getID();
							//locate the feature associated w/ first hop...
							for (unsigned k = 0; k < NUM_FEATURES; k++) {
								catomID kthFeatureNeighbor = currCatom->C.getNeighbor(k);
								if (firstHop == kthFeatureNeighbor) {
									//send routed message to first hop
									WatchpointMessage *msg = new WatchpointMessage(watchpointMessagingBox, newWP);
									featureMap[k].getNetworkAdapter()->sendMessage(msg);
								}
							}
						}
					}
					break;
			}
			delete (*i);			
		}
		swap(tempSet,activeMatchers);	
	}
	
	//spread rerouted matchers to our 1-hop neighbors
	for (set<Watchpoint*>::const_iterator i = reroutedMatchers.begin(); i != reroutedMatchers.end(); i++) {
		//send to neighbors here
		set<catomID> possible = possibleNeighbors(*i);
		for (unsigned k = 0; k < NUM_FEATURES; k++) {
			catomID kthFeatureNeighbor = currCatom->C.getNeighbor(k);
			if (kthFeatureNeighbor && (possible.find(kthFeatureNeighbor) != possible.end())) {
				//send (w/ clone)
				WatchpointMessage *msg = new WatchpointMessage(watchpointMessagingBox, (*i)->clone());
				featureMap[k].getNetworkAdapter()->sendMessage(msg);
			}
		}
		delete (*i);
	}
	reroutedMatchers.clear();
}

bool DistributedWatchpoint::watchpointMsgHandler(WatchpointMessage* msg) {
	//if desired, temporal shift matcher
#ifdef TEMPORAL_SHIFT
	msg->payload->tShift();
#endif
	if (msg->payload->inTransit) {
		//are we the current destination?
		if (worldPtr->catomHash[hostCatom] == msg->payload->destination) {
			if (isDuplicateMultihop(msg)) { //if the message is a duplicate (see below), delete it
				delete msg->payload;
			}
			else {
				msg->payload->inTransit = false;
				//queue to propagate to our 1-hop neighbors (don't fill state info again)
				reroutedMatchers.insert(msg->payload);
			}
		}
		else {
			//forward on to next hop
			Feature* featureMap = worldPtr->catomHash[hostCatom]->C.getFeatureMap();
			CatomSim* nextStep = msg->payload->nextRoutingStep(worldPtr->catomHash[hostCatom]);
			if (nextStep->isTouching(worldPtr->catomHash[hostCatom])) {
				featureID outFeature = worldPtr->catomHash[hostCatom]->C.getFeatureTouching(nextStep->C.getID());
				WatchpointMessage *newMsg = new WatchpointMessage(watchpointMessagingBox, msg->payload);
				featureMap[outFeature].getNetworkAdapter()->sendMessage(newMsg);
			}
			else {
				delete msg->payload;
			}
		}
	}
	else {
		//add incoming matcher to local set (state info will be filled)
		activeMatchers.insert(msg->payload);
	}
  return false;
}

float DistributedWatchpoint::getState(string varname) {
	return WPManager::getState(worldPtr->catomHash[hostCatom],varname);
}

float DistributedWatchpoint::lookupSavedState(string varname,int offset) {
	return savedState[varname][maxTemporal - offset];
}

//calculate 1-hop neighbors
set<catomID> DistributedWatchpoint::possibleNeighbors(Watchpoint *wp) {
	set<catomID>candidates;
	CatomSim *currCatom = worldPtr->catomHash[hostCatom];
	//get all neighbors of current catom
	for (unsigned i = 0; i < NUM_FEATURES; i++) {
		catomID currNeighbor = currCatom->C.getNthNeighbor(i);
		if (currNeighbor) 
			candidates.insert(currNeighbor);
		else break;
	}
	//remove the catoms already in the wp
	for (unsigned i = 0; i < wp->c_list->names.size(); i++) {
		if (wp->c_list->names[i].second) {
			candidates.erase(wp->c_list->names[i].second->C.getID());
		}
	}
	//cout << candidates.size() << endl;
	return candidates;
}

//calculate distant neighbors
set<catomID> DistributedWatchpoint::possibleDistantNeighbors(Watchpoint*wp) {
	set<catomID>candidates;
	if (linearOnly) return candidates; //never spawn any multihop messages

	//add every other catom in watchpoint to set 	
	for (unsigned i = 0; i < wp->c_list->names.size(); i++) {
		if (wp->c_list->names[i].second) {
	 		candidates.insert(wp->c_list->names[i].second->C.getID());
	 	}
	}
	//remove this catom
	candidates.erase(worldPtr->catomHash[hostCatom]->C.getID());

	if (!enableNeighborSelect) return candidates; //don't cull impossible remote neighbors
		
	set<catomID> passed;
	//try each candidate, and add those that passed to the set
	 for (set<catomID>::iterator j = candidates.begin();j != candidates.end(); j++) {
	 	if (wp->isValidCandidate(worldPtr->catomHash[*j],wp->nextSlot()))
	 		passed.insert(*j);
	 }
	return passed;
}

//is the current catom the neighbor of two (or more) catoms in the wp?
//did the message not come from the highest slot numbered of those catoms?
//else false
bool DistributedWatchpoint::isDuplicateMultihop(WatchpointMessage* msg) {
	CatomSim *currCatom = worldPtr->catomHash[hostCatom];
	CatomList *cList = msg->payload->c_list;
	catomID senderCatom = currCatom->C.getNeighbor(msg->arrivalContact);
	if (!senderCatom) return false;
	int senderIdx = cList->slotIdx(worldPtr->catomHash[senderCatom]); //get slot index of sender
	//find max slot index of other neighbors (if any)
	int maxIdx = senderIdx;
	for (int i = 0; i < (int)cList->names.size(); i++) {
		if ((cList->names[i].second) &&
			  (currCatom->C.getFeatureTouching(cList->names[i].second->C.getID()))) {
				maxIdx = i;
			}
	}
	if (maxIdx > senderIdx) return true; //message should properly come from max numbered neighbor
	return false;
}
