///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright (C) 2006 by Intel Coproration and Carnegie Mellon University    //
// Contacts: casey.j.helfrich @ intel.com                                    //
//           bdr @ cs.cmu.edu                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "DPRHierarchy.hxx"

#include "CatomWorld.hxx"
#include "CatomSim.hxx"
#include "SpeculationManager.hxx"
#include "StatsManager.hxx"

extern StatsManager* statsMgrPtr;

static bool cutthroughEnabled = false;
static bool pref_useFanoutControl = true;
static long beaconingFrequency = 0;
static unsigned long maxVisibilityDistance = 0;
static string speculationMode;
static unsigned long speculationLatency = 0;

using namespace std;

LandmarkMsg::LandmarkMsg(DPRHierarchy* _hierarchy, catomID dest, 
			 Message* _payload, unsigned _distTraveled) : 
  Message(_hierarchy->hierarchyMessageBox), 
  destination(dest), 
  payload(_payload),
  distanceTraveled(_distTraveled) { }
LandmarkMsg::LandmarkMsg(MailboxID _mailboxID, catomID dest, 
			 Message* _payload, unsigned _distTraveled) : 
  Message(_mailboxID), 
  destination(dest), 
  payload(_payload),
  distanceTraveled(_distTraveled) { }
LandmarkMsg::LandmarkMsg(DPRHierarchy* _hierarchy, unsigned long long _msgID,
			 catomID dest, 
			 Message* _payload, unsigned _distTraveled) : 
  Message(_hierarchy->hierarchyMessageBox, _msgID), 
  destination(dest), 
  payload(_payload),
  distanceTraveled(_distTraveled) { }
LandmarkMsg::LandmarkMsg(MailboxID _mailboxID, unsigned long long _msgID,
			 catomID dest, 
			 Message* _payload, unsigned _distTraveled) : 
  Message(_mailboxID, _msgID), 
  destination(dest), 
  payload(_payload),
  distanceTraveled(_distTraveled) { }

LandmarkMsg::~LandmarkMsg() {
  if(payload)
    delete payload;
  statsMgrPtr->log("landmarkMsg-distanceTraveled", distanceTraveled, "n/a");
}

static bool landmarkMsgSizeParamsChecked = false;
static long long landmarkMsgHeaderSizeOverride = -1;
static long long landmarkMsgPayloadSizeOverride = -1;
static void checkLandmarkMsgSizeParams() {
  if(landmarkMsgSizeParamsChecked)
    return;

  string headerString = worldPtr->search_key_value_list("HEADERSIZE");
  if(headerString != "")
    landmarkMsgHeaderSizeOverride = strtoll(headerString.c_str(), NULL, 0);

  string payloadString = worldPtr->search_key_value_list("PAYLOADSIZE");
  if(payloadString != "")
    landmarkMsgPayloadSizeOverride = strtoll(payloadString.c_str(), NULL, 0);

  landmarkMsgSizeParamsChecked = true;
}

unsigned long long LandmarkMsg::headerSize() {
  if(!landmarkMsgSizeParamsChecked)
    checkLandmarkMsgSizeParams();

  if(landmarkMsgHeaderSizeOverride >= 0)
    return landmarkMsgHeaderSizeOverride;
  else
    return sizeof(catomID);
}

unsigned long long LandmarkMsg::payloadSize() {
  if(!landmarkMsgSizeParamsChecked)
    checkLandmarkMsgSizeParams();

  if(landmarkMsgPayloadSizeOverride >= 0)
    return landmarkMsgPayloadSizeOverride;
  else
    return sizeof(distanceTraveled) + payload->headerSize() + payload->payloadSize();
}

RequestPromotionMsg::RequestPromotionMsg(DPRHierarchy* _hchy) :
  Message(_hchy->reqPromotionBox) { }

RequestPromotionMsg::RequestPromotionMsg(MailboxID _mbox) :
  Message(_mbox) { }

LandmarkBeacon::LandmarkBeacon(DPRHierarchy* _hierarchy, catomID _id, 
			       long int _beaconVersionID, 
			       unsigned int _visDistance, 
			       unsigned int _distanceSoFar, 
			       int _level, catomID _parent) : 
  Message(_hierarchy->landmarkBeaconBox), 
  id(_id), 
  beaconVersionID(_beaconVersionID), 
  visDistance(_visDistance), 
  distanceSoFar(_distanceSoFar), 
  level(_level), 
  parent(_parent) { }
LandmarkBeacon::LandmarkBeacon(MailboxID _mailboxID, catomID _id, 
			       long int _beaconVersionID, 
			       unsigned int _visDistance, 
			       unsigned int _distanceSoFar, 
			       int _level, catomID _parent) : 
  Message(_mailboxID), 
  id(_id), 
  beaconVersionID(_beaconVersionID), 
  visDistance(_visDistance), 
  distanceSoFar(_distanceSoFar), 
  level(_level), 
  parent(_parent) { }

BeaconRequestBeacon::BeaconRequestBeacon(MailboxID _mailboxID,
					 simtime _requestTime,
					 unsigned _ttl,
					 unsigned _distSoFar) :
  Message(_mailboxID),
  requestTime(_requestTime),
  ttl(_ttl),
  distSoFar(_distSoFar) { }

DPRHierarchy::DPRHierarchy(catomID _hostCatom, string _hierarchyName) :
  Hierarchy(_hostCatom, _hierarchyName),
  hierarchyMessageBox("_hchy_" + _hierarchyName),
  landmarkBeaconBox("_lm_" + _hierarchyName),
  routeLandmarkMsgBox("_rtlm_" + _hierarchyName),
  reqPromotionBox("_rp_" + _hierarchyName),
  beaconRequestBox("_brq_" + _hierarchyName),
  numHopsToParent(ULONG_MAX),
  visibilityDistance(2),
  lastAnnouncementTime(-1),
  nextBeaconVersionID(1)
{
  statsMgrPtr->setCompletionStats("aggMessages", STATS_VALUE);
  statsMgrPtr->setCompletionStats("landmarkMsg-distanceTraveled", STATS_MEAN|STATS_DIST);
}

DPRHierarchy::~DPRHierarchy() { }

void DPRHierarchy::simulationStart() {
  Hierarchy::simulationStart();

  speculationMode = worldPtr->search_key_value_list("SPECULATIONMODE");
  if(speculationMode == "")
    speculationMode = "perfect";

  string useCutthroughString = worldPtr->search_key_value_list("CUTTHROUGH");
  if(useCutthroughString != "")
    cutthroughEnabled = true;

  string beaconingFrequencyString = worldPtr->search_key_value_list("BEACONINGFREQUENCY");
  if(beaconingFrequencyString != "")
    beaconingFrequency = strtol(beaconingFrequencyString.c_str(), NULL, 0);

  string speculationLatencyString = worldPtr->search_key_value_list("SPECLATENCY");
  if(speculationLatencyString != "")
    speculationLatency = strtoul(speculationLatencyString.c_str(), NULL, 0);

  string useFanoutControlString = worldPtr->search_key_value_list("USEFANOUTCONTROL");
  if(useFanoutControlString == "false")
    pref_useFanoutControl = false;

  //maxVisibilityDistance = (unsigned long)(worldPtr->catomHash.size() * 1.25);
  maxVisibilityDistance = (unsigned long)ceil(sqrt(worldPtr->catomHash.size()) * sqrt(2.0) * 1.01);

  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(hierarchyMessageBox,
								   this,
								   (msgHandlerPtr)&DPRHierarchy::hierarchyMessageIntentHandler,
								   MSGINTENT,
								   (speculationLatency > 0));
  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(hierarchyMessageBox,
								   this,
								   (msgHandlerPtr)&DPRHierarchy::hierarchyMessageHeaderHandler,
								   MSGHEADER);
  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(hierarchyMessageBox,
								   this,
								   (msgHandlerPtr)&DPRHierarchy::hierarchyMessagePayloadHandler,
								   MSGPAYLOAD,
								   false);
  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(landmarkBeaconBox,
								   this,
								   (msgHandlerPtr)&DPRHierarchy::landmarkBeaconHandler,
								   MSGPAYLOAD);

  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(routeLandmarkMsgBox,
								   this,
								   (msgHandlerPtr)&DPRHierarchy::routeLandmarkMsg);

  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(reqPromotionBox,
								   this,
								   (msgHandlerPtr)&DPRHierarchy::reqPromotionHandler,
								   MSGPAYLOAD,
								   false);

  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(beaconRequestBox,
								   this,
								   (msgHandlerPtr)&DPRHierarchy::beaconRequestHandler);

  // Add our physical neighbors to the peers set to seed it
  for(featureID i=1; i<=NUM_FEATURES; i++) {
    catomID neighbor = worldPtr->catomHash[hostCatom]->C.getNeighbor(i);
    if(neighbor != 0)
      peers.insert(neighbor);
  }
}

void DPRHierarchy::dumpStats() {
  worldPtr->oStart();
  cerr << worldPtr->current_time << " DPRHierarchy(" << hierarchyName << ") "  << hostCatom << ":"
       << " peers.size " << peers.size()
       << " visibility " << visibilityDistance
       << endl << flush;
  worldPtr->oEnd();

  Hierarchy::dumpStats();
}

void DPRHierarchy::endTick() {
  // Only consider self-promotion/beaconing changes if:
  // (a) we don't already have a parent
  // (b) there's been time for our beacon to go all the way out, and conversely, get beacons from peers (with a bit of extra slack)
  if((getParent() == INVALID_CATOM_ID) && ((worldPtr->current_time - lastAnnouncementTime) > 1.05*visibilityDistance+1)) {
    // 1. Consider self-electing if:
    // (a) We have a respectable number of peers, OR
    // (b) We're already visible to the entire object and have *any* peers
    // 2. Otherwise, grow our visibility radius to try to get more peers
    
    if((peers.size() > 4) || // option (a)
       ((peers.size() > 0) && (visibilityDistance >= maxVisibilityDistance))) { // option (b)
      // Declare ourselves a parent with some probability inversely proportional to
      //     the number of peers and our visibility distance
      // Ideally, we want only one leader of this level in this area
      //float threshold = 1.0f / ((peers.size()+1) * visibilityDistance);
      float threshold = 1.0f / (4.0f * visibilityDistance); // Temp: 4 = fanout of physical layer
      
      float val = (float)((double)random() / (double)RAND_MAX);
      if(val <= threshold) {
	// we hit the probability, so we should elect ourselves
	promoteSelf();
      } else {
	#if 0
	worldPtr->oStart();
	cerr << worldPtr->current_time << " " << hostCatom << " misses threshold of " << threshold << endl;
	worldPtr->oEnd();
	#endif
      }
    } else {
      // 2. Otherwise, grow our visibility radius to try to get more peers
      
      if(visibilityDistance < maxVisibilityDistance) {
	// Currently, grow by 25% (parameterize and test?)
	visibilityDistance = (int)ceil(1.5*visibilityDistance);
	if(visibilityDistance > maxVisibilityDistance)
	  visibilityDistance = maxVisibilityDistance;
	landmarkAnnounce();
      }
    }
  }
  
  // broadcast an update maintenance beacon if 'beaconingFrequency' ticks have passed
  if((beaconingFrequency > 0) &&
     (worldPtr->current_time > pref_fastStartEndTime) &&
     ((worldPtr->current_time - lastAnnouncementTime) >= beaconingFrequency))
    landmarkAnnounce();

  // Fanout control: If fanout is too high (for congestion purposes), start telling children to promote
  if(pref_useFanoutControl) {
    // during fast start, beacon every 250 ticks, regardless
    if((level > 0) &&
       (worldPtr->current_time < (pref_fastStartEndTime-350)) && // stop 350 before end to let them drain
       (((unsigned)worldPtr->current_time % 250) == ((unsigned)hostCatom % 250)))
      landmarkAnnounce();
    

    if((worldPtr->current_time < (pref_fastStartEndTime-350)) &&
       (((unsigned)worldPtr->current_time % 150) == ((unsigned)hostCatom % 150)) &&
       (childCatomIDs().size() > 6)) {
      list<catomID> childIDs = childCatomIDs();
      vector<catomID> childIDVec; childIDVec.insert(childIDVec.begin(), childIDs.begin(), childIDs.end());
      set<catomID> childrenToPromote;
      
      while(childrenToPromote.size() < floor(childIDs.size() / 4)) {
	// Choose a random child to promote
	// Vector doesn't seem to have a delete(int) method, so I guess we'll do it this way
	vector<catomID>::size_type i = rand() % childIDVec.size();
	childrenToPromote.insert(childIDVec[i]);
      }
      
      for(set<catomID>::iterator i = childrenToPromote.begin(); i != childrenToPromote.end(); i++) {
	worldPtr->oStart();
	cerr << worldPtr->current_time << " " << hostCatom << " telling " << *i << " to promote (fanout "
	     << childIDs.size() << ")\n";
	worldPtr->oEnd();
	worldPtr->catomHash[*i]->C.mailboxManager.fileMessage(new RequestPromotionMsg(this));
      }
    }
  }

  Hierarchy::endTick();
}

void DPRHierarchy::promoteSelf() {
#if 1
  worldPtr->oStart();
  cerr << worldPtr->current_time << " " << hostCatom << " promotes self to level " << level+1 << endl;
  worldPtr->oEnd();
#endif

  if(worldPtr->current_time > pref_fastStartEndTime) {
    worldPtr->oStart();
    cerr << "Catom " << hostCatom << " self-promoting after fast start ended\n";
    worldPtr->oEnd();

    exit(-1);
  }

  // send ourselves up a level
  level++;
  
  // recalculate our visibility distance
  visibilityDistance = (unsigned)ceil(visibilityDistance*2);
  if(visibilityDistance > maxVisibilityDistance)
    visibilityDistance = maxVisibilityDistance;
  
  // send a new beacon
  landmarkAnnounce();
  
  // reset peers
  peers.clear();
  // timeLastPeerAdded = worldPtr->current_time;
}

featureID DPRHierarchy::calcRoutingDest(catomID destination) {
  if(destination == hostCatom) {
    // return 0 if it's meant for us
    return 0;
  } else {
    // otherwise, check in the routing table
    featureID contact = routingTable[destination];
    
    if(contact == 0) {
      // Routing failure!
      cerr << "Routing failure at " << hostCatom << endl;
      //throw 0;
      return 0;
    }

    return contact;
  }
}

bool DPRHierarchy::hierarchyMessageIntentHandler(Message* _msg) {
  LandmarkMsg* msg = (LandmarkMsg*)_msg;

  // Give the speculation manager a chance to guess, if one exists
  SpeculationManager* specMgr = CODE_MODULE(SpeculationManager);
  if(specMgr) {
    featureID actualDest = calcRoutingDest(msg->destination);
    if(speculationMode != "perfect") {
      featureID speculatedDest = specMgr->speculateDestFromSourceWithMode(msg->arrivalContact, speculationMode);

      if(speculatedDest != actualDest) {
	// spin off a misspeculated header
	// TODO
	return false;
      }
    }

    if(actualDest == 0) {
      // If it's going to this catom, then there's nothing to do at this point; need to wait for payload
      return false;
    }

    // Speculation succeeded
    // Log the fact that we've already sent it on
    msgsForwardedBySpeculation.insert(msg->messageID);
    // And go ahead and send it
    return routeLandmarkMsg(msg);
  }

  return false;
}

bool DPRHierarchy::hierarchyMessageHeaderHandler(Message* _msg) {
  LandmarkMsg* msg = (LandmarkMsg*)_msg;

  // Implement cutthrough
  if(cutthroughEnabled) {
    if(msgsForwardedBySpeculation.find(msg->messageID) != msgsForwardedBySpeculation.end()) {
      // It was already handled speculatively
      // We can remove it from the set now
      msgsForwardedBySpeculation.erase(msg->messageID);
      return false;
    } else {
      // Wasn't handled by speculation
      featureID actualDest = calcRoutingDest(msg->destination);
      if(actualDest == 0) {
	// If it's going to this catom, then there's nothing to do at this point; need to wait for payload
	return false;
      }

      return routeLandmarkMsg(msg);
    }
  }

  // If we're not doing cutthrough, nothing to do
  return false;
}

bool DPRHierarchy::hierarchyMessagePayloadHandler(Message* _msg) {
  LandmarkMsg* msg = (LandmarkMsg*)_msg;
  
  // If it's for us, then it won't have been handled before now; go ahead and route it to our correct mailbox
  featureID actualDest = calcRoutingDest(msg->destination);
  if(actualDest == 0)
    return routeLandmarkMsg(msg);

  // If it's not for us, see if it's been done before
  // Was it handled by speculation?
  if(msgsForwardedBySpeculation.find(msg->messageID) != msgsForwardedBySpeculation.end()) {
    // It was already handled speculatively
    // We can remove it from the set now
    msgsForwardedBySpeculation.erase(msg->messageID);
    return false;
  }
  // Was it handled by cutthrough?
  if(cutthroughEnabled)
    return false;

  // Wasn't already handled at a previous phase
  // Route it on (next tick)
  msg->mailboxID = routeLandmarkMsgBox;
  worldPtr->catomHash[hostCatom]->C.mailboxManager.fileMessage(msg);
  // We want to keep it (as we're passing it on), so return true
  return true;
}

bool DPRHierarchy::routeLandmarkMsg(Message* _msg) {
  LandmarkMsg* msg = (LandmarkMsg*)_msg;
  msg->mailboxID = hierarchyMessageBox;
  msg->distanceTraveled++;

  featureID destFeature = calcRoutingDest(msg->destination);
  if((destFeature == 0) && (msg->destination != hostCatom)) {
    // routing failure
    delete msg;
    return true;
  }

  SpeculationManager* specMgr = CODE_MODULE(SpeculationManager);
  if(specMgr)
    specMgr->messageRouted(msg->arrivalContact, destFeature);

  if(destFeature == 0) {
    // it's meant for us; handle the payload
    worldPtr->catomHash[hostCatom]->C.mailboxManager.fileMessage((Message*)msg->payload->clone());
    delete msg;
  } else {
    worldPtr->catomHash[hostCatom]->C.getFeatureMap()[destFeature].getNetworkAdapter()->sendMessage(msg);
  }

  return true;
}

bool DPRHierarchy::landmarkBeaconHandler(Message* _msg) {
  LandmarkBeacon* msg = (LandmarkBeacon*)_msg;

#if 0
  worldPtr->oStart();
  cerr << worldPtr->current_time << " " << hostCatom << " receives beacon from "
       << msg->id << "(level " << msg->level
       << ") (traveled " << msg->distanceSoFar << "/" << msg->visDistance << ")\n";
  worldPtr->oEnd();
#endif
  
  if(msg->id == hostCatom) {
    return false;
  }
  
  bool didChange = updateLandmarkRoutingTables(msg->id, msg->arrivalContact, msg->beaconVersionID, msg->distanceSoFar);

  // If it's from our parent, update info accordingly
  if(msg->id == getParent()) {
    parentLevel = msg->level;
  } else if((msg->level > level) &&            // must be higher level than us, AND
     ((getParent() == INVALID_CATOM_ID) ||     //  - we don't have a parent yet, or
      (msg->level < parentLevel) ||            //  - it's closer to our own level than our old parent, or
      ((msg->level == parentLevel) &&          //  - it's the same level as the old parent, and
       (msg->distanceSoFar < numHopsToParent)) //    - it's closer to us than our old parent
      )
     ) {
    // OK; adopt as our parent!
    setParent(msg->id);
    numHopsToParent = msg->distanceSoFar;
    parentLevel = msg->level;

#if 1
    worldPtr->oStart();
    cerr << worldPtr->current_time << " " << hostCatom << " becomes a child of " << getParent() << "(level " << msg->level << ")\n";
    worldPtr->oEnd();
#endif
    
    unsigned int neededDistance = (int)ceil(msg->distanceSoFar * 1.05); // give a 5% leeway
    if(neededDistance > visibilityDistance) {
      visibilityDistance = neededDistance;
      if(visibilityDistance > maxVisibilityDistance)
	visibilityDistance = maxVisibilityDistance;
      landmarkAnnounce();
    } else if (lastAnnouncementTime < 0) {
      landmarkAnnounce();
    }
  }
  
  // see if they're a peer
  if(msg->level == level) {
    peers.insert(msg->id).second;
    
    // possibly change our visibility if we're still in the running for being a parent
    if((getParent() == INVALID_CATOM_ID) && (msg->visDistance > visibilityDistance)) {
      visibilityDistance = msg->visDistance;
      if(visibilityDistance > maxVisibilityDistance)
	visibilityDistance = maxVisibilityDistance;
      landmarkAnnounce();
    }
  }
  
  // propagate the announcement
  if(didChange && (msg->distanceSoFar < msg->visDistance)) {
    for(unsigned int i=1; i<=NUM_FEATURES; i++) {
      Feature* contact = &((worldPtr->catomHash[hostCatom]->C.getFeatureMap())[i]);
      LandmarkBeacon* newMsg = new LandmarkBeacon(this,
						  msg->id,
						  msg->beaconVersionID,
						  msg->visDistance,
						  msg->distanceSoFar+1,
						  msg->level,
						  msg->parent);
      contact->getNetworkAdapter()->sendMessage(newMsg);
    }
  } 
#if 0
  else if(!didChange) {
    cerr << "Not forwarding because no change\n";
  } else {
    cerr << "Not forwarding because max distance\n";
  }
#endif 

  return false;
}

bool DPRHierarchy::reqPromotionHandler(Message* _msg) {
  // We can't be our parent's child anymore if we're going to end up the same level :-/
  if(parentLevel == level+1)
    setParent(INVALID_CATOM_ID);

  this->promoteSelf();

  return false;
}

bool DPRHierarchy::beaconRequestHandler(BeaconRequestBeacon* msg) {
  if(lastAnnouncementTime < msg->requestTime) {
    landmarkAnnounce();

    // Propagate beacon
    if(msg->distSoFar < msg->ttl) {
      for(unsigned int i=1; i<=NUM_FEATURES; i++) {
	Feature* contact = &((worldPtr->catomHash[hostCatom]->C.getFeatureMap())[i]);
	BeaconRequestBeacon* newMsg = new BeaconRequestBeacon(this->beaconRequestBox,
							      msg->requestTime,
							      msg->ttl,
							      msg->distSoFar+1);
	contact->getNetworkAdapter()->sendMessage(newMsg);
      }
    }
  }

  return false;
}

bool DPRHierarchy::updateLandmarkRoutingTables(catomID _landmarkID, featureID _contact, long _beaconID, unsigned long _hopcount) {
  long lastID = beaconIDs[_landmarkID];

  // if it's not at least as new as the last beacon we heard, then quit
  if(lastID > _beaconID)
    return false;
  
  if((lastID == 0) || (_beaconID > lastID) || // If it's a new beacon,
     (_hopcount < beaconHopcounts[_landmarkID])) { // or a shorter path than we already had
    // then we definitely need to update
    beaconIDs[_landmarkID] = _beaconID;
    beaconHopcounts[_landmarkID] = _hopcount;
    routingTable[_landmarkID] = _contact;

    return true;
  }

  // Otherwise, we know here that it's the same beacon we've already seen

  // Throw it out if it's just a longer path
  if(_hopcount > beaconHopcounts[_landmarkID])
    return false;

  // Otherwise, it's not only the same beacon, it's the same path length
  // Break the tie with a hash of contact number and landmark id
  if((_contact+_landmarkID)%NUM_FEATURES < (routingTable[_landmarkID]+_landmarkID)%NUM_FEATURES) {
    // We update in this case to get stability
    beaconIDs[_landmarkID] = _beaconID;
    beaconHopcounts[_landmarkID] = _hopcount;
    routingTable[_landmarkID] = _contact;
  }

  // But even if we updated, we still return false here, because it doesn't need to be propagated
  return false;

  /*
  if(worldPtr->current_time > 150) {
    worldPtr->oStart();
    cerr << worldPtr->current_time << " " << hostCatom << " changed its routing table late in the game:"
	 << " landmarkID " << _landmarkID 
	 << " old_contact " << routingTable[_landmarkID]
	 << " new_contact " << _contact
	 << " old_send_time " << lastSendTime
	 << " new_send_time " << _msgSendTime
	 << endl;
    worldPtr->oEnd();
  }
  */
}

void DPRHierarchy::landmarkAnnounce() {
  // Short circuit out if we've already sent something this tick
  if(lastAnnouncementTime >= worldPtr->current_time)
    return;

  nextBeaconVersionID++;
  lastAnnouncementTime = worldPtr->current_time;

#if 1
  worldPtr->oStart();
  cerr << worldPtr->current_time << " " << hostCatom << " beacons with visDistance " << visibilityDistance << endl;
  worldPtr->oEnd();
#endif
  
  Feature* featureMap = worldPtr->catomHash[hostCatom]->C.getFeatureMap();
  for(unsigned int i=1; i<=NUM_FEATURES; i++) {
    LandmarkBeacon* newMsg = new LandmarkBeacon(this,
						hostCatom,
						nextBeaconVersionID,
						visibilityDistance,
						1,
						level,
						getParent());
    
    featureMap[i].getNetworkAdapter()->sendMessage(newMsg);
  }
}

// API/Utilities

bool DPRHierarchy::sendMsgToParent(Message* msg) {
  if(getParent() == INVALID_CATOM_ID) {
    delete msg;
    return false;
  }
  
  LandmarkMsg* lMsg = new LandmarkMsg(this, getParent(), msg);
#if 0
  worldPtr->oStart();
  cerr << "Landmark packet distribution; header " << lMsg->headerSize() << " bytes, payload " << lMsg->payloadSize() << " bytes\n";
  worldPtr->oEnd();
#endif

  worldPtr->oStart();
  cerr << worldPtr->current_time << " hchyTrace " << hostCatom << " " << getParent() << " " << msg->messageID << " " << msg->logString() << endl;
  worldPtr->oEnd();

  routeLandmarkMsg(lMsg);
  
  return true;
}
