#include "DPRHierarchy-HotBackups.hxx"

#include "CatomWorld.hxx"
#include "DPRSim.hxx"

CODE_MODULE_DECLARATION( DPRHierarchyHotBackups, DPRHierarchyHotBackups, "DPRHierarchyHotBackups" );

static string hotBackupRequestMailbox = "HotBackup Request";
static string hotBackupRandomWalkRequestMailbox = "HotBackup Random Walk Request";
static string hotBackupMyHBAnnouncementMailbox = "HotBackup My HB Announcement";
static string hotBackupResponseMailbox = "HotBackup Response";
static string hotBackupUpdateMailbox = "HotBackup Update";
string hotBackupKeepaliveMailbox = "HotBackup Keepalive";
string hotBackupFailoverBeaconMailbox = "HotBackup Failover";

simtime pref_keepaliveInterval = 1;
static bool pref_randFailParent = false;
static float pref_parentFailProb = 0.0;
static map<catomID,simtime> pref_deadCatoms = map<catomID,simtime>();
static bool pref_informChildrenOfHB = false;

HotBackupRequest::HotBackupRequest(catomID _primaryCatom) :
  Message(hotBackupRequestMailbox),
  primaryCatom(_primaryCatom) { }

HotBackupRandomWalkRequest::HotBackupRandomWalkRequest(catomID _primCatom, unsigned _ttl, unsigned _trav) :
  Message(hotBackupRandomWalkRequestMailbox),
  primaryCatom(_primCatom),
  ttl(_ttl),
  distTraveled(_trav) { }

HotBackupMyHBAnnouncement::HotBackupMyHBAnnouncement(catomID _me, catomID _hb, unsigned _ttl, unsigned _trav) :
  Message(hotBackupMyHBAnnouncementMailbox),
  original(_me),
  hb(_hb),
  ttl(_ttl),
  distTraveled(_trav) { }

HotBackupResponse::HotBackupResponse(catomID _backupCatom) :
  Message(hotBackupResponseMailbox),
  backupCatom(_backupCatom) { }

HotBackupKeepalive::HotBackupKeepalive(unsigned _visDistance) :
  Message(hotBackupKeepaliveMailbox),
  visDistance(_visDistance) { }

HotBackupFailoverBeacon::HotBackupFailoverBeacon(catomID _oldCatom, catomID _newCatom) :
  Message(hotBackupFailoverBeaconMailbox),
  oldCatom(_oldCatom),
  newCatom(_newCatom) { }

HotBackupUpdate::HotBackupUpdate(catomID _childID, ChildData* _childData)  :
  Message(hotBackupUpdateMailbox),
  childID(_childID), childData(_childData) { }

void DPRHierarchyHotBackups::simulationStart() {
  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(hotBackupRequestMailbox,
								   this,
								   (msgHandlerPtr)&DPRHierarchyHotBackups::requestHandler);
  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(hotBackupRandomWalkRequestMailbox,
								   this,
								   (msgHandlerPtr)&DPRHierarchyHotBackups::randomWalkRequestHandler);
  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(hotBackupResponseMailbox,
								   this,
								   (msgHandlerPtr)&DPRHierarchyHotBackups::responseHandler);

  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(hotBackupUpdateMailbox,
								   this,
								   (msgHandlerPtr)&DPRHierarchyHotBackups::updateHandler);

  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(hotBackupKeepaliveMailbox,
								   this,
								   (msgHandlerPtr)&DPRHierarchyHotBackups::keepaliveHandler);

  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(hotBackupFailoverBeaconMailbox,
								   this,
								   (msgHandlerPtr)&DPRHierarchyHotBackups::failoverBeaconHandler);
  
  worldPtr->catomHash[hostCatom]->C.mailboxManager.registerHandler(hotBackupMyHBAnnouncementMailbox,
								   this,
								   (msgHandlerPtr)&DPRHierarchyHotBackups::hbAnnouncementHandler);

  string catomsToDie = worldPtr->search_key_value_list("HBDEADCATOMS");
  if( catomsToDie != "" ) {
    class ParseException {};

    try {
      for( unsigned i = 0; i < catomsToDie.size(); i++ ) {
	if( catomsToDie[i] == ',' ) i++;
	if( catomsToDie[i] != '(' ) throw ParseException();

	i++;
	string catom_id = "";
	do {
	  catom_id = catom_id + catomsToDie[i];
	  i++;
	  if( i >= catomsToDie.size() ) throw ParseException();
	} while( catomsToDie[i] != ',' );

	cerr << "Catom ID parsed: " << catom_id << endl;

	i++;
	if( i >= catomsToDie.size() ) throw ParseException();

	string time_to_die = "";
	do {
	  time_to_die = time_to_die + catomsToDie[i];
	  i++;
	  if( i >= catomsToDie.size() ) throw ParseException();
	} while( catomsToDie[i] != ')' );

	cerr << "Time to kill parsed: " << time_to_die << endl;

	// Put this catom/time pair in the map.
	{
	  long cid_long = atol(catom_id.c_str());
	  long ttd_long = atol(time_to_die.c_str());
	  pref_deadCatoms[cid_long] = ttd_long;
	}
      }
    } catch(ParseException pe) {
      cerr << "Could not parse list of dead catoms." << endl;
    }
  }
  
  string randFail = worldPtr->search_key_value_list("RANDFAILPARENT");
  if( randFail == "TRUE" || randFail == "True" || randFail == "true" )
    pref_randFailParent = true;
  else 
    pref_randFailParent = false;

  string pFailProb = worldPtr->search_key_value_list("PARENTFAILPROB");
  if( pFailProb != "" )
    pref_parentFailProb = strtod(pFailProb.c_str(), NULL);

  string keepaliveString = worldPtr->search_key_value_list("HBKEEPALIVE");
  if(keepaliveString != "")
    pref_keepaliveInterval = strtoul(keepaliveString.c_str(), NULL, 0);

  string informChildrenString = worldPtr->search_key_value_list("HBSENDTOHB");
  if(informChildrenString != "")
    pref_informChildrenOfHB = true;

  DPRHierarchy::simulationStart();
}

bool DPRHierarchyHotBackups::requestHandler(HotBackupRequest* msg) {
  if(hasFailed)
    return false;
  
  // Unless we're a leaf, not currently claimed, and not currently promoting...tough.
  if((level != 0) ||
     claimedAsHotBackup != INVALID_CATOM_ID ||
     (hotBackupRequestTTL >= 0))
    return false;

  worldPtr->oStart();
  cerr << worldPtr->current_time << " " << hostCatom << " agrees to handle backups for " << msg->primaryCatom << endl;
  worldPtr->oEnd();

  // Then tell them "I've got your back(up)"
  HotBackupResponse* response = new HotBackupResponse(hostCatom);
  LandmarkMsg* responseMsg = new LandmarkMsg(this, msg->primaryCatom, response);
  routeLandmarkMsg(responseMsg);

  claimedAsHotBackup = msg->primaryCatom;

  // Make sure we don't have a parent
  setParent(INVALID_CATOM_ID);

  return false;
}

bool DPRHierarchyHotBackups::randomWalkRequestHandler(HotBackupRandomWalkRequest* msg) {
  if(hasFailed || 
     ((routingTable.find(msg->primaryCatom) == routingTable.end()) &&
      (msg->primaryCatom != hostCatom))) {
    // Send the beacon back the way it came, with the same TTL
    worldPtr->catomHash[hostCatom]->C.getFeatureMap()[msg->arrivalContact].getNetworkAdapter()->sendMessage(msg->clone());
    return false;
  }

  if(msg->ttl == 0) {
    // It's trying to settle down
    // Do we fit?

    if((level != 0) ||
       claimedAsHotBackup != INVALID_CATOM_ID ||
       (hotBackupRequestTTL >= 0)) {
      // don't fit; just fall through to bounce it around to the next place
    } else {
      // We accept their hot backup request
      worldPtr->oStart();
      cerr << hostCatom << " agrees to handle backups for " << msg->primaryCatom << endl;
      worldPtr->oEnd();
      
      // Then tell them "I've got your back(up)"
      HotBackupResponse* response = new HotBackupResponse(hostCatom);
      LandmarkMsg* responseMsg = new LandmarkMsg(this, msg->primaryCatom, response);
      routeLandmarkMsg(responseMsg);
      
      claimedAsHotBackup = msg->primaryCatom;

      if(visibilityDistance <= (msg->distTraveled*1.01)) {
	visibilityDistance = (double)msg->distTraveled*1.01;
	landmarkAnnounce();
      }
      
      // Make sure we don't have a parent
      setParent(INVALID_CATOM_ID);

      return false;
    }
  }

  // Send it on to random walk another step
  featureID randomFeature = 0;
  HotBackupRandomWalkRequest* nextMessage = NULL;
  do {
    randomFeature = (random() % NUM_FEATURES) + 1;
    nextMessage = new HotBackupRandomWalkRequest(msg->primaryCatom,
						 (msg->ttl > 0) ? (msg->ttl - 1) : 0,
						 msg->distTraveled + 1);
  } while(!worldPtr->catomHash[hostCatom]->C.getFeatureMap()[randomFeature].getNetworkAdapter()->sendMessage(nextMessage));

  return false;
}

bool DPRHierarchyHotBackups::responseHandler(HotBackupResponse* msg) {
  if(hasFailed)
    return false;

  // If we already have a hot backup, we don't need another
  // TODO: we probably want to tell the other person not to bother
  if(hotBackupCatom != INVALID_CATOM_ID)
    return false;

  hotBackupCatom = msg->backupCatom;

  worldPtr->oStart();
  cerr << worldPtr->current_time << hostCatom << " has a new hot backup of " << hotBackupCatom << endl;
  worldPtr->oEnd();

  // Since we were waiting on it, it's time to do the actual promotion
  promoteSelf();

  return false;
}

bool DPRHierarchyHotBackups::keepaliveHandler(HotBackupKeepalive* msg) {
  lastKeepaliveHeardTime = worldPtr->current_time;

  if((msg->visDistance+1) > visibilityDistance) {
    visibilityDistance = msg->visDistance+1;
    landmarkAnnounce();
  }

  return false;
}

bool DPRHierarchyHotBackups::updateHandler(HotBackupUpdate* msg) {
  if(hasFailed)
    return false;

  assert(hotBackupCatom == INVALID_CATOM_ID);
  this->setChildData(msg->childID, msg->childData);

#if 0
  worldPtr->oStart();
  cerr << hostCatom << " got some hot backup update data from somebody" << endl;
  worldPtr->oEnd();
#endif  

  return false;
}

bool DPRHierarchyHotBackups::failoverBeaconHandler(HotBackupFailoverBeacon* msg) {
  if(getParent() == msg->oldCatom) {
    setParent(msg->newCatom);
    cerr << hostCatom << " now knows that " << msg->newCatom << " is its new parent, Time: " << worldPtr->current_time << endl;
  }

  if(routingTable.find(msg->oldCatom) != routingTable.end()) {
    // Yank it from the routing table
    // (also serves as our marker in the future that the beacon doesn't need to be propagated)
    routingTable.erase(msg->oldCatom);

    // Propagate the beacon
    for(unsigned int i=1; i<=NUM_FEATURES; i++) {
      Feature* contact = &((worldPtr->catomHash[hostCatom]->C.getFeatureMap())[i]);
      contact->getNetworkAdapter()->sendMessage(msg->clone());
    }
  }
  
  return false;
}

bool DPRHierarchyHotBackups::hbAnnouncementHandler(HotBackupMyHBAnnouncement* msg) {
  map<catomID, catomID>::iterator currentHB = knownHotBackups.find(msg->original);
  if((currentHB == knownHotBackups.end()) ||
     (currentHB->second != msg->hb)) {
    // It's an update
    knownHotBackups[msg->original] = msg->hb;

    // Propagate beacon
    for(unsigned int i=1; i<=NUM_FEATURES; i++) {
      Feature* contact = &((worldPtr->catomHash[hostCatom]->C.getFeatureMap())[i]);
      HotBackupMyHBAnnouncement* newMsg =
	new HotBackupMyHBAnnouncement(msg->original,
				      msg->hb,
				      msg->ttl,
				      msg->distTraveled+1);
      contact->getNetworkAdapter()->sendMessage(newMsg);
    }
  }

  return false;
}

DPRHierarchyHotBackups::DPRHierarchyHotBackups(catomID _hostCatom, string _hierarchyName) :
  DPRHierarchy(_hostCatom, _hierarchyName),
  claimedAsHotBackup(INVALID_CATOM_ID),
  hotBackupCatom(INVALID_CATOM_ID),
  hotBackupRequestTTL(-1),
  hasFailed(false),
  failoverStart(0),
  isConsistent(true),
  lastKeepaliveSendTime(0),
  lastKeepaliveHeardTime(0) { 

}

void DPRHierarchyHotBackups::endTick() {
  if(!hasFailed) {
    // See if you need to fail.
    bool will_die = pref_deadCatoms.count(hostCatom) > 0;
    simtime t = will_die ? pref_deadCatoms[hostCatom] : 0;
    
    if( will_die && 
	(t == worldPtr->current_time) ) {

      hasFailed = true;
      if( childCatomIDs().size() == 0 || 
	  claimedAsHotBackup != INVALID_CATOM_ID ) {
	
	cerr << hostCatom << " Time: " << worldPtr->current_time 
	     << " Backup: " << hotBackupCatom
	     << " FAILED ****************\n";
	exit(0);
      }
      else {
	cerr << hostCatom << " Time: " << worldPtr->current_time 
	     << " Backup: " << hotBackupCatom
	     << " Level: " << getLevel()
	     << " PARENTFAILED ****************\n";
      }
      
    }
    else if( pref_randFailParent && 
	     worldPtr->current_time >= pref_fastStartEndTime ) {
      if( childCatomIDs().size() != 0 && 
	  claimedAsHotBackup == INVALID_CATOM_ID ) {
	// So yes, I am a parent.
	float val = (float)((double)random() / (double)RAND_MAX);
	if( val <= pref_parentFailProb ) {
	  hasFailed = true;
	  cerr << hostCatom << " Time: " << worldPtr->current_time 
	     << " Backup: " << hotBackupCatom
	     << " Level: " << getLevel()
	     << " PARENTFAILED ****************\n";
	}
      }
    }
  }

  // Let the superclass do its thing
  DPRHierarchy::endTick();

  if(hasFailed)
    return;

  // See if we need to ask somebody else to be our hot backup
  if((hotBackupRequestTTL == 0) && (hotBackupCatom == INVALID_CATOM_ID)) {
    // Yep.  It's time.
    worldPtr->oStart();
    cerr << hostCatom << " timed out in the search for a hot backup; trying next feature\n";
    worldPtr->oEnd();
    askForHotBackup();
  }

  if(hotBackupRequestTTL > 0)
    hotBackupRequestTTL--;

  // If we already have a hot backup, send a keepalive if it's time
  if((hotBackupCatom != INVALID_CATOM_ID) &&
     ((worldPtr->current_time - lastKeepaliveSendTime) >= pref_keepaliveInterval)) {
    routeLandmarkMsg(new LandmarkMsg(this, hotBackupCatom, new HotBackupKeepalive(visibilityDistance)));
  }

  // See if we haven't heard from our main dude recently
  if(claimedAsHotBackup != INVALID_CATOM_ID &&
     (lastKeepaliveHeardTime > 0) &&
     ((worldPtr->current_time - lastKeepaliveHeardTime) == (pref_keepaliveInterval+1))) {
    
    failoverStart = worldPtr->current_time;
    isConsistent = false;
    takeOverFromMainCatom();
  }
}

void DPRHierarchyHotBackups::takeOverFromMainCatom() {
  
  cerr << hostCatom << " declares its main catom dead. Time: "
       << worldPtr->current_time << endl;
  
  // Propagate the beacon
  HotBackupFailoverBeacon* msg = 
    new HotBackupFailoverBeacon(claimedAsHotBackup,hostCatom);
  
  for(unsigned int i=1; i<=NUM_FEATURES; i++) {
    Feature* contact = &((worldPtr->catomHash[hostCatom]->C.getFeatureMap())[i]);
    contact->getNetworkAdapter()->sendMessage(msg->clone());
  }
}

void DPRHierarchyHotBackups::promoteSelf() {
  if(hasFailed)
    return;

  // If we *do* have a hot backup, then just pass it through to the superclass
  if(hotBackupCatom != INVALID_CATOM_ID) {
    DPRHierarchy::promoteSelf();
    return;
  }

  // If we're somebody else's hot backup, don't allow promotion
  if(claimedAsHotBackup != INVALID_CATOM_ID)
    return;

  // If we haven't done it before, start a request for a hot backup
  if(hotBackupRequestTTL == -1) {
    assert(hotBackupCatom == INVALID_CATOM_ID);
    // Yep.  It's time.
    worldPtr->oStart();
    cerr << hostCatom << " starts the first hot backup search\n";
    worldPtr->oEnd();
    askForHotBackup();
    return;
  }

  // Otherwise, we're still waiting.  Just chill.
}

void DPRHierarchyHotBackups::askForHotBackup() {
  if(hasFailed)
    return;

  if(lastAnnouncementTime == -1) {
    landmarkAnnounce();
    hotBackupRequestTTL = 2;
    return;
  }

  string requestMethod = worldPtr->search_key_value_list("HBSELECTMETHOD");
  if(requestMethod == "RW") {
    HotBackupRandomWalkRequest* req = new HotBackupRandomWalkRequest(hostCatom, (random() % visibilityDistance) + 1);
    this->randomWalkRequestHandler(req);
    delete req;
    hotBackupRequestTTL = 50000;
  } else {
    // Send it on to random walk another step
    featureID randomFeature = 0;
    HotBackupRequest* nextMessage = NULL;
    do {
      randomFeature = (random() % NUM_FEATURES) + 1;
      nextMessage = new HotBackupRequest(hostCatom);
    } while(!worldPtr->catomHash[hostCatom]->C.getFeatureMap()[randomFeature].getNetworkAdapter()->sendMessage(nextMessage));

    hotBackupRequestTTL = 5;
  }
}

void DPRHierarchyHotBackups::setChildData(catomID childID, ChildData* childData) {
  if(hasFailed)
    return;

  // First up, call superclass to do normal setting
  DPRHierarchy::setChildData(childID, childData);

  // Just return if we're using direct sending from children
  if(pref_informChildrenOfHB)
    return;

  // If we don't have a hot backup (though we should), just return
  if(hotBackupCatom == INVALID_CATOM_ID)
    return;

  // Otherwise, send the data to our hot backup
  HotBackupUpdate* msg = new HotBackupUpdate(childID,
					     childData ? childData->clone() : NULL);
  routeLandmarkMsg(new LandmarkMsg(this, hotBackupCatom, msg));
}

bool DPRHierarchyHotBackups::sendMsgToParent(Message* msg) {
  Message* myCopy = msg->clone();

  if(!DPRHierarchy::sendMsgToParent(msg)) {
    delete myCopy;
    return false;
  }
  
  map<catomID, catomID>::iterator it = knownHotBackups.find(getParent());
  catomID parentHB = (it != knownHotBackups.end()) ? it->second : INVALID_CATOM_ID;
  if(parentHB == INVALID_CATOM_ID) {
    delete myCopy;
    return true;
  }

  worldPtr->oStart();
  cerr << worldPtr->current_time << " hchyTrace " << hostCatom << " " << getParent() << " " << myCopy->messageID << endl;
  worldPtr->oEnd();
  
  LandmarkMsg* lMsg = new LandmarkMsg(this, parentHB, myCopy);
  routeLandmarkMsg(lMsg);
  return true;
}


void DPRHierarchyHotBackups::landmarkAnnounce() {
  DPRHierarchy::landmarkAnnounce();

  if(pref_informChildrenOfHB && (hotBackupCatom != INVALID_CATOM_ID)) {
    for(unsigned int i=1; i<=NUM_FEATURES; i++) {
      Feature* contact = &((worldPtr->catomHash[hostCatom]->C.getFeatureMap())[i]);
      contact->getNetworkAdapter()->sendMessage(new HotBackupMyHBAnnouncement(hostCatom,
									      hotBackupCatom,
									      visibilityDistance));
    }
  }
}
