// Belt.cxx
// Nels Beckman
// April 3rd, 2007
//
// An application that works like a conveyor belt of catoms.

#include "Belt.hxx"

CODE_MODULE_DECLARATION( Belt, Belt );

Belt::Belt(catomID _hostCatom) : 
  RPCCodeModule<Belt>(_hostCatom),
  token(false)
{
  pthread_mutex_init( &mutex, NULL );
}

void wakeMe() {
  catomID c = RPCCodeModuleStatic::getThreadHome();
  RPCCodeModuleStatic::setThreadWake(c);
}

void Belt::thread() {
  do {
    start();
    
    for( int wait_time = rand() % 10; wait_time != 0; wait_time-- ){
      wakeMe();
      doRPCYield();
    }

  } while(true);
}

void Belt::start() {
  
  try {
    if( getToken() ) {
      if( getNeighbor(NORTH) != NULL &&
	  getNeighbor(SOUTH) == NULL ) {
	
	// We can start!
	worldPtr->oStart();
	cout << "Catom " << hostCatom 
	     << ": I am at the front. (" << HOSTCATOM.getLocation() 
	     << ")" << endl;
	worldPtr->oEnd();
	
	HOSTCATOM.setColor( 128, 0, 0, 0 );
	
	if( getNeighbor(NORTH)->get()->recurse( NextMove(hostCatom,TOP) ) ) {
	  
	  waitForTopAndMove( NextMove(hostCatom, SOUTH) );
	  
	  while( getNeighbor(SOUTH) == NULL ) {
	    wakeMe();
	    doRPCYield();
	  }
	  
	  getNeighbor(SOUTH)->get()->urDone();
	}
      }
      
      
      releaseToken();
    }  
  } catch( SendFail sf) {}
}

bool Belt::getToken() {
  pthread_mutex_lock( &mutex );
  bool ret = !token;
  if( !token ) token = true;
  pthread_mutex_unlock( &mutex );

  return ret;
}

void Belt::releaseToken() {
  pthread_mutex_lock( &mutex );
  token = false;
  pthread_mutex_unlock( &mutex );
  HOSTCATOM.setColor( 128, 128, 128, 0 );
}

void Belt::scheduleMove(NextMove nm) {
  // CALL PRIMITIVE
  sched_move( nm.c, nm.f );
}

void Belt::waitForTopAndMove(NextMove nm) {
  worldPtr->oStart();
  cout << "Catom " << hostCatom << ": waitForTopAndMove." << endl;
  worldPtr->oEnd();

  HOSTCATOM.setColor( 0, 255, 0, 0 );
  while( getNeighbor(TOP) == NULL ) {
    wakeMe();
    doRPCYield();
  }

  worldPtr->oStart();
  cout << "Catom " << hostCatom 
       << ": waitForTopAndMove, top was non-null!" << endl;
  worldPtr->oEnd();

  this->getNeighbor(TOP)->get()->remoteMove(nm);
}

bool Belt::changeColor() {
  HOSTCATOM.setColor( 0,0,128,0 );
  return true;
}

unsigned int Belt::randEorW() {
  return (rand() % 2) == 0 ? EAST : WEST;
}

bool Belt::recurse(NextMove nm) {
  ENTER_RPC_METHOD;

  worldPtr->oStart();
  cout << "Catom " << hostCatom << ": received recurse call." << endl;
  worldPtr->oEnd();

  if( getToken() ) {
    while(true) {
      try {
	
	if( getNeighbor(NORTH) != NULL ) {
	  HOSTCATOM.setColor( 0, 128, 0, 0 );
	  if( getNeighbor(NORTH)->get()->recurse(NextMove(hostCatom,TOP) )) {
	    waitForTopAndMove(nm);
	    releaseToken();
	    RPC_RETURN true;
	  }
	  else {
	    releaseToken();
	    RPC_RETURN false;
	  }
	}
	else {
	  unsigned int first_choice = randEorW();
	  unsigned int secnd_choice = first_choice == EAST ? WEST : EAST;
	  
	  if( getNeighbor(first_choice) != NULL &&
	      changeColor() &&
	      getNeighbor(first_choice)->get()->recurse(NextMove(hostCatom,
								 TOP)) ) {
	    // WE FOUND OUR FIRST CHOICE PATH
	    waitForTopAndMove(nm);
	    releaseToken();
	    RPC_RETURN true;
	  }
	  else if( getNeighbor(secnd_choice) != NULL &&
		   changeColor() &&
		   getNeighbor(secnd_choice)->get()->recurse(NextMove(hostCatom,
								      TOP)) ) {
	    // WE FOUND OUR SECOND CHOICE PATH
	    waitForTopAndMove(nm);
	    releaseToken();
	    RPC_RETURN true;
	  }
	  else {
	    // MOVE YOURSELF!
	    HOSTCATOM.setColor( 0, 0, 128,0 );
	    scheduleMove(nm);
	    RPC_RETURN true;	
	  }
	}
      } catch( SendFail sf ) {}
    }
  }
  else {
    RPC_RETURN false;
  }
  
}

void Belt::urDone() {
  ENTER_RPC_METHOD;

  releaseToken();

  RPC_RETURN;
}

void Belt::remoteMove(NextMove nm) {
  ENTER_RPC_METHOD;

  scheduleMove(nm);

  RPC_RETURN;
}
