///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright (C) 2006 by Intel Corporation and Carnegie Mellon University    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <iostream>
#include <errno.h>
#include <string.h>

#include "AntennaSim.hxx"
#include "CatomWorld.hxx"
#include "CatomSim.hxx"

CODE_MODULE_DECLARATION( Antenna, Antenna );

using namespace std;

static void* foo = (void*) 0xABC000AB;
static double DB_ADD = 10;
static double DB_SCALE = 2;
static uint8 R = 128;
static uint8 G = 0;
static uint8 B = 128;
static uint8 A = 96;
static double Scale = 0.005;
static unsigned long MHZ = 2400;
static double Thick = 0.1;
static string ANTSIMCMD="DATA/antsim.sh";

static void display_output() {
  poly_list pl;
  vector<Point3D> face;
  
  // remove previous displayed output
  list<poly_list>::iterator pl_i;
  for ( pl_i=worldPtr->poly_lists.begin(); pl_i!=worldPtr->poly_lists.end(); pl_i++ ) {
    if (pl_i->data == foo ) {
      worldPtr->poly_lists.erase( pl_i );
      break;
    }
  }
  
  // read file
  fstream fs;
  fs.open( "ant_out.txt", fstream::in );
  double x,y,z,old_y;
  vector< vector<Point3D> > points;
  vector<Point3D> row;
  while (fs.good()) {
    fs >> x >> y >> z;
    if (!fs.good()) break;
    if ((int)(10*y)!=(int)(10*old_y)) {
       if (row.size()) points.push_back(row);
       row.clear();
       old_y = y;
    }
    z = (z+DB_ADD)*DB_SCALE;
    if (z<0) z=0;
    x = 90-x;   // convert to lattitude, as used by Point3D
    x = x*M_PI/180;
    y = y*M_PI/180;
    row.push_back( Point3D(z,x,y,1) );
  }
  if (row.size()) points.push_back(row);
  fs.close();
  
  // generate surface for display
  unsigned int a, b;
  for ( a=0; a<points.size(); a++ ) {
    for ( b=0; b<points[a].size()-1; b++ ) {
      unsigned int a2=a+1;
      unsigned int b2=b+1;
      if (a2==points.size()) a2=0;
      if (b2==points[a].size()) b2=0;
      //if (points[a].size()!=points[a2].size()) cerr << "size error " << points[a].size() << " " << points[a2].size()<< "\n";
      face.clear();
      face.push_back( points[a][b] );
      face.push_back( points[a][b2] );
      face.push_back( points[a2][b2] );
      face.push_back( points[a2][b] );
      pl.list.push_back( face );
    }
  }
  pl.data = foo;
  pl.body = 0;
  pl.red = R;
  pl.green = G;
  pl.blue = B;
  pl.alpha = A;
  worldPtr->poly_lists.push_back( pl );

}


static fstream outfs;
static set< pair< CatomSim*, CatomSim* > > edges;
static int wirenum;
static double mindist;
static CatomSim* closest_catom;

static void process_one_catom( pair< catomID, CatomSim* > ch ) {
  unsigned int i;
  CatomSim* c = ch.second;
  catomID cid = c->C.getID();
  for ( i=0; i<NUM_NEIGHBORS; i++) {
    catomID nid = c->C.getNthNeighbor( i );
    if (nid) {
      CatomSim* n=worldPtr->catomHash[nid];
      if (nid<cid) edges.insert( pair< CatomSim*, CatomSim*>(n,c) );
      else edges.insert( pair<CatomSim*, CatomSim*>(c,n) );
    } else break;
  }
  double dist = c->C.getLocation().norm();
  if (dist<mindist) {
    mindist = dist;
    closest_catom = c;
  }
}

static void process_one_edge( pair< CatomSim*, CatomSim* > e ) {
  Point3D p = e.first->C.getLocation();
  outfs << "GW " << (++wirenum) << " 1 " << (p.getX()) << " " << (p.getY()) << " " << (p.getZ());
  p = e.second->C.getLocation();
  outfs << " " << (p.getX()) << " " << (p.getY()) << " " << p.getZ() << " " << Thick << "\n";
}

static void write_to_file() {
  outfs.open( "ant_in.txt", fstream::out );
  wirenum = 0;
  mindist = 1000000000;
  for_each( worldPtr->catomHash.begin(), worldPtr->catomHash.end(), process_one_catom );
  for_each( edges.begin(), edges.end(), process_one_edge );
  // add wire from origin to closest catom, inject signal
  Point3D p = closest_catom->C.getLocation();
  outfs << "GW " << (++wirenum) << " 1 0. 0. 0. " << (p.getX()) << " " << (p.getY()) << " " << p.getZ() << " " << Thick << "\n";
  // add scaling factor
  outfs << "GS 0 0 " << Scale << "\n";
  outfs << "GE\n";
  outfs << "FR 0 1 0 0 " << MHZ << "\n";
  outfs << "EX 0 " << wirenum << " 1 0 1. 0. 0.\n";
  outfs << "RP 0 72 72 1000 0. 0. 5. 5.\n";
  outfs << "EN\n";
  edges.clear();
  outfs.close();
}

static void execute_sim() {
  system( ANTSIMCMD.c_str() );
}


void Antenna::oracle() {
  // should read parameters from experiment file
  string tmp;
  tmp = worldPtr->search_key_value_list( "Antenna_CMD" );
  if (tmp!="") ANTSIMCMD=tmp;
  tmp = worldPtr->search_key_value_list( "Antenna_R" );
  if (tmp!="") R = strtoul( tmp.c_str(), 0, 0 );
  tmp = worldPtr->search_key_value_list( "Antenna_G" );
  if (tmp!="") G = strtoul( tmp.c_str(), 0, 0 );
  tmp = worldPtr->search_key_value_list( "Antenna_B" );
  if (tmp!="") B = strtoul( tmp.c_str(), 0, 0 );
  tmp = worldPtr->search_key_value_list( "Antenna_A" );
  if (tmp!="") A = strtoul( tmp.c_str(), 0, 0 );
  tmp = worldPtr->search_key_value_list( "Antenna_MHZ" );
  if (tmp!="") MHZ = strtoul( tmp.c_str(), 0, 0 );
  tmp = worldPtr->search_key_value_list( "Antenna_DB_ADD" );
  if (tmp!="") DB_ADD = strtod( tmp.c_str(), 0 );
  tmp = worldPtr->search_key_value_list( "Antenna_DB_SCALE" );
  if (tmp!="") DB_SCALE = strtod( tmp.c_str(), 0 );
  tmp = worldPtr->search_key_value_list( "Antenna_SCALE" );
  if (tmp!="") Scale = strtod( tmp.c_str(), 0 );
  tmp = worldPtr->search_key_value_list( "Antenna_THICK" );
  if (tmp!="") Thick = strtod( tmp.c_str(), 0 );
  oracleYield();
  oracleYield();
  while (1) {
    //cout << "Oracle: running concurrently with newTick, endTick\n";
    oracleYield();
    //cout << "Oracle: running with other oracles only\n";
    write_to_file();
    execute_sim();
    display_output();
    oracleYield();
  }
}
