///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright (C) 2006 by Intel Corporation and Carnegie Mellon University    //
// Contacts: casey.j.helfrich @ intel.com                                    //
//           bdr @ cs.cmu.edu                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "Catom.hxx"
#include "CatomSim.hxx"
#include "CatomWorld.hxx"
#include "Util.hxx"

#include <errno.h>

extern CatomWorld *worldPtr;

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// MailboxManager Implementations                                            //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

MailboxManager::MailboxManager() {
  pthread_mutexattr_t mutex_attrs;
  pthread_mutexattr_init(&mutex_attrs);
  pthread_mutexattr_settype(&mutex_attrs, PTHREAD_MUTEX_RECURSIVE);
  pthread_mutex_init(&boxesLock, &mutex_attrs);
  pthread_mutexattr_destroy(&mutex_attrs);
}

MailboxManager::~MailboxManager() {
  lock();
  
  for(unsigned phase=0; phase<NUMMSGPHASES; phase++) {
    // Clear all messages
    map<unsigned long long, map<MailboxID, queue<Message*> > >::iterator ticksIter = messages[phase].begin();
    while( ticksIter != messages[phase].end() ) {
      // For each tick that we have messages for
      unsigned long long tick = ticksIter->first;
      
      map<MailboxID, queue<Message*> >::iterator boxesIter = messages[phase][tick].begin();
      while( boxesIter != messages[phase][tick].end() ) {
	// For each mailbox within that tick
	MailboxID boxID = boxesIter->first;
	
	while(!messages[phase][tick][boxID].empty()) {
	  delete messages[phase][tick][boxID].front();
	  messages[phase][tick][boxID].pop();
	}
	
	boxesIter++;
      }
      
      ticksIter++;
    }
  }

  unlock();
  
  pthread_mutex_destroy(&boxesLock);
}

void MailboxManager::lock() {
  int err = pthread_mutex_lock(&boxesLock);
  if(err) {
    cerr << EINVAL << " " << EBUSY << " " << EAGAIN << " " 
	 << EDEADLK << " " << EPERM << endl;
    cerr << "can't lock mutex : " << strerror(errno) << endl;
    exit(1);
  }
}

void MailboxManager::unlock() {
  int err = pthread_mutex_unlock(&boxesLock);
  if(err) {
    cerr << "can't unlock mutex : " << strerror(errno) << endl;
    exit(1);
  }
}

///////////////////////////////////////////////////////////////////////////////
// MailboxManager::fileMessage(Message *)
//
// Purpose: Inserts the given message into the appropriate queue.
//
// Returns: Success status of the insertion.
//

bool MailboxManager::fileMessage( Message* msg, MessagePhase phase ) {
  //cerr << "'s MailboxManager::fileMessage()" << endl;

  // If no delay, handle right now
  if(!boxHasDelay[phase][msg->mailboxID])
    return handleMessage(msg,phase);
  
  // Otherwise, queue for handling on the next tick
  
  lock();

  messages[phase][worldPtr->current_time+1][msg->mailboxID].push(msg);
  
  unlock();
  
  return true;
}

///////////////////////////////////////////////////////////////////////////////
// MailboxManager::handleMessage()
//
// Purpose: Calls handler for any arbitrary message.  
//          Called from within catom code OS thread.
// 
// Returns: whether a message was handled
//

bool MailboxManager::handleMessage() {
  for(unsigned phase=0; phase<NUMMSGPHASES; phase++) {
    map<MailboxID, queue<Message*> >::iterator iter = messages[phase][worldPtr->current_time].begin();
  
    // might be nice to introduce some randomness into this beyond
    // just that which comes from the hash table
    while( iter != messages[phase][worldPtr->current_time].end() ) {
      if( handleMessage(iter->first, (MessagePhase)phase) )
	return true;
      iter++;
    }

    messages[phase].erase(worldPtr->current_time - 1);
  }
  return false;
}

///////////////////////////////////////////////////////////////////////////////
// MailboxManager::handleMessage(MailboxID,MessagePhase)
//
// Purpose: Calls handler for a particular mailbox
//
// Returns: whether a message was handled
//

bool MailboxManager::handleMessage( MailboxID boxID, MessagePhase phase ) {
  lock();
  
  if(messages[phase][worldPtr->current_time][boxID].empty()) {
    unlock();
    return false;
  }

  Message* msg = messages[phase][worldPtr->current_time][boxID].front();
  messages[phase][worldPtr->current_time][boxID].pop();

  unlock();

  return handleMessage(msg, phase);
}

///////////////////////////////////////////////////////////////////////////////
// MailboxManager::handleMessage(Message *, MessagePhase)
//
// Purpose: Immediately calls hander for a particular message.  
//          Doesn't involve mailboxes, just dispatches
//
// Returns: whether the message was handled
//

bool MailboxManager::handleMessage( Message* msg, MessagePhase phase ) {
  //worldPtr->oStart();
  //cerr << "Received message in handleMessage\n";
  //worldPtr->oEnd();
  if( !msg )
    return false;
  
  pair<CodeModule*, msgHandlerPtr> handler = messageHandlers[phase][msg->mailboxID];
  
  if( !handler.first || !handler.second ) {
    if(phase == MSGPAYLOAD) {
      worldPtr->oStart();
      cerr << "No CodeModule/handler found to handle message for " << msg->mailboxID <<"!  Make sure you register your handlers (usually in simulationStart()) before sending any messages of this type.\n";
      worldPtr->oEnd();
      exit(-1);
    } else {
      delete msg;
      return false;
    }
  }
  
  if(!(handler.first->*(handler.second))(msg)) {
    delete msg;
  }
  
  return true;
}

///////////////////////////////////////////////////////////////////////////////
// MailboxManager::registerHandler(MailboxID, CodeModule*, msgHandlerPtr, MessagePhase)
//
// Purpose: Defines the handler for a particular mailbox name
//

void MailboxManager::registerHandler( MailboxID boxName, CodeModule* module, 
				      msgHandlerPtr fnPtr,
				      MessagePhase phase,
				      bool hasDelay ) {
  make_pair(module, fnPtr);
  lock();
  messageHandlers[phase][boxName] = make_pair( module, fnPtr );
  boxHasDelay[phase][boxName] = hasDelay;
  unlock();
}


///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// END MailboxManager Implementation                                         //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Catom Method Implementation                                               //
//                                                                           //
// The following methods are all that are available to the Catom             //
// programmer in their Code Modules                                          //
//                                                                           //
// Catom::Catom(CatomID)                                                     //
// Catom::~Catom()                                                           //
//                                                                           //
// Catom::getID()                                                            //
// Catom::getLocation()                                                      //
// Catom::getBatteryLife()                                                   //
// Catom::getNeighborCenter()                                                //
// Catom::getNeighbors()                                                     //
// Catom::getNeighbor(featureID)                                             //
//                                                                           //
// Catom::amPowered()                                                        //
// Catom::amOnFloor()                                                        //
// Catom::amOnSurface()                                                      //
// Catom::amMobile()                                                         //
//                                                                           //
// Catom::moveTo(Catom*, featureID)                                          //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// Catom::Catom(unsigned long)
// 
// Purpose: Catom object constructor
//

Catom::Catom( catomID i ) : 
  ID(i), 
  batteryLife(1),
  sel(false) {
  map = new Feature[NUM_FEATURES+1];
  neighbors = new featureID[NUM_NEIGHBORS];
  neighborID = new catomID[NUM_NEIGHBORS];
  get_feature_map( LATTICE_TYPE, i, map );
  for (int i = 0; i < NUM_MAGNETS; i++) {
    LEVEL0_magnetstate[i] = OFF;
  }
}

///////////////////////////////////////////////////////////////////////////////
// Catom::~Catom()
//
// Purpose: Catom object destructor
//

Catom::~Catom() {
  delete [] map;
  delete [] neighbors;
  delete [] neighborID;
}

///////////////////////////////////////////////////////////////////////////////
// Catom::getID()
//
// Returns: Unique identifier for this catom
//

catomID Catom::getID() {
  return ID;
}

///////////////////////////////////////////////////////////////////////////////
// Catom::getFeatureMap()
//
// Returns: Feature map
//

Feature* Catom::getFeatureMap() {
  return map;
}

///////////////////////////////////////////////////////////////////////////////
// Catom::getFeature(featueID f)
//
// Returns my feature with featureID f
//

Feature* Catom::getFeature(featureID f) {
  return &(map[f]);
}

///////////////////////////////////////////////////////////////////////////////
// Catom::getLocation()
//
// Returns: Location of current catom
//

WorldPoint3D Catom::getLocation() {
  return WorldPoint3D( dBodyGetPosition( worldPtr->catomHash[ID]->body ) );
}

///////////////////////////////////////////////////////////////////////////////
// Catom::getBatteryLife()
//
// Returns: The internal value for batteryLife
//

double Catom::getBatteryLife() {
  return batteryLife;
}

///////////////////////////////////////////////////////////////////////////////
// Catom::getNeighborCenter()
//
// Returns: The Location of a neighbor attached to my feature f
//          (Even if the neighbor doesn't exist!)
//

Point3D Catom::getNeighborCenter( featureID f ) {
  Point3D fl = map[f].getLocation();
  double r = 2*CatomSim::catom_radius;
  dVector3 loc;
  dBodyGetRelPointPos( worldPtr->catomHash[ID]->body,
		r*fl.getX(), r*fl.getY(), r*fl.getZ(), loc );
  return Point3D( loc[0], loc[1], loc[2] );
}

///////////////////////////////////////////////////////////////////////////////
// Catom::getNearestFeature( Point3D p, catomID hostid=0 )
//
// Returns: FeatureID of feature closest to absolute location p
//
//

featureID Catom::getNearestFeature( Point3D p, catomID hostid ) {
  unsigned int i, best;
  double dist, bestdist=1000000;
  if ( hostid==0 ) hostid=ID;
  CatomSim* c = worldPtr->catomHash[hostid];
  Feature* featuremap = c->C.getFeatureMap();
  Point3D rloc, absloc;
  dVector3 loc;
  for (i=1; i<=NUM_FEATURES; i++) {
    rloc = featuremap[i].getLocation();
    dBodyGetRelPointPos( c->body, rloc.getX(), rloc.getY(), rloc.getZ(), loc );
    absloc = Point3D( loc );
    dist = absloc.distanceFrom( p );
    if (dist<bestdist) {
      bestdist=dist;
      best = i;
    }
  }
  return best;
}

///////////////////////////////////////////////////////////////////////////////
// Catom::getFeatureTouching( neighborid,  hostID=0  )
//
// Returns: feature touching catom neighborid
//
//

featureID Catom::getFeatureTouching( catomID neighborid, catomID hostID ) {
  unsigned int i;
  for ( i=0; i<NUM_NEIGHBORS; i++ )
    if ( neighborID[i] == neighborid ) return neighbors[i];
  return 0;
}

///////////////////////////////////////////////////////////////////////////////
// Catom::setColor(r,g,b,a)
//
// Set the visual color of the catom.  Valid input values are [0..1]
//

void Catom::setColor( uint8 r, uint8 g, uint8 b, uint8 a ) {
  worldPtr->catomHash[ID]->red = r;
  worldPtr->catomHash[ID]->green = g;
  worldPtr->catomHash[ID]->blue = b;
  worldPtr->catomHash[ID]->alpha = a;
  return;
}

///////////////////////////////////////////////////////////////////////////////
// Catom::getNeighbor(featureID)
//
// Returns: Pointer to the Catom object in contact with feature f
//          (or NULL if there is no Catom)
//

catomID Catom::getNeighbor( featureID f ) {
  unsigned int i;
  for (i=0; i<NUM_NEIGHBORS; i++) {
    if (neighbors[i]==f) return neighborID[i];
  }
  return 0;
}

///////////////////////////////////////////////////////////////////////////////
// Catom::clearNeighbors()
// empties neighbor list.  should not normally be used, only when moving
//	should be called by magic move, physics, etc.

void Catom::clearNeighbors() {
  unsigned int i;
  for (i=0; i<NUM_NEIGHBORS; i++) {
    neighbors[i]=0;
    neighborID[i]=0;
  }
  for (i=1; i<=NUM_FEATURES; i++) {
    map[i].clearRemoteFeatures();
  }
}

///////////////////////////////////////////////////////////////////////////////
// Catom::removeNeighbor( catomID targetCatom )
// removes neighbor from list if it is there.  
//      should not normally be used, only when moving
//	should be called by magic move, physics, etc.

void Catom::removeNeighbor( catomID targetCatom ) {
  unsigned int i,j;
  for (i=0,j=0; i<NUM_NEIGHBORS; i++,j++) {
    if ( neighborID[i]==targetCatom) {
      map[neighbors[i]].clearRemoteFeatures();
      j++;
    } 
    if (j>=NUM_NEIGHBORS) {
      neighbors[i]=0;
      neighborID[i]=0;
    } else if (i!=j) {
      neighbors[i] = neighbors[j];
      neighborID[i] = neighborID[j];
    }
  }
}

///////////////////////////////////////////////////////////////////////////////
// Catom::addNeighbor( featureID f, catomID targetCatom, featureID t_f )
// adds neighbor to the list.  
//      returns -1 if feature already present
//      should not normally be used, only when moving
//	should be called by magic move, physics, etc.

int Catom::addNeighbor( featureID f, catomID targetCatom, featureID t_f ) {
  unsigned int i;
  map[f].addRemoteFeature( &(worldPtr->catomHash[targetCatom]->C.map[t_f]) );
  for (i=0; i<NUM_NEIGHBORS; i++) {
    if (neighbors[i]==0) {
      neighbors[i]=f;
      neighborID[i]=targetCatom;
      return 0;
    }
    if (neighbors[i]==f) break;
  }
  return -1;
}


///////////////////////////////////////////////////////////////////////////////
// Catom::setBatteryLife(double)
//
// Purpose: Manually updates internal battery Data with supplied value
//

void Catom::setBatteryLife( double b ) { 
  batteryLife = b; 
}

///////////////////////////////////////////////////////////////////////////////
// Catom::amPowered()
//
// Returns: True if batteryLife is non-zero
//

bool Catom::amPowered() {
  return (batteryLife > 0);
}

///////////////////////////////////////////////////////////////////////////////
// Catom::amOnFloor()
//
// Purpose: In reality this will be done through some sensing, but this is
//          sufficient in simulation. Checks to see if your center is within 
//          one Catom radius of the floor
//
// Returns: True if this catom is on the floor
//

bool Catom::amOnFloor() {
  return ( dBodyGetPosition( worldPtr->catomHash[ID]->body )[2] < (CatomSim::catom_radius + EPSILON));
}


///////////////////////////////////////////////////////////////////////////////
// Catom::moveTo(catomID targetCatom, featureID targetFeature)
//
// Purpose: Actually move the Catom in the simulated world.
//
// There are no rules associated with the values for Catom* and featureID.
// Therefore, we call this Magic-Move.  When you call this function you WILL
// be moved to the location you ask for instantly.
//
// In the future the simulator may only allow target to be a catom you are 
// currently in contact with.  And more agressively, dest may be forced to 
// only be a direct featureNeighbor on the FM, only allowing small, direct 
// hops.
// 
// Returns: True if the move was successful
//	Normally fails if target location collides with some other catom.
//	Can override this check by providing parameter force as true (default: false)
//
// WARNING: You ought to hold a lock on the world when using this function
//  or restrict concurrent use appropriately.  Otherwise BAD THINGS can happen.
//
// Automatic locking features use two flags for lock and unlock:
//   true true (default): lock and unlock
//   true false:  lock, and leave locked
//   false false: leave lock alone
//   false true:  don't call lock, but unlock on exit
// This allows doing multiple magic move calls atomically with respect
//   to other magic moves
//

static pthread_mutex_t magic_move_lock = PTHREAD_MUTEX_INITIALIZER;

bool Catom::moveTo( catomID target, featureID tarFeat, 
                    bool roll, bool lock, bool unlock, bool matchvel, bool force ) {
  bool retval;
  if (lock) pthread_mutex_lock(&magic_move_lock);
  // Calculate new real location
  Point3D newLoc = worldPtr->catomHash[target]->C.getNeighborCenter(tarFeat);
  // Update orientation as needed
  //  This assumes sphere-in-sphere rolling about an immediate neighbor
  if (roll) {
    dQuaternion orient, rot;
    Point3D curLoc = getLocation();
    Point3D neiLoc = worldPtr->catomHash[target]->C.getLocation();
    double lenA = (curLoc-neiLoc).norm();
    double lenB = (newLoc-neiLoc).norm();
    double lenC = (newLoc-curLoc).norm();
    if ( lenC < lenA+lenB ) {
      Point3D axis = (curLoc-neiLoc).cross(newLoc-neiLoc);
      axis /= axis.norm();
      double angle = acos( (lenA*lenA + lenB*lenB - lenC*lenC)/(2*lenA*lenB) );
      dQFromAxisAndAngle( rot, axis.getX(), axis.getY(), axis.getZ(), 2*angle );
      dQMultiply0( orient, rot, 
         dBodyGetQuaternion( worldPtr->catomHash[ID]->body ) );
      retval = moveTo( newLoc, orient, false, unlock, true, force );
    }
  } else retval = moveTo( newLoc, NULL, false, unlock, true, force );
  if ( retval && matchvel ) {
    const dReal *vel = dBodyGetLinearVel( worldPtr->catomHash[target]->body );
    dBodySetLinearVel( worldPtr->catomHash[ID]->body, vel[0], vel[1], vel[2] );
  }
  return retval;
}

// Warp to location variants

bool Catom::moveTo( Point3D dest, dQuaternion orient, bool lock, bool unlock,
			bool clearvel, bool force ) {
  return moveTo( dest.getX(), dest.getY(), dest.getZ(), orient, lock, unlock, clearvel, force );
}


static void moveToCollide( void *data, dGeomID o1, dGeomID o2 ) {
  bool *is_collided = (bool*)data;
  CatomSim *c1 = (CatomSim*) dGeomGetData( o1 );
  CatomSim *c2 = (CatomSim*) dGeomGetData( o2 );
  if (c1==c2 || c1==NULL || c2==NULL) return;
  double dist = (c1->C.getLocation() - c2->C.getLocation()).norm();
  // WARNING - this collision test is approximate for spheres and cylinders in 2D
  if ( dist < 2*c1->catom_radius-EPSILON ) *is_collided = true;
}

bool Catom::moveTo( double dest_x, double dest_y, double dest_z, 
                    dQuaternion orient, bool lock, bool unlock,
		    bool clearvel, bool force ) {

  if (lock) pthread_mutex_lock(&magic_move_lock);
  
  // Remove old neighbors from this catom's list and vice versa
  worldPtr->catomHash[ID]->preMoveUnlinkNeighbors();
  // Save current position
  Point3D oldpos = getLocation();
  // Update CatomSim location
  dBodySetPosition(  worldPtr->catomHash[ID]->body, dest_x, dest_y, dest_z );
  // Collision Detection Here
  // If collided, do not update position and return false.
  bool is_collided=false;
  if ( !force ) dSpaceCollide2( worldPtr->catomHash[ID]->shape, (dGeomID) worldPtr->coll_space,
  			(void*) &is_collided, moveToCollide );
  if (is_collided) {
    // restore old position
    dBodySetPosition( worldPtr->catomHash[ID]->body, oldpos.getX(), oldpos.getY(), oldpos.getZ() );
  } else {
    if (clearvel) {
      // Kill velocities
      dBodySetLinearVel( worldPtr->catomHash[ID]->body, 0, 0, 0 );
      dBodySetAngularVel( worldPtr->catomHash[ID]->body, 0, 0, 0 );
    }
    // Update orientation if needed
    if (orient) {
      dBodySetQuaternion( worldPtr->catomHash[ID]->body, orient );
    }
  }
  // Update new neighbors
  worldPtr->catomHash[ID]->postMoveRelinkNeighbors();
  if ( unlock ) pthread_mutex_unlock(&magic_move_lock);

  return (!is_collided);
}

int selectedCatom = 0;

void Catom::selected() {
  this->sel = !this->sel;
  selectedCatom = this->ID;
}

static catomID lastSpeak = 0;
			 
// #include "CodeModules/DcPlanner.hxx"			 
			 
bool Catom::speak(int speakSize, char *speakBuf) {
  if (this->sel) {
    snprintf(speakBuf, speakSize, "GUID=%d (%3.3f,%3.3f,%3.3f)", this->getID(), this->getLocation().getX(), this->getLocation().getY(), this->getLocation().getZ());
    if (lastSpeak != ID) {
	//NOTE: insert app-specific debug code here
	lastSpeak = ID;
	//cout << REMOTE_CODE_MODULE(ID,DcPlanner)->state << " " << REMOTE_CODE_MODULE(ID,DcPlanner)->inside << " ";
	//cout << REMOTE_CODE_MODULE(ID,DcPlanner)->parent << " " << REMOTE_CODE_MODULE(ID,DcPlanner)->notChildOf.size() << endl;
// 	copy(REMOTE_CODE_MODULE(ID,DcPlanner)->locks.begin(),
// 	     REMOTE_CODE_MODULE(ID,DcPlanner)->locks.end(),
// 				ostream_iterator<float>(cout, " "));
	cout << endl;
    }
    return true;
  }
  return false;
}

// LEVEL 0

double Catom::LEVEL0_getMagnetState(unsigned int mag) {
  return LEVEL0_magnetstate[mag];
}


bool Catom::LEVEL0_setMagnetState(unsigned int mag, double value) {
  LEVEL0_magnetstate[mag] = value;
  string outFile = "LEVEL_0_SIM_OUT";
  fstream fs;
  fs.open(outFile.c_str(), fstream::out | fstream::trunc);
  fs << "MAGNET " << ID << " " << mag << " " << value << "\n";
  fs.close();
  return true;
}

double Catom::LEVEL0_getSensorState(unsigned int sen) {
  return LEVEL0_sensorstate[sen];
}


bool Catom::LEVEL0_setSensorState(unsigned int sen, double value) {
  LEVEL0_sensorstate[sen] = value;
  string outFile = "LEVEL_0_SIM_OUT";
  fstream fs;
  fs.open(outFile.c_str(), fstream::out | fstream::trunc);
  fs << "SENSOR " << ID << " " << sen << " "<< value << "\n";
  fs.close();
  return true;
}

bool Catom::LEVEL0_toggleLED() {
  string outFile = "LEVEL_0_SIM_OUT";
  fstream fs;
  fs.open(outFile.c_str(), fstream::out | fstream::trunc);
  fs << "TOGGLE " << ID << "\n";
  fs.close();
  return true;
}
