// Spatial Index Library
//
// Copyright (C) 2004 Navel Ltd.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Contact information:
//  Mailing address:
//    Marios Hadjieleftheriou
//    University of California, Riverside
//    Department of Computer Science
//    Surge Building, Room 310
//    Riverside, CA 92521
//
//  Email:
//    marioh@cs.ucr.edu

#ifndef __spatialindex_tprtree_node_h
#define __spatialindex_tprtree_node_h

namespace SpatialIndex
{
	namespace TPRTree
	{
		class TPRTree;
		class Leaf;
		class Index;
		class Node;

		typedef Tools::PoolPointer<Node> NodePtr;

		class Node : public SpatialIndex::INode
		{
		public:
			virtual ~Node();

			//
			// Tools::IObject interface
			//
			virtual Tools::IObject* clone() throw (Tools::NotSupportedException);

			//
			// Tools::ISerializable interface
			//
			virtual unsigned long getByteArraySize();
			virtual unsigned long loadFromByteArray(byte* const data);
			virtual void storeToByteArray(unsigned long& len, byte** data);

			//
			// SpatialIndex::IEntry interface
			//
			virtual long getIdentifier() const;
			virtual void getShape(IShape** out) const throw (std::exception);

			//
			// SpatialIndex::INode interface
			//
			virtual unsigned long getChildrenCount() const;
			virtual long getChildIdentifier(unsigned long index)  const throw (Tools::IndexOutOfBoundsException);
			virtual void getChildShape(unsigned long index, IShape** out)  const throw (Tools::IndexOutOfBoundsException, std::exception);
			virtual unsigned long getLevel() const;
			virtual bool isIndex() const;
			virtual bool isLeaf() const;

		private:
			Node();
			Node(TPRTree* pTree, long id, unsigned long level, unsigned long capacity);

			virtual Node& operator=(const Node&);

			virtual bool insertEntry(unsigned long dataLength, byte* pData, MovingRegion& mbr, long id);
			virtual void deleteEntry(unsigned long index);

			virtual bool insertData(unsigned long dataLength, byte* pData, MovingRegion& mbr, long id, std::stack<long>& pathBuffer, byte* overflowTable);
			virtual void reinsertData(unsigned long dataLength, byte* pData, MovingRegion& mbr, long id, std::vector<long>& reinsert, std::vector<long>& keep);

			virtual void rstarSplit(unsigned long dataLength, byte* pData, MovingRegion& mbr, long id, std::vector<long>& group1, std::vector<long>& group2);

			//virtual void pickSeeds(unsigned long& index1, unsigned long& index2);

			virtual void condenseTree(std::stack<NodePtr>& toReinsert, std::stack<long>& pathBuffer, NodePtr& ptrThis);

			virtual NodePtr chooseSubtree(const MovingRegion& mbr, unsigned long level, std::stack<long>& pathBuffer) = 0;
			virtual NodePtr findLeaf(const MovingRegion& mbr, long id, std::stack<long>& pathBuffer) = 0;

			virtual void split(unsigned long dataLength, byte* pData, MovingRegion& mbr, long id, NodePtr& left, NodePtr& right) = 0;

			TPRTree* m_pTree;
				// Parent of all nodes.

			unsigned long m_level;
				// The level of the node in the tree.
				// Leaves are always at level 0.

			long m_identifier;
				// The unique ID of this node.

			unsigned long m_children;
				// The number of children pointed by this node.

			unsigned long m_capacity;
				// Specifies the node capacity.

			MovingRegion m_nodeMBR;
				// The minimum bounding region enclosing all data contained in the node.

			byte** m_pData;
				// The data stored in the node.

			MovingRegionPtr* m_ptrMBR;
				// The corresponding data MBRs.

			long* m_pIdentifier;
				// The corresponding data identifiers.

			unsigned long* m_pDataLength;

			unsigned long m_totalDataLength;

			class RstarSplitEntry
			{
			public:
				MovingRegion* m_pRegion;
				long m_id;
				unsigned long m_sortDim;

				RstarSplitEntry(MovingRegion* pr, long id, unsigned long dimension)
					: m_pRegion(pr), m_id(id), m_sortDim(dimension) {}

				static int compareLow(const void* pv1, const void* pv2)
				{
					RstarSplitEntry* pe1 = * (RstarSplitEntry**) pv1;
					RstarSplitEntry* pe2 = * (RstarSplitEntry**) pv2;

					if (pe1->m_pRegion->m_pLow[pe1->m_sortDim] < pe2->m_pRegion->m_pLow[pe1->m_sortDim]) return -1;
					if (pe1->m_pRegion->m_pLow[pe1->m_sortDim] > pe2->m_pRegion->m_pLow[pe1->m_sortDim]) return 1;
					return 0;
				}

				static int compareHigh(const void* pv1, const void* pv2)
				{
					RstarSplitEntry* pe1 = * (RstarSplitEntry**) pv1;
					RstarSplitEntry* pe2 = * (RstarSplitEntry**) pv2;

					if (pe1->m_pRegion->m_pHigh[pe1->m_sortDim] < pe2->m_pRegion->m_pHigh[pe1->m_sortDim]) return -1;
					if (pe1->m_pRegion->m_pHigh[pe1->m_sortDim] > pe2->m_pRegion->m_pHigh[pe1->m_sortDim]) return 1;
					return 0;
				}

				static int compareVLow(const void* pv1, const void* pv2)
				{
					RstarSplitEntry* pe1 = * (RstarSplitEntry**) pv1;
					RstarSplitEntry* pe2 = * (RstarSplitEntry**) pv2;

					if (pe1->m_pRegion->m_pVLow[pe1->m_sortDim] < pe2->m_pRegion->m_pVLow[pe1->m_sortDim]) return -1;
					if (pe1->m_pRegion->m_pVLow[pe1->m_sortDim] > pe2->m_pRegion->m_pVLow[pe1->m_sortDim]) return 1;
					return 0;
				}

				static int compareVHigh(const void* pv1, const void* pv2)
				{
					RstarSplitEntry* pe1 = * (RstarSplitEntry**) pv1;
					RstarSplitEntry* pe2 = * (RstarSplitEntry**) pv2;

					if (pe1->m_pRegion->m_pVHigh[pe1->m_sortDim] < pe2->m_pRegion->m_pVHigh[pe1->m_sortDim]) return -1;
					if (pe1->m_pRegion->m_pVHigh[pe1->m_sortDim] > pe2->m_pRegion->m_pVHigh[pe1->m_sortDim]) return 1;
					return 0;
				}
			}; // RstarSplitEntry

			class ReinsertEntry
			{
			public:
				long m_id;
				double m_dist;

				ReinsertEntry(long id, double dist) : m_id(id), m_dist(dist) {}

				static int compareReinsertEntry(const void* pv1, const void* pv2)
				{
					ReinsertEntry* pe1 = * (ReinsertEntry**) pv1;
					ReinsertEntry* pe2 = * (ReinsertEntry**) pv2;

					if (pe1->m_dist < pe2->m_dist) return -1;
					if (pe1->m_dist > pe2->m_dist) return 1;
					return 0;
				}
			}; // ReinsertEntry

			// Needed to access protected members without having to cast from Node.
			// It is more efficient than using member functions to access protected members.
			friend class TPRTree;
			friend class Leaf;
			friend class Index;
			friend class Tools::PointerPool<Node>;
		}; // Node
	}
}

#endif /*__spatialindex_tprtree_node_h*/
