package edu.cmu.hcii.problemreports;

import java.util.Collections;
import java.util.HashSet;
import java.util.Vector;

public class QueryTable {

	private int totalSentences, totalWords, totalWordsInQuery;
	private Vector<Row> rows;

	private enum Column { COUNT, TAG, WORD };
	private Column columnToSortBy;

	private boolean expand = false;
	
	public QueryTable(String sortBy, boolean expand, int totalSentences, int totalWords, int totalWordsInQuery) {

		columnToSortBy = sortBy.equals("count") ? Column.COUNT : sortBy.equals("tag") ? Column.TAG : sortBy.equals("word") ? Column.WORD : null; 
		if(columnToSortBy == null) System.err.println("Warning: no column to sort by for " + sortBy);

		this.expand = expand;
		
		this.totalSentences = totalSentences;
		this.totalWords = totalWords;
		this.totalWordsInQuery = totalWordsInQuery;
		rows = new Vector<Row>();

	}
	
	public void addRow(Vector<Word> row) {

		Row newRow = new Row(row, (float)row.size() / totalWords, (float)row.size() / totalWordsInQuery);
		rows.add(newRow);
		
	}
	
	public void print() {
		
		Collections.sort(rows);
		
		System.out.println("***\tindex\tcount\t%sents\t%words\t%query\tsum(%query)\ttag\tword");
		System.out.println("--------------------------------------------------------------\n");
		float cumulativeQueryPercent = 0.0f;
		int i = 1;
		for(Row row : rows) {
			cumulativeQueryPercent += row.percentOfQuery;
			System.out.println(row.toRowString(i++, cumulativeQueryPercent));
		}
		
		System.out.println("--------------------------------------------------------------\n");
		System.out.println("" + totalSentences + " sentences, " + totalWords + " words, " + totalWordsInQuery + " words in result of query.");
		System.out.println("" + rows.size() + " unique word/tag pairs");
		
	}
	
	class Row implements Comparable<Row> {
		
		public Vector<Word> words;
		public final float percentOfAll;
		public final float percentOfQuery;
		public final float percentOfSentences;

		public Row(Vector<Word> words, float percentOfAll, float percentOfQuery) {
			
			this.words = words;
			this.percentOfAll = percentOfAll;
			this.percentOfQuery = percentOfQuery;
			
			// How many sentences does it appear in?
			HashSet<Sentence> sentences = new HashSet<Sentence>();
			for(Word word : words) sentences.add(word.getSentence());
			this.percentOfSentences = (float)sentences.size() / totalSentences;
			
		}
		
		public int getCount() { return words.size(); }
		public float getPercentOfAll() { return percentOfAll; }
		public float getPercentOfQuery() { return percentOfQuery; }
		public String getWord() { return words.firstElement().word; }
		public String getTag() { return words.firstElement().tag; }

		public int compareTo(Row row) {
			
			if(columnToSortBy == Column.COUNT)
				return (new Integer(getCount())).compareTo(row.getCount());			
			else if(columnToSortBy == Column.TAG)
				return getTag().compareTo(row.getTag());
			else if(columnToSortBy == Column.WORD)
				return getWord().compareTo(row.getWord());
			else return -1;

		}
		
		public String toRowString(int index, float cumQuery) {
			
			String result = "***\t" + 
				index + "\t" + 
				getCount() + "\t" + 
				(((int)(10000 * percentOfSentences)) / 100.0) + "%" + "\t" + 
				(((int)(10000 * percentOfAll)) / 100.0) + "%" + "\t" + 
				(((int)(10000 * percentOfQuery)) / 100.0)+ "%" + "\t" + 
				(((int)(10000 * cumQuery)) / 100.0) + "%" + "\t" + 
				words.firstElement().tag + "\t" + 
				words.firstElement().word;
			
			if(expand) {
				for(Word word : words)
					result = result + "\n\t" + word.getSentence().toUntaggedString();
				result = result + "\n";
			}

			return result;
			
		}

	}
	
	
}