/***********************************************************************/
/*                                                                     */
/*   svm_struct_api_types.h                                            */
/*                                                                     */
/*   *************************************************************     */
/*   Definition of API for attaching implementing SVM learning of      */
/*   structures (e.g. parsing, multi-label classification, HMM)        */
/*                                                                     */
/*   Author: Thorsten Joachims                                         */
/*   Date: 13.10.03                                                    */
/*                                                                     */
/*   Copyright (c) 2003  Thorsten Joachims - All rights reserved       */
/*   *************************************************************     */
/*                                                                     */
/*                                                                     */
/*   *************************************************************     */
/*   Data structure implementation for conducting structured           */
/*   prediction on Web page segmentation task                          */
/*                                                                     */
/*   Author: Lidong Bing                                               */
/*   Date: 11 Jun, 2014                                                */
/*                                                                     */
/*   Copyright (c) 2014  Lidong Bing       - All rights reserved       */
/*   *************************************************************     */
/*                                                                     */
/*   This software is available for non-commercial use only. It must   */
/*   not be modified and distributed without prior permission of the   */
/*   author. The author is not responsible for implications from the   */
/*   use of this software.                                             */
/*                                                                     */
/***********************************************************************/

#ifndef svm_struct_api_types
#define svm_struct_api_types

# include "svm_light/svm_common.h"
# include "svm_light/svm_learn.h"

# define INST_NAME          "SVM-Page-Segmentation"
# define INST_VERSION       "V1.00"
# define INST_VERSION_DATE  "11 JUN 2014"

/* default precision for solving the optimization problem */
# define DEFAULT_EPS         0.1 
/* default loss rescaling method: 1=slack_rescaling, 2=margin_rescaling */
# define DEFAULT_RESCALING   2
/* default loss function: 1=informative block based loss, 2=hamming distance loss*/
# define DEFAULT_LOSS_FCT    2
/* default optimization algorithm to use:
 *
 0: n-slack algorithm described in [1]
 1: n-slack algorithm with shrinking heuristic
 2: 1-slack algorithm (primal) described in [5]
 3: 1-slack algorithm (dual) described in [5]
 4: 1-slack algorithm (dual) with constraint cache [5]
 9: custom algorithm in svm_struct_learn_custom.c*/
# define DEFAULT_ALG_TYPE    3
/* store Psi(x,y) (for ALG_TYPE 1) instead of recomputing it every time: */
# define USE_FYCACHE         1
/* decide whether to evaluate sum before storing vectors in constraint
 cache:
 0 = NO,
 1 = YES (best, if sparse vectors and long vector lists),
 2 = YES (best, if short vector lists),
 3 = YES (best, if dense vectors and long vector lists) */
# define COMPACT_CACHED_VECTORS 1
/* minimum absolute value below which values in sparse vectors are
 rounded to zero. Values are stored in the FVAL type defined in svm_common.h
 RECOMMENDATION: assuming you use FVAL=float, use
 10E-15 if COMPACT_CACHED_VECTORS is 1
 10E-10 if COMPACT_CACHED_VECTORS is 2 or 3
 */
# define COMPACT_ROUNDING_THRESH 10E-15



/*
 * One possible cutting line in a page
 */
typedef struct cline {
	long line_ID; /*start from 1 */
	SVECTOR *fvec;

	long parent_line_ID[2]; /* -1: not a parent, at most two parent */

	/* 1=a cut, 2=not a cut*/
	int label;

	/*
	 * Indicate whether the current line is a very important one.
	 * Very important line: the boundary of the informative
	 * blocks. These lines have label 1.
	 * 0(NOT_VIP_LINE):no
	 * 1(VIP_LINE):yes
	 */
	int vipline;

} CLINE;

typedef struct pattern {
	/* this defines the x-part of a training example, e.g. the structure
	 for storing the cutting line graph of a page.
	The last position is a flag with line_ID==-1 always
	The line with ID n is stored in position n-1 in "lines" */
	CLINE *lines;

	/* the total number of cutting line in this graph*/
	long graph_size;

	char file[1024];
} PATTERN;

typedef struct label {
	/* this defines the y-part (the label) of a training example,
	 e.g. the labels for each cutting line.
	 label 1 = this is a cut
	 label 2 = not a cut */
	long *labels;
	PATTERN *x;
	long graph_size;
} LABEL;

typedef struct structmodel {
	double *w; /* pointer to the learned weights */
	MODEL *svm_model; /* the learned SVM model */
	long sizePsi; /* maximum number of weights in w */
	double walpha;
	/* other information that is needed for the stuctural model can be
	 added here, e.g. the grammar rules for NLP parsing */
	int add_your_variables_here;
} STRUCTMODEL;

typedef struct struct_learn_parm {
	double epsilon; /* precision for which to solve
	 quadratic program */
	double newconstretrain; /* number of new constraints to
	 accumulate before recomputing the QP
	 solution (used in w=1 algorithm) */
	int ccache_size; /* maximum number of constraints to
	 cache for each example (used in w=4
	 algorithm) */
	double batch_size; /* size of the mini batches in percent
	 of training set size (used in w=4
	 algorithm) */
	double C; /* trade-off between margin and loss */
	char custom_argv[50][300]; /* storage for the --* command line options */
	int custom_argc; /* number of --* command line options */
	int slack_norm; /* norm to use in objective function
	 for slack variables; 1 -> L1-norm,
	 2 -> L2-norm */
	int loss_type; /* selected loss type from -r
	 command line option. Select between
	 slack rescaling (1) and margin
	 rescaling (2) */
	int loss_function; /* select between different loss
	 functions via -l command line
	 option. Loss function: 1=informative block based loss, 2=hamming distance loss */
	/* further parameters that are passed to init_struct_model() */
	long num_features; /* number of features in each
	 individual node vector */

	long num_classes; /* number of classes that each node can take */
} STRUCT_LEARN_PARM;

typedef struct struct_test_stats {
	/* you can add variables for keeping statistics when evaluating the
	 test predictions in svm_struct_classify. This can be used in the
	 function eval_prediction and print_struct_testing_stats. */
} STRUCT_TEST_STATS;

# define TRAIN_FILE_SUFFIX          ".train"
# define TEST_FILE_SUFFIX          ".test"
# define NON_LABEL          		-1
# define CUT_LABEL					1
# define NOT_CUT_LABEL				2
# define VIP_LINE				1
# define NOT_VIP_LINE				0
# define HAMMING_LOSS		2
# define INFORMATIVE_BLOCK_LOSS		1

#endif
