/***********************************************************************/
/*                                                                     */
/*   svm_struct_api.c                                                  */
/*                                                                     */
/*   *************************************************************     */
/*   Definition of API for attaching implementing SVM learning of      */
/*   structures (e.g. parsing, multi-label classification, HMM)        */
/*                                                                     */
/*   Author: Thorsten Joachims                                         */
/*   Date: 03.07.04                                                    */
/*                                                                     */
/*   Copyright (c) 2004  Thorsten Joachims - All rights reserved       */
/*   *************************************************************     */
/*                                                                     */
/*                                                                     */
/*   *************************************************************     */
/*   Implementation for conducting structured   prediction             */
/*   on Web page segmentation task                                     */
/*                                                                     */
/*   Author: Lidong Bing                                               */
/*   Date: 11 Jun, 2014                                                */
/*                                                                     */
/*   Copyright (c) 2014  Lidong Bing       - All rights reserved       */
/*   *************************************************************     */
/*                                                                     */
/*   This software is available for non-commercial use only. It must   */
/*   not be modified and distributed without prior permission of the   */
/*   author. The author is not responsible for implications from the   */
/*   use of this software.                                             */
/*                                                                     */
/***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/stat.h>

#include "svm_struct/svm_struct_common.h"
#include "svm_struct_api.h"

#define MAX(x,y)      ((x) < (y) ? (y) : (x))
#define MIN(x,y)      ((x) > (y) ? (y) : (x))
#define SIGN(x)       ((x) > (0) ? (1) : ((x) < (0) ? (-1) : (0)))
#define FEATURE_MAP_INDEX(num_feat,class,feat_id)(num_feat*(class-1)+feat_id)

/********************begin my functions********************** */
/*read one cutting line graph example from a file
 * return 1: one example successfully got
 * return 0: fail for some reason
 */
int read_one_example(char *file, long *total_features, int *maxlabel,
		EXAMPLE *exam);

/* Returns a feature vector describing the match between CLINE line
 and label y. */
SVECTOR *psi_line(CLINE line_i, int y_i, STRUCT_LEARN_PARM *sparm);

/*
 * Construct the label for a DOM's ground truth label
 */

void construct_org_label(PATTERN *x, LABEL *y);

/*
 * write the LP problem to a file
 */
void write_LP_problem(char *file, double *coeff, PATTERN *x);

void write_LP_problem_with_fixedVs(char *file, double *coeff, PATTERN *x,
		long * vip_set, long * subset);

int solve_LP_problem(char *file, double *coeff, long * sol, PATTERN *x);

int solve_LP_problem_with_fixedVs(char *file, double *coeff, long * sol,
		PATTERN *x, long * vip_set, long * subset);

int load_LP_solution(char *file, long *sol, long graph_size);

int solve_LP(char *infile, char *outfile);

void construct_empty_label(PATTERN *x, LABEL *y);

void lpSol_to_label(LABEL *y);

int is_subset_legal(long *vip_set, long size_full, long *subset, long size_sub,
		PATTERN *x);

int is_pattern_legal(PATTERN *x);

LABEL classify_struct_example_with_partial_label(PATTERN x, STRUCTMODEL *sm,
		STRUCT_LEARN_PARM *sparm, long * vip_set, long * subset);

double obj_Fv(PATTERN x, LABEL y, STRUCTMODEL *sm, STRUCT_LEARN_PARM *sparm);

double partial_obj_Fv(CLINE line_i, int y_i, STRUCTMODEL *sm,
		STRUCT_LEARN_PARM *sparm);

/********************end begin my functions********************** */

/*============my utility functions ================== */

/*get files names in a particular folder
 * the last one is NULL as flag
 */
void get_files(char *dir, char ***files);

/*
 * load the lines from file
 */
char ** load_lines(char * file);

/*
 * when max_idx is -1, the last line of lines should be
 * NULL as flag
 */
void free_charSS(char ** lines, int max_idx);

void free_arrLL(long ** lines, int max_idx);

/*
 * parse the parent lines's ID from pattern "plid1:x plid2:y"
 */
void parse_parents(char *parent_str, long *parent_1, long *parent_2,
		int *vipline);

double loss_token(long y, long ybar);

void addOne_subset(int *flgs, int size, int sub_size, long *cnt_sub,
		long ***subsets, long * vip_set);
long ** get_subsets(int size, int sub_size, long * vip_set);

/*============ end my utility functionss ================== */

void svm_struct_learn_api_init(int argc, char* argv[]) {
	/* Called in learning part before anything else is done to allow
	 any initializations that might be necessary. */
}

void svm_struct_learn_api_exit() {
	/* Called in learning part at the very end to allow any clean-up
	 that might be necessary. */
}

void svm_struct_classify_api_init(int argc, char* argv[]) {
	/* Called in prediction part before anything else is done to allow
	 any initializations that might be necessary. */
}

void svm_struct_classify_api_exit() {
	/* Called in prediction part at the very end to allow any clean-up
	 that might be necessary. */
}

/*
 * read the training examples from a folder
 * author: bing
 */
SAMPLE read_struct_examples(char *dir, STRUCT_LEARN_PARM *sparm) {
	/* Reads struct examples and returns them in sample. The number of
	 examples must be written into sample.n */
	/* Each file in the "dir" is a single training example */

	if (dir[strlen(dir) - 1] != '/') {
		strcat(dir, "/");
	}
	SAMPLE sample; /* sample */
	EXAMPLE *examples = NULL;

	long n = 0; /* number of examples */
	long file_no = 0;

	long total_feature = 0;
	int maxlabel = 0;
	int sucFlg;

	char file_name[1024];
	char **files = NULL;
	get_files(dir, &files);

	file_no = 0;
	while (files[file_no]) {
		file_no++;
	}
	examples = (EXAMPLE *) my_malloc(sizeof(EXAMPLE) * file_no);

	file_no = 0;
	n = 0;
	while (files[file_no]) {
		if (strstr(files[file_no], TRAIN_FILE_SUFFIX) || strstr(files[file_no],
				TEST_FILE_SUFFIX)) {
			strcpy(file_name, dir);
			strcat(file_name, files[file_no]);
			sucFlg = read_one_example(file_name, &total_feature, &maxlabel,
					&(examples[n]));
			if (sucFlg == 1) {
				construct_org_label(&examples[n].x, &examples[n].y);
				n++;

			}
		}
		file_no++;
	}

	if (struct_verbosity >= 1) {
		printf(" %ld examples, %ld features, %d classes... ", n, total_feature,
				maxlabel);
		fflush(stdout);
	}

	examples = (EXAMPLE *) realloc(examples, sizeof(EXAMPLE) * n);
	sample.n = n;
	sample.examples = examples;

	free_charSS(files, -1);
	return (sample);
}

void init_struct_model(SAMPLE sample, STRUCTMODEL *sm,
		STRUCT_LEARN_PARM *sparm, LEARN_PARM *lparm, KERNEL_PARM *kparm) {
	/* Initialize structmodel sm. The weight vector w does not need to be
	 initialized, but you need to provide the maximum size of the
	 feature space in sizePsi. This is the maximum number of different
	 weights that can be learned. Later, the weight vector w will
	 contain the learned weights for the model. */

	EXAMPLE *examples = sample.examples;
	PATTERN *x;
	LABEL *y;
	long maxlabel = 0, maxfeat = 0, i, j;
	WORD *ai;
	double sizePsi;

	/* find number of classes and number of features in training set */
	for (i = 0; i < sample.n; i++) {
		x = &examples[i].x;
		y = &examples[i].y;
		for (j = 0; j < y->graph_size; j++) {
			maxlabel = MAX(maxlabel,y->labels[j]);
			if (y->labels[j] < 1) {
				printf(
						"ERROR: Found label ID '%ld'. Label IDs must be greater or equal to 1!\n",
						y->labels[j]);
				exit(1);
			}
			for (ai = x->lines[j].fvec->words; ai->wnum; ai++) {
				maxfeat = MAX(maxfeat,ai->wnum);
			}
		}
	}

	sparm->num_classes = maxlabel;
	sparm->num_features = maxfeat;

	sizePsi = sparm->num_features * sparm->num_classes;

	if (struct_verbosity >= 2) {
		printf("Dimensionality of Psi: %.0lf\n", sizePsi);
		fflush(stdout);
	}
	if (sizePsi > FNUM_MAX) {
		printf(
				"ERROR: Number of features exceeds maximum (change type of FNUM in svm_common.h\n");
		exit(1);
	}

	sm->sizePsi = (long) (sizePsi + 0.5);

	sm->walpha = 1;

}

CONSTSET init_struct_constraints(SAMPLE sample, STRUCTMODEL *sm,
		STRUCT_LEARN_PARM *sparm) {
	/* Initializes the optimization problem. Typically, you do not need
	 to change this function, since you want to start with an empty
	 set of constraints. However, if for example you have constraints
	 that certain weights need to be positive, you might put that in
	 here. The constraints are represented as lhs[i]*w >= rhs[i]. lhs
	 is an array of feature vectors, rhs is an array of doubles. m is
	 the number of constraints. The function returns the initial
	 set of constraints. */
	CONSTSET c;
	long sizePsi = sm->sizePsi;
	long i;
	WORD words[2];

	if (1) { /* normal case: start with empty set of constraints */
		c.lhs = NULL;
		c.rhs = NULL;
		c.m = 0;
	} else { /* add constraints so that all learned weights are
	 positive. WARNING: Currently, they are positive only up to
	 precision epsilon set by -e. */
		c.lhs = my_malloc(sizeof(DOC *) * sizePsi);
		c.rhs = my_malloc(sizeof(double) * sizePsi);
		for (i = 0; i < sizePsi; i++) {
			words[0].wnum = i + 1;
			words[0].weight = 1.0;
			words[1].wnum = 0;
			/* the following slackid is a hack. we will run into problems,
			 if we have move than 1000000 slack sets (ie examples) */
			c.lhs[i] = create_example(i, 0, 1000000 + i, 1, create_svector(
					words, "", 1.0));
			c.rhs[i] = 0.0;
		}
	}
	return (c);
}

LABEL classify_struct_example(PATTERN x, STRUCTMODEL *sm,
		STRUCT_LEARN_PARM *sparm) {
	/* Finds the label yhat for pattern x that scores the highest
	 according to the linear evaluation function in sm, especially the
	 weights sm.w. The returned label is taken as the prediction of sm
	 for the pattern x. The weights correspond to the features defined
	 by psi() and range from index 1 to index sm->sizePsi. If the
	 function cannot find a label, it shall return an empty label as
	 recognized by the function empty_label(y). */

	LABEL ybar;
	long i, ret_flg;
	double partial_obj_yes, partial_obj_no;
	double *coeff = (double *) my_malloc(sizeof(double) * (x.graph_size + 1));
	/* the coefficient of the LP problem.
	 coeff[i] =F_i - \overline{F'}_i,
	 where F_i = <\Psi(x_i, CUT_LABEL), sm.w>
	 is the partial F value achieved by x_i with label CUT_LABEL,
	 \overline{F}_i = <\Psi(x_i, NOT_CUT_LABEL), sm.w>*/

	for (i = 0; i < x.graph_size; i++) {

		partial_obj_yes = partial_obj_Fv(x.lines[i], CUT_LABEL, sm, sparm);
		partial_obj_no = partial_obj_Fv(x.lines[i], NOT_CUT_LABEL, sm, sparm);

		coeff[i] = partial_obj_yes - partial_obj_no;

	}

	construct_empty_label(&x, &ybar);
	ret_flg = solve_LP_problem("./tmp_lp_file", coeff, ybar.labels, &x);
	if (ret_flg != 1) {
		printf(
				"ERROR: in solving the LP solution in find_most_violated_constraint_marginrescaling function for example %s.\n",
				x.file);
		fflush(stdout);
		exit(0);
	}
	lpSol_to_label(&ybar);

	free(coeff);
	return (ybar);

}

LABEL classify_struct_example_with_partial_label(PATTERN x, STRUCTMODEL *sm,
		STRUCT_LEARN_PARM *sparm, long * vip_set, long * subset) {
	/* Finds the label yhat for pattern x that scores the highest
	 according to the linear evaluation function in sm.
	 This inference should satisfy some conditions given by vip_set and subset.
	 Namely, the lines with IDs in subset should have NOT_CUT_LABLE, the lines
	 in vip_set but not in subset should have CUT_LABEL.
	 To do so, we can specify the variables, in LP, corresponding to subset
	 to have value 0, and the variables, in LP, corresponding to vip_set but not
	 subset to have value 1 */

	LABEL ybar;
	long i, ret_flg;
	double partial_obj_yes, partial_obj_no;
	double *coeff = (double *) my_malloc(sizeof(double) * (x.graph_size + 1));
	/* the coefficient of the LP problem.
	 coeff[i] =F_i - \overline{F'}_i,
	 where F_i = <\Psi(x_i, CUT_LABEL), sm.w>
	 is the partial F value achieved by x_i with label CUT_LABEL,
	 \overline{F}_i = <\Psi(x_i, NOT_CUT_LABEL), sm.w>*/

	for (i = 0; i < x.graph_size; i++) {

		partial_obj_yes = partial_obj_Fv(x.lines[i], CUT_LABEL, sm, sparm);
		partial_obj_no = partial_obj_Fv(x.lines[i], NOT_CUT_LABEL, sm, sparm);

		coeff[i] = partial_obj_yes - partial_obj_no;

	}

	construct_empty_label(&x, &ybar);
	ret_flg = solve_LP_problem_with_fixedVs("./tmp_lp_file", coeff,
			ybar.labels, &x, vip_set, subset);
	if (ret_flg != 1) {
		printf(
				"ERROR: in solving the LP solution in find_most_violated_constraint_marginrescaling function for example %s.\n",
				x.file);
		fflush(stdout);
		exit(0);
	}
	lpSol_to_label(&ybar);

	free(coeff);
	return (ybar);

}

LABEL find_most_violated_constraint_slackrescaling(PATTERN x, LABEL y,
		STRUCTMODEL *sm, STRUCT_LEARN_PARM *sparm) {
	/* Finds the label ybar for pattern x that that is responsible for
	 the most violated constraint for the slack rescaling
	 formulation. For linear slack variables, this is that label ybar
	 that maximizes

	 argmax_{ybar} loss(y,ybar)*(1-psi(x,y)+psi(x,ybar))

	 Note that ybar may be equal to y (i.e. the max is 0), which is
	 different from the algorithms described in
	 [Tschantaridis/05]. Note that this argmax has to take into
	 account the scoring function in sm, especially the weights sm.w,
	 as well as the loss function, and whether linear or quadratic
	 slacks are used. The weights in sm.w correspond to the features
	 defined by psi() and range from index 1 to index
	 sm->sizePsi. Most simple is the case of the zero/one loss
	 function. For the zero/one loss, this function should return the
	 highest scoring label ybar (which may be equal to the correct
	 label y), or the second highest scoring label ybar, if
	 Psi(x,ybar)>Psi(x,y)-1. If the function cannot find a label, it
	 shall return an empty label as recognized by the function
	 empty_label(y). */

	/*
	 * slack rescaling is paired with informative block loss, this condition should be checked first
	 */
	if (sparm->loss_function != INFORMATIVE_BLOCK_LOSS) {
		printf(
				"ERROR: Margin rescaling has to be paired with Hamming loss function!\n");
		exit(1);
	}

	LABEL ytmp;
	LABEL ybar;

	double Fy;
	double Fy_est;
	double loss_v;
	double maxH, tmpH;
	long i, j;
	long vip_cnt;
	long wrong_vip_cnt;
	long sub_size;
	long *vip_set;

	ybar = classify_struct_example(x, sm, sparm);

	Fy_est = obj_Fv(x, ybar, sm, sparm);
	Fy = obj_Fv(x, y, sm, sparm);

	loss_v = loss(y, ybar, sparm);
	maxH = loss_v * (1 + Fy_est - Fy);

	vip_cnt = 0;
	for (i = 0; i < x.graph_size; i++)
		if (x.lines[i].vipline == VIP_LINE)
			vip_cnt++;

	vip_set = (long *) malloc(sizeof(long) * (vip_cnt + 1));
	vip_set[vip_cnt] = -1;

	j = 0;
	wrong_vip_cnt = 0;
	for (i = 0; i < x.graph_size; i++) {
		if (x.lines[i].vipline == VIP_LINE) {
			vip_set[j] = x.lines[i].line_ID;
			j++;
			// the vip line is wrongly labeled (vip line must be CUT)
			if (ybar.labels[i] == NOT_CUT_LABEL) {
				wrong_vip_cnt++;
			}
		}

	}

	for (sub_size = wrong_vip_cnt + 1; sub_size <= vip_cnt; sub_size++) {
		// each sub set in subsets is ended with '-1' element as flag
		long ** subsets = get_subsets(vip_cnt, sub_size, vip_set);
		j = 0;
		while (subsets[j] != NULL) {
			if (is_subset_legal(vip_set, vip_cnt, subsets[j], sub_size, &x)) {
				ytmp = classify_struct_example_with_partial_label(x, sm, sparm,
						vip_set, subsets[j]);

				Fy_est = obj_Fv(x, ytmp, sm, sparm);
				loss_v = loss(y, ytmp, sparm);
				tmpH = loss_v * (1 + Fy_est - Fy);
				if (tmpH > maxH) {
					free_label(ybar);
					ybar = ytmp;
				} else {
					free_label(ytmp);
				}
			}
			j++;
		}

		free_arrLL(subsets, -1);
	}

	if (struct_verbosity >= 4) {
		printf("  --> w*Psi(x,y) = %lf\n", Fy);
		Fy_est = obj_Fv(x, ybar, sm, sparm);
		printf("  --> w*Psi(x,ybar) = %lf\n", Fy_est);
		loss_v = loss(y, ybar, sparm);
		printf("  --> (1- w*(Psi(x,y)-Psi(x,ybar)))delta(y,ybar) = %lf\n",
				loss_v * (1 + Fy_est - Fy));
	}

	free(vip_set);
	//free others
	return (ybar);
}

LABEL find_most_violated_constraint_marginrescaling(PATTERN x, LABEL y,
		STRUCTMODEL *sm, STRUCT_LEARN_PARM *sparm) {
	/* Finds the label ybar for pattern x that that is responsible for
	 the most violated constraint for the margin rescaling
	 formulation. For linear slack variables, this is that label ybar
	 that maximizes

	 ===== the constant item -F(psi(x,y),sm.w) omitted. by bing ======
	 argmax_{ybar} { H'(ybar) = loss(y,ybar)+F(psi(x,ybar),sm.w)}

	 Note that ybar may be equal to y (i.e. the max is 0), which is
	 different from the algorithms described in
	 [Tschantaridis/05]. Note that this argmax has to take into
	 account the scoring function in sm, especially the weights sm.w,
	 as well as the loss function, and whether linear or quadratic
	 slacks are used. The weights in sm.w correspond to the features
	 defined by psi() and range from index 1 to index
	 sm->sizePsi. Most simple is the case of the zero/one loss
	 function. For the zero/one loss, this function should return the
	 highest scoring label ybar (which may be equal to the correct
	 label y), or the second highest scoring label ybar, if
	 Psi(x,ybar)>Psi(x,y)-1. If the function cannot find a label, it
	 shall return an empty label as recognized by the function
	 empty_label(y). */

	/*
	 * Margin rescaling is paired with hamming loss, this condition should be checked first
	 */
	if (sparm->loss_function != HAMMING_LOSS) {
		printf(
				"ERROR: Margin rescaling has to be paired with Hamming loss function!\n");
		exit(1);
	}

	LABEL ybar;
	long i, ret_flg;
	double partial_objH_yes, partial_objH_no;
	double *coeff = (double *) my_malloc(sizeof(double) * (x.graph_size + 1));
	/* the coefficient of the LP problem.
	 coeff[i] = H'_i - \overline{H'}_i,
	 where H'_i = <\Psi(x_i, CUT_LABEL), sm.w> + \Delta(CUT_LABEL, c^*_i)
	 is the partial H' value achieved by x_i with label CUT_LABEL,
	 \overline{H'}_i = <\Psi(x_i, NOT_CUT_LABEL), sm.w> + \Delta(NOT_CUT_LABEL, c^*_i)*/

	for (i = 0; i < x.graph_size; i++) {

		partial_objH_yes = partial_obj_Fv(x.lines[i], CUT_LABEL, sm, sparm);
		partial_objH_yes += loss_token(y.labels[i], CUT_LABEL);

		partial_objH_no = partial_obj_Fv(x.lines[i], NOT_CUT_LABEL, sm, sparm);
		partial_objH_no += loss_token(y.labels[i], NOT_CUT_LABEL);

		coeff[i] = partial_objH_yes - partial_objH_no;

	}

	construct_empty_label(&x, &ybar);
	ret_flg = solve_LP_problem("./tmp_lp_file", coeff, ybar.labels, &x);
	if (ret_flg != 1) {
		printf(
				"ERROR: in solving the LP solution in find_most_violated_constraint_marginrescaling function for example %s.\n",
				x.file);
		fflush(stdout);
		exit(0);
	}

	lpSol_to_label(&ybar);

	if (struct_verbosity >= 4) {
		double scorey = obj_Fv(x, y, sm, sparm);
		printf("  --> w*Psi(x,y) = %lf\n", scorey);
		double score = obj_Fv(x, ybar, sm, sparm);
		printf("  --> w*Psi(x,ybar) = %lf\n", score);
		score += loss(y, ybar, sparm);
		printf("  --> w*Psi(x,ybar)+delta(y,ybar) = %lf\n", score);
		printf("  --> w*(Psi(x,y)-Psi(x,ybar))+delta(y,ybar) = %lf\n", score
				- scorey);
	}

	free(coeff);
	return (ybar);
}

int empty_label(LABEL y) {
	/* Returns true, if y is an empty label. An empty label might be
	 returned by find_most_violated_constraint_???(x, y, sm) if there
	 is no incorrect label that can be found for x, or if it is unable
	 to label x at all */
	return (0);
}

SVECTOR *psi(PATTERN x, LABEL y, STRUCTMODEL *sm, STRUCT_LEARN_PARM *sparm) {
	/* Returns a feature vector describing the match between pattern x
	 and label y. The feature vector is returned as a list of
	 SVECTOR's. Each SVECTOR is in a sparse representation of pairs
	 <featurenumber:featurevalue>, where the last pair has
	 featurenumber 0 as a terminator. Featurenumbers start with 1 and
	 end with sizePsi. Featuresnumbers that are not specified default
	 to value 0. As mentioned before, psi() actually returns a list of
	 SVECTOR's. Each SVECTOR has a field 'factor' and 'next'. 'next'
	 specifies the next element in the list, terminated by a NULL
	 pointer. The list can be though of as a linear combination of
	 vectors, where each vector is weighted by its 'factor'. This
	 linear combination of feature vectors is multiplied with the
	 learned (kernelized) weight vector to score label y for pattern
	 x. Without kernels, there will be one weight in sm.w for each
	 feature. Note that psi has to match
	 find_most_violated_constraint_???(x, y, sm) and vice versa. In
	 particular, find_most_violated_constraint_???(x, y, sm) finds
	 that ybar!=y that maximizes psi(x,ybar,sm)*sm.w (where * is the
	 inner vector product) and the appropriate function of the
	 loss + margin/slack rescaling method. See that paper for details. */
	SVECTOR *fvec = NULL;

	SVECTOR *fshift = NULL;
	long fnum = sparm->num_features;
	long i, j;
	WORD tfeat[2];

	WORD word_ij;
	CLINE line_i;

	tfeat[1].wnum = 0;
	fvec = NULL;

	for (i = 0; i < x.graph_size; i++) {
		line_i = x.lines[i];
		for (j = 0; line_i.fvec->words[j].wnum != 0; j++) {
			word_ij = line_i.fvec->words[j];
			tfeat[0].wnum = FEATURE_MAP_INDEX(fnum, y.labels[i],word_ij.wnum);

			tfeat[0].weight = word_ij.weight;
			fshift = create_svector(tfeat, NULL, 1.0);
			append_svector_list(fshift, fvec);
			fvec = fshift;
		}
	}

	return (fvec);
}

SVECTOR *psi_line(CLINE line_i, int y_i, STRUCT_LEARN_PARM *sparm) {
	/* Returns a feature vector describing the match between CLINE line
	 and label y. */
	SVECTOR *fvec = NULL;

	SVECTOR *fshift = NULL;
	long fnum = sparm->num_features;
	long j;
	WORD tfeat[2];

	WORD word_ij;

	tfeat[1].wnum = 0;
	fvec = NULL;

	for (j = 0; line_i.fvec->words[j].wnum != 0; j++) {
		word_ij = line_i.fvec->words[j];
		tfeat[0].wnum = FEATURE_MAP_INDEX(fnum, y_i ,word_ij.wnum);

		tfeat[0].weight = word_ij.weight;
		fshift = create_svector(tfeat, NULL, 1.0);
		append_svector_list(fshift, fvec);
		fvec = fshift;
	}

	return (fvec);
}

double loss(LABEL y, LABEL ybar, STRUCT_LEARN_PARM *sparm) {
	/* loss for correct label y and predicted label ybar.
	 sparm->loss_function is set with the -l option. */

	long i;
	int cnt_vipline, cnt_wrg_vipline;
	int cnt_uncorrect;
	if (sparm->loss_function == 0) { /* type 0 loss: 0/1 loss */
		/* return 0, if y==ybar. return 1 else */
		printf("ERROR: 0/1 loss function is not supported!\n");
		exit(1);
		//		for (i = 0; i < y.graph_size; i++) {
		//			if (y.labels[i] != ybar.labels[i])
		//				return 1;
		//		}
		//		return 0;
	} else if (sparm->loss_function == INFORMATIVE_BLOCK_LOSS) { /* type 1 loss: informative block based loss */
		/* Put your code for different loss functions here. But then
		 find_most_violated_constraint_???(x, y, sm) has to return the
		 highest scoring label with the largest loss. */
		cnt_vipline = 0;
		cnt_wrg_vipline = 0;
		for (i = 0; i < y.graph_size; i++) {
			if (y.x->lines[i].vipline == VIP_LINE) {
				cnt_vipline++;
				if (ybar.labels[i] == NOT_CUT_LABEL)
					cnt_wrg_vipline++;
			}
		}
		if (cnt_vipline == 0) {
			printf(
					"ERROR: no very important line found, so cannot use type 1 loss! Please check the training example %s\n",
					y.x->file);
			exit(1);
		}
		return exp(((double) cnt_wrg_vipline) / cnt_vipline);
	} else if (sparm->loss_function == HAMMING_LOSS) { /* type 2 loss: hamming distance loss */
		cnt_uncorrect = 0;
		for (i = 0; i < y.graph_size; i++) {
			if (y.labels[i] != ybar.labels[i])
				cnt_uncorrect++;
		}
		return ((double) cnt_uncorrect);
	} else {
		printf("ERROR: Unknown type of loss function!\n");
		exit(1);
	}
}

int finalize_iteration(double ceps, int cached_constraint, SAMPLE sample,
		STRUCTMODEL *sm, CONSTSET cset, double *alpha, STRUCT_LEARN_PARM *sparm) {

	/* This function is called just before the end of each cutting plane iteration.
	 * ceps is the amount by which the most violated constraint found in the current
	 * iteration was violated. cached_constraint is true if the added constraint was
	 * constructed from the cache. If the return value is FALSE, then the algorithm
	 * is allowed to terminate. If it is TRUE, the algorithm will keep iterating
	 * even if the desired precision sparm->epsilon is already reached. */
	return (0);
}

void print_struct_learning_stats(SAMPLE sample, STRUCTMODEL *sm, CONSTSET cset,
		double *alpha, STRUCT_LEARN_PARM *sparm) {
	/* This function is called after training and allows final touches to
	 the model sm. But primarly it allows computing and printing any
	 kind of statistic (e.g. training error) you might want. */
	MODEL *model;

	/* Replace SV with single weight vector */
	model = sm->svm_model;
	if (model->kernel_parm.kernel_type == LINEAR) {
		if (struct_verbosity >= 1) {
			printf("Compacting linear model...");
			fflush(stdout);
		}
		sm->svm_model = compact_linear_model(model);
		sm->w = sm->svm_model->lin_weights; /* short cut to weight vector */
		free_model(model, 1);
		if (struct_verbosity >= 1) {
			printf("done\n");
			fflush(stdout);
		}
	}

	//	LABEL classify_struct_example(PATTERN x, STRUCTMODEL *sm,
	//			STRUCT_LEARN_PARM *sparm)

	int i = 0;
	/* calculate the training error */
	if (sparm->loss_function == HAMMING_LOSS) {
		double all = 0, wrong = 0;
		for (i = 0; i < sample.n; i++) {
			LABEL ybar = classify_struct_example(sample.examples[i].x, sm,
					sparm);
			all += sample.examples[i].x.graph_size;
			wrong += loss(sample.examples[i].y, ybar, sparm);
		}
		printf("The error rate of wrongly labeled boundary lines: %.2f%%\n", (float) 100.0 *wrong / all);
	}
	if (sparm->loss_function == INFORMATIVE_BLOCK_LOSS) {
		double cnt_vipline = 0, cnt_wrg_vipline = 0;
		int j;
		for (i = 0; i < sample.n; i++) {
			LABEL ybar = classify_struct_example(sample.examples[i].x, sm,
					sparm);
			for (j = 0; j < sample.examples[i].x.graph_size; j++) {
				if (sample.examples[i].x.lines[j].vipline == VIP_LINE) {
					cnt_vipline++;
					if (ybar.labels[j] == NOT_CUT_LABEL)
						cnt_wrg_vipline++;
				}
			}
		}
		printf("The error rate of wrongly labeled VIP boundary lines: %.2f%%\n",
				(float) 100.0 *cnt_wrg_vipline / cnt_vipline);
	}
}

void print_struct_testing_stats(SAMPLE sample, STRUCTMODEL *sm,
		STRUCT_LEARN_PARM *sparm, STRUCT_TEST_STATS *teststats) {
	/* This function is called after making all test predictions in
	 svm_struct_classify and allows computing and printing any kind of
	 evaluation (e.g. precision/recall) you might want. You can use
	 the function eval_prediction to accumulate the necessary
	 statistics for each prediction. */
}

void eval_prediction(long exnum, EXAMPLE ex, LABEL ypred, STRUCTMODEL *sm,
		STRUCT_LEARN_PARM *sparm, STRUCT_TEST_STATS *teststats) {
	/* This function allows you to accumlate statistic for how well the
	 predicition matches the labeled example. It is called from
	 svm_struct_classify. See also the function
	 print_struct_testing_stats. */
	if (exnum == 0) { /* this is the first time the function is
	 called. So initialize the teststats */
	}
	if (sparm->loss_function == HAMMING_LOSS) {
		double all = 0, wrong = 0;
		all = ex.x.graph_size;
		wrong = loss(ex.y, ypred, sparm);
		printf("The error rate of wrongly labeled boundary lines in %s: %.2f%%\n", ex.x.file, (float) 100.0 *wrong / all);
	}
	if (sparm->loss_function == INFORMATIVE_BLOCK_LOSS) {
		double cnt_vipline = 0, cnt_wrg_vipline = 0;
		int j;
		for (j = 0; j < ex.x.graph_size; j++) {
			if (ex.x.lines[j].vipline == VIP_LINE) {
				cnt_vipline++;
				if (ypred.labels[j] == NOT_CUT_LABEL)
					cnt_wrg_vipline++;
			}
		}

		printf("The error rate of wrongly labeled VIP boundary lines in %s: %.2f%%\n", ex.x.file,
				(float)100.0 * cnt_wrg_vipline / cnt_vipline);
	}
}

void write_struct_model(char *file, STRUCTMODEL *sm, STRUCT_LEARN_PARM *sparm) {
	/* Writes structural model sm to file file. */

	FILE *modelfl;
	long j, i, sv_num;
	SVECTOR *v;
	MODEL *model = sm->svm_model;

	if (verbosity >= 1) {
		printf("Writing model file...");
		fflush(stdout);
	}
	if ((modelfl = fopen(file, "w")) == NULL) {
		perror(file);
		exit(1);
	}
	fprintf(modelfl, "SVM-Page-Segmentation Version %s\n", INST_VERSION);
	fprintf(modelfl, "%ld # kernel type\n", model->kernel_parm.kernel_type);
	fprintf(modelfl, "%ld # kernel parameter -d \n",
			model->kernel_parm.poly_degree);
	fprintf(modelfl, "%.8g # kernel parameter -g \n",
			model->kernel_parm.rbf_gamma);
	fprintf(modelfl, "%.8g # kernel parameter -s \n",
			model->kernel_parm.coef_lin);
	fprintf(modelfl, "%.8g # kernel parameter -r \n",
			model->kernel_parm.coef_const);
	fprintf(modelfl, "%s# kernel parameter -u \n", model->kernel_parm.custom);
	fprintf(modelfl, "%ld # highest mapping feature index \n", model->totwords);
	fprintf(modelfl, "%ld # number of features \n", sparm->num_features);
	fprintf(modelfl, "%ld # number of classes \n", sparm->num_classes);
	//	fprintf(modelfl, "%ld # HMM order of transitions \n",
	//			sparm->hmm_trans_order);
	//	fprintf(modelfl, "%ld # HMM order of emissions \n", sparm->hmm_emit_order);
	fprintf(modelfl, "%d # loss function \n", sparm->loss_function);

	sv_num = 1;
	for (i = 1; i < model->sv_num; i++) {
		for (v = model->supvec[i]->fvec; v; v = v->next)
			sv_num++;
	}
	fprintf(modelfl, "%ld # number of support vectors plus 1 \n", sv_num);
	fprintf(
			modelfl,
			"%.8g # threshold b, each following line is a SV (starting with alpha*y)\n",
			model->b);

	for (i = 1; i < model->sv_num; i++) {
		for (v = model->supvec[i]->fvec; v; v = v->next) {
			fprintf(modelfl, "%.32g ", model->alpha[i] * v->factor);
			for (j = 0; (v->words[j]).wnum; j++) {
				fprintf(modelfl, "%ld:%.8g ", (long) (v->words[j]).wnum,
						(double) (v->words[j]).weight);
			}
			if (v->userdefined)
				fprintf(modelfl, "#%s\n", v->userdefined);
			else
				fprintf(modelfl, "#\n");
			/* NOTE: this could be made more efficient by summing the
			 alpha's of identical vectors before writing them to the
			 file. */
		}
	}
	fclose(modelfl);
	if (verbosity >= 1) {
		printf("done\n");
	}

}

STRUCTMODEL read_struct_model(char *file, STRUCT_LEARN_PARM *sparm) {
	/* Reads structural model sm from file file. This function is used
	 only in the prediction module, not in the learning module. */

	FILE *modelfl;
	long i, queryid, slackid;
	double costfactor;
	long max_sv, max_words, ll, wpos;
	char *line, *comment;
	WORD *words;
	char version_buffer[100];
	MODEL *model;
	STRUCTMODEL sm;

	nol_ll(file, &max_sv, &max_words, &ll); /* scan size of model file */
	max_words += 2;
	ll += 2;

	words = (WORD *) my_malloc(sizeof(WORD) * (max_words + 10));
	line = (char *) my_malloc(sizeof(char) * ll);
	model = (MODEL *) my_malloc(sizeof(MODEL));
	sm.svm_model = model;

	if ((modelfl = fopen(file, "r")) == NULL) {
		perror(file);
		exit(1);
	}

	fscanf(modelfl, "SVM-Page-Segmentation Version %s\n", version_buffer);
	if (strcmp(version_buffer, INST_VERSION)) {
		perror(
				"Version of model-file does not match version of the current classifier!");
		exit(1);
	}
	fscanf(modelfl, "%ld%*[^\n]\n", &model->kernel_parm.kernel_type);
	fscanf(modelfl, "%ld%*[^\n]\n", &model->kernel_parm.poly_degree);
	fscanf(modelfl, "%lf%*[^\n]\n", &model->kernel_parm.rbf_gamma);
	fscanf(modelfl, "%lf%*[^\n]\n", &model->kernel_parm.coef_lin);
	fscanf(modelfl, "%lf%*[^\n]\n", &model->kernel_parm.coef_const);
	fscanf(modelfl, "%[^#]%*[^\n]\n", model->kernel_parm.custom);

	fscanf(modelfl, "%ld%*[^\n]\n", &model->totwords);
	sm.sizePsi = model->totwords;
	fscanf(modelfl, "%ld%*[^\n]\n", &sparm->num_features);
	fscanf(modelfl, "%ld%*[^\n]\n", &sparm->num_classes);
	//	fscanf(modelfl, "%ld%*[^\n]\n", &sparm->hmm_trans_order);
	//	fscanf(modelfl, "%ld%*[^\n]\n", &sparm->hmm_emit_order);
	fscanf(modelfl, "%d%*[^\n]\n", &sparm->loss_function);
	fscanf(modelfl, "%ld%*[^\n]\n", &model->sv_num);
	fscanf(modelfl, "%lf%*[^\n]\n", &model->b);

	model->supvec = (DOC **) my_malloc(sizeof(DOC *) * model->sv_num);
	model->alpha = (double *) my_malloc(sizeof(double) * model->sv_num);
	model->index = NULL;
	model->lin_weights = NULL;

	for (i = 1; i < model->sv_num; i++) {
		fgets(line, (int) ll, modelfl);
		if (!parse_document(line, words, &(model->alpha[i]), &queryid,
				&slackid, &costfactor, &wpos, max_words, &comment)) {
			printf("\nParsing error while reading model file in SV %ld!\n%s",
					i, line);
			exit(1);
		}
		model->supvec[i] = create_example(-1, 0, 0, 0.0, create_svector(words,
				comment, 1.0));
	}
	fclose(modelfl);
	free(line);
	free(words);
	if (verbosity >= 1) {
		fprintf(stdout, "(%d support vectors read)..",
				(int) (model->sv_num - 1));
	}
	sm.w = model->lin_weights;

	return (sm);
}

void write_label(FILE *fp, LABEL y) {
	/* Writes label y to file handle fp. */
	long i;
	for (i = 0; i < y.graph_size; i++) {
		fprintf(fp, "%ld\t%ld\n", i + 1, y.labels[i]);
	}
	fprintf(fp, "\n");

}

void free_pattern(PATTERN x) {
	/* Frees the memory of x. */
	long i;
	for (i = 0; i < x.graph_size; i++) {
		if (x.lines[i].fvec)
			free_svector(x.lines[i].fvec);
	}
	free(x.lines);
}

void free_label(LABEL y) {
	/* Frees the memory of y. */
	free(y.labels);
}

void free_struct_model(STRUCTMODEL sm) {
	/* Frees the memory of model. */
	/* if(sm.w) free(sm.w); *//* this is free'd in free_model */
	if (sm.svm_model)
		free_model(sm.svm_model, 1);
	/* add free calls for user defined data here */
}

void free_struct_sample(SAMPLE s) {
	/* Frees the memory of sample s. */
	int i;
	for (i = 0; i < s.n; i++) {
		free_pattern(s.examples[i].x);
		free_label(s.examples[i].y);
	}
	free(s.examples);
}

void print_struct_help() {
	/* Prints a help text that is appended to the common help text of
	 svm_struct_learn. */
	printf(
			"         --* string  -> custom parameters that can be adapted for struct\n");
	printf(
			"                        learning. The * can be replaced by any character\n");
	printf(
			"                        and there can be multiple options starting with --.\n");
}

void parse_struct_parameters(STRUCT_LEARN_PARM *sparm) {
	/* Parses the command line parameters that start with -- */
	int i;

	for (i = 0; (i < sparm->custom_argc) && ((sparm->custom_argv[i])[0] == '-'); i++) {
		switch ((sparm->custom_argv[i])[2]) {
		case 'a':
			i++; /* strcpy(learn_parm->alphafile,argv[i]); */
			break;
		case 'e':
			i++; /* sparm->epsilon=atof(sparm->custom_argv[i]); */
			break;
		case 'k':
			i++; /* sparm->newconstretrain=atol(sparm->custom_argv[i]); */
			break;
		default:
			printf("\nUnrecognized option %s!\n\n", sparm->custom_argv[i]);
			exit(0);
		}
	}
}

void print_struct_help_classify() {
	/* Prints a help text that is appended to the common help text of
	 svm_struct_classify. */
	printf(
			"         --* string -> custom parameters that can be adapted for struct\n");
	printf(
			"                       learning. The * can be replaced by any character\n");
	printf(
			"                       and there can be multiple options starting with --.\n");
}

void parse_struct_parameters_classify(STRUCT_LEARN_PARM *sparm) {
	/* Parses the command line parameters that start with -- for the
	 classification module */
	int i;

	for (i = 0; (i < sparm->custom_argc) && ((sparm->custom_argv[i])[0] == '-'); i++) {
		switch ((sparm->custom_argv[i])[2]) {
		/* case 'x': i++; strcpy(xvalue,sparm->custom_argv[i]); break; */
		default:
			printf("\nUnrecognized option %s!\n\n", sparm->custom_argv[i]);
			exit(0);
		}
	}
}

/****************************implement my function *************************/
void get_files(char *dir, char *** files) {
	int file_num = 0;
	int cnt = 0;
	DIR *dirr;
	struct dirent *ptr;

	dirr = opendir(dir);
	while ((ptr = readdir(dirr)) != NULL) {
		file_num++;
	}
	closedir(dirr);

	/*
	 * malloc(sizeof(char *) * (file_num + 1)): allocate (file_num + 1) positions for (char *)
	 * the beginning of this memory block is type of (char **)
	 */
	(*files) = (char **) my_malloc(sizeof(char *) * (file_num + 1));

	dirr = opendir(dir);
	while ((ptr = readdir(dirr)) != NULL) {
		(*files)[cnt] = (char *) my_malloc(sizeof(char) * (strlen(ptr-> d_name)
				+ 1));
		strcpy((*files)[cnt], ptr-> d_name);
		//		printf("file   :   %s\n ", (*files)[cnt]);
		cnt++;
	}
	closedir(dirr);
	/*ending flag	 */
	(*files)[cnt] = NULL;

	ptr = NULL;
	dirr = NULL;
}

void free_arrLL(long ** lines, int max_idx) {

	if (lines == NULL)
		return;

	int i = 0;
	if (max_idx != -1) {
		for (i = 0; i <= max_idx; i++) {
			free(lines[i]);
		}
	} else {
		i = 0;
		while (lines[i] != NULL) {
			free(lines[i]);
			i++;
		}
	}
	free(lines);
}

/*
 * when max_idx is -1, the last line of lines should be
 * NULL as flag
 */
void free_charSS(char ** lines, int max_idx) {
	if (lines == NULL)
		return;

	int i = 0;
	if (max_idx != -1) {
		for (i = 0; i <= max_idx; i++) {
			free(lines[i]);
		}
	} else {
		i = 0;
		while (lines[i] != NULL) {
			free(lines[i]);
			i++;
		}
	}
	free(lines);
}

int read_one_example(char *file, long *total_features, int *maxlabel,
		EXAMPLE *example) {
	PATTERN *x = &example->x;

	/*
	 * total_nodes: total number of cutting lines
	 * totwords: total number of features
	 */
	long tmp_total_features = 0, total_nodes = 0, i, line_id;
	DOC **examples_flat = NULL; //each example is one cutting line
	double *labels_flat = NULL;
	long max_node_id = 0;
	long parent1, parent2;
	int vipline;

	read_documents(file, &examples_flat, &labels_flat, &tmp_total_features,
			&total_nodes);

	line_id = 0;
	for (i = 0; i < total_nodes; i++) {
		if (examples_flat[i]->queryid < 1) {
			printf(
					"ERROR (Line %ld): illegal cutting line ID (<1) in file: %s\n",
					i + 1, file);
			fflush(stdout);
			free(examples_flat);
			free(labels_flat);
			exit(1);
		}
		if (line_id > examples_flat[i]->queryid) {
			printf(
					"ERROR (Line %ld): cutting line ID's have to be in increasing order. In file: %s\n",
					i + 1, file);
			fflush(stdout);
			free(examples_flat);
			free(labels_flat);
			exit(1);
		}
		if (examples_flat[i]->queryid > max_node_id) {
			max_node_id = examples_flat[i]->queryid;
		}
		line_id = examples_flat[i]->queryid;
	}

	if (max_node_id != total_nodes) {
		printf(
				"ERROR: Max cutting line ID (%ld) != total number of lines (%ld) in file: %s\n",
				max_node_id, total_nodes, file);
		fflush(stdout);
		free(examples_flat);
		free(labels_flat);
		exit(1);
	}

	x->lines = (CLINE *) my_malloc(sizeof(CLINE) * (total_nodes + 1));

	// The last position is a flag with line_ID==-1 always
	for (i = 0; i <= total_nodes; i++) {
		x->lines[i].line_ID = -1;
		x->lines[i].parent_line_ID[0] = -1;
		x->lines[i].parent_line_ID[1] = -1;
		x->lines[i].label = -1;
		x->lines[i].fvec = NULL;
	}

	for (i = 0; i < total_nodes; i++) {
		//line ID starts from 1. Line with ID 1 stored in x->lines[0]
		line_id = examples_flat[i]->queryid;

		if (x->lines[line_id - 1].line_ID != -1) {
			printf("ERROR: Duplicated cutting line ID (%ld) in: %s \n",
					line_id, file);
			free(x->lines);
			free(examples_flat);
			free(labels_flat);
			exit(1);
		}

		x->lines[line_id - 1].line_ID = line_id;
		x->lines[line_id - 1].label = labels_flat[i];
		x->lines[line_id - 1].fvec = examples_flat[i]->fvec;

		// parse parent of the lines
		parse_parents(examples_flat[i]->fvec->userdefined, &parent1, &parent2,
				&vipline);
		x->lines[line_id - 1].parent_line_ID[0] = parent1;
		x->lines[line_id - 1].parent_line_ID[1] = parent2;
		if (vipline == VIP_LINE && labels_flat[i] != CUT_LABEL) {
			printf(
					"ERROR: The label of important block's boundary must be CUT_LABEL. Line ID (%ld) in: %s \n",
					line_id, file);
			free(x->lines);
			free(examples_flat);
			free(labels_flat);
			exit(1);
		}
		x->lines[line_id - 1].vipline = vipline;

		*maxlabel = MAX(*maxlabel, labels_flat[i]);

	}

	*total_features = MAX(*total_features, tmp_total_features);

	x->graph_size = total_nodes;
	strcpy(x->file, file);

	if (is_pattern_legal(x) == 0) {
		printf(
				"ERROR: The example is illegal, because of illegal IDs or missing IDs in: %s \n",
				file);
		free(x->lines);
		free(examples_flat);
		free(labels_flat);
		exit(1);
	}

	free(examples_flat);
	free(labels_flat);

	return 1;
}

void parse_parents(char *parent_str, long *parent_1, long *parent_2,
		int *vipline) {
	int pos = 0;
	char word[100];
	while (space_or_null((int) parent_str[pos]))
		pos++;

	*parent_1 = -1;
	*parent_2 = -1;
	*vipline = 0;
	while ((pos += read_word(parent_str + pos, word)) && (word[0])) {
		sscanf(word, "plid1:%ld", parent_1);
		sscanf(word, "plid2:%ld", parent_2);
		sscanf(word, "vipline:%d", vipline);
	}
}

void construct_org_label(PATTERN *x, LABEL *y) {
	y->labels = (long *) my_malloc(sizeof(long) * (x->graph_size + 1));
	long i = 0;
	for (i = 0; i <= x->graph_size; i++) {
		y->labels[i] = x->lines[i].label;
	}
	y->x = x;
	y->graph_size = x->graph_size;

}

int solve_LP_problem_with_fixedVs(char *file, double *coeff, long * sol,
		PATTERN *x, long * vip_set, long * subset) {
	int ret_flg;

	write_LP_problem_with_fixedVs("./tmp_lp_file", coeff, x, vip_set, subset);

	// to solve this lp problem
	// call solution to label function


	system("export LD_LIBRARY_PATH=./glpk/lib");
	ret_flg
			= system(
					"./glpk/bin/glpsol --lp ./tmp_lp_file -o ./tmp_lp_file.sol > sh_out.tmp");

	if (ret_flg != 0) {
		printf(
				"ERROR: in solving the LP in find_most_violated_constraint_marginrescaling function for example %s.\n",
				x->file);
		fflush(stdout);
		exit(0);
	}

	ret_flg = load_LP_solution("./tmp_lp_file.sol", sol, x->graph_size);
	if (ret_flg == 0) {
		printf(
				"ERROR: in loading the LP solution in find_most_violated_constraint_marginrescaling function for example %s.\n",
				x->file);
		fflush(stdout);
		exit(0);
	}

	return 1;
}

int solve_LP_problem(char *file, double *coeff, long * sol, PATTERN *x) {

	int ret_flg;

	write_LP_problem("./tmp_lp_file", coeff, x);

	// to solve this lp problem
	// call solution to label function


	system("export LD_LIBRARY_PATH=./glpk/lib");
	ret_flg
			= system(
					"./glpk/bin/glpsol --lp ./tmp_lp_file -o ./tmp_lp_file.sol > sh_out.tmp");

	if (ret_flg != 0) {
		printf(
				"ERROR: in solving the LP in find_most_violated_constraint_marginrescaling function for example %s.\n",
				x->file);
		fflush(stdout);
		exit(0);
	}

	ret_flg = load_LP_solution("./tmp_lp_file.sol", sol, x->graph_size);
	if (ret_flg == 0) {
		printf(
				"ERROR: in loading the LP solution in find_most_violated_constraint_marginrescaling function for example %s.\n",
				x->file);
		fflush(stdout);
		exit(0);
	}

	return 1;
}

void write_LP_problem_with_fixedVs(char *file, double *coeff, PATTERN *x,
		long * vip_set, long * subset) {

	FILE *LP_file;
	long i, j;
	int in_subset, in_vip_set;
	if ((LP_file = fopen(file, "w")) == NULL) {
		perror(file);
		printf("ERROR: error when writing the LP problem to the tmp file ");
		exit(1);
	}

	fprintf(LP_file, "Maximize \n");
	for (i = 0; i < x->graph_size; i++) {
		if (coeff[i] >= 0) {
			fprintf(LP_file, "+ %.8g x%ld ", coeff[i], x->lines[i].line_ID);
		} else {
			fprintf(LP_file, "- %.8g x%ld ", coeff[i] * -1, x->lines[i].line_ID);
		}
	}

	fprintf(LP_file, "\n\n");

	fprintf(LP_file, "Subject To \n");

	for (i = 0; i < x->graph_size; i++) {
		if (x->lines[i].parent_line_ID[0] != -1) {
			fprintf(LP_file, "x%ld - x%ld >= 0 \n",
					x->lines[i].parent_line_ID[0], x->lines[i].line_ID);
		}
		if (x->lines[i].parent_line_ID[1] != -1) {
			fprintf(LP_file, "x%ld - x%ld >= 0 \n",
					x->lines[i].parent_line_ID[1], x->lines[i].line_ID);
		}

	}

	fprintf(LP_file, "\n");

	fprintf(LP_file, "Bounds \n");

	for (i = 0; i < x->graph_size; i++) {
		in_subset = 0;
		in_vip_set = 0;
		j = 0;
		while (subset[j] != -1) {
			if (subset[j] == x->lines[i].line_ID) {
				in_subset = 1;
				break;
			}
			j++;
		}
		if (in_subset == 1) {
			// in subset, have value 0, i.e. label NOT_CUT_LABEL
			fprintf(LP_file, "x%ld = 0 \n", x->lines[i].line_ID);
		} else {
			j = 0;
			while (vip_set[j] != -1) {
				if (vip_set[j] == x->lines[i].line_ID) {
					in_vip_set = 1;
					break;
				}
				j++;
			}
			if (in_vip_set == 1) {
				// in vip_set but not in subset, have value 1, i.e. label CUT_LABEL
				fprintf(LP_file, "x%ld = 1 \n", x->lines[i].line_ID);
			} else {
				fprintf(LP_file, "0 <= x%ld <= 1 \n", x->lines[i].line_ID);
			}
		}
	}

	fprintf(LP_file, "\n");
	fprintf(LP_file, "end \n");
	fclose(LP_file);
}

void write_LP_problem(char *file, double *coeff, PATTERN *x) {

	FILE *LP_file;
	long i;
	if ((LP_file = fopen(file, "w")) == NULL) {
		perror(file);
		printf("ERROR: error when writing the LP problem to the tmp file ");
		exit(1);
	}

	fprintf(LP_file, "Maximize \n");
	for (i = 0; i < x->graph_size; i++) {
		if (coeff[i] >= 0) {
			fprintf(LP_file, "+ %.8g x%ld ", coeff[i], x->lines[i].line_ID);
		} else {
			fprintf(LP_file, "- %.8g x%ld ", coeff[i] * -1, x->lines[i].line_ID);
		}
	}

	fprintf(LP_file, "\n\n");

	fprintf(LP_file, "Subject To \n");

	for (i = 0; i < x->graph_size; i++) {
		if (x->lines[i].parent_line_ID[0] != -1) {
			fprintf(LP_file, "x%ld - x%ld >= 0 \n",
					x->lines[i].parent_line_ID[0], x->lines[i].line_ID);
		}
		if (x->lines[i].parent_line_ID[1] != -1) {
			fprintf(LP_file, "x%ld - x%ld >= 0 \n",
					x->lines[i].parent_line_ID[1], x->lines[i].line_ID);
		}

	}

	fprintf(LP_file, "\n");

	fprintf(LP_file, "Bounds \n");

	for (i = 0; i < x->graph_size; i++) {
		fprintf(LP_file, "0 <= x%ld <= 1 \n", x->lines[i].line_ID);
	}

	fprintf(LP_file, "\n");
	fprintf(LP_file, "end \n");
	fclose(LP_file);
}

char ** load_lines(char * file) {
	long num_line = 0;
	long max_num_word = 0;
	long longest_line = 0;
	char **lines, *line;
	long cnt = 0;

	nol_ll(file, &num_line, &max_num_word, &longest_line);

	FILE *docfl;
	if ((docfl = fopen(file, "r")) == NULL) {
		printf("error in loading boundary lines of %s \n", file);
		exit(1);
	}

	//	num_line += 2;
	//	max_num_word += 2;
	longest_line += 2;

	line = (char *) my_malloc(sizeof(char) * (longest_line + 2));
	lines = (char **) my_malloc(sizeof(char*) * (num_line + 2));

	while ((!feof(docfl)) && fgets(line, (int) longest_line, docfl)) {
		if (line[0] == '#')
			continue; /* line contains comments */
		else {
			lines[cnt++] = line;
			line = (char *) my_malloc(sizeof(char) * (longest_line + 2));
		}
	}
	// end flag
	lines[cnt] = NULL;
	fclose(docfl);

	free(line);
	return lines;
}

int load_LP_solution(char *file, long *sol, long graph_size) {

	int i = 0, j = 0, pos = 0;
	long num_var;
	int flg_undefined;
	char ** lines = load_lines(file);
	char word[500];

	num_var = 0;
	flg_undefined = 0;
	i = 0;
	while (lines[i] != NULL) {
		if (strstr(lines[i], "Columns:")) {
			pos = strlen("Columns:");
			if ((pos += read_word(lines[i] + pos, word)) && (word[0])) {
				num_var = atol(word);
			}
		}
		if (strstr(lines[i], "Status:")) {
			if (strstr(lines[i], "UNDEFINED")) {
				flg_undefined = 1;
			}
			break;
		}
		i++;
	}

	if (num_var == 0) {
		printf(
				"ERROR:No column number found in %s. in load_LP_solution function. \n",
				file);
		fflush(stdout);
		return 0;
	}

	if (flg_undefined == 1) {
		printf(
				"ERROR:The LP problem is unsolvable %s. in load_LP_solution function. \n",
				file);
		fflush(stdout);
		return 0;
	}

	if (num_var != graph_size) {
		printf("ERROR: in load_LP_solution function for solution %s.\n", file);
		printf(
				"   LP solution's column number (%ld) is not equal to the boundary line number (%ld) \n",
				num_var, graph_size);
		fflush(stdout);
		return 0;
	}

	while (lines[i] != NULL) {
		if (strstr(lines[i], "Column name")) {
			i = i + 2;
			break;
		}
		i++;
	}

	if (lines[i] == NULL) {
		printf(
				"ERROR: No Column matrix section found in %s. in load_LP_solution function.\n",
				file);
		fflush(stdout);
		return 0;
	}

	j = i;
	for (; i < j + num_var; i++) {
		pos = 0;
		//skip the No. Column
		if (!((pos += read_word(lines[i] + pos, word)) && (word[0]))) {
			printf(
					"ERROR: in Column line %d: %s. in load_LP_solution function.\n",
					i, lines[i]);
			fflush(stdout);
			return 0;
		}
		//skip the variable name Column
		if (!((pos += read_word(lines[i] + pos, word)) && (word[0]))) {
			printf(
					"ERROR: in Column line %d: %s. in load_LP_solution function.\n",
					i, lines[i]);
			fflush(stdout);
			return 0;
		}
		//skip the St Column
		if (!((pos += read_word(lines[i] + pos, word)) && (word[0]))) {
			printf(
					"ERROR: in Column line %d: %s. in load_LP_solution function.\n",
					i, lines[i]);
			fflush(stdout);
			return 0;
		}
		//read the solution for one variable
		if (!((pos += read_word(lines[i] + pos, word)) && (word[0]))) {
			printf(
					"ERROR: in Column line %d: %s. in load_LP_solution function.\n",
					i, lines[i]);
			fflush(stdout);
			return 0;
		}

		sol[i - j] = (long) (atof(word) + 0.5);

	}

	free_charSS(lines, -1);
	return 1;
}

void construct_empty_label(PATTERN *x, LABEL *y) {
	y->labels = (long *) my_malloc(sizeof(long) * (x->graph_size + 1));
	y->graph_size = x->graph_size;
	long i = 0;
	for (i = 0; i <= x->graph_size; i++) {
		y->labels[i] = -1;
	}
}

void lpSol_to_label(LABEL *y) {
	int i = 0;
	for (; i < y->graph_size; i++) {
		if (y->labels[i] == 0)
			y->labels[i] = NOT_CUT_LABEL;
	}
}

double loss_token(long y, long ybar)
/* Computes the per label loss */
{
	if (y == ybar)
		return (0);
	else
		return (1);
}

/*
 * The lines in subset has label NOT_CUT_LABEL.
 * The lines in vip_set but not in subset has label CUT_LABEL.
 * Check the constraint if x_i ->> x_j, x_j's label cannot conflict with x_i.
 * In other words, if x_j is in vip_set but not in subset, i.e. having CUT_LABEL,
 * all its ancestors should also not be in subset, i.e., having CUT_LABEL.
 * NOTE that all lines in vip_set have true label CUT_LABEL.
 */
int is_subset_legal(long *vip_set, long size_full, long *subset, long size_sub,
		PATTERN *x) {
	long i, j;
	int flg_not_in_sub;
	int flg_legal = 1;
	long* ancestor_IDs;
	long parent_1, parent_2;
	long anc_size, anc_i;
	long currr_id;

	int k = 0;

	//	printf("\n");
	//	printf("===================");
	//	for (k = 0; k < size_full; k++) {
	//		printf("%ld ", vip_set[k]);
	//	}
	//	printf("\n");
	//	for (k = 0; k < size_sub; k++) {
	//		printf("%ld ", subset[k]);
	//	}
	//	printf("\n");

	ancestor_IDs = (long *) malloc(sizeof(long) * (x->graph_size + 2));
	for (i = 0; i < size_full; i++) {
		flg_not_in_sub = 1;
		j = 0;
		for (j = 0; j < size_sub; j++) {
			if (vip_set[i] == subset[j]) {
				flg_not_in_sub = 0;
				break;
			}
		}
		// if vip_set[i] not in subset
		if (flg_not_in_sub) {

			ancestor_IDs[0] = vip_set[i];
			ancestor_IDs[1] = -1;
			anc_size = 1;
			anc_i = 0;
			// get all ancestors of vip_set[i]
			do {
				currr_id = ancestor_IDs[anc_i];
				parent_1 = x->lines[currr_id - 1].parent_line_ID[0];
				parent_2 = x->lines[currr_id - 1].parent_line_ID[1];
				if (parent_1 != -1) {
					j = 0;
					for (j = 0; j < anc_size; j++) {
						if (ancestor_IDs[j] == parent_1) {
							break;
						}
					}
					if (j == anc_size) {
						ancestor_IDs[anc_size] = parent_1;
						anc_size++;
						ancestor_IDs[anc_size] = -1;
					}
				}
				if (parent_2 != -1) {
					j = 0;
					for (j = 0; j < anc_size; j++) {
						if (ancestor_IDs[j] == parent_2) {
							break;
						}
					}
					if (j == anc_size) {
						ancestor_IDs[anc_size] = parent_2;
						anc_size++;
						ancestor_IDs[anc_size] = -1;
					}
				}
				anc_i++;
			} while (ancestor_IDs[anc_i] != -1);

			// all ancestors of set[i] are not in subset
			for (anc_i = 0; anc_i < anc_size; anc_i++) {
				j = 0;
				for (j = 0; j < size_sub; j++) {
					if (ancestor_IDs[anc_i] == subset[j]) {
						flg_legal = 0;

						//						int k = 0;
						//						for (k = 0; k < anc_size; k++) {
						//							printf("%ld ", ancestor_IDs[k]);
						//						}
						//						printf("\n");
						//						printf("=== NOT legal ===\n");


						free(ancestor_IDs);
						return flg_legal;
					}
				}
			}
		}
	}

	//	printf("=== legal ===\n");

	free(ancestor_IDs);
	return flg_legal;
}

void addOne_subset(int *flgs, int size, int sub_size, long *cnt_sub,
		long ***subsets, long * vip_set) {

	long * one_sub = (long *) malloc(sizeof(long) * (sub_size + 1));

	int i;
	int j = 0;
	for (i = 0; i < size; ++i)
		if (*(flgs + i) == 1) {
			one_sub[j++] = vip_set[i];
		}
	one_sub[j] = -1;

	*subsets = (long **) realloc(*subsets, sizeof(long *) * (*cnt_sub + 2));
	(*subsets)[*cnt_sub] = one_sub;
	(*cnt_sub)++;

}

long ** get_subsets(int size, int sub_size, long * vip_set) {

	long ** subsets = (long **) malloc(sizeof(long*));
	long cnt_sub = 0;

	int *flgs, i, j, count;
	flgs = (int*) malloc(sizeof(int) * size);
	for (i = 0; i < sub_size; ++i)
		*(flgs + i) = 1;
	if (size == sub_size) {
		addOne_subset(flgs, size, sub_size, &cnt_sub, &subsets, vip_set);
		//                one_sub = (long *) malloc(sizeof(long) * (sub_size + 1));
		//                generate_subset(flgs, size, one_sub);
		//                subsets = (long **) realloc(subsets, sizeof(long *) * (cnt_sub + 2));
		//                subsets[cnt_sub] = one_sub;
		//                cnt_sub++;
		subsets[cnt_sub] = NULL;
		return subsets;
	}
	for (i = sub_size; i < size; ++i)
		*(flgs + i) = 0;
	addOne_subset(flgs, size, sub_size, &cnt_sub, &subsets, vip_set);
	while (1) {
		count = 0;

		for (i = 0; i < size - 1; ++i) {
			if (*(flgs + i) == 1 && *(flgs + i + 1) == 0) {
				*(flgs + i) = 0;
				*(flgs + i + 1) = 1;
				break;
			}
			if (*(flgs + i) == 1)
				++count;
		}

		for (j = 0; j < i; ++j) {
			if (j < count)
				*(flgs + j) = 1;
			else
				*(flgs + j) = 0;
		}
		addOne_subset(flgs, size, sub_size, &cnt_sub, &subsets, vip_set);
		int ii, break_flg = 1;
		for (ii = size - 1; ii > size - sub_size - 1; --ii) {
			if (*(flgs + ii) == 0) {
				break_flg = 0;
				break;
			}
		}
		if (break_flg == 1)
			break;
	}
	free(flgs);
	subsets[cnt_sub] = NULL;
	return subsets;
}

int is_pattern_legal(PATTERN *x) {
	int i = 0;
	for (; i < x->graph_size; i++) {
		if (x->lines[i].line_ID != (i + 1))
			return 0;
	}
	return 1;
}
double obj_Fv(PATTERN x, LABEL y, STRUCTMODEL *sm, STRUCT_LEARN_PARM *sparm) {
	double fv;
	SVECTOR *ppy = psi(x, y, sm, sparm);
	SVECTOR *py = add_list_ss(ppy);
	fv = sprod_ns(sm->w, py);
	free_svector(ppy);
	free_svector(py);
	return fv;
}

double partial_obj_Fv(CLINE line_i, int y_i, STRUCTMODEL *sm,
		STRUCT_LEARN_PARM *sparm) {
	double fv;
	SVECTOR *pp = psi_line(line_i, y_i, sparm);
	SVECTOR *p = add_list_ss(pp);
	fv = sprod_ns(sm->w, p);
	free_svector(pp);
	free_svector(p);
	return fv;
}

char * get_file_name(char * abs_file_name) {
	char * ret = NULL;
	long len = strlen(abs_file_name);
	for (; len > 0; len--) {
		if (abs_file_name[len] == '/') {
			ret = &abs_file_name[len + 1];
			break;
		}
	}
	return ret;
}
