/*
 * util.c
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#include <ctype.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>

#include "verify.h"

#include "util.h"

#define STATIC_SPRINTF_SIZE 4096

char * xstrdup(const char * restrict str) {
  char * dup;
  verify(dup = strdup(str));
  return dup;
}

void * xmalloc(size_t size) { 
  void * ptr = malloc(size);
  verifyf(ptr, "malloc of size %zu failed.\n", size);
  return ptr;
}

void * xcalloc(size_t n, size_t size) {
  void * ptr = calloc(n, size);
  verifyf(ptr, "calloc of %zu by %zu failed.\n", n, size);
  return ptr;
}

void * xrealloc(void * restrict ptr, size_t size) {
  assert(ptr);
  verifyf(ptr = realloc(ptr, size), "realloc of size %zu failed.\n", size);
  return ptr;
}

int remove_comment(FILE * restrict stream) {
  int ch;

  assert(stream);
  assert(!feof(stream));

  for(;;) {
    do {
      ch = fgetc(stream);
    } while(!(ch == EOF || !isspace(ch)));
    if (ch == '#') {
      do {
	ch = fgetc(stream);
      } while(!(ch == '\n' || ch == EOF));
      if (ch == EOF) {
	return 0;
      }
    } else if (ch == EOF) {
      return 0;
    } else {
      ungetc(ch, stream);
      return 1;
    }
  }
}

FILE * open_stream(const char * restrict path, const char * restrict mode) {
  FILE * stream;
  
  assert(path);
  assert(mode);

  if (!strcmp(path, "-")) {
    if (*mode == 'r') {
      stream = stdin;
    } else if (*mode == 'w') {
      stream = stdout;
    } else {
      verify(0);
    }
  } else {
    verify(stream = fopen(path, mode));
  }

  return stream;
}

void close_stream(const char * restrict path, FILE * restrict stream) {
  if (strcmp(path, "-")) {
    fclose(stream);
  }
}

char * static_sprintf(const char * restrict format, ...) {
  static char buffer[STATIC_SPRINTF_SIZE];
  va_list args;

  va_start(args, format);
  vsnprintf(buffer, STATIC_SPRINTF_SIZE, format, args);
  va_end(args);

  return buffer;
}
