/*
 * timer.h
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#ifndef _TIMER_H_

#include <sys/time.h>

#include "attributes.h"

#include "verify.h"

typedef struct {
  struct timeval start;
  double elapsed;
  int running;
} timer_t;

const char * format_elapsed(double elapsed, char * restrict buffer);

static inline void init_timer(timer_t * restrict timer) NONNULL(1);
static inline void start_timer(timer_t * restrict timer) NONNULL(1);
static inline double elapsed(timer_t * restrict timer) NONNULL(1);
static inline void stop_timer(timer_t * restrict timer) NONNULL(1);
static inline void update_timer(timer_t * restrict timer) NONNULL(1);

static inline void init_timer(timer_t * restrict timer) {
  assert(timer);
  timer->elapsed = 0;
  timer->running = 0;
}

static inline void start_timer(timer_t * restrict timer) {
  assert(timer);
  assert(!timer->running);
  gettimeofday(&timer->start, NULL);
  timer->running = 1;
}

static inline double elapsed(timer_t * restrict timer) {
  update_timer(timer);
  return timer->elapsed;
}

static inline void stop_timer(timer_t * restrict timer) {
  update_timer(timer);
  timer->running = 0;
}

static inline void update_timer(timer_t * restrict timer) {
  struct timeval now;
  double delta;

  assert(timer);

  if (timer->running) {
    gettimeofday(&now, NULL);
    delta = now.tv_sec - timer->start.tv_sec + 1e-6*(now.tv_usec - timer->start.tv_usec);
    timer->elapsed += delta;
    timer->start    = now;
  }
}

#endif /* _TIMER_H_ */
